C      ALGORITHM 693, COLLECTED ALGORITHMS FROM ACM.
C      THIS WORK PUBLISHED IN TRANSACTIONS ON MATHEMATICAL SOFTWARE,
C      VOL. 17, NO. 2, PP. 273-283.  JUNE, 1991.
C
C     FM 1.0                  David M Smith                  1-06-90
C
C
C  The FM package performs floating point multiple precision arithmetic.
C
C  Before calling any routine in the package, several variables in the
C  common blocks /FMUSER/, /FM/, and /FMSAVE/ must be initialized.
C  These three common blocks contain information which must be saved
C  between calls, so they should be declared in the main program.
C
C  Subroutine FMSET initializes these variables to default values and
C  defines all machine-dependent values in the package.  After calling
C  FMSET once at the start of a program, the user may sometimes want
C  to reset some of the variables in common block /FMUSER/.
C
C
C  JBASE is the base in which the arithmetic is done.
C        JBASE must be bigger than one, and less than or equal
C        to the square root of the largest representable integer.
C        For best efficiency JBASE should be about 1/4 to 1/2 as big as
C        the square root of the largest representable integer.
C        Input and output conversion are much faster when JBASE is a
C        power of ten.
C
C  NDIG  is the number of base JBASE digits which are carried in the
C        multiple precision numbers.  NDIG must be at least two.
C        The upper limit for NDIG is defined in the PARAMETER statement
C        at the top of each routine and is restricted only by the amount
C        of memory available.
C
C
C  There are two representations for a floating multiple precision
C  number.  The unpacked representation used by the routines while
C  doing the computations is base JBASE and is stored in NDIG+1 words.
C  A packed representation is available to store the numbers in the
C  user's program in compressed form.  In this format, the NDIG
C  (base JBASE) digits of the mantissa are packed two per word to
C  conserve storage.  Thus the external, packed form of a number
C  requires (NDIG+1)/2+1 words.
C
C  The unpacked format of a floating multiple precision number is as
C  follows.  The number is kept in an integer array with the first
C  element containing the exponent and each of the succeeding NDIG
C  locations containing one digit of the mantissa, expressed in base
C  JBASE.  The exponent is a power of JBASE and the implied radix point
C  is immediately before the first digit of the mantissa.  Every nonzero
C  number is normalized so that the second array element (the first
C  digit of the mantissa) is nonzero.
C
C  In both representations the sign of the number is carried on the
C  second array element only.  Elements 3,4,... are always nonnegative.
C  The exponent is a signed integer and may be as large in magnitude as
C  MXEXP (defined in FMSET).
C
C  For JBASE = 10,000 and NDIG = 4 the number -pi would have these
C  representations:
C                   Word 1         2         3         4         5
C
C         Unpacked:      1        -3      1415      9265      3590
C         Packed:        1    -31415  92653590
C
C  Because of normalization, the equivalent number of base 10
C  significant digits for an FM number may be as small as
C  LOG10(JBASE)*(NDIG-1) + 1.
C
C
C  Subroutine FMOUT performs conversion of an FM number to base 10 and
C  formats it for output as a character array.
C  The user sets JFORM1 and JFORM2 to determine the output format.
C
C  JFORM1 = 0     E   format       ( .314159M+6 )
C         = 1     1PE format       ( 3.14159M+5 )
C         = 2     F   format       ( 314159.000 )
C
C  JFORM2 = Number of significant digits to display (if JFORM1 = 0, 1).
C           If JFORM2.EQ.0 then a default number of digits is chosen.
C           The default is roughly the full precision of the number.
C  JFORM2 = Number of digits after the decimal point (if JFORM1 = 2).
C           See the FMOUT documentation for more details.
C
C
C  KW is the unit number to be used for all output from the package,
C     including error and warning messages, and trace output.
C
C
C  NTRACE and LVLTRC control trace printout from the package.
C
C  NTRACE =  0   No printout except warnings and errors.
C         =  1   The result of each call to one of the routines
C                   is printed in base 10, using FMOUT.
C         = -1   The result of each call to one of the routines
C                   is printed in internal base JBASE format.
C         =  2   The input arguments and result of each call to one
C                   of the routines is printed in base 10, using FMOUT.
C         = -2   The input arguments and result of each call to one
C                   of the routines is printed in base JBASE format.
C
C
C  LVLTRC defines the call level to which the trace is done.  LVLTRC = 1
C         means only FM routines called directly by the user are traced,
C         LVLTRC = 2 also prints traces for FM routines called by other
C         FM routines called directly by the user, etc.
C
C  In the above description internal JBASE format means the number is
C  printed as it appears in the array as an exponent followed by NDIG
C  base JBASE digits.
C
C
C  KFLAG is a condition parameter returned by the package.  Negative
C        values indicate conditions for which a warning message will
C        be printed unless KWARN = 0.  Positive values indicate
C        conditions which may be of interest but are not errors.
C        No warning message is printed if KFLAG is nonnegative.
C
C    KFLAG =  0     Normal operation.
C
C          =  1     One of the operands in FMADD or FMSUB was
C                       insignificant with respect to the other, so
C                       that the result was equal to the argument of
C                       larger magnitude.
C          =  2     In converting an FM number to a one word integer
C                       in FMM2I the FM number was not exactly an
C                       integer.  The next integer toward zero was
C                       returned.
C
C          = -1     NDIG was less than 2 or more than MXNDIG.
C          = -2     JBASE was less than 2 or more than MXBASE.
C          = -3     An exponent was out of range.
C          = -4     Invalid input argument(s) to an FM routine.
C                        UNKNOWN was returned.
C          = -5     + or - OVERFLOW was generated as a result from an
C                        FM routine.
C          = -6     + or - UNDERFLOW was generated as a result from an
C                        FM routine.
C          = -7     The input string to FMINP was not legal.
C          = -8     The output array for FMOUT was not large enough.
C          = -9     Precision could not be raised enough to provide all
C                        requested guard digits.  UNKNOWN was returned.
C          = -10    An FM input argument was too small in magnitude to
C                        convert in FMM2SP or FMM2DP.  Zero was
C                        returned.
C
C  When a negative KFLAG condition is encountered the routine calls
C  FMWARN, which uses the value of KWARN as follows.
C
C  KWARN = 0     Execution continues and no message is printed.
C        = 1     A warning message is printed and execution continues.
C        = 2     A warning message is printed and execution stops.
C
C  When an overflow or underflow is generated for an operation in which
C  an input argument was already an overflow or underflow, no additional
C  message is printed.  When an unknown result is generated and an input
C  argument was already unknown, no additional message is printed.  In
C  these cases the negative KFLAG value is still returned.
C
C
C  KRAD = 0     Causes all angles in the trigonometric functions and
C                  inverse functions to be measured in degrees.
C       = 1     Causes all angles to be measured in radians.
C
C
C  KROUND = 0   Causes all final results to be chopped (rounded toward
C                  zero).  Intermediate results are rounded.
C         = 1   Causes all results to be rounded to the nearest FM
C                  number, or to the value with an even last digit if
C                  the result is halfway between two FM numbers.
C
C
C  Here is a list of the routines in FM which are designed to be called
C  by the user.  All are subroutines except logical function FMCOMP.
C  MA, MB, MC refer to FM format numbers.
C
C  FMABS(MA,MB)         MB = ABS(MA)
C  FMACOS(MA,MB)        MB = ACOS(MA)
C  FMADD(MA,MB,MC)      MC = MA + MB
C  FMASIN(MA,MB)        MB = ASIN(MA)
C  FMATAN(MA,MB)        MB = ATAN(MA)
C  FMATN2(MA,MB,MC)     MC = ATAN2(MA,MB)
C  FMBIG(MA)            MA = Biggest FM number less than overflow.
C  FMCOMP(MA,LREL,MB)   Logical comparison of MA and MB.  LREL is a
C                            CHARACTER*2 value identifying which
C                            comparison is made.
C                            Example:  IF (FMCOMP(MA,'GE',MB)) ...
C  FMCOS(MA,MB)         MB = COS(MA)
C  FMCOSH(MA,MB)        MB = COSH(MA)
C  FMDIG(NSTACK,KST)    Find a set of precisions to use during Newton
C                            iteration for finding a simple root
C                            starting with about double precision
C                            accuracy.
C  FMDIM(MA,MB,MC)      MC = DIM(MA,MB)
C  FMDIV(MA,MB,MC)      MC = MA/MB
C  FMDIVI(MA,INT,MB)    MB = MA/INT for one word integer INT.
C  FMDP2M(X,MA)         MA = X conversion from double precision to FM.
C  FMEQU(MA,MB,NDA,NDB) MB = MA where MA has NDA digits and MB has
C                            NDB digits.
C  FMEXP(MA,MB)         MB = EXP(MA)
C  FMI2M(INT,MA)        MA = INT conversion from one word integer to FM.
C  FMINP(LINE,MA,LA,LB) MA = LINE input conversion of LINE(LA) through
C                            LINE(LB) from characters to FM.
C  FMINT(MA,MB)         MB = INT(MA) integer part of MA.
C  FMIPWR(MA,INT,MB)    MB = MA**INT raise FM number to a one word
C                            integer power.
C  FMLG10(MA,MB)        MB = LOG10(MA)
C  FMLN(MA,MB)          MB = LOG(MA)
C  FMLNI(INT,MA)        MA = LOG(INT) natural log of one word integer.
C  FMM2DP(MA,X)         X  = MA conversion from FM to double precision.
C  FMM2I(MA,INT)        INT = MA conversion from FM to one word integer.
C  FMM2SP(MA,X)         X  = MA conversion from FM to single precision.
C  FMMAX(MA,MB,MC)      MC = MAX(MA,MB)
C  FMMIN(MA,MB,MC)      MC = MIN(MA,MB)
C  FMMOD(MA,MB,MC)      MC = MA mod MB
C  FMMPY(MA,MB,MC)      MC = MA*MB
C  FMMPYI(MA,INT,MB)    MB = MA*INT multiplication by one word integer.
C  FMNINT(MA,MB)        MB = NINT(MA) nearest integer.
C  FMOUT(MA,LINE,LB)    LINE = MA conversion from FM to character.  LB
C                              is the size of array LINE.
C  FMPI(MA)             MA = pi
C  FMPRNT(MA)           Print MA using current format.
C  FMPWR(MA,MB,MC)      MC = MA**MB
C  FMSET(NPREC)         Set default values and machine-dependent
C                            variables to give at least NPREC base 10
C                            digits plus three base 10 guard digits.
C  FMSIGN(MA,MB,MC)     MC = SIGN(MA,MB) sign transfer.
C  FMSIN(MA,MB)         MB = SIN(MA)
C  FMSINH(MA,MB)        MB = SINH(MA)
C  FMSP2M(X,MA)         MA = X conversion from single precision to FM.
C  FMSQRT(MA,MB)        MB = SQRT(MA)
C  FMSUB(MA,MB,MC)      MC = MA - MB
C  FMTAN(MA,MB)         MB = TAN(MA)
C  FMTANH(MA,MB)        MB = TANH(MA)
C  FMULP(MA,MB)         MB = 1 Unit in the Last Place of MA.
C
C  For each of these routines there is also a version available for
C  which the argument list is the same but all FM numbers are in packed
C  format.  The packed versions have the same names except 'FM' is
C  replaced by 'FP' at the start of each name.
C
C
C  NOTES ON ARRAY DIMENSIONS.
C
C  The dimensions of the arrays in the FM package are defined using
C  a PARAMETER statement at the top of each routine.  The size of
C  these arrays depends on the values of parameters MXNDIG and NBITS.
C  MXNDIG is the maximum value the user may set for NDIG.
C  NBITS is the number of bits used to represent integers.
C
C  FM numbers in packed format have size LPACK, and those in unpacked
C  format have size LUNPCK.
C
C
C  PORTABILITY NOTES.
C
C  In routines FMEQU and FMSUB there is code which attempts to
C  determine if two input arrays refer to identical memory locations.
C  Some optimizing compilers assume the arrays must be distinct and
C  may remove code which would then be redundant.  This code removal
C  could cause errors, so the tests are done in a way which should
C  keep the compiler from removing code. The current version works
C  correctly on all compilers tested.  Computing SIN(1.0) in radian
C  mode should reveal whether other compilers handle it correctly.
C  If there is a problem, SIN(1) gives 0.999... instead of 0.841....
C  To fix such a problem, MB can be copied to a local temporary array
C  and then negated in FMSUB before calling FMADD2.  For FMEQU
C  simply set KSAME = 0 after the section which tries to determine if
C  MA and MB are the same array. This makes both routines run slower.
C  A simpler fix which often works is to re-compile at a lower
C  optimization (e.g., OPT=0).
C
C  In FMSET there is machine-dependent code which attempts to
C  approximate the largest one word integer value.  The current code
C  works on all machines tested, but if an FM run fails, check the
C  MAXINT loop in FMSET in addition to the three routines mentioned
C  above.  Values for SPMAX and DPMAX are also defined which should
C  be set to values near overflow for single precision and double
C  precision.
C
C  Using the CFT77 compiler on a Cray X-MP computer there are
C  problems using a large value for JBASE when the default 46-bit
C  integer arithmetic mode is used.  In particular, FMSET chooses
C  too large a JBASE value since some of the arithmetic in the
C  MAXINT loop is done with 64-bit precision.  Setting JBASE = 10**6
C  or less may be ok, but the preferred solution is to select the
C  64-bit integer compiler option.  Then JBASE = 10**9 can be used.
C
C --------------------------------------------------------------------
C
C
      SUBROUTINE FMSET(NPREC)
C
C  Initialize the values in common which must be set before calling
C  other FM routines.
C
C  Base and precision will be set to give at least NPREC+3 decimal
C  digits of precision (giving the user three base ten guard digits).
C
C  JBASE is set to the largest permissible power of ten.
C  JFORM1 and JFORM2 are set to 1PE format displaying NPREC significant
C  digits.
C
C  The trace option is set off, and the mode for angles in trig
C  functions is set to radians.  The rounding mode is set to symmetric
C  rounding.
C
C  KW, the unit number for all FM output, is set to 6.
C
C  Start of gie revision for identical params file
CC  The size of all arrays is controlled by defining two parameters:
CC  MXNDIG is the maximum value the user can set NDIG,
CC  NBITS  is the number of bits per integer word.
CC
C      PARAMETER ( MXNDIG=256 , NBITS=32 ,
CC
CC             Define the array sizes:
CC
C     *          LPACK  = (MXNDIG+1)/2 + 1 , LUNPCK = (6*MXNDIG)/5 + 20,
C     *          LMWA   = 2*LUNPCK         , LJSUMS = 8*LUNPCK,
C     *          LMBUFF = ((LUNPCK+3)*(NBITS-1)*301)/2000 + 6  )
C end of revision after next line
	include 'params.inc'
C
C  Here are all the common blocks used in FM.
C
C  /FMUSER/, /FM/, and /FMSAVE/ should also be declared in the main
C  program, because some compilers allocate and free space used for
C  labelled common which is declared only in subprograms.  This causes
C  the saved information to be lost.
C
C             FMUSER contains values which may need to be changed by
C             the calling program.
C
      COMMON /FMUSER/ NDIG,JBASE,JFORM1,JFORM2,KRAD,
     *                KW,NTRACE,LVLTRC,KFLAG,KWARN,KROUND
C
C             FM contains the work array used by the low-level
C             arithmetic routines, definitions for overflow and
C             underflow thresholds, etc.
C
      DOUBLE PRECISION DPMAX
C
      COMMON /FM/ MWA(LMWA),NCALL,MXEXP,MXEXP2,KARGSW,KEXPUN,KEXPOV,
     *            KUNKNO,IUNKNO,RUNKNO,MXBASE,MXNDG2,KSTACK(19),
     *            MAXINT,SPMAX,DPMAX
C
C             FMSAVE contains information about saved constants.
C
      COMMON /FMSAVE/ NDIGPI,NJBPI,NDIGE,NJBE,NDIGLB,NJBLB,NDIGLI,NJBLI,
     *                MPISAV(LUNPCK),MESAV(LUNPCK),MLBSAV(LUNPCK),
     *                MLN1(LUNPCK),MLN2(LUNPCK),MLN3(LUNPCK),
     *                MLN4(LUNPCK)
C
C             MJSUMS is an array which can contain several FM numbers
C             being used to accumulate the concurrent sums in FMEXP2
C             and FMSIN2.  When MXNDIG is 256 eight is about the maximum
C             number of sums needed (but this depends on JBASE).  For
C             larger MXNDIG dimensioning MJSUMS to hold more than eight
C             FM numbers could speed the functions up.
C
      COMMON /FMSUMS/ MJSUMS(LJSUMS)
C
C             MBUFF is a character array used by FMPRNT for printing
C             output from FMOUT.  This array may also be used for calls
C             to FMOUT from outside the FM package.
C
      CHARACTER MBUFF
C
      COMMON /FMBUFF/ MBUFF(LMBUFF)
C
C             FM1 contains scratch arrays for temporary storage of FM
C             numbers while computing various functions.
C
      COMMON /FM1/ M01(LUNPCK),M02(LUNPCK),M03(LUNPCK),M04(LUNPCK),
     *             M05(LUNPCK),M06(LUNPCK)
C
C             FMPCK contains scratch arrays used to hold input arguments
C             in unpacked format when the packed versions of functions
C             are used.
C
      COMMON /FMPCK/ MPA(LUNPCK),MPB(LUNPCK)
C
      DOUBLE PRECISION TEMP
C
C             KW is the unit number for all output from the FM package.
C                This includes trace output and error messages.
C
      KW = 6
C
C             MAXINT should be set to a very large integer, possibly
C                    the largest representable integer for the current
C                    machine.  For most 32-bit machines MAXINT is set to
C                    2**31 - 1 = 2 147 483 647.
C
C                    Setting MAXINT to a smaller number is ok, but this
C                    unnecessarily restricts the permissible range of
C                    JBASE and MXEXP.  Too small a value of MAXINT will
C                    also slow the elementary functions like SIN, EXP,
C                    etc., since MXBASE = SQRT(MAXINT) is used to
C                    determine how many terms can be combined when
C                    summing series.
C
C                    The following code should set MAXINT to the largest
C                    representable number of the form 2**J - 1.
C
C          WARNING:  This loop causes integer overflow to occur, so it
C                    is a likely place for the program to fail when
C                    run on a different machine.  The loop below has
C                    been used successfully on a Fortran-77 compiler
C                    on each of these machines:
C                    IBM 3090, CDC 176, CRAY XMP, MAGNUSON, MIPS 800,
C                    SUN 4/260, IBM PC, MACINTOSH, MACINTOSH II,
C                    COMPAQ 386/20, TRS-80/16.
C                    However, even different versions of the same
C                    compiler may react differently, so check the value
C                    of MAXINT if there are problems installing FM on
C                    a new machine.
C
      MAXINT = 3
  110 L = 2*MAXINT + 1
      IF (INT(L/2).EQ.MAXINT) THEN
          MAXINT = L
          GO TO 110
      ENDIF
C
C             DPMAX should be set to a value near the machine's double
C                   precision overflow threshold, so that DPMAX and
C                   1.0D0/DPMAX are both representable in double
C                   precision.
C
      DPMAX = 1.0D+74
C
C             SPMAX should be set to a value near the machine's single
C                   precision overflow threshold, so that 1.01*SPMAX
C                   and 1.0/SPMAX are both representable in single
C                   precision.
C
      SPMAX = 1.0E+37
C
C             MXNDG2 is the maximum value for NDIG which can be used
C                    internally.  Several of the FM routines may raise
C                    NDIG above MXNDIG temporarily, in order to
C                    compute correctly rounded results.
C                    In the definition of LUNPCK The '6/5' condition
C                    allows for converting from a large base to the
C                    (smaller) largest power of ten base for output
C                    conversion.
C                    The '+ 20' condition allows for the need to carry
C                    many guard digits when using a small base like 2.
C
      MXNDG2 = LUNPCK - 1
C
C             MXBASE is the maximum value for JBASE.
C
      TEMP = MAXINT
      MXBASE = SQRT(TEMP)
C
C             JBASE is the currently used base for arithmetic.
C
      K = LOG10(REAL(MXBASE))
      JBASE = 10**K
C
C             NDIG is the number of digits currently being carried.
C
      NPSAVE = NPREC
      NDIG = 2 + (NPREC+2)/K
      IF (NDIG.LT.2 .OR. NDIG.GT.MXNDIG) THEN
          NDIG = MAX(2,MIN(MXNDIG,NDIG))
          WRITE (KW,120) NPREC,NDIG
  120     FORMAT(//' PRECISION OUT OF RANGE WHEN CALLING FMSET.',
     *           '  NPREC =',I20/' THE NEAREST VALID NDIG WILL BE USED',
     *           ' INSTEAD:   NDIG =',I6//)
          NPSAVE = 0
      ENDIF
C
C             KFLAG is the flag for error conditions.
C
      KFLAG = 0
C
C             NTRACE is the trace switch.  Default is no printing.
C
      NTRACE = 0
C
C             LVLTRC is the trace level.  Default is to trace only
C                    routines called directly by the user.
C
      LVLTRC = 1
C
C             NCALL is the call stack pointer.
C
      NCALL = 0
C
C             Some constants which are often needed are stored in the
C             maximum precision which they have been computed with the
C             currently used base.  This speeds up the trig, log, power,
C             and exponential functions.
C
C             NDIGPI is the number of digits available in the currently
C                    stored value of pi (MPISAV).
C
      NDIGPI = 0
C
C             NJBPI is the value of JBASE for the currently stored
C                   value of pi.
C
      NJBPI = 0
C
C             NDIGE is the number of digits available in the currently
C                   stored value of e (MESAV).
C
      NDIGE = 0
C
C             NJBE is the value of JBASE for the currently stored
C                  value of e.
C
      NJBE = 0
C
C             NDIGLB is the number of digits available in the currently
C                    stored value of LN(JBASE) (MLBSAV).
C
      NDIGLB = 0
C
C             NJBLB is the value of JBASE for the currently stored
C                   value of LN(JBASE).
C
      NJBLB = 0
C
C             NDIGLI is the number of digits available in the currently
C                    stored values of the four logarithms used by FMLNI
C                    MLN1 - MLN4.
C
      NDIGLI = 0
C
C             NJBLI is the value of JBASE for the currently stored
C                   values of MLN1 - MLN4.
C
      NJBLI = 0
C
C             MXEXP  is the current maximum exponent.
C             MXEXP2 is the internal maximum exponent. This is used to
C                    define the overflow and underflow thresholds.
C
C             These values are chosen so that FM routines can raise the
C             overflow/underflow limit temporarily while computing
C             intermediate results, and so that EXP(MAXINT) is greater
C             than MXBASE**(MXEXP2+1).
C
C             The overflow threshold is JBASE**(MXEXP+1), and the
C             underflow threshold is JBASE**(-MXEXP-1).
C             This means the valid exponents in the first word of an FM
C             number can range from -MXEXP to MXEXP+1 (inclusive).
C
      TEMP = MXBASE
      MXEXP = (TEMP*TEMP)/(2.0*LOG(TEMP)) - 1.0
      MXEXP2 = 2*MXEXP + MXEXP/100
C
C             KARGSW is a switch used by some of the elementary function
C                    routines to disable argument checking while doing
C                    calculations where no exceptions can occur.
C                    See FMARGS for a description of argument checking.
C                    KARGSW = 0 is the normal setting,
C                           = 1 means argument checking is disabled.
C
      KARGSW = 0
C
C             KEXPUN is the exponent used as a special symbol for
C                    underflowed results.
C
      KEXPUN = -MXEXP2 - 5*MXNDIG
C
C             KEXPOV is the exponent used as a special symbol for
C                    overflowed results.
C
      KEXPOV = -KEXPUN
C
C             KUNKNO is the exponent used as a special symbol for
C                    unknown FM results (1/0, SQRT(-3.0), etc).
C
      KUNKNO = KEXPOV + 5*MXNDIG
C
C             RUNKNO is returned from FM to real or double conversion
C                    routines when no valid result can be expressed in
C                    real or double precision.  On systems which provide
C                    a value for undefined results (e.g., Not A Number)
C                    setting RUNKNO to that value is reasonable.  On
C                    other systems set it to a value which is likely to
C                    make any subsequent results obviously wrong which
C                    use it.  In either case a KFLAG = -4 condition is
C                    also returned.
C
      RUNKNO = -1.01*SPMAX
C
C             IUNKNO is returned from FM to integer conversion routines
C                    when no valid result can be expressed as a one word
C                    integer.  KFLAG = -4 is also set.
C
      IUNKNO = -MXBASE*MXBASE
C
C             JFORM1 indicates the format used by FMOUT.
C
      JFORM1 = 1
C
C             JFORM2 indicates the number of digits used in FMOUT.
C
      JFORM2 = NPSAVE
C
C             KRAD = 1 indicates that trig functions use radians,
C                  = 0 means use degrees.
C
      KRAD = 1
C
C             KWARN = 0 indicates that no warning message is printed
C                       and execution continues when UNKNOWN or another
C                       exception is produced.
C                   = 1 means print a warning message and continue.
C                   = 2 means print a warning message and stop.
C
      KWARN = 1
C
C             KROUND = 1   Causes all results to be rounded to the
C                          nearest FM number, or to the value with
C                          an even last digit if the result is halfway
C                          between two FM numbers.
C                    = 0   Causes all results to be chopped.
C
      KROUND = 1
C
      RETURN
      END
