#ifndef CGEMATRIX_H
#define CGEMATRIX_H
#pragma once 

/*
 *	Declarations for DComplex general matricies.
 *
 *	Copyright (C) 1988, 1989.
 *
 *	Dr. Thomas Keffer
 *	Rogue Wave Associates
 *	P.O. Box 85341
 *	Seattle WA 98145-1341
 *
 *	Permission to use, copy, modify, and distribute this
 *	software and its documentation for any purpose and
 *	without fee is hereby granted, provided that the
 *	above copyright notice appear in all copies and that
 *	both that copyright notice and this permission notice
 *	appear in supporting documentation.
 *	
 *	This software is provided "as is" without any
 *	expressed or implied warranty.
 *
 *
 *	@(#)CGEMatrix.h	2.2	9/26/89
 */

/*
 *	This class is derived from class DComplexVec.  Data is stored
 *	FORTRAN style: by columns.
 *
 *	Defining the preprocessor directive "BOUNDS_CHECK" will invoke
 *	bounds checking.
 */

#include "DComplexVec.h"

class CGEMatrix : public DComplexVec {
  int ncols;			// Number of columns
  int nrows;			// Number of rows
protected:
  void 			assertColRange(int);
  void 			assertRowRange(int);
  void			assertRowCol(const CGEMatrix&);
  void			assertLength(const DComplexVec&);
  void			assertSquare();
  void			assertProduct(const CGEMatrix&);
  void			assertProduct(const DComplexVec&);
public:
  CGEMatrix();
  CGEMatrix(int rows, int cols);
  CGEMatrix(int rows, int cols, DComplex initval);
  CGEMatrix(const DComplex* dat, int, int);  // Copy of dat will be made
  CGEMatrix(const DComplexVec& v, int, int); // Reference to v will be made
  CGEMatrix(const CGEMatrix& m);	   // Reference to m will be made

  DComplex*		data()		{return DComplexVec::data();}
  int			cols();
  int			rows();

  CGEMatrix&		reference(CGEMatrix& m); // Reference self to m
  CGEMatrix		deepCopy();	// copy of self with distinct instance variables 
  CGEMatrix		copy()		{return deepCopy();} // Synonym for deepCopy()
  void			deepenShallowCopy();	// Guarantee that references==1:
  void			resize(unsigned,unsigned);

  void			scanFrom(istream& s);	// Format 2 x 2 [ 3 2 1 4 ]
  void			printOn(ostream& s);
  void			readFrom(istream&);
  void			storeOn(ostream&);
  void			readFrom(fileDescTy&);
  void			storeOn(fileDescTy&);

  DComplexVec		operator[](int j);	// Return a col as a slice
  DComplexVec		col(int j);		// Return a col as a slice
  DComplexVec		row(int i);		// Return a row as a slice
  DComplexVec		diagonal(int idiag=0);	// Return a diagonal as a slice
  DComplex&		operator()(int i, int j); // Subscripting

// Math functions
  CGEMatrix		product(const CGEMatrix&); // Inner product
  DComplexVec		product(const DComplexVec&);

// Assignment operators --- self must be same size as m
  CGEMatrix&		operator=(const CGEMatrix& m);
  CGEMatrix&		operator=(DComplex);
  CGEMatrix&		operator+=(const CGEMatrix& m);
  CGEMatrix&		operator+=(DComplex);
  CGEMatrix&		operator-=(const CGEMatrix& m);
  CGEMatrix&		operator-=(DComplex);
  CGEMatrix&		operator*=(const CGEMatrix& m);
  CGEMatrix&		operator*=(DComplex);
  CGEMatrix&		operator/=(const CGEMatrix& m);
  CGEMatrix&		operator/=(DComplex);

// Increment/decrement operators
//CGEMatrix&		operator++();
//CGEMatrix&		operator--();

// Friendly arithmetic operators; Notice that operator* is an element-by-
// element multiply, NOT a matrix multiply.
  friend CGEMatrix	operator-(const CGEMatrix&);	// Unary minus
  friend CGEMatrix	operator+(const CGEMatrix&);	// Unary plus
  friend CGEMatrix	operator*(const CGEMatrix&, const CGEMatrix&);
  friend CGEMatrix	operator/(const CGEMatrix&, const CGEMatrix&);
  friend CGEMatrix	operator+(const CGEMatrix&, const CGEMatrix&);
  friend CGEMatrix	operator-(const CGEMatrix&, const CGEMatrix&);
  friend CGEMatrix	operator*(const CGEMatrix&, DComplex);
  friend CGEMatrix	operator*(DComplex, const CGEMatrix&);
  friend CGEMatrix	operator/(const CGEMatrix&, DComplex);
  friend CGEMatrix	operator/(DComplex, const CGEMatrix&);
  friend CGEMatrix	operator+(const CGEMatrix&, DComplex);
  friend CGEMatrix	operator+(DComplex, const CGEMatrix&);
  friend CGEMatrix	operator-(const CGEMatrix&, DComplex);
  friend CGEMatrix	operator-(DComplex, const CGEMatrix&);

};

// Other (related) declarations:
ostream&		operator<<(ostream& s, const CGEMatrix& m);
CGEMatrix		transpose(const CGEMatrix&);

/******************* I N L I N E S **************************/

Inline int CGEMatrix::cols() { return ncols;}
Inline int CGEMatrix::rows() { return nrows;}
Inline void CGEMatrix::deepenShallowCopy(){DComplexVec::deepenShallowCopy();}
Inline CGEMatrix operator+(const CGEMatrix& m)		{ return m; }
Inline CGEMatrix operator*(DComplex d, const CGEMatrix& m){ return m*d; }
Inline CGEMatrix operator+(DComplex d, const CGEMatrix& m){ return m+d; }

// Return a column
Inline DComplexVec CGEMatrix::operator[](int j){
#if BOUNDS_CHECK
  assertColRange(j);
#endif
  return DComplexVec::slice(j*nrows,nrows,1);
}

Inline DComplexVec CGEMatrix::col(int j){	// Same as above
#if BOUNDS_CHECK
  assertColRange(j);
#endif
  return DComplexVec::slice(j*nrows,nrows,1);
}

Inline DComplexVec CGEMatrix::row(int i){
#if BOUNDS_CHECK
  assertRowRange(i);
#endif
  return DComplexVec::slice(i, ncols, nrows);
}

Inline DComplexVec CGEMatrix::diagonal(int i){
  register int iabs=ABSOLUTE(i);
#if BOUNDS_CHECK
  assertSquare();
  assertRowRange(iabs);
#endif
  return DComplexVec::slice(i>0 ? i*nrows : iabs, nrows-iabs, nrows+1);
}

Inline DComplex& CGEMatrix::operator()(int i, int j){
#if BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return DComplexVec::operator()(j*nrows+i);
}

#endif CGEMATRIX_HXX
