#############################################################################
##
#A  Matrix Group and G-module library                   Derek Holt
#A                                                      Charles Leedham-Green
#A                                                      Eamonn O'Brien
#A                                                      Sarah Rees 
##
#A  @(#)$Id$
##
#Y  Copyright 1994 -- School of Mathematical Sciences, ANU   
##
#H  $Log$
##
############################################################################
##
#F  InfoSemiLinear (...)  . . . . . . . . . . . for debugging assistance
##
##
if not IsBound(InfoSemiLinear)  then InfoSemiLinear := Ignore;  fi;

#############################################################################
##
#F  SemiLinearDecomp( module,S,C,e )  . . applied when S acts irreducibly.
## module is the module of the group G, and <S> is a subgroup of G.
##  C centralises the action of <S> on the underlying vector space and
## acts as multiplication by a scalar x
## in GF(q^e) for some embedding of a d/e-dimensional vector
## space over GF(q^e) in the d-dimensional space, where x is a field generator
## of GF(q^e). Thus, provided C centralises the action of <S>^G, the normal
## closure of <S>, <S>^G embeds in GL(d/e,q^e), and  G 
## embeds in GammaL(d/e,q^e). We test for that by trying to construct a map from
## G to Aut(GF(q^e).    
## We check to see if G can be embedded in GammaL(d/e,q^e), using the function
## powermaps. If powermaps returns false it must be because C doesn't centralise
## the action of <S>^G, so there is some conjugate of an element of S which is 
## not centralised by C. We find such a conjugate and return false.
## If powermaps returns true we return true.
## If true is returned then the SemiLinearFlag is set to true, the 
## FieldExtDegFlag is set to e, the LinearPartFlag is set to S, and the
## FrobeniusAutosFlag is set to the sequence of integers i such that
## multiplication of C by the generator g corresponds to the action of
## the field automorphism x -> x^q^i(g) on the corresponding element of
## GF(q^e). 
## 
SemiLinearDecomp := function ( module,S,C,e) 
   local dim,q,g,powermaps;

   dim := DimFlag(module);
   q := FieldFlag(module).size;
   InfoSemiLinear("Looking for powermaps.\n");
   powermaps := PowerMaps(module,C,e);
   if powermaps <> false then
     SetSemiLinearFlag(module,true);
     SetFieldExtDegFlag(module,e);
     SetLinearPartFlag(module,S);
     SetCentMatFlag(module,C);
     SetFrobeniusAutosFlag(module,powermaps);
     return true;
   else
# Enlarge S by a random conjugate (of an element of S) that doesn't
# commute with C.
     repeat g := RandomConj(MatricesFlag(module),S);
     until g*C <> C*g;
     Add(S,g);
   fi;

end;


#############################################################################
##
#F  PowerMaps( module,C,e )  . . part of test to see if 
##  GL(d/e,q^e)<G<=GammaL(d/e,q^e)
## module is d-dimensional, acted on by G, and C is a dxd matrix, which
## acts as multiplication by a scalar x ( a field generator of GF(q^e))
## for some embedding of a d/e-dimensional vector
## space over GF(q^e) in the d-dimensional space.
## G acts as a semilinear group of automorphisms on the d/e-dimensional
## space if and only if, for each generator g of G, there is an integer i = i(g)
## such that Cg = gC^{q^i}, i.e. g corresponds to the
## field automorphism x -> x^(q^i). 
## Then we have a map from G to the (cyclic) group Aut(GF(q^e), and C centralises
## the the action of the kernel of this map,  which thus lies in GL(d,q^e)
## We test this by first, if possible, finding such i=i(g) 
## such that wCg = wgC^(q^i) for a single vector w of the d-dimensional space
## (in fact the first vector of the standard basis) and then checking that the
## vCg = vgC^(q^i) for all other vector v in the basis.
## This function returns a list, powermaps, consisting of the integers
## found, or false if no such integers can be found..
PowerMaps := function ( module,C,e ) 
   local powermaps,g,matrices,found,v,F,q,dim,
         L,M,N,R,s,i;

   matrices := MatricesFlag(module);
   dim := DimFlag(module);
   F := FieldFlag(module);
   q := F.size;
   
   powermaps := [];
   for g in matrices do
     found := false;
     v := F.zero * C[1]; # this sets v equal to the zero vector of
                         # the appropriate length 
     v[1] := F.one; #  Now v is the vector (1,0,0....0).
     L := v*C*g;
     M := v*g;
     N := C;
     s:=0;
     repeat 
       R := M*N;
       if L=R then found:=true;
       else N:=N^q; s := s+1;
       fi;
     until found=true or s=e; 
     if s=e then
       InfoSemiLinear("No powermap found.\n"); return false;
     fi;
     for i in [2..dim] do
       v := F.zero * C[1]; # this sets v equal to the zero vector of
                         # the appropriate length 
       v[i] := F.one; #  Now v is the vector (0,0..,1,..0).
       M:= v*g;
       L := v*C*g;
       R := M*N; 
       if L<>R then
         InfoSemiLinear("No consistent powermap found.\n"); 
         return false;
       fi;
     od;
     Add(powermaps,s);
   od;
   return powermaps;
end;


