/*##############################################################################

FUNNNELWEB COPYRIGHT
====================
FunnelWeb is a literate-programming macro preprocessor.

Copyright (C) 1992 Ross N. Williams.

   Ross N. Williams
   ross@spam.adelaide.edu.au
   16 Lerwick Avenue, Hazelwood Park 5066, Australia.

This program is free software; you can redistribute it and/or modify
it under the terms of Version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See Version 2 of the GNU General Public License for more details.

You should have received a copy of Version 2 of the GNU General Public
License along with this program. If not, you can FTP the license from
prep.ai.mit.edu/pub/gnu/COPYING-2 or write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

Section 2a of the license requires that all changes to this file be
recorded prominently in this file. Please record all changes here.

Programmers:
   RNW  Ross N. Williams  ross@spam.adelaide.edu.au
   ABC  Anthony B. Coates coates@physics.uq.edu.au

Changes:
   07-May-1992  RNW  Program prepared for release under GNU GPL V2.
   31-Aug-1993  ABC  Added support for EL_SPEC
   11-Jan-1994  ABC  Added MAXINV which defines the maximum
                     number of macro invocation citations
		     which should be put on the one TeX line.
   15-Apr-1994  ABC  Added hypertext typesetter types
                     TR_HTXT and TR_HTML, which are to HTML hypertext
                     what TR_NONE and TR_TEX are to TeX respectively.
   15-Apr-1994  ABC  Added doc_incQ flag for inclusion or not of
                     FunnelWeb TeX/HTML header.  If FALSE, also
                     suppresses the "\bye" at the end of TeX files
                     and the <HTML>...</HTML> markup surrounding
                     HTML files.
   20-May-1994  ABC  Added raw_namQ flag for allowing or not of
                     raw TeX/HTML in macro/section names
                     (TR_TEX or TR_HTML modes only).

##############################################################################*/


/******************************************************************************/
/*                                   WEAVE.C                                  */
/******************************************************************************/

#include <ctype.h>
#include "style.h"

#include "as.h"
#include "data.h"
#include "lister.h"
#include "machin.h"
#include "misc.h"
#include "table.h"
#include "texhead.h"
/* ABC { */
#include "htmlhead.h"
/* } ABC */
#include "weave.h"

/* ABC { */
/* MAXINV is the maximum number of macro invocation references */
/* that should be put on one TeX line */
/* This stops line to big for the TeX buffer from being created. */
#define MAXINV 20
/* } ABC */

/* IMPORTANT WARNING: The Macintosh THINK C compiler sets up strings that     */
/* start with \p or \P as PASCAL strings rather than C strings. So be very    */
/* careful not to start any strings with \p or \P.                            */

/******************************************************************************/

LOCVAR wf_t f_t;          /* Variable representing documentation file.        */
LOCVAR bool literal;      /* True if we are inside a literal @{.              */
/* ABC { */
LOCVAR bool title1Q;      /* True if a title is the first for the document.   */
LOCVAR bool lastEOLQ;     /* last written text scrap finished with EOL?       */
/* } ABC */

/******************************************************************************/

LOCAL void wv_ex P_((p_ells_t));

/******************************************************************************/

LOCAL char * fn_nodir P_((char *));
LOCAL char * fn_nodir (p)
/* Given a pointer to a string, returns a pointer to the character just past  */
/* the last occurrance of FN_DELIM, or a pointer to the start of the string   */
/* if it does not occur. In other words, given a filename, return a pointer   */
/* to the start of the name field.                                            */
char *p;
{
 char *q = strrchr(p,FN_DELIM);
 if (q==NULL)
    return p;
 else
    return q+1;
}

/******************************************************************************/

LOCAL void wf_dec P_((p_wf_t,ulong));
LOCAL void wf_dec(p_wf,n)
/* Writes the unsigned number n to the specified output stream as a decimal.  */
/* This function has been named wf_dec so that it can be moved to the wf      */
/* package if it turns out to be popular.                                     */
p_wf_t p_wf;
ulong  n;
{
 char t[20];
 sprintf(t,"%lu",n);
 wf_wr(p_wf,t);
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_txugch P_((int));
LOCAL void wv_txugch(ch)
/* WeaVe TeX UGly CHaracter.                                                  */
/* } ABC */
/* Sometimes it is necessary to output text without causing the typesetting   */
/* program (TeX) to interpret it as control sequences and generate errors.    */
/* There are two ways of doing this. First, we can get the typesetter to do   */
/* the work by making it less sensitive. This has been done with the          */
/* typesetting of the main macro text. The other approach is to convert each  */
/* offending character into the typesetter control sequence that will simply  */
/* output the character. For example, in TeX, "$" becomes "\$". This function */
/* accepts a single character and writes out its safe equivalent.             */
/* The term "ugly" is used to describe potentially harmful text/characters.   */
int ch;
{
 /* End of lines are the only non-printable allowed. */
 if (ch==EOL) {wf_chr(&f_t,EOL); return;}

 /* All other non-printables result in a bullet. In the absence of bugs, the  */
 /* only way a non-printable can get to us here is as a result of the user    */
 /* explicitly inserting one using the scanner's @^ control sequence.         */
 if (!isascprn(ch))
    {
     wf_wr(&f_t,"$\\bullet$");
     return;
    }
 /* If it IS printable, then print it ... carefully! */
 switch (ch)
   {
    /* Go through the printables in ASCII order. */
    /*   ' ':  is an ordinary character. */
    /*   '!':  is an ordinary character. */
    case '\"':  wf_wr(&f_t,"\\char`\\\"");  break;
    case '#':  wf_wr(&f_t,"\\#");  break;
    case '$':  wf_wr(&f_t,"\\$");  break;
    /* Note: The string "\\%" triggers a unknown sequence "\%" in GNUC.       */
    case '%':  wf_chr(&f_t,'\\'); wf_chr(&f_t,'%'); break;
    case '&':  wf_wr(&f_t,"\\&");  break;
    /*   ''': is an ordinary character. */
    /*   '(': is an ordinary character. */
    /*   ')': is an ordinary character. */
    /*   '*': is an ordinary character. */
    /*   '+': is an ordinary character. */
    /*   ',': is an ordinary character. */
    /*   '-': is an ordinary character. */
    /*   '.': is an ordinary character. */
    /*   '/': is an ordinary character. */
    /* '0-9': are   ordinary characters.*/
    /*   ':': is an ordinary character. */
    /*   ';': is an ordinary character. */
              /* Note: Some \chars work well in roman. Others don't. */
    case '<': if (literal)
                 wf_wr(&f_t,"\\char`\\<");
              else
                 wf_wr(&f_t,"$<$");
              break;
    /*   '=': is an ordinary character. */
    case '>': if (literal)
                 wf_wr(&f_t,"\\char`\\>");
              else
                 wf_wr(&f_t,"$>$");
              break;
    /*   '?': is an ordinary character. */
    /*   '@': is an ordinary character. */
    /* 'A-Z': are   ordinary characters.*/
    /*   '[': is an ordinary character. */
    case '\\': if (literal)
                  wf_wr(&f_t,"\\char`\\\\");
               else
                  wf_wr(&f_t,"$\\backslash$");
               break;
    /*   ']': is an ordinary character. */
    case '^':  wf_wr(&f_t,"\\char`\\^");  break;
    case '_':  wf_wr(&f_t,"\\_");  break;
    /*   '`': is an ordinary character. */
    /* 'a-z': are   ordinary characters.*/
    case '{':  if (literal)
                  wf_wr(&f_t,"\\char`\\{");
               else
                  wf_wr(&f_t,"$\\{$");
               break;
    case '|':  if (literal)
                  wf_wr(&f_t,"\\char`\\|");
               else
                  wf_wr(&f_t,"$|$");
               break;
    case '}':  if (literal)
                  wf_wr(&f_t,"\\char`\\}");
               else
                  wf_wr(&f_t,"$\\}$");
               break;
    case '~':  wf_wr(&f_t,"\\char`\\~");  break;
    default:   wf_chr(&f_t,ch);    break;
   }
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htugch P_((int));
LOCAL void wv_htugch(ch)
/* WeaVe HTml UGly CHaracter.                                            */
/* Sometimes it is necessary to output text without causing the typesetting   */
/* program (HTML displayer) to interpret it as control sequences and generate */
/* errors.                                                                    */
/* There are two ways of doing this. First, we can get the typesetter to do   */
/* the work by making it less sensitive. This has been done with the          */
/* typesetting of the main macro text. The other approach is to convert each  */
/* offending character into the typesetter control sequence that will simply  */
/* output the character. For example, in HTML, "<" becomes "&lt;". This       */
/* function accepts a single character and writes out its safe equivalent.    */
/* The term "ugly" is used to describe potentially harmful text/characters.   */
int ch;
{
 /* End of lines are the only non-printable allowed. */
 if (ch==EOL) {wf_chr(&f_t,EOL); return;}

 /* All other non-printables result in an inverted question mark. In the      */
 /* absence of bugs, the only way a non-printable can get to us here is as a  */
 /* result of the user explicitly inserting one using the scanner's @^        */
 /* control sequence.                                                         */
 if (!isascprn(ch))
    {
     wf_wr(&f_t,"&#191;"); /* inverted question mark */
     return;
    }
 /* If it IS printable, then print it ... carefully! */
 switch (ch)
   {
    /* Go through the printables in ASCII order. */
    /*   ' ':  is an ordinary character. */
    /*   '!':  is an ordinary character. */
    /*  '\"':  is an ordinary character. */
    /*   '#':  is an ordinary character. */
    /*   '$':  is an ordinary character. */
    /*   '%':  is an ordinary character. */
    case '&': wf_wr(&f_t,"&amp;");
              break;
    /*   ''': is an ordinary character. */
    /*   '(': is an ordinary character. */
    /*   ')': is an ordinary character. */
    /*   '*': is an ordinary character. */
    /*   '+': is an ordinary character. */
    /*   ',': is an ordinary character. */
    /*   '-': is an ordinary character. */
    /*   '.': is an ordinary character. */
    /*   '/': is an ordinary character. */
    /* '0-9': are   ordinary characters.*/
    /*   ':': is an ordinary character. */
    /*   ';': is an ordinary character. */
    case '<': wf_wr(&f_t,"&lt;");
              break;
    /*   '=': is an ordinary character. */
    case '>': wf_wr(&f_t,"&gt;");
              break;
    /*   '?': is an ordinary character. */
    /*   '@': is an ordinary character. */
    /* 'A-Z': are   ordinary characters.*/
    /*   '[': is an ordinary character. */
    /*  '\\': is an ordinary character. */
    /*   ']': is an ordinary character. */
    /*   '^': is an ordinary character. */
    /*   '_': is an ordinary character. */
    /*   '`': is an ordinary character. */
    /* 'a-z': are   ordinary characters.*/
    /*   '{': is an ordinary character. */
    /*   '|': is an ordinary character. */
    /*   '}': is an ordinary character. */
    /*   '~': is an ordinary character. */
    default:   wf_chr(&f_t,ch);    break;
   }
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_ugch P_((int));
LOCAL void wv_ugch(ch)
/* WeaVe TeX UGly CHaracter.                                                  */
/* Selects either the TeX or HTML ugly character pre-processing function.     */
int ch;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txugch(ch);  break;
  case TR_HTXT :
  case TR_HTML : wv_htugch(ch);  break;
  default      : as_bomb("wv_ugch:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

LOCAL void wv_ugstr P_((char *));
LOCAL void wv_ugstr(s)
/* Writes out a potentially ugly string. */
char *s;
{
 while (*s != EOS)
    wv_ugch(*s++);
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_txugsc P_((p_sc_t));
LOCAL void wv_txugsc(p_sc)
/* } ABC */
/* Writes out a potentially ugly TeX text scrap. */
p_sc_t p_sc;
{
 char *p      = p_sc->sc_first;
 char *p_last = p_sc->sc_last;

 while (p<=p_last)
    wv_ugch(*p++);
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htugsc P_((p_sc_t));
LOCAL void wv_htugsc(p_sc)
/* Writes out a potentially ugly HTML text scrap.         */
/* This is where paragraph (<P>) markups may be added for */
/* hypertext mode wherever groups of contiguous blank     */
/* lines are left in the text (not inside macros).        */
p_sc_t p_sc;
{
 char *p_first = p_sc->sc_first;
 char *p_last  = p_sc->sc_last;
 char *p = p_first;

 if (no_pmrkQ) {
   while (p<=p_last) wv_ugch(*p++);
 } else {
   if ((f_t.wf_last == EOL) && (*p == EOL)) {
     /* if two EOLs in a row are found, it signals */
     /* the start of a new HTML paragraph          */
     wf_wl(&f_t, "<P>");
     /* scan past further EOLs in a row */
     while ((*(++p) == EOL) && (p <= p_last));
   } /* end if */
   while (p<=p_last) {
     /* if two EOLs in a row are found, it signals */
     /* the start of a new HTML paragraph          */
     if ((*p == EOL) && (p != p_first) && (*(p-1) == EOL)) {
       wf_wl(&f_t, "<P>");
       /* scan past further EOLs in a row */
       while ((*(++p) == EOL) && (p <= p_last));
     } else {
       wv_ugch(*p++);
     } /* end if */
   } /* end while */
 } /* end if */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_htsc P_((p_sc_t));
LOCAL void wv_htsc(p_sc)
/* Writes out a potentially ugly HTML text code scrap.    */
/* This function does not insert paragraph markups.       */
p_sc_t p_sc;
{
 char *p_first = p_sc->sc_first;
 char *p_last  = p_sc->sc_last;
 char *p = p_first;

 while (p<=p_last) wv_ugch(*p++);
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_ugsc P_((p_sc_t));
LOCAL void wv_ugsc(p_sc)
/* Writes out a potentially ugly text scrap. */
p_sc_t p_sc;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txugsc(p_sc);  break;
  case TR_HTXT :
  case TR_HTML : wv_htugsc(p_sc);  break;
  default      : as_bomb("wv_ugsc:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

LOCAL void wv_ugscls P_((p_scls_t));
LOCAL void wv_ugscls(p_scls)
/* This function writes the specified text list to the output stream. It      */
/* writes it cleaning up all ugly characters.                                 */
p_scls_t p_scls;
{
 ls_fir(p_scls);
 while (TRUE)
   {
    p_sc_t p_sc;
    ls_nxt(p_scls,PPV &p_sc);
    if (p_sc==NULL) break;
    wv_ugsc(p_sc);
   }
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_txmanm P_((p_ma_t,ulong));
LOCAL void wv_txmanm(p_ma,part)
/* } ABC */
/* This function writes the specified macro name and no. in TeX. */
/* The part number is the part to write out.                     */
p_ma_t p_ma;
ulong part;
{
 p_bp_t p_bp;

 /* ABC { */
 if (p_ma->ma_defn.md_isfil) {
    wf_wr(&f_t,"\\fwfilename{");
    wv_ugstr(p_ma->ma_name);
 } else {
    wf_wr(&f_t,"\\fwmacroname{");
   if (raw_namQ && (tr_codes==TR_TEX)) {
     wf_wr(&f_t,p_ma->ma_name);
   } else {
     wv_ugstr(p_ma->ma_name);
   } /* end if */
  } /* end if */
 /* } ABC */
 wf_wr(&f_t,"}{");
 ls_loo(p_ma->ma_defn.md_body,part,PPV &p_bp);
 wf_dec(&f_t,(ulong) p_bp->bp_seq);
 wf_wr(&f_t,"}");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htmanm P_((p_ma_t,ulong));
LOCAL void wv_htmanm(p_ma,part)
/* This function writes the specified macro name and no. in HTML. */
/* The part number is the part to write out.                      */
p_ma_t p_ma;
ulong part;
{
 p_bp_t p_bp;

 if (p_ma->ma_defn.md_isfil) {
   wf_wr(&f_t,"<STRONG>");
   wv_ugstr(p_ma->ma_name);
 } else {
   wf_wr(&f_t,"<CITE>");
   if (raw_namQ && (tr_codes==TR_HTML)) {
     wf_wr(&f_t,p_ma->ma_name);
   } else {
     wv_ugstr(p_ma->ma_name);
   } /* end if */
 } /* end if */
 if (p_ma->ma_defn.md_isfil)
    wf_wr(&f_t,"</STRONG>");
 else
    wf_wr(&f_t,"</CITE>");
 wf_wr(&f_t,"[");
 ls_loo(p_ma->ma_defn.md_body,part,PPV &p_bp);
 wf_dec(&f_t,(ulong) p_bp->bp_seq);
 wf_wr(&f_t,"]");
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_manm P_((p_ma_t,ulong));
LOCAL void wv_manm(p_ma,part)
/* This function writes the specified macro name and no. */
/* The part number is the part to write out.             */
p_ma_t p_ma;
ulong part;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txmanm(p_ma,part);  break;
  case TR_HTXT :
  case TR_HTML : wv_htmanm(p_ma,part);  break;
  default      : as_bomb("wv_manm:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_mano P_((p_ma_t,ulong));
LOCAL void wv_mano(p_ma,part)
/* This function writes the specified macro no. */
/* The part number is the part to write out.    */
p_ma_t p_ma;
ulong part;
{
 p_bp_t p_bp;

 ls_loo(p_ma->ma_defn.md_body,part,PPV &p_bp);
 wf_dec(&f_t,(ulong) p_bp->bp_seq);
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txsecnn P_((p_ty_t));
LOCAL void wv_txsecnn(p_ty)
/* } ABC */
/* Write out the section number and name, each in curly braces. */
p_ty_t p_ty;
{
 /* Write as arguments the section number and name. */
 wf_chr(&f_t,'{');
 sn_str(&p_ty->ty_sn,linet1);
 wf_wr(&f_t,linet1);
 wf_wr(&f_t,"}{");
 /* ABC { */
 if (raw_namQ && (tr_codes==TR_TEX)) {
   wf_wr(&f_t,p_ty->ty_name);
 } else {
   wv_ugstr(p_ty->ty_name);
 } /* end if */
 /* } ABC */
 wf_chr(&f_t,'}');
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htsecnn P_((p_ty_t));
LOCAL void wv_htsecnn(p_ty)
/* Write out the section number and name. */
p_ty_t p_ty;
{
 /* Write as arguments the section number and name. */
 sn_str(&p_ty->ty_sn,linet1);
 wf_wr(&f_t,linet1);
 wf_wr(&f_t," ");
 /* ABC { */
 if (raw_namQ && (tr_codes==TR_HTML)) {
   wf_wr(&f_t,p_ty->ty_name);
 } else {
   wv_ugstr(p_ty->ty_name);
 } /* end if */
 /* } ABC */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_secnn P_((p_ty_t));
LOCAL void wv_secnn(p_ty)
/* Write out the section number and name. */
p_ty_t p_ty;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txsecnn(p_ty);  break;
  case TR_HTXT :
  case TR_HTML : wv_htsecnn(p_ty);  break;
  default      : as_bomb("wv_secnn:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txecho P_((p_scls_t));
LOCAL void wv_txecho(p_scls)
/* This function writes the specified text list to the output stream. It      */
/* writes it just as it is, with no modifications whatsoever.                 */
p_scls_t p_scls;
{
 ls_fir(p_scls);
 while (TRUE)
   {
    p_sc_t p_sc;
    ls_nxt(p_scls,PPV &p_sc);
    if (p_sc==NULL) break;
    wf_blk(&f_t,p_sc->sc_first,(size_t)
           (p_sc->sc_last-p_sc->sc_first+1));
   }
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_htecho P_((p_scls_t));
LOCAL void wv_htecho(p_scls)
/* This function writes the specified text list to the output stream. It      */
/* writes it just as it is, except that '<' --> "&lt;" and '>' --> "&gt;"     */
/* and '&' --> "&amp;".                                                       */
p_scls_t p_scls;
{
 ls_fir(p_scls);
 while (TRUE)
   {
    p_sc_t p_sc;
    ls_nxt(p_scls,PPV &p_sc);
    if (p_sc==NULL) break;
    wv_htsc(p_sc);
   }
}
/* } ABC */

/******************************************************************************/

/* ABC { */
/* This function writes the specified text list to the output stream. It      */
/* writes it just as it is, changing as few characters as possible       .    */
LOCAL void wv_echo P_((p_scls_t));
LOCAL void wv_echo(p_scls)
p_scls_t p_scls;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txecho(p_scls);  break;
  case TR_HTXT :
  case TR_HTML : wv_htecho(p_scls);  break;
  default      : as_bomb("wv_echo:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txverb P_((p_scls_t));
LOCAL void wv_txverb(p_scls)
/* } ABC */
/* Writes a TeX text list verbetim, but brackets it with sequences that       */
/* instruct the typesetter not to interpret the text in any special way.      */
p_scls_t p_scls;
{
 if (ls_len(p_scls) > 0)
   {
    wf_wr(&f_t,"\\fwbtx[");
    wv_echo(p_scls);
    wf_wl(&f_t,"]fwetx=%");
   }
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htverb P_((p_scls_t));
LOCAL void wv_htverb(p_scls)
/* } ABC */
/* Writes an HTML text list verbetim, but brackets it with sequences that     */
/* instruct the typesetter not to interpret the text in any special way.      */
p_scls_t p_scls;
{
 if (ls_len(p_scls) > 0)
   {
    wf_wr(&f_t,"<CODE>");
    wv_echo(p_scls);
    wf_wr(&f_t,"</CODE>");
   }
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_verb P_((p_scls_t));
LOCAL void wv_verb(p_scls)
/* } ABC */
/* Writes a text list verbetim, but brackets it with sequences that           */
/* instruct the typesetter not to interpret the text in any special way.      */
p_scls_t p_scls;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txverb(p_scls);  break;
  case TR_HTXT :
  case TR_HTML : wv_htverb(p_scls);  break;
  default      : as_bomb("wv_verb:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txhead P_((void));
LOCAL void wv_txhead()
/* } ABC */
{
 wf_wl(&f_t,"");
 wf_wl(&f_t,"%*******************************************************************************");
 wf_wl(&f_t,"%*                    START OF AUTOMATICALLY GENERATED TEX FILE                *");
 wf_wl(&f_t,"%*******************************************************************************");
 wf_wl(&f_t,"%*                                                                             *");
 wf_wl(&f_t,"%* This TeX file was automatically generated by the FunnelWeb preprocessor.    *");
 wf_wl(&f_t,"%* You can typeset this file to produce printed documentation by running it    *");
 wf_wl(&f_t,"%* through the TeX typesetter using a command such as:                         *");
 /* ABC { */
 if (doc_incQ) {
   wf_wl(&f_t,"%*    tex thisfilename                                                         *");
 } else {
   wf_wl(&f_t,"%*    latex thisfilename                                                       *");
   wf_wl(&f_t,"%* (the standard TeX header was not included, due to the presence of a         *");
   wf_wl(&f_t,"%*  `@p no_doc_header' pragma in the original FunnelWeb source file.           *");
   wf_wl(&f_t,"%*  It is hence assumed that the FunnelWeb code was in the form of a LaTeX     *");
   wf_wl(&f_t,"%*  file starting with a line of the form                                      *");
   wf_wl(&f_t,"%*    \\documentstyle[fw-latex,...]{...}                                        *");
   wf_wl(&f_t,"%*  If not, don't blame FunnelWeb for the resulting TeX or LaTeX errors ...    *");
   wf_wl(&f_t,"%*  The file `fw-latex.sty' can be found at the ftp archive noted below.)      *");
 } /* end if */
 /* } ABC */
 wf_wl(&f_t,"%* The resultant file thisfilename.dvi can be printed using a command such as: *");
 wf_wl(&f_t,"%*    lpr -Pcslw -d thisfilename.dvi                                           *");
 wf_wl(&f_t,"%*                                                                             *");
 wf_wl(&f_t,"%* FunnelWeb is a preprocessor that allows programmers to weave programs and   *");
 wf_wl(&f_t,"%* their documentation together in a single document. The FunnelWeb program    *");
 wf_wl(&f_t,"%* analyses such documents producing both program files and typeset            *");
 wf_wl(&f_t,"%* documentation such as this TeX file.                                        *");
 wf_wl(&f_t,"%* FunnelWeb was created by Ross Williams.                                     *");
 wf_wl(&f_t,"%*                                                                             *");
 wf_wl(&f_t,"%* For more information on FunnelWeb look in the following FTP archive:        *");
 wf_wl(&f_t,"%*    Machine  : sirius.itd.adelaide.edu.au [IP=129.127.40.3].                 *");
 wf_wl(&f_t,"%*    Directory: ~pub/funnelweb/                                               *");
 wf_wl(&f_t,"%*               (or some other appropriately named directory).                *");
 wf_wl(&f_t,"%* or email Ross Williams at ross@spam.adelaide.edu.au                         *");
 if (!doc_incQ) {
   wf_wl(&f_t,"%*     (Tony Coates at coates@physics.uq.edu.au for pragma `no_doc_header') *");
 } /* end if */
 if (raw_namQ) {
   wf_wl(&f_t,"%*     (Tony Coates at coates@physics.uq.edu.au for pragma `raw_macro_and_section_names') *");
 } /* end if */
 wf_wl(&f_t,"%*                                                                             *");
 wf_wl(&f_t,"%*******************************************************************************");
 wf_wl(&f_t,"");
 wf_wl(&f_t,"");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_hthead P_((void));
LOCAL void wv_hthead()
{
 wf_wl(&f_t,"");
 wf_wl(&f_t,"<!--*************************************************************************-->");
 wf_wl(&f_t,"<!--                  START OF AUTOMATICALLY GENERATED HTML FILE             -->");
 wf_wl(&f_t,"<!--*************************************************************************-->");
 wf_wl(&f_t,"<!--                                                                         -->");
 wf_wl(&f_t,"<!--This HTML file was automatically generated by the FunnelWeb preprocessor.-->");
 wf_wl(&f_t,"<!--You can view this file using an HTML viewer such as `Mosaic'.            -->");
 wf_wl(&f_t,"<!--FunnelWeb is a preprocessor that allows programmers to weave programs and-->");
 wf_wl(&f_t,"<!--their documentation together in a single document. The FunnelWeb program -->");
 wf_wl(&f_t,"<!--analyses such documents producing both program files and typeset         -->");
 wf_wl(&f_t,"<!--documentation such as this HTML file.                                    -->");
 wf_wl(&f_t,"<!--FunnelWeb was created by Ross Williams.                                  -->");
 wf_wl(&f_t,"<!--                                                                         -->");
 wf_wl(&f_t,"<!--For more information on FunnelWeb look in the following FTP archive:     -->");
 wf_wl(&f_t,"<!--  Machine  : sirius.itd.adelaide.edu.au [IP=129.127.40.3].               -->");
 wf_wl(&f_t,"<!--  Directory: ~pub/funnelweb/                                             -->");
 wf_wl(&f_t,"<!--             (or some other appropriately named directory).              -->");
 wf_wl(&f_t,"<!--or email Ross Williams at ross@spam.adelaide.edu.au                      -->");
 wf_wl(&f_t,"<!--   (Tony Coates at coates@physics.uq.edu.au for HTML support)             -->");
 wf_wl(&f_t,"<!--                                                                         -->");
 wf_wl(&f_t,"<!--*************************************************************************-->");
 wf_wl(&f_t,"");
 wf_wl(&f_t,"");
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_head P_((void));
LOCAL void wv_head()
{
 if ((tr_codes==TR_HTXT) || (tr_codes==TR_HTML)) wv_hthead();
 else                                            wv_txhead();
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txtail P_((void));
LOCAL void wv_txtail()
/* } ABC */
{
 wf_wl(&f_t,"");
 wf_wl(&f_t,"");
 wf_wl(&f_t,"%*******************************************************************************");
 wf_wl(&f_t,"%*                    END OF AUTOMATICALLY GENERATED TEX FILE                  *");
 wf_wl(&f_t,"%*******************************************************************************");
 wf_wl(&f_t,"");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_httail P_((void));
LOCAL void wv_httail()
{
 wf_wl(&f_t,"");
 wf_wl(&f_t,"");
 wf_wl(&f_t,"<!--*************************************************************************-->");
 wf_wl(&f_t,"<!--                  END OF AUTOMATICALLY GENERATED HTML FILE               -->");
 wf_wl(&f_t,"<!--*************************************************************************-->");
 wf_wl(&f_t,"");
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_tail P_((void));
LOCAL void wv_tail()
{
 if ((tr_codes==TR_HTXT) || (tr_codes==TR_HTML)) wv_httail();
 else                                            wv_txtail();
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txnsec P_((p_ty_t));
LOCAL void wv_txnsec(p_ty)
/* } ABC */
/* Weaves the start of a new TeX section. */
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_NSEC,"wv_txnsec: Not a section!");

 /* Work out what kind of section macro should be used. */
 switch (sn_lev(&p_ty->ty_sn))
   {
    case 1: wf_wr(&f_t,"\\fwseca"); break;
    case 2: wf_wr(&f_t,"\\fwsecb"); break;
    case 3: wf_wr(&f_t,"\\fwsecc"); break;
    case 4: wf_wr(&f_t,"\\fwsecd"); break;
    case 5: wf_wr(&f_t,"\\fwsece"); break;
    default: as_bomb("wv_txnsec:case defaulted.");
   }

 /* Write out the section number and name. */
 wv_secnn(p_ty);
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htnsec P_((p_ty_t));
LOCAL void wv_htnsec(p_ty)
/* Weaves the start of a new HTML section. */
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_NSEC,"wv_htnsec: Not a section!");

 /* Work out what kind of section macro should be used. */
 switch (sn_lev(&p_ty->ty_sn))
   {
    case 1: wf_wr(&f_t,"<H1>"); break;
    case 2: wf_wr(&f_t,"<H2>"); break;
    case 3: wf_wr(&f_t,"<H3>"); break;
    case 4: wf_wr(&f_t,"<H4>"); break;
    case 5: wf_wr(&f_t,"<H5>"); break;
    default: as_bomb("wv_htnsec:case defaulted.");
   }

 /* Write out an HTML anchor, which will be attached to
  * the index if any
  */
 wf_wr(&f_t,"<A NAME=\"");
 wv_secnn(p_ty);
 wf_wr(&f_t,"\">");

 /* Write out the section number and name. */
 wv_secnn(p_ty);

 /* Close anchor */
 wf_wr(&f_t,"</A>");

 /* Close section heading. */
 switch (sn_lev(&p_ty->ty_sn))
   {
    case 1: wf_wr(&f_t,"</H1>"); break;
    case 2: wf_wr(&f_t,"</H2>"); break;
    case 3: wf_wr(&f_t,"</H3>"); break;
    case 4: wf_wr(&f_t,"</H4>"); break;
    case 5: wf_wr(&f_t,"</H5>"); break;
    default: as_bomb("wv_htnsec:case defaulted.");
   }
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_nsec P_((p_ty_t));
LOCAL void wv_nsec(p_ty)
/* Weaves the start of a new section. */
p_ty_t p_ty;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txnsec(p_ty);  break;
  case TR_HTXT :
  case TR_HTML : wv_htnsec(p_ty);  break;
  default      : as_bomb("wv_nsec:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txolit P_((p_ty_t));
LOCAL void wv_txolit(p_ty)
/* } ABC */
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_OLIT,"wv_txolit: Not an open literal!");
 wf_wr(&f_t,"\\fwlit{");
 literal=TRUE;
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htolit P_((p_ty_t));
LOCAL void wv_htolit(p_ty)
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_OLIT,"wv_htolit: Not an open literal!");
 wf_wr(&f_t,"<CODE>");
 literal=TRUE;
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_olit P_((p_ty_t));
LOCAL void wv_olit(p_ty)
p_ty_t p_ty;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txolit(p_ty);  break;
  case TR_HTXT :
  case TR_HTML : wv_htolit(p_ty);  break;
  default      : as_bomb("wv_olit:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txclit P_((p_ty_t));
LOCAL void wv_txclit(p_ty)
/* } ABC */
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_CLIT,"wv_txclit: Not a close literal!");
 wf_wr(&f_t,"}");
 literal=FALSE;
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htclit P_((p_ty_t));
LOCAL void wv_htclit(p_ty)
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_CLIT,"wv_htclit: Not a close literal!");
 wf_wr(&f_t,"</CODE>");
 literal=FALSE;
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_clit P_((p_ty_t));
LOCAL void wv_clit(p_ty)
p_ty_t p_ty;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txclit(p_ty);  break;
  case TR_HTXT :
  case TR_HTML : wv_htclit(p_ty);  break;
  default      : as_bomb("wv_clit:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txoemp P_((p_ty_t));
LOCAL void wv_txoemp(p_ty)
/* } ABC */
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_OEMP,"wv_txoemp: Not an open emphasize!");
 wf_wr(&f_t,"\\fwemp{");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htoemp P_((p_ty_t));
LOCAL void wv_htoemp(p_ty)
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_OEMP,"wv_htoemp: Not an open emphasize!");
 wf_wr(&f_t,"<EM>");
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_oemp P_((p_ty_t));
LOCAL void wv_oemp(p_ty)
p_ty_t p_ty;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txoemp(p_ty);  break;
  case TR_HTXT :
  case TR_HTML : wv_htoemp(p_ty);  break;
  default      : as_bomb("wv_oemp:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txcemp P_((p_ty_t));
LOCAL void wv_txcemp(p_ty)
/* } ABC */
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_CEMP,"wv_txcemp: Not a close emphasize!");
 wf_wr(&f_t,"}");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htcemp P_((p_ty_t));
LOCAL void wv_htcemp(p_ty)
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_CEMP,"wv_htcemp: Not a close emphasize!");
 wf_wr(&f_t,"</EM>");
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_cemp P_((p_ty_t));
LOCAL void wv_cemp(p_ty)
p_ty_t p_ty;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txcemp(p_ty);  break;
  case TR_HTXT :
  case TR_HTML : wv_htcemp(p_ty);  break;
  default      : as_bomb("wv_cemp:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txnpag P_((p_ty_t));
LOCAL void wv_txnpag(p_ty)
/* } ABC */
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_NPAG,"wv_txnpag: Not a new page!");
 wf_wl(&f_t,"");
 wf_wl(&f_t,"\\fwnewpage");
 wf_wl(&f_t,"");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htnpag P_((p_ty_t));
LOCAL void wv_htnpag(p_ty)
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_NPAG,"wv_htnpag: Not a new page!");
 /* new pages are currently ignored for HTML */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_npag P_((p_ty_t));
LOCAL void wv_npag(p_ty)
p_ty_t p_ty;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txnpag(p_ty);  break;
  case TR_HTXT :
  case TR_HTML : wv_htnpag(p_ty);  break;
  default      : as_bomb("wv_npag:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txtocs P_((p_ty_t));
LOCAL void wv_txtocs(p_ty)
/* } ABC */
/* Writes a TeX table of contents to the weave output file. */
p_ty_t p_ty;
{
 /* We need to save the document list position as the rest of the weaver is   */
 /* in the process of traversing it!!!                                        */
 p_void p_mark = ls_mar(document_list);

 as_cold(p_ty->ty_kind==TY_TOCS,"wv_txtocs: Not a table of contents!");

 wf_wl(&f_t,"");
 wf_wl(&f_t,"\\fwtocstart{}");
 ls_fir(document_list);
 while (TRUE)
   {
    p_dc_t p_dc;
    ls_nxt(document_list,PPV &p_dc);
    if (p_dc == NULL) break;
    if (p_dc->dc_kind==DC_TYPE && p_dc->dc_ty.ty_kind==TY_NSEC)
      {
       switch(sn_lev(&p_dc->dc_ty.ty_sn))
         {
          case 1: wf_wr(&f_t,"\\fwtoca"); break;
          case 2: wf_wr(&f_t,"\\fwtocb"); break;
          case 3: wf_wr(&f_t,"\\fwtocc"); break;
          case 4: wf_wr(&f_t,"\\fwtocd"); break;
          case 5: wf_wr(&f_t,"\\fwtoce"); break;
          default: as_bomb("wv_txtocs: switch defaulted.");
         }
       wv_secnn(&p_dc->dc_ty);
       wf_wl(&f_t,"");
      }
   }
 wf_wl(&f_t,"\\fwtocfinish{}");
 wf_wl(&f_t,"");

 /* Restore the document list position. */
 ls_set(document_list,p_mark);
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_httocs P_((p_ty_t));
LOCAL void wv_httocs(p_ty)
/* Writes a table of contents to the weave output file. */
p_ty_t p_ty;
{
 /* We need to save the document list position as the rest of the weaver is   */
 /* in the process of traversing it!!!                                        */
 p_void p_mark = ls_mar(document_list);

 as_cold(p_ty->ty_kind==TY_TOCS,"wv_httocs: Not a table of contents!");

 wf_wl(&f_t,"<HR>");
 wf_wl(&f_t,"<H1>Table of Contents</H1>");
 wf_wl(&f_t,"<HR>");
 ls_fir(document_list);
 while (TRUE)
   {
    p_dc_t p_dc;
    ls_nxt(document_list,PPV &p_dc);
    if (p_dc == NULL) break;
    if (p_dc->dc_kind==DC_TYPE && p_dc->dc_ty.ty_kind==TY_NSEC)
      {
       switch(sn_lev(&p_dc->dc_ty.ty_sn))
         {
          case 1: wf_wr(&f_t,"<H2>"); break;
          case 2: wf_wr(&f_t,"<H3>"); break;
          case 3: wf_wr(&f_t,"<H4>"); break;
          case 4: wf_wr(&f_t,"<H5>"); break;
          case 5: wf_wr(&f_t,"<H6>"); break;
          default: as_bomb("wv_httoc: switch defaulted.");
         }
       wf_wr(&f_t,"<A HREF=\"#");
       wv_secnn(&p_dc->dc_ty);
       wf_wr(&f_t,"\">");
       wv_secnn(&p_dc->dc_ty);
       wf_wr(&f_t,"</A>");
       switch(sn_lev(&p_dc->dc_ty.ty_sn))
         {
          case 1: wf_wl(&f_t,"</H2>"); break;
          case 2: wf_wl(&f_t,"</H3>"); break;
          case 3: wf_wl(&f_t,"</H4>"); break;
          case 4: wf_wl(&f_t,"</H5>"); break;
          case 5: wf_wl(&f_t,"</H6>"); break;
          default: as_bomb("wv_httoc: switch defaulted.");
         }
      }
   }
 wf_wl(&f_t,"<HR>");

 /* Restore the document list position. */
 ls_set(document_list,p_mark);
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_tocs P_((p_ty_t));
LOCAL void wv_tocs(p_ty)
/* Writes a table of contents to the weave output file. */
p_ty_t p_ty;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txtocs(p_ty);  break;
  case TR_HTXT :
  case TR_HTML : wv_httocs(p_ty);  break;
  default      : as_bomb("wv_tocs:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txskip P_((p_ty_t));
LOCAL void wv_txskip(p_ty)
/* } ABC */
/* Writes a vertical space to the output file. */
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_SKIP,"wv_txskip: Not a skip directive!");

 sprintf(linet1,"\\fwvskip{%lu}",(ulong) p_ty->ty_mm);
 wf_wl(&f_t,linet1);
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htskip P_((p_ty_t));
LOCAL void wv_htskip(p_ty)
/* Writes a vertical space to the output file. */
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_SKIP,"wv_htskip: Not a skip directive!");

 /* no skipping is currently done for HTML files */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_skip P_((p_ty_t));
LOCAL void wv_skip(p_ty)
/* Writes a vertical space to the output file. */
p_ty_t p_ty;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txskip(p_ty);  break;
  case TR_HTXT :
  case TR_HTML : wv_htskip(p_ty);  break;
  default      : as_bomb("wv_skip:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txtitl P_((p_ty_t));
LOCAL void wv_txtitl(p_ty)
/* Writes a TeX title line to the output file. */
/* } ABC */
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_TITL,"wv_txtitl: Not a title!");

 switch(p_ty->ty_align)
   {
    case LR_LEFT: wf_wr(&f_t,"\\fwleftline{");   break;
    case LR_RIGH: wf_wr(&f_t,"\\fwrightline{");  break;
    case LR_CENT: wf_wr(&f_t,"\\fwcenterline{"); break;
    default: as_bomb("wv_txtitle: Alignment switch defaulted.");
   }
 switch(p_ty->ty_font)
   {
    case FT_NORM: wf_wr(&f_t,"\\fwfontnormal{");     break;
    case FT_TITL: wf_wr(&f_t,"\\fwfonttitle{");      break;
    case FT_STIT: wf_wr(&f_t,"\\fwfontsmalltitle{"); break;
    default: as_bomb("wv_txtitle: Font switch defaulted.");
   }
 wv_ugsc(&p_ty->ty_sc);
 wf_wl(&f_t,"}}");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_httitl P_((p_ty_t));
LOCAL void wv_httitl(p_ty)
/* Writes an HTML title line to the output file,
 * creating a document title if the title line is the
 * first for the document.
 */
p_ty_t p_ty;
{
 as_cold(p_ty->ty_kind==TY_TITL,"wv_httitl: Not a title!");

 if (title1Q) {
   title1Q = FALSE;
   wf_wl(&f_t, "<TITLE>");
   switch(p_ty->ty_align)
     {
      /* Title alignment is currently ignored */
      case LR_LEFT:
      case LR_RIGH:
      case LR_CENT: break;
      default: as_bomb("wv_httitle: Alignment switch defaulted.");
     }
   switch(p_ty->ty_font)
     {
      /* Title font is currently ignored */
      case FT_NORM:
      case FT_TITL:
      case FT_STIT: break;
      default: as_bomb("wv_httitle: Font switch defaulted.");
     }
   wv_ugsc(&p_ty->ty_sc);
   if (*(p_ty->ty_sc.sc_last) != '.') {
     /* there may be more title lines to come */
     wf_wr(&f_t," ...");
   } /* end if */
   wf_wl(&f_t,"\n</TITLE>");
 } /* end if */

 wf_wl(&f_t, "<H1>");
 switch(p_ty->ty_align)
   {
    /* Title alignment is currently ignored */
    case LR_LEFT:
    case LR_RIGH:
    case LR_CENT: break;
    default: as_bomb("wv_httitle: Alignment switch defaulted.");
   }
 switch(p_ty->ty_font)
   {
    /* Title font is currently ignored */
    case FT_NORM:
    case FT_TITL:
    case FT_STIT: break;
    default: as_bomb("wv_httitle: Font switch defaulted.");
   }
 wv_ugsc(&p_ty->ty_sc);
 wf_wl(&f_t,"\n</H1>");
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_titl P_((p_ty_t));
LOCAL void wv_titl(p_ty)
/* Writes a title line to the output file. */
p_ty_t p_ty;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txtitl(p_ty);  break;
  case TR_HTXT :
  case TR_HTML : wv_httitl(p_ty);  break;
  default      : as_bomb("wv_titl:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

LOCAL void wv_dcty P_((p_dc_t));
LOCAL void wv_dcty(p_dc)
/* Weaves a document typesetter element. */
p_dc_t p_dc;
{
 p_ty_t p_ty = &p_dc->dc_ty;

 as_cold(p_dc->dc_kind==DC_TYPE,"wv_dcty: Not a typesetter component!");

 switch (p_ty->ty_kind)
   {
    case TY_NSEC: wv_nsec(p_ty); break;
    case TY_OLIT: wv_olit(p_ty); break;
    case TY_CLIT: wv_clit(p_ty); break;
    case TY_OEMP: wv_oemp(p_ty); break;
    case TY_CEMP: wv_cemp(p_ty); break;
    case TY_NPAG: wv_npag(p_ty); break;
    case TY_TOCS: wv_tocs(p_ty); break;
    case TY_SKIP: wv_skip(p_ty); break;
    case TY_TITL: wv_titl(p_ty); break;
    default     : as_bomb("wv_dcty: Switch defaulted.");
   }
}

/******************************************************************************/

LOCAL void wv_dctx P_((p_dc_t));
LOCAL void wv_dctx(p_dc)
/* Write out a document text item. */
p_dc_t p_dc;
{
 as_cold(p_dc->dc_kind==DC_TEXT,"wv_dctx: Not text.");

/* ABC { */
 if (literal || (tr_codes==TR_NONE) || (tr_codes==TR_HTXT))
/* } ABC */
    wv_ugscls(p_dc->dc_text);
 else
    wv_echo(p_dc->dc_text);
}

/******************************************************************************/

LOCAL void wv_eltx P_((p_el_t));
LOCAL void wv_eltx(p_el)
/* Weaves a text element. */
p_el_t p_el;
{
 as_cold(p_el->el_kind==EL_TEXT,"wv_eltx: Not text.");
 wv_verb(p_el->el_text);
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_ohtdf P_((void));
LOCAL void wv_ohtdf()
/* Write a macro definition tag opening */
{
  wf_wr(&f_t, "FWDEF[");
}
/* ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_chtdf P_((void));
LOCAL void wv_chtdf()
/* Write a macro definition tag closing */
{
  wf_wr(&f_t, "]DEFFW");
}
/* ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txelin P_((p_el_t));
LOCAL void wv_txelin(p_el)
/* } ABC */
/* Weaves a TeX invocation element. */
p_el_t p_el;
{
 as_cold(p_el->el_kind==EL_INVC,"wv_txelin: Not an invocation.");

 /* Write out the macro call proper. */
 wv_manm(p_el->el_p_mac,1L);

 /* Write out the actual parameter list, if any. */
 if (ls_len(p_el->el_parls) > 0)
   {
    ulong    parnum;
    wf_wr(&f_t,"\\fwoparen ");
    ls_fir(p_el->el_parls);  /* List of actual parameter expressions. */
    ls_fir(p_el->el_pretx);  /* List of crud before each expression.  */
    ls_fir(p_el->el_postx);  /* List of crud after  each expression.  */
    parnum=1;
    while (TRUE)
      {
       p_ells_t *pp_exp;
       p_scls_t *pp_scls1;
       p_scls_t *pp_scls2;
       ls_nxt(p_el->el_parls,PPV &pp_exp);
       ls_nxt(p_el->el_pretx,PPV &pp_scls1);
       ls_nxt(p_el->el_postx,PPV &pp_scls2);
       if (pp_exp==NULL) break;
       if (parnum++>1) wf_wr(&f_t,"\\fwcomma ");
       wv_verb(*pp_scls1);
       wf_wr(&f_t,"\\fwoquote"); wv_ex(*pp_exp); wf_wr(&f_t,"\\fwcquote");
       wv_verb(*pp_scls2);
      }
    wf_wr(&f_t,"\\fwcparen ");
   }
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htelin P_((p_el_t));
LOCAL void wv_htelin(p_el)
/* Weaves an HTML invocation element. */
p_el_t p_el;
{
 as_cold(p_el->el_kind==EL_INVC,"wv_htelin: Not an invocation.");

 /* Write an anchor to reference the definition */
 wf_wr(&f_t, "<A HREF=\"#");
 wv_ohtdf();

 /* Write out the macro no. */
 wv_mano(p_el->el_p_mac,1L);

 /* Close the anchor name */
 wv_chtdf();
 wf_wr(&f_t, "\">");

 /* Write out the macro call proper. */
 wv_manm(p_el->el_p_mac,1L);

 /* Close anchor */
 wf_wr(&f_t, "</A>");

 /* Write out the actual parameter list, if any. */
 if (ls_len(p_el->el_parls) > 0)
   {
    ulong    parnum;
    wf_wr(&f_t,"(");
    ls_fir(p_el->el_parls);  /* List of actual parameter expressions. */
    ls_fir(p_el->el_pretx);  /* List of crud before each expression.  */
    ls_fir(p_el->el_postx);  /* List of crud after  each expression.  */
    parnum=1;
    while (TRUE)
      {
       p_ells_t *pp_exp;
       p_scls_t *pp_scls1;
       p_scls_t *pp_scls2;
       ls_nxt(p_el->el_parls,PPV &pp_exp);
       ls_nxt(p_el->el_pretx,PPV &pp_scls1);
       ls_nxt(p_el->el_postx,PPV &pp_scls2);
       if (pp_exp==NULL) break;
       if (parnum++>1) wf_wr(&f_t,",");
       wv_verb(*pp_scls1);
       wf_wr(&f_t,"`"); wv_ex(*pp_exp); wf_wr(&f_t,"'");
       wv_verb(*pp_scls2);
      }
    wf_wr(&f_t,")");
   }
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_elin P_((p_el_t));
LOCAL void wv_elin(p_el)
/* Weaves an invocation element. */
p_el_t p_el;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txelin(p_el);  break;
  case TR_HTXT :
  case TR_HTML : wv_htelin(p_el);  break;
  default      : as_bomb("wv_elin:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txelpr P_((p_el_t));
LOCAL void wv_txelpr(p_el)
/* } ABC */
/* Weaves a TeX parameter element. */
p_el_t p_el;
{
 as_cold(p_el->el_kind==EL_PARM,"wv_txelpr: Not a parameter.");
 wf_wr (&f_t,"\\fwparam{");
 wf_dec(&f_t,(ulong) p_el->el_parno);
 wf_wr (&f_t,"}");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htelpr P_((p_el_t));
LOCAL void wv_htelpr(p_el)
/* Weaves an HTML parameter element. */
p_el_t p_el;
{
 as_cold(p_el->el_kind==EL_PARM,"wv_htelpr: Not a parameter.");
 wf_wr (&f_t,"#");
 wf_dec(&f_t,(ulong) p_el->el_parno);
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_elpr P_((p_el_t));
LOCAL void wv_elpr(p_el)
/* Weaves a parameter element. */
p_el_t p_el;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txelpr(p_el);  break;
  case TR_HTXT :
  case TR_HTML : wv_htelpr(p_el);  break;
  default      : as_bomb("wv_elpr:case defaulted.");
  } /* end switch */
}
/* } ABC */

/* ABC { */
/******************************************************************************/

LOCAL void wv_elsp P_((p_el_t));
LOCAL void wv_elsp(p_el)
/* Weaves a special function element. */
p_el_t p_el;
{
 as_cold(p_el->el_kind==EL_SPEC,"wv_elsp: Not an internal function.");
}

/* } ABC */

/******************************************************************************/

LOCAL void wv_ex(p_exp)
/* Weaves a expression. */
p_ells_t p_exp;
{

 ls_fir(p_exp);
 while (TRUE)
   {
    p_el_t p_el;
    ls_nxt(p_exp,PPV &p_el);
    if (p_el==NULL) break;
    switch (p_el->el_kind)
      {
       case EL_TEXT: wv_eltx(p_el); break;
       case EL_INVC: wv_elin(p_el); break;
       case EL_PARM: wv_elpr(p_el); break;
       /* ABC { */
       case EL_SPEC: wv_elsp(p_el); break;
       /* } ABC */
       default: as_bomb("wv_ex: Case defaulted.");
      }
   }
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_txusage P_((p_ma_t));
LOCAL void wv_txusage(p_ma)
/* } ABC */
/* Weaves a TeX list of all the sections in which the specified macro is used. */
p_ma_t p_ma;
{
 ulong  no_calls = ls_len(p_ma->ma_calls);
 ulong  callno;

 /* If it is a file macro then it is not allowed to be called. */
 if (p_ma->ma_defn.md_isfil)
   {
    wf_wl(&f_t,"\\fwisafile{This macro is attached to an output file.}");
    return;
   }

 if (no_calls==0)
   {
    wf_wl(&f_t,"\\fwusedin{This macro is NEVER invoked.}");
    return;
   }

 wf_wr(&f_t,"\\fwusedin{This macro is invoked in definition");
 if (no_calls>1) wf_chr(&f_t,'s');
 wf_chr(&f_t,' ');
 ls_fir(p_ma->ma_calls);
 for (callno=1;callno<=no_calls;callno++)
   {
    p_mc_t p_mc;
    ls_nxt(p_ma->ma_calls,PPV &p_mc);
    as_cold(p_mc!=NULL,"wv_txusage: Awk!");
    wf_dec(&f_t,(ulong) p_mc->mc_seq);
    if (callno < no_calls-1) {
       wf_wr(&f_t,", ");
       /* ABC{ */
       /* add return if necessary [ABC] */
       if (callno % MAXINV == 0) wf_wr(&f_t,"\n");
       /* } ABC */
    } else
       if (callno==no_calls-1)
          if (no_calls>2)
             wf_wr(&f_t,", and ");
          else
             wf_wr(&f_t," and ");
   }
 wf_wl(&f_t,".}");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htusage P_((p_ma_t));
LOCAL void wv_htusage(p_ma)
/* Weaves an HTML list of all the sections in which the specified macro is used. */
p_ma_t p_ma;
{
 ulong  no_calls = ls_len(p_ma->ma_calls);
 ulong  callno;

 /* If it is a file macro then it is not allowed to be called. */
 if (p_ma->ma_defn.md_isfil)
   {
    wf_wr(&f_t,"<CITE>");
    wf_wr(&f_t,"This macro is attached to an output file.");
    wf_wl(&f_t,"</CITE>");
    return;
   }

 if (no_calls==0)
   {
    wf_wl(&f_t,"<CITE>This macro is NEVER invoked.</CITE>");
    return;
   }

 wf_wr(&f_t,"<CITE>This macro is invoked in definition");
 if (no_calls>1) wf_chr(&f_t,'s');
 wf_chr(&f_t,' ');
 ls_fir(p_ma->ma_calls);
 for (callno=1;callno<=no_calls;callno++)
   {
    p_mc_t p_mc;
    ls_nxt(p_ma->ma_calls,PPV &p_mc);
    as_cold(p_mc!=NULL,"wv_htusage: Awk!");
    wf_wr(&f_t, "<A HREF=\"#");
    wv_ohtdf();
    wf_dec(&f_t,(ulong) p_mc->mc_seq);
    wv_chtdf();
    wf_wr(&f_t, "\">");
    wf_dec(&f_t,(ulong) p_mc->mc_seq);
    wf_wr(&f_t, "</A>");
    if (callno < no_calls-1) {
       wf_wr(&f_t,", ");
       /* ABC{ */
       /* add HTML return if necessary [ABC] */
       if (callno % MAXINV == 0) wf_wr(&f_t,"\n");
       /* } ABC */
    } else
       if (callno==no_calls-1)
          if (no_calls>2)
             wf_wr(&f_t,", and ");
          else
             wf_wr(&f_t," and ");
   }
 wf_wl(&f_t,".</CITE>");
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_usage P_((p_ma_t));
LOCAL void wv_usage(p_ma)
/* Weaves a list of all the sections in which the specified macro is used. */
p_ma_t p_ma;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txusage(p_ma);  break;
  case TR_HTXT :
  case TR_HTML : wv_htusage(p_ma);  break;
  default      : as_bomb("wv_usage:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txdefin P_((p_ma_t));
LOCAL void wv_txdefin(p_ma)
/* } ABC */
/* Just after each macro body part definition comes a note listing the        */
/* containing the macro's body parts. This function weaves that list.         */
p_ma_t p_ma;
{
 ulong  no_parts = ls_len(p_ma->ma_defn.md_body);
 ulong  partno;

 /* If it's not an additive macro, we don't write anything. */
 if (!p_ma->ma_defn.md_isadd) return;

 wf_wr(&f_t,"\\fwseealso{This macro is defined in definitions ");
 ls_fir(p_ma->ma_defn.md_body);
 for (partno=1;partno<=no_parts;partno++)
   {
    p_bp_t p_bp;
    ls_nxt(p_ma->ma_defn.md_body,PPV &p_bp);
    as_cold(p_bp!=NULL,"wv_txdefin: Awk!");
    wf_dec(&f_t,(ulong) p_bp->bp_seq);
    if (partno<no_parts-1)
       wf_wr(&f_t,", ");
    else
       if (partno==no_parts-1)
          if (no_parts>2)
             wf_wr(&f_t,", and ");
          else
             wf_wr(&f_t," and ");
   }
 wf_wl(&f_t,".}");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htdefin P_((p_ma_t));
LOCAL void wv_htdefin(p_ma)
/* Just after each macro body part definition comes a note listing the        */
/* containing the macro's body parts. This function weaves that list.         */
p_ma_t p_ma;
{
 ulong  no_parts = ls_len(p_ma->ma_defn.md_body);
 ulong  partno;

 /* If it's not an additive macro, we don't write anything. */
 if (!p_ma->ma_defn.md_isadd) return;

 wf_wr(&f_t,"<CITE>This macro is defined in definitions ");
 ls_fir(p_ma->ma_defn.md_body);
 for (partno=1;partno<=no_parts;partno++)
   {
    p_bp_t p_bp;
    ls_nxt(p_ma->ma_defn.md_body,PPV &p_bp);
    as_cold(p_bp!=NULL,"wv_htdefin: Awk!");
    wf_wr(&f_t, "<A HREF=\"#");
    wv_ohtdf();
    wf_dec(&f_t,(ulong) p_bp->bp_seq);
    wv_chtdf();
    wf_wr(&f_t, "\">");
    wf_dec(&f_t,(ulong) p_bp->bp_seq);
    wf_wr(&f_t, "</A>");
    if (partno<no_parts-1)
       wf_wr(&f_t,", ");
    else
       if (partno==no_parts-1)
          if (no_parts>2)
             wf_wr(&f_t,", and ");
          else
             wf_wr(&f_t," and ");
   }
 wf_wl(&f_t,".</CITE>");
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_defin P_((p_ma_t));
LOCAL void wv_defin(p_ma)
/* Just after each macro body part definition comes a note listing the        */
/* containing the macro's body parts. This function weaves that list.         */
p_ma_t p_ma;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txdefin(p_ma);  break;
  case TR_HTXT :
  case TR_HTML : wv_htdefin(p_ma);  break;
  default      : as_bomb("wv_defin:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

/* ABC { */
LOCAL void wv_txdcma P_((p_dc_t));
LOCAL void wv_txdcma(p_dc)
/* } ABC */
/* Weaves a macro definition in TeX. */
p_dc_t p_dc;
{
 p_bp_t p_bp;

 /* Macro calls sandwich the definition itsef. */
 wf_wl(&f_t,"\\fwbeginmacro");

 /* Write out the macro's name. */
 wv_manm(p_dc->dc_p_ma,p_dc->dc_part);

 /* Write the formal parameter list. */
 if (p_dc->dc_p_ma->ma_defn.md_npar>0)
   {
    wf_wr(&f_t,"\\fwparams{");
    wf_dec(&f_t,(ulong) p_dc->dc_p_ma->ma_defn.md_npar);
    wf_wr(&f_t,"}");
   }

 /* Zero and many options are indicated by bold letters. */
 if (p_dc->dc_p_ma->ma_defn.md_iszer) wf_wr(&f_t,"\\fwzero{}");
 if (p_dc->dc_p_ma->ma_defn.md_isman) wf_wr(&f_t,"\\fwmany{}");

 /* Write the "equals". */
 if (p_dc->dc_p_ma->ma_defn.md_isadd)
    wf_wr(&f_t,"\\fwplusequals ");
 else
    wf_wr(&f_t,"\\fwequals ");

 /* Write out this particular body part. */
 ls_loo(p_dc->dc_p_ma->ma_defn.md_body,p_dc->dc_part,PPV &p_bp);
 wf_wr(&f_t,"\\fwodef ");
 wv_ex(p_bp->bp_ex);
 wf_wl(&f_t,"\\fwcdef ");

 /* After the macro definition, write out notes on the macro. */
 wf_wl(&f_t,"\\fwbeginmacronotes");
 wv_defin(p_dc->dc_p_ma);            /* A list of the macro's body parts.  */
 wv_usage(p_dc->dc_p_ma);            /* A list of where the macro is used. */
 wf_wl(&f_t,"\\fwendmacronotes");

 /* The macro definition is sandwiched by macro calls. */
 wf_wl(&f_t,"\\fwendmacro");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_htdcma P_((p_dc_t));
LOCAL void wv_htdcma(p_dc)
/* } ABC */
/* Weaves a macro definition in HTML. */
p_dc_t p_dc;
{
 p_bp_t p_bp;

 /* Write an anchor for the macro */
 wf_wr(&f_t, "<PRE><A NAME=\"");
 wv_ohtdf();

 /* Write out the macro's number. */
 wv_mano(p_dc->dc_p_ma,p_dc->dc_part);

 /* Finish anchor name */
 wv_chtdf();
 wf_wr(&f_t, "\">");

 /* Write out the macro's name. */
 wv_manm(p_dc->dc_p_ma,p_dc->dc_part);

 /* Close anchor */
 wf_wr(&f_t, "</A>");

 /* Write the formal parameter list. */
 if (p_dc->dc_p_ma->ma_defn.md_npar>0)
   {
    wf_wr(&f_t,"(#");
    wf_dec(&f_t,(ulong) p_dc->dc_p_ma->ma_defn.md_npar);
    wf_wr(&f_t,")");
   }

 /* Zero and many options are indicated by bold letters. */
 if (p_dc->dc_p_ma->ma_defn.md_iszer)
   wf_wr(&f_t,"<STRONG>Z</STRONG>");
 if (p_dc->dc_p_ma->ma_defn.md_isman)
   wf_wr(&f_t,"<STRONG>M</STRONG>");

 /* Write the "equals". */
 if (p_dc->dc_p_ma->ma_defn.md_isadd)
    wf_wl(&f_t,"+=");
 else
    wf_wl(&f_t,"=");

 /* Write out this particular body part. */
 ls_loo(p_dc->dc_p_ma->ma_defn.md_body,p_dc->dc_part,PPV &p_bp);
 wf_wr(&f_t,"{");
 wv_ex(p_bp->bp_ex);
 wf_wl(&f_t,"}");

 /* After the macro definition, write out notes on the macro. */
 wv_defin(p_dc->dc_p_ma);            /* A list of the macro's body parts.  */
 wv_usage(p_dc->dc_p_ma);            /* A list of where the macro is used. */
 wf_wl(&f_t,"</PRE>");
}

/******************************************************************************/

/* ABC { */
LOCAL void wv_dcma P_((p_dc_t));
LOCAL void wv_dcma(p_dc)
/* } ABC */
/* Weaves a macro definition. */
p_dc_t p_dc;
{
  switch (tr_codes) {
  case TR_NONE :
  case TR_TEX  : wv_txdcma(p_dc);  break;
  case TR_HTXT :
  case TR_HTML : wv_htdcma(p_dc);  break;
  default      : as_bomb("wv_dcma:case defaulted.");
  } /* end switch */
}
/* } ABC */

/******************************************************************************/

LOCAL void wv_dc P_((p_dc_t));
LOCAL void wv_dc(p_dc)
/* Weaves a document component (p_dc). */
p_dc_t p_dc;
{
 switch (p_dc->dc_kind)
   {
    case DC_TEXT: wv_dctx(p_dc); break;
    case DC_TYPE: wv_dcty(p_dc); break;
    case DC_MACR: wv_dcma(p_dc); break;
    default     : as_bomb("wv_dc: Case defaulted.");
   }
}

/******************************************************************************/

LOCAL void wv_docnt P_((void));
LOCAL void wv_docnt()
/* Weaves the entire document. This functions and most functions above assume */
/* that the output stream f_t is prepared for writing.                        */
{
 /* Write the typeset header. */
 wv_head();

 /* Writes the library macros. */
 /* ABC { */
 if (doc_incQ) {
   switch (tr_codes) {
   case TR_NONE :
   case TR_TEX  : tex_head(&f_t);  break;
   case TR_HTXT :
   case TR_HTML : html_head(&f_t);  break;
   default      : as_bomb("wv_docnt:case defaulted.");
   } /* end switch */
 } /* end if */
 if (
   (tr_codes==TR_HTXT)
   || ((tr_codes==TR_HTML) && doc_incQ)
 ) {
   /* Writes the opening HTML markup */
   wf_wl(&f_t,"<HTML>");
 } /* end if */
 /* } ABC */

 ls_fir(document_list);
 literal=FALSE;
 while (TRUE)
   {
    p_dc_t p_dc;
    ls_nxt(document_list,PPV &p_dc);
    if (p_dc == NULL) break;
    wv_dc(p_dc);
   }
 /* ABC { */
 if (
   (tr_codes==TR_NONE)
   || ((tr_codes==TR_TEX) && doc_incQ)
 ) {
   /* Writes the closing TeX command */
   wf_wl(&f_t,"\\bye");
 } else if (
   (tr_codes==TR_HTXT)
   || ((tr_codes==TR_HTML) && doc_incQ)
 ) {
   /* Writes the closing HTML markup */
   wf_wl(&f_t,"</HTML>");
 } /* end if */
 /* } ABC */

 /* Write the typeset trailer. */
 wv_tail();
}

/******************************************************************************/

EXPORT void weave()
/* Writes out the documentation file. */
{
 fn_t  fn_tmp;   /* Name of temporary file.                                   */
 fn_t  fn_targ;  /* Name of documentation file.                               */
 bool  renfil;   /* Do we wish to rename output file?                         */
 bool  istarg;   /* Does a target file already exist?                         */

 /* Writing the documentation file differs from some other output files. With */
 /* non critical files such as the listing file that are really just logs,    */
 /* generation of half a listing file is acceptable if not desirable. However */
 /* in the case of the documentation file, it is very bad to generate half an */
 /* output file; far better to generate none at all. For this reason, and     */
 /* also because of the presence of the D option (which prevents the writing  */
 /* of output files identical to existing files (to prevent MAKE              */
 /* propagations)) it is best to write a temporary file and then rename it.   */

 /* We shouldn't be here if the weave option is off! */
 as_cold(option.op_t_b,"weave: option.op_t_b==FALSE!");

 /* ABC { */
 /* There are no titles yet, so the next is the first */
 title1Q = TRUE;
 /* No text scraps have been written yet */
 lastEOLQ = FALSE;
 /* } ABC */

 /* Work out what the typeset file's name should be.                          */
 strcpy(fn_targ,"");              /* Start with an empty string.              */
 fn_ins(fn_targ,option.op_f_s);
 /* ABC { */
 switch (tr_codes) {
 case TR_NONE :
 case TR_TEX  : fn_ins(fn_targ,".tex");  break;
 case TR_HTXT :
 case TR_HTML :
#if PC
   fn_ins(fn_targ,".htm");
#else /* !PC */
   fn_ins(fn_targ,".html");
#endif /* PC */
   break;
 default      : as_bomb("weave:case defaulted.");
 } /* end switch */
 /* } ABC */
 fn_ins(fn_targ,option.op_t_s);

 /* The temporary file has to inherit too, because the output directory may   */
 /* not be the default directory and some computers can't rename across       */
 /* directories (and we have to rename it later).                             */
 strcpy(fn_tmp,fn_targ);
 fn_ins(fn_tmp,fn_temp());

 /* Expand the macro to the temporary file. */
 wf_ini(&f_t,TRUE);
 wf_ope(&f_t,fn_tmp);
 if (wf_err(&f_t))
   {
    sprintf(linet1,"Error creating temporary documentation file \"%s\".",&fn_tmp[0]);
    wl_sjl(linet1);
    (void) remove(fn_tmp);
    goto severe;
   }

 wv_docnt();

 /* Make sure that there weren't any errors writing to the temporary file. */
 if (wf_err(&f_t))
   {
    sprintf(linet1,"Error writing to temporary documentation file \"%s\".",&fn_tmp[0]);
    wl_sjl(linet1);
    (void) remove(fn_tmp);
    goto severe;
   }

 /* Close the temporary file. */
 wf_clo(&f_t);
 if (wf_err(&f_t))
   {
    sprintf(linet1,"Error closing temporary documentation file \"%s\".",&fn_tmp[0]);
    wl_sjl(linet1);
    remove(fn_tmp);
    goto severe;
   }

 /* The rest of the code in this function copes with the renaming. */

 /* By default, we wish to rename the temporary file. */
 renfil=TRUE;

 /* Deal with any existing file of the target name. */
 istarg=fexists(fn_targ);
 if (istarg && option.op_d_b)
   {
    /* A target already exists, and the D option is on. If the target is      */
    /* identical to the temporary, we can simply delete the temporary!        */
    char *errstr;
    bool  same;
    errstr=eq_files(fn_tmp,fn_targ,&same);
    if (errstr != NULL)
      {
       wl_sjl("Error comparing temporary file with previous documentation file.");
       wl_sjl("(A comparison was attempted because the D option was turned on.)");
       wl_sjl("Error from comparison routine was as follows (first=temp):");
       wr_sjl("   ");wl_sjl(errstr);
       sprintf(linet1,"Temporary file name was \"%s\".",&fn_tmp[0]);
       wl_sjl(linet1);
       sprintf(linet1,"Output    file name was \"%s\".",&fn_targ[0]);
       wl_sjl(linet1);
       wl_sjl("FunnelWeb will leave both files intact so you can look at them.");
       goto severe;
      }
    /* If the two files are the same, we can simply delete the temporary. */
    if (same)
      {
       int status;
       status=remove(fn_tmp);
       if (status != REMOVE_S)
         {
          sprintf(linet1,"Error deleting (under +D option) temporary file \"%s\".",&fn_tmp[0]);
          wl_sjl(linet1);
          goto severe;
         }
       sprintf(linet1,"Deleted identical documentation file \"%s\".",&fn_targ[0]);
       wl_sjl(linet1);
       renfil=FALSE;
      }
   }

 if (renfil)
   {
    int status;
    /* We need to delete any existing file of the target name. */
    if (istarg)
      {
       status=remove(fn_targ);
       if (status != REMOVE_S)
         {
          sprintf(linet1,"Error deleting existing documentation file \"%s\".",&fn_targ[0]);
          wl_sjl(linet1);
          goto severe;
         }
      }
    /* Rename the temporary file to the output file. */
    status=rename(fn_tmp,fn_targ);
    if (status != RENAME_S)
      {
       wl_sjl("Error renaming temporary documentation file to documentation file.");
       sprintf(linet1,"Temporary file name was \"%s\".",&fn_tmp[0]);
       wl_sjl(linet1);
       sprintf(linet1,"Output    file name was \"%s\".",&fn_targ[0]);
       wl_sjl(linet1);
       wl_sjl("FunnelWeb will leave both files intact so you can look at them.");
       goto severe;
      }
   }

 /* Tell everyone that we have written an output file. */
 sprintf(linet1,"Weave : Completed %s.",fn_nodir(&fn_targ[0]));
 wl_s(linet1);
 if (option.op_b7_b)
    sprintf(linet1,"Weave: Completed %s.",SUPPNAME);
 wl_j(linet1);
 wl_l(linet1);
 return;

 /* Jump here is a nasty file error occurs. */
 severe:
 sprintf(linet1,"A problem occurred during the generation of documentation file \"%s\".",&fn_targ[0]);
 wl_sjl(linet1);
 wl_sjl("S: Aborting...");
 num_sev++;
 return;
}

/******************************************************************************/

