package zpplet.data;

import zpplet.machine.ZMachine;
import zpplet.misc.ZError;

public class ZObjectTree
	{
	protected ZMachine zm;
	protected int table;
	protected int tree;

	public ZObjectTree(ZMachine zm)
		{
		this.zm = zm;
		table = zm.hd.getObjectTableAddr();
		tree = table + numProperties() * 2;
		}

	final public int defaultProperty(int prop)
		{
		return zm.getWord(table + (prop - 1) * 2);
		}

	protected int numProperties()
		{
		return 31;
		}

	protected int getEntryLoc(int obj)
		{
		return tree + (obj - 1) * 9;
		}

	final public boolean getAttribute(int obj, int attr)
		{
		int entryloc = getEntryLoc(obj);
		int bytenum = attr >> 3;
		int bitmask = 1 << (7 - (attr & 7));
		return ((zm.getByte(entryloc + bytenum) & bitmask) != 0);
		}

	final public void setAttribute(int obj, int attr, boolean value)
		{
		int entryloc = getEntryLoc(obj);
		int bytenum = attr >> 3;
		int bitmask = 1 << (7 - (attr & 7));
		if (value)
			zm.setByte(entryloc + bytenum, zm.getByte(entryloc + bytenum) | bitmask);
		else
			zm.setByte(entryloc + bytenum, zm.getByte(entryloc + bytenum) & ~bitmask);
		}

	public int getParent(int obj)
		{
		return obj == 0 ? 0 : zm.getByte(getEntryLoc(obj) + 4);
		}

	public int getSibling(int obj)
		{
		return obj == 0 ? 0 : zm.getByte(getEntryLoc(obj) + 5);
		}

	public int getChild(int obj)
		{
		return obj == 0 ? 0 : zm.getByte(getEntryLoc(obj) + 6);
		}

	public void setParent(int obj, int newparent)
		{
		if (obj != 0)
			zm.setByte(getEntryLoc(obj) + 4, newparent);
		}

	public void setSibling(int obj, int newsibling)
		{
		if (obj != 0)
			zm.setByte(getEntryLoc(obj) + 5, newsibling);
		}

	public void setChild(int obj, int newchild)
		{
		if (obj != 0)
			zm.setByte(getEntryLoc(obj) + 6, newchild);
		}

	public int getPropertyTableAddr(int obj)
		{
		return obj == 0 ? 0 : zm.getWord(getEntryLoc(obj) + 7);
		}

	final public int getShortNameAddr(int obj)
		{
		return (getPropertyTableAddr(obj) + 1);
		}

	public int getPropEntryAddr(int obj, int prop)
		{
		if (obj == 0)
			return 0;
		int addr = getPropertyTableAddr(obj);
		addr += zm.getByte(addr) * 2 + 1;
		int sizebyte = zm.getByte(addr);
		while (sizebyte != 0)
			{
			if ((sizebyte & 31) == prop)
				return addr;
			else if ((sizebyte & 31) < prop)
				return 0;
			addr += (sizebyte >> 5) + 2;
			sizebyte = zm.getByte(addr);
			}
		return 0;
		}

	public int getNextProp(int obj, int prop)
			throws ZError
		{
		int addr;

		if (prop == 0)
			{ // first property
			addr = getPropertyTableAddr(obj);
			if (addr == 0)
				throw new ZError(
						"Tried to get next property for object with no properties");
			addr += zm.getByte(addr) * 2 + 1;
			}

		else
			{
			addr = getPropEntryAddr(obj, prop);
			if (addr == 0)
				throw new ZError(
						"Tried to get next property for nonexistent property");
			addr += (zm.getByte(addr) >> 5) + 2;
			}

		return zm.getByte(addr) & 31;
		}

	public int getPropAddr(int obj, int prop)
		{
		int addr = getPropEntryAddr(obj, prop);
		if (addr == 0)
			return 0;
		return addr + 1;
		}

	public int getPropLen(int addr)
		{
		if (addr == 0)
			return 0;

		int sizebyte = zm.getByte(addr - 1);
		return (sizebyte >> 5) + 1;
		}

	final public int getProp(int obj, int prop)
		{
		int addr = getPropEntryAddr(obj, prop);
		if (addr == 0)
			return defaultProperty(prop);

		int size = (zm.getByte(addr) >> 5) + 1;
		if (size == 1)
			return zm.getByte(addr + 1);

		return zm.getWord(addr + 1);
		}

	final public void setProp(int obj, int prop, int val)
			throws ZError
		{
		int addr = getPropEntryAddr(obj, prop);
		if (addr == 0)
			throw new ZError("Tried to set nonexistent property");

		int size = (zm.getByte(addr) >> 5) + 1;
		if (size == 1)
			zm.setByte(addr + 1, val);
		else
			zm.setWord(addr + 1, val);
		}

	private void debugDumpObject(int obj, int indent)
		{
		String shortname;
		try
			{
			shortname = zm.getStringAt(getShortNameAddr(obj));
			}
		catch (Exception e)
			{
			shortname = "<error>";
			}
		for (int i = 0; i < indent; i++)
			System.out.print(" ");
		System.out.println("[" + obj + "] " + shortname);
		int child = getChild(obj);
		while (child != 0)
			{
			debugDumpObject(child, indent + 4);
			child = getSibling(child);
			}
		}

	public void debugDumpTree()
		{
		int firstproperty = zm.getMem().length;
		int obj = 0;
		while (getEntryLoc(obj) < firstproperty)
			{
			if (getParent(obj) == 0)
				debugDumpObject(obj, 0);
			int propaddr = getPropertyTableAddr(obj);
			if ((propaddr != 0) && (propaddr < firstproperty))
				firstproperty = propaddr;
			obj++;
			}
		}
	}

