package zpplet.system;

import zpplet.header.ZHeader;
import zpplet.machine.ZMachine;
import zpplet.misc.*;

public class ZWindow
	{
	protected ZMachine zm;
	protected ZScreen screen;
	public int id; // window number

	public int top, left, width, height; // window position in pixels
	int xpos, ypos; // cursor position in pixels (window relative)
	protected boolean wrap, scroll, transcript, buffer;
	int leftmargin, rightmargin;

	public ZFont font;
	private ZFont morefont;

	private StringBuilder linebuffer;
	private boolean time_for_more;

	protected int line_counter;
	protected int interrupt_routine;
	protected int interrupt_count;
	protected boolean pausing;

	public ZWindow(ZMachine zm, int id)
		{
		this.zm = zm;
		this.screen = zm.s;
		this.id = id;
		width = 1;
		height = 1;
		resetLineCount();
		buffer = true;
		wrap = true;
		scroll = true;
		transcript = true;
		linebuffer = new StringBuilder();
		font = new ZFont(screen);

		morefont = new ZFont(screen);
		morefont.set(ZFont.FIXED_FONT);
		morefont.setStyle(ZFont.REVERSE);
		}

	public void resetLineCount()
		{
		if (line_counter != -999) line_counter = getLines() - 1;
		time_for_more = false;
		}
	
	protected void drawMore()
		{
		ZFont f = font;
		font = morefont;
		screen.drawText(left + xpos + leftmargin, top + ypos, "[MORE]", this);
		font = f;
		}

	protected void promptForMore()
		{
		resetLineCount();
		if (zm.input == null)
			{
			pausing = true;
			drawMore();
			screen.repaint();
			screen.readKeyCode(0, 0);
			pausing = false;
			eraseLine();
			}
		}

	protected boolean inBounds(int x)
		{
		return (x >= leftmargin) && (x < (width - rightmargin));
		}

	public void eraseLine()
		{
		screen.eraseRegion(left + xpos, top + ypos, width - xpos, screen.getLineHeight(), font.getBack());
		}

	public void eraseLineTo(int right)
		{
		if (right > (width - rightmargin - xpos)) right = width - rightmargin - xpos;
		screen.eraseRegion(left + xpos, top + ypos, right, screen.getLineHeight(), font.getBack());
		}

	public void setWrapMode(boolean value)
		{
		flush();
		wrap = value;
		}

	public void setBufferMode(boolean value)
		{
		flush();
		buffer = value;
		}

	public void setTranscripting(boolean value)
		{
		transcript = value;
		}

	public void setScroll(boolean value)
		{
		scroll = value;
		resetLineCount();
		}

	public void moveTo(int left, int top)
		{
		flush();
		left = left != 0 ? (left - 1) * screen.unitx : this.left;
		top = top != 0 ? (top - 1) * screen.unity : this.top;
		xpos += this.left - left;
		ypos += this.top - top;
		this.left = left;
		this.top = top;
		if (!inBounds(xpos) || (ypos < 0))
			{
			xpos = leftmargin;
			ypos = 0;
			}
		}

	public int getTextWidth()
		{
		return width - leftmargin - rightmargin;
		}

	public void resize(int width, int height)
		{
		flush();
		this.width = width * screen.unitx;
		this.height = height * screen.unity;
		if (!inBounds(xpos) || (ypos >= this.height))
			{
			xpos = leftmargin;
			ypos = 0;
			}
		resetLineCount();
		}

	public int getLines()
		{
		return height / screen.getLineHeight();
		}

	public int getCursorX()
		{
		return xpos / screen.unitx + 1;
		}

	public int getCursorY()
		{
		return ypos / screen.unity + 1;
		}

	public int getPosX()
		{
		return left + xpos;
		}

	public int getPosY()
		{
		return top + ypos;
		}

	public void moveCursor(int x, int y)
		{
		flush();
		xpos = x == 0 ? xpos : (x - 1) * screen.unitx;
		ypos = y == 0 ? ypos : (y - 1) * screen.unity;
		if (!inBounds(xpos)) xpos = leftmargin;
		resetLineCount();
		}

	public void printZAscii(int zascii)
		{
		char unicode = zm.zc.toOutput(zascii);
		if (unicode == 0) // unprintable
			return;

		if (buffer)
			bufferChar(unicode);
		else
			{
			if (wrap && !inBounds(xpos + screen.charWidth(unicode))) newLineNoFlush();
			drawString(String.valueOf(unicode));
			}
		}
	
	public void flush()
		{
		if (buffer)
			{
			drawString(linebuffer.toString());
			linebuffer.setLength(0);
			}
		}

	public void newLine()
		{
		flush();
		newLineNoFlush();
		}

	void countdown()
		{
		if (interrupt_count != 0) if (--interrupt_count == 0) zm.zi.interrupt(interrupt_routine);
		}

	protected void newLineNoFlush()
		{
		if (zm.hd.newlineatstart) countdown();

		xpos = leftmargin;
		if (ypos < (height - screen.getLineHeight() * 2 + 1))
			ypos += screen.getLineHeight();
		else if (scroll)
			screen.scrollUp(this, screen.getLineHeight());
		else
			ypos = 0;

		if (scroll && (line_counter != -999) && time_for_more) promptForMore();

		if (line_counter != -999) time_for_more = (--line_counter == 0) || time_for_more;

		if (!zm.hd.newlineatstart) countdown();
		}

	public int getStringWidth(String s)
		{
		return screen.getStringWidth(s, font);
		}

	public int splitText(String s, int maxwidth)
	// returns length of initial part of string that will fit in the given width
	// returns 0 if entire string fits
		{
		if (screen.getStringWidth(s, font) <= maxwidth) return 0;

		int last = s.length();
		while (true)
			{
			int space = s.lastIndexOf(' ', last);

			if (space == -1) // no spaces; must break in middle of line
				{
				while ((last > 1) && (screen.getStringWidth(s.substring(0, last), font) > maxwidth))
					last--;
				return last;
				}

			while ((space > 0) && (s.charAt(space - 1) == ' '))
				// get first non-space
				space--;

			if (screen.getStringWidth(s.substring(0, space), font) <= maxwidth) // this fits
				return space;

			last = space - 1; // trim to a prior space
			}
		}

	protected synchronized void bufferChar(int ch)
		{
		linebuffer.append((char)ch);

		if (wrap)
			{
			int split = splitText(linebuffer.toString(), width - rightmargin - xpos);
			if (split > 0)
				{
				drawString(linebuffer.substring(0, split));
				while ((split < linebuffer.length()) && linebuffer.charAt(split) == ' ')
					split++;
				linebuffer.delete(0, split);
				newLineNoFlush();
				}
			}
		}

	public void clear()
		{
		screen.eraseRegion(left, top, width, height, font.getBack());
		}

	public void setColor(int fg, int bg)
		{
		flush();
		if (fg == ZColor.Z_UNDERCURSOR)
			{
			fg = ZColor.Z_CURRENT;
			font.color.fg = screen.getColorAt(getPosX(), getPosY());
			}
		if (bg == ZColor.Z_UNDERCURSOR)
			{
			bg = ZColor.Z_CURRENT;
			font.color.bg = screen.getColorAt(getPosX(), getPosY());
			}
		font.color.setZColor(fg, bg);
		}

	public void setStyle(int style)
		{
		flush();
		font.setStyle(style);
		}
	
	public int setFont(int zfont)
		{
		flush();
		return font.set(zfont);
		}

	private void drawString(String s)
		{
		if (scroll && time_for_more) promptForMore();
		if (s.length() != 0) xpos += screen.drawText(left + xpos, top + ypos, s, this);
		}

	public void setMargins(int left, int right)
		{
		flush();
		leftmargin = left;
		rightmargin = right;
		if (!inBounds(xpos)) xpos = leftmargin;
		}

	public int getWindowStyle()
		{
		int curflags = 0;
		if (wrap) curflags |= (1 << 0);
		if (scroll) curflags |= (1 << 1);
		if (transcript) curflags |= (1 << 2);
		if (buffer) curflags |= (1 << 3);
		return curflags;
		}

	public void setWindowStyle(int flags, int op)
			throws ZError
		{
		// flag bits: 0-wrap, 1-scroll, 2-to printer, 3-buffer
		// op: 0=set as given, 1=set if set, 2=clear if set, 3=toggle if set
		int curflags = getWindowStyle();
		switch (op)
			{
			case 0:
				curflags = flags;
				break;
			case 1:
				curflags |= flags;
				break;
			case 2:
				curflags &= ~flags;
				break;
			case 3:
				curflags ^= flags;
				break;
			default:
				throw new ZError("Invalid window setStyle operator");
			}

		// specific kludges
		if ((id == 0) && (zm.hd.story == ZHeader.ZORK_ZERO) && (zm.hd.getRelease() == 366)) curflags |= 1;
		if ((id == 0) && (zm.hd.story == ZHeader.SHOGUN) && (zm.hd.getRelease() <= 295)) curflags |= 1;

		setWrapMode((curflags & (1 << 0)) != 0);
		setScroll((curflags & (1 << 1)) != 0);
		setTranscripting((curflags & (1 << 2)) != 0);
		setBufferMode((curflags & (1 << 3)) != 0);
		}

	public int getProp(int prop)
			throws ZError
		{
		switch (prop)
			{
			case 0:
				return top + 1;
			case 1:
				return left + 1;
			case 2:
				return height;
			case 3:
				return width;
			case 4:
				flush();
				return ypos + 1;
			case 5:
				flush();
				return xpos + 1;
			case 6:
				return leftmargin;
			case 7:
				return rightmargin;
			case 8:
				return interrupt_routine;
			case 9:
				return interrupt_count;
			case 10:
				return font.getZStyle();
			case 11:
				return font.getColorBytes();
			case 12:
				return font.getZFont();
			case 13:
				return (screen.getCharWidth() & 0xFF) + ((screen.getLineHeight() & 0xFF) << 8);
			case 14:
				return getWindowStyle();
			case 15:
				return line_counter;
			default:
				throw new ZError("Unknown window property " + prop);
			}
		}

	public void setProp(int prop, int value)
			throws ZError
		{
		switch (prop)
			{
			case 8:
				interrupt_routine = value;
				break;
			case 9:
				interrupt_count = (short)value;
				break;
			case 10:
				setStyle(value);
				break;
			case 15:
				line_counter = (short)value;
				break;
			default:
				throw new ZError("Tried to set window property " + prop);
			}
		}
	
	public void postInputLine(String s)
		{
		// do nothing; used for scrollback
		}
	}