/********************************************************************
 *                                                                  *
 *  "Printer Graphics Interface" Demonstration Program              *
 *                                                                  *
 *  This program demonstrates how to use various functions          *
 *  available in the PGRAPH library.                                *
 *                                                                  *
 *  Author: F van der Hulst                                         *
 *                                                                  *
 * Revisions:                                                       *
 * 27 March 1991: Initial release (Turbo C v2.0 only)               *
 * 07 April 1991: Ported to MicroSoft C v5.1                        *
 *                                                                  *
 ********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>

#include "pgraph.h"

void shapes_demo(void);
void stroked_fonts_demo(void);
void default_font_demo(void);
void horiz_text_demo(void);
void vert_text_demo(void);
void text_scaling_demo(void);
void flood_fill_demo(void);
void lines_demo(void);
void shape_fill_demo(void);
void image_demo(void);
void pie_demo(void);
void view_demo(void);

void STAR_DRIVER(void);			/* PGI Object entries */
void LASERJET_DRIVER(void);

#define MAX_WIDTH 801			/* Maximum width of any PGRAPH viewport defined in the program */

extern int page_height, page_width;	/* Size of page in pixels */
extern FILE *prn;				/* Output device */

struct {								/* Page sizes in 1/100 of an inch available on various printers */
	int x;
	int y;
} page_size[] = {
	{0, 0},
	{800, 1100},					/* STAR */
	{780, 1088},               /* LaserJet */
	{800, 1100},					/* Epson LX400 */
	{800, 1100},					/* USER1 */
	{800, 1100} };					/* USER2 */


/********************************************************************
 Process command line arguments. */

void get_args(int argc, char *argv[], int *driver, int *mode, char *dev_name, char *demos)
{
int i, low, high;

	for (i = 1; i < argc; i++) {
		strupr(argv[i]);
		if (argv[i][0] != '/' && argv[i][0] != '-') {
			printf("Invalid command line switch: %s\n(Must start with '-' or '/')\n", argv[i]);
			exit(1);
		}
		if (argv[i][1] == '?' || argv[i][1] == 'H') {
			printf("Command syntax: %s [/O=outputdevice][/P=printer][/M=mode][/D=demos]\n\n", argv[0]);
			printf("outputdevice may be PRN, or a filename\n");
			printf("printer may be STAR, LASERJET, LX-400, USER1, or USER2\n");
			printf("\tIf you use LX-400, USER1, USER2, the corresponding .PGI file\n\tmust be in the current directory\n");
			printf("mode is an integer in the range 0 to the maximum mode for the selected printer\n");
			printf("demos is a series of letters, identifying which demos to print\n");
			printf("\nDefault values are PRN, LASERJET, and 2 (150x150 dpi)\n\n");
			exit(0);
		}
		if (argv[i][2] != '=') {
			printf("Invalid command line switch: %s\n(Must be /%c=VALUE)\n", &argv[i][1]);
			exit(1);
		}
		switch(argv[i][1]) {
		case 'O':
		case 'o':
			strcpy(dev_name, &argv[i][3]);
			break;
		case 'P':
		case 'p':
			if (strcmp(&argv[i][3], "STAR") == 0)				*driver = STAR;
			else if (strcmp(&argv[i][3], "LX-400") == 0) 		*driver = LX400;
			else if (strcmp(&argv[i][3], "LASERJET") == 0) 	*driver = LASERJET;
			else if (strcmp(&argv[i][3], "USER1") == 0) 		*driver = USER1;
			else if (strcmp(&argv[i][3], "USER2") == 0) 		*driver = USER2;
			else {
				printf("Unknown printer type: %s\n", &argv[i][3]);
				exit(1);
			}
			break;
		case 'M':
		case 'm':
			*mode = atoi(&argv[i][3]);
			p_getmoderange(*driver, &low, &high);
			if (*mode > high || *mode < low) {
				printf("Invalid mode: %d\n", *mode);
				exit(1);
			}
			break;
		case 'D':
		case 'd':
			strcpy(demos, &argv[i][3]);
			break;
		default:
			printf("Invalid command line switch: %s\n(Must be /D, /O, /P, or /M)\n", argv[i]);
			exit(1);
		}
	}
}

/********************************************************************
 Find the best mode (the worst X resolution that will display
 MAX_WIDTH bits) for the selected printer. */

int best_mode(void)
{
int i;
int xres, yres, best_x, best_y;
int mode;

	mode = 0;
	best_y = best_x = 1000;
	for (i = 0; i <= p_getmaxmode(); i++) {
		p_setgraphmode(i);
		p_getresolution(&xres, &yres);
		if (xres*(long)page_width/100 >= MAX_WIDTH+7)
			if ((xres < best_x) || (xres == best_x && yres < best_y)) {
				best_y = yres;
				best_x = xres;
				mode = i;
			}
	}
	return mode;
}

void main(int argc, char *argv[])
{
int driver = LASERJET, mode = -1, dummy_mode, errorcode;
int xres, yres;
static char filename[80] = "PRN";
static char selection[80] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

	if (p_registerfarbgidriver(STAR_DRIVER) < 0) {
		printf("Couldn't register STAR");
		return;
	}
	if (p_registerbgidriver(LASERJET_DRIVER) < 0) {
		printf("Couldn't register LASERJET");
		return;
	}
	get_args(argc, argv, &driver, &mode, filename, selection);
	if (strcmp(filename, "PRN") != 0) {
		prn = fopen(filename, "wb");
		if (prn == NULL) {
			printf("Couldn't open %s for output\n", filename);
			exit(1);
		}
	}

	dummy_mode = mode == -1 ? 0 : mode;
printf("Calling p_initgraph(%d, %d, NULL)\n", driver, dummy_mode);
    p_initgraph(&driver, &dummy_mode, NULL);
printf("Returned from p_initgraph(%d, %d, NULL)\n", driver, dummy_mode);
    errorcode = p_graphresult();             /* preserve error return */
	if (errorcode != grOk) {					/* error? */
		printf("Graphics error: %d", errorcode);
		return;
	}
	page_height = page_size[driver].y;
	page_width = page_size[driver].x;
	if (mode < 0 || mode > p_getmaxmode())		mode = best_mode();
	p_setgraphmode(mode);
	p_getresolution(&xres, &yres);
	printf("Currently set to mode %d (%d by %ddpi).\n", mode, xres, yres);

	if (strchr(selection, 'A') != NULL) shapes_demo();
	if (strchr(selection, 'B') != NULL) stroked_fonts_demo();
	if (strchr(selection, 'C') != NULL) default_font_demo();
	if (strchr(selection, 'D') != NULL) horiz_text_demo();
	if (strchr(selection, 'E') != NULL) vert_text_demo();
	if (strchr(selection, 'F') != NULL) text_scaling_demo();
	if (strchr(selection, 'G') != NULL) shape_fill_demo();
	if (strchr(selection, 'H') != NULL) flood_fill_demo();
	if (strchr(selection, 'I') != NULL) lines_demo();
	if (strchr(selection, 'J') != NULL) image_demo();
	if (strchr(selection, 'K') != NULL) pie_demo();
	if (strchr(selection, 'L') != NULL) view_demo();
	p_closegraph();

	if (strcmp(filename, "PRN") != 0)	fclose(prn);
}
