/*****************************************************************************
File    : Image2PDF.h
Author  : Utility Warrior, (c) 2005
Project : Image to PDF (Image2PDF) Dynamic Link Library
Purpose : This file provides C prototypes for all DLL functions

Notes  : a) #include this header file at the top of any C code that references
            any of these DLL functions
		 b) Link with Image2PDF.lib which provides the necessary linkage to
		    automatically generate the correct DLL call for the function, plus
			it will ensure that the Image2PDF.DLL file is loaded when the
			calling application is run (NB there is no need to use LoadLibrary)
		 c) Ensure that Image2PDF.DLL is either in the same directory as the
		    calling application or on the system path

Please refer to the accompanying documentation for full details.
******************************************************************************/


#ifndef _Image2PDF_h
#define _Image2PDF_h

/*****************************************************************************
   This allows the support of multiple versions of a DLL (if and when
   available) so that functions that might only be available in a later version
   can then only be called if the returned version currently being used is one 
   that supports those functions.
   
   Return value: integer value representing the DLL version, a value of 100 is
                 equivalent to version 1.00
******************************************************************************/
extern __declspec(dllimport) UINT I2PDF_GetDLLVersion(void);


/*****************************************************************************
Required and Optional
******************************************************************************/
extern __declspec(dllimport) void I2PDF_License(char *code);
/* This is required to remove the "Evaluation Mode" message that is stamped 
   over the top of every page in the output PDF document.   
   
   To purchase a license code please visit the Utility Warrior web site which 
   has the latest information and prices on the licenses that are available. 

   Return value: none 
*/

extern __declspec(dllimport) UINT I2PDF_AddImage(char *image);
/* Specifies the name and location of an image to embed within the output PDF 
   document.

   The image can be one of the following types: JPG, PNG, GIF, BMP, PCX or TGA.

   Up to 1000 images can be included.

   Images are embedded in the output PDF document in the order that they are 
   referenced.

   Return value:
   0 - success
   1 - invalid parameter
   2 - maximum number of images already added
   3 - invalid image type
*/

#define VERTICAL_ALIGNMENT_TOP				1
#define VERTICAL_ALIGNMENT_MIDDLE			2
#define VERTICAL_ALIGNMENT_BOTTOM			3
#define HORIZONTAL_ALIGNMENT_LEFT			1
#define HORIZONTAL_ALIGNMENT_CENTER			2
#define HORIZONTAL_ALIGNMENT_RIGHT			3
extern __declspec(dllimport) UINT I2PDF_ImageStamp(char *image, UINT verticalPosition, UINT horizontalPosition, double margin);
/* Specifies the name and location of an image that is "stamped" at the specified location on each page of the output PDF document.
   A margin between the image stamp and the edge of the page can be specified if required.

   Return value:
   0 - success
   1 - invalid image parameter
   2 - invalid vertical position
   3 - invalid horizontal position
   4 - invalid margin
*/

extern __declspec(dllimport) UINT I2PDF_StampURL(char *url);
/* This is optional and specifies the URL that the image stamp will open in the default browser when it is clicked on.

   A maximum of 128 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

#define SLIDESHOW_WIPE_RIGHT						1
#define SLIDESHOW_WIPE_UP							2
#define SLIDESHOW_WIPE_LEFT							3
#define SLIDESHOW_WIPE_DOWN							4
#define SLIDESHOW_BARN_DOORS_HORIZONTAL_OUT			5
#define SLIDESHOW_BARN_DOORS_HORIZONTAL_IN			6
#define SLIDESHOW_BARN_DOORS_VERTICAL_OUT			7
#define SLIDESHOW_BARN_DOORS_VERTICAL_IN			8
#define SLIDESHOW_BOX_OUT							9
#define SLIDESHOW_BOX_IN							10
#define SLIDESHOW_BLINDS_HORIZONTAL					11
#define SLIDESHOW_BLINDS_VERTICAL					12
#define SLIDESHOW_DISSOLVE							13
#define SLIDESHOW_GLITTER_RIGHT						14
#define SLIDESHOW_GLITTER_DOWN						15
#define SLIDESHOW_GLITTER_TOP_LEFT_TO_BOTTOM_RIGHT	16
#define SLIDESHOW_REPLACE							17
extern __declspec(dllimport) UINT I2PDF_SlideShow(UINT transition, double transitionDuration, double pageDisplayDuration);
/* Forces the PDF document to be displayed full screen with the chosen screen transition between each page.

   Return value:
   0 - success
   1 - invalid transition
   2 - invalid transition duration
   3 - invalid page display duration
*/

extern __declspec(dllimport) void I2PDF_DeleteImagesOnConvert(void);
/* When the PDF document has been successfully produced all of the images converted to PDF are deleted.

   Return value: none 
*/


/*****************************************************************************
PDF Settings
******************************************************************************/
#define OUTLINE_FORMAT_FILENAME	1
#define OUTLINE_FORMAT_NAME		2
#define STYLE_NONE				0
#define STYLE_LOWERCASE			1
#define STYLE_UPPERCASE			2
#define STYLE_CAPITALISE		3
extern __declspec(dllimport) UINT I2PDF_SetOutline(UINT format, UINT style);
/* This turns on the generation of a bookmark for each image that is in the 
   output PDF document.

   format can be either "filename" (this is the name of the image file including 
   the image extension but not including the full filepath) or "name" (same as 
   filename but without the image extension).

   style can be: "lowercase", "uppercase", "capitalise" or "none".

   Return value:
   0 - success
   1 - invalid format parameter
   2 - invalid style parameter
*/

extern __declspec(dllimport) void I2PDF_SetPermissionEdit(void);
/* Allows editing/changing of the PDF document but NOT adding or changing of text 
   notes and AcroForm fields.

   Return value: none 
*/

extern __declspec(dllimport) void I2PDF_SetPermissionEditExtra(void);
/* Allows the adding and changing of text notes and AcroForm fields within a PDF document.

   Return value: none 
*/

extern __declspec(dllimport) void I2PDF_SetPermissionCopy(void);
/* Allows the copying of text and graphics from the PDF document.

   Return value: none 
*/

extern __declspec(dllimport) void I2PDF_SetPermissionPrint(void);
/* Allows printing of the PDF document.

   Return value: none 
*/

extern __declspec(dllimport) UINT I2PDF_SetOwnerPassword(char *password);
/* The owner of a PDF document would enter this password in order to view it
   and make changes and perform actions that may be restricted to a normal user.

   A maximum of 32 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT I2PDF_SetUserPassword(char *password);
/* In order that only the intended user can view the PDF document, they must type
   in this password, otherwise the user is not allowed to read the PDF document.

   A maximum of 32 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT I2PDF_SetDPI(UINT dpi);
/* PDF documents use a default resolution setting of 72 DPI. Images created for 
   screen display under Windows will probably have a resolution of 96 DPI. If you 
   have created images for printing then you will probably need to use 300 or 600 DPI.
   
   If you do not adjust the default resolution setting then when displayed within a 
   PDF document on screen a 96 DPI image will appear 25% larger than expected. This 
   gets much worse for a 300 DPI image which will appear 416% larger than expected.

   Return value:
   0 - success
   1 - invalid parameter - dpi specified is less than 9
   2 - invalid parameter - dpi specified is greater than 2880
*/


/*****************************************************************************
PDF Meta Information
******************************************************************************/
extern __declspec(dllimport) UINT I2PDF_SetProducer(char *producer);
/* The name of the application that converted the document from its native format to PDF.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT I2PDF_SetCreator(char *creator);
/* If the document was converted into a PDF document from another form, this is 
   usually the name of the application that created the original document.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT I2PDF_SetAuthor(char *author);
/* The name of the person who created the PDF document.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT I2PDF_SetTitle(char *title);
/* The PDF document's title.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT I2PDF_SetSubject(char *subject);
/* The PDF document's subject.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT I2PDF_SetKeywords(char *keywords);
/* A list of keywords associated with the PDF document.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) void I2PDF_SetCreationDate(void);
/* The date that the PDF document was created.

   This is set using the current local time of the PC.

   Return value: none
*/


/*****************************************************************************
PDF Viewer Preferences
******************************************************************************/
extern __declspec(dllimport) void I2PDF_SetViewerPreferenceCenterWindow(void);
/* The window displaying the PDF document should be displayed in the center of 
   the computer's screen.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetViewerPreferenceHideToolbar(void);
/* The PDF viewer's toolbar should be hidden when the PDF document is active.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetViewerPreferenceHideMenubar(void);
/* The PDF viewer's menubar should be hidden when the PDF document is active.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetViewerPreferenceHideWindowUI(void);
/* The user interface elements within the PDF document's window should be hidden.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetViewerPreferenceFitWindow(void);
/* The PDF viewer should resize the window displaying the PDF document to fit the 
   size of the first displayed page of the PDF document.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetPageModeOutlines(void);
/* The PDF viewer should show the outline (ie bookmark) navigation item when the 
   PDF document is open.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetPageModeNone(void);
/* The PDF viewer should not show the outline (ie bookmark) or thumbnail navigation
   items when the PDF document is open.

   This is the default if no other page mode is specified.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetPageModeUseThumbs(void);
/* The PDF viewer should show the thumbnail navigation item when the PDF document 
   is open.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetPageModeFullScreen(void);
/* The PDF viewer should open the PDF document in full-screen mode without showing 
   the menu bar, window controls or any other window.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetPageLayoutSinglePage(void);
/* Display the pages one page at a time.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetPageLayoutOneColumn(void);
/* Display the pages in one column.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetPageLayoutTwoColumnLeft(void);
/* Display the pages in two columns, with odd-numbered pages on the left.

   Return value: none
*/

extern __declspec(dllimport) void I2PDF_SetPageLayoutTwoColumnRight(void);
/* Display the pages in two columns, with odd-numbered pages on the right.

   Return value: none
*/


/*****************************************************************************
Specifies the name and location of the produced PDF document, 
******************************************************************************/
extern __declspec(dllimport) UINT I2PDF_MakePDF(char *output, BOOL bOpenPDF, BOOL bReset, char *errorText, UINT maxErrorTextSize);
/* Generate the output PDF according to the settings already selected through
   the Application Programming Interface functions above.

   To have the output PDF open in the default PDF viewer set the bOpenPDF
   parameter to TRUE instead of FALSE.

   To have all settings applied reset after PDF has been generated set the bReset
   parameter to TRUE instead of FALSE (does NOT reset the License code).

   Return value:
   0 - success
   1 - invalid parameter
   2 - output pdf could not be opened
*/

#endif
