% withesis.cls
% LaTeX Style file for the University of Wisconsin-Madison Thesis Format
% Adapted from the Purdue University Thesis Format
% Originally by Dave Kraynie
% Edits by Darrell McCauley
% Adapted to UW-Madison format by Eric Benedict  (Noted with <EB>)
% Updated to LaTeX2e by Eric Benedict 24 July 00
% 
%=============================================================================
% Licensed under the Perl Artistic License.
% see: http://www.ctan.org/tex-archive/help/Catalogue/licenses.artistic.html
% for more info...
%=============================================================================
\NeedsTeXFormat{LaTeX2e}
\ProvidesClass{withesis}[2000/07/24 Univ. of Wisconsin-Madison Thesis Format]
\typeout{Document Class `withesis' v2.0 <24 July 00>}


%=============================================================================
%  Set up the paper size
%=============================================================================
\setlength\paperheight{11in}
\setlength\paperwidth{8.5in}


%=============================================================================
%  The type size option is handled by reading a different file for each
%  size, as follows, to define font size-specific commands:
%       10pt : withe10, 11pt : withe11, 12pt : withe12
%  Performed by defining \@ptsize to hold the last digit
%=============================================================================
\newcommand\@ptsize{}
\DeclareOption{10pt}{\renewcommand\@ptsize{0}}  % 10pt option reads in withe10.STY
\DeclareOption{11pt}{\renewcommand\@ptsize{2}  % 10pt option reads in withe10.STY
     \typeout{11pt not available, subsituting 12pt}}   % 11pt option reads in withe12.STY
\DeclareOption{12pt}{\renewcommand\@ptsize{2}}   % 12pt option reads in withe12.STY


%=============================================================================
% Two-side or one-side printing.
% \@twosidefalse                      % Default is one-sided printing.
%=============================================================================
\DeclareOption{twoside}{\@twosidetrue % Defines twoside option.
           \@mparswitchtrue}          % Marginpars go on outside of page.

%=============================================================================
%  Margin Check option
% \overfullrule = 0pt                 % Default is don't mark overfull hboxes.
%=============================================================================
\DeclareOption{margincheck}{\overfullrule 5pt}  % Causes overfull hboxes to be marked.


%=============================================================================
%  Masters Thesis margin option
%  \@msmarginfalse                    % Default is 1" margins all around
%  The margins must be modified in the \documentstyle because they will have
%  already been set by the time the pre-amble is processed.  The \@msmargin
%  flag is also used to change the defaults for the \thesis and \degree
%=============================================================================
\newif\if@msmargin                % Creates switch, default is false
\DeclareOption{msthesis}{\@msmargintrue       % Defines 1.5" left margins
      \typeout{Using UW Master's Thesis Margins and Defaults}}



%=============================================================================
% The \ProcessOptions\relax command causes the execution of every command
% which is defined with \DeclareOption{FOO} and for which the user typed the
% FOO option in his \documentclass command.  For every option BAR he typed for
% which BAR is not defined, the file BAR.sty will be read after the present
% (main) .STY file is executed.
%=============================================================================
\ExecuteOptions{10pt}
\ProcessOptions\relax
\RequirePackage{withe1\@ptsize}



%=============================================================================
%  Set up the font definition a la Standard LaTeX
%
%=============================================================================
\DeclareOldFontCommand{\rm}{\normalfont\rmfamily}{\mathrm}
\DeclareOldFontCommand{\sf}{\normalfont\sffamily}{\mathsf}
\DeclareOldFontCommand{\tt}{\normalfont\ttfamily}{\mathtt}
\DeclareOldFontCommand{\bf}{\normalfont\bfseries}{\mathbf}
\DeclareOldFontCommand{\it}{\normalfont\itshape}{\mathit}
\DeclareOldFontCommand{\sl}{\normalfont\slshape}{\@nomath\sl}
\DeclareOldFontCommand{\sc}{\normalfont\scshape}{\@nomath\sc}






%=============================================================================
% LISTS
%=============================================================================

%=============================================================================
% ENUMERATE
%  Enumeration is done with four counters: enumi, enumii, enumiii
%  and enumiv, where enumN controls the numbering of the Nth level
%  enumeration.  The label is generated by the commands \labelenumi
%  ... \labelenumiv.  The expansion of \p@enumN\theenumN defines the
%  output of a \ref command.
%=============================================================================
\def\labelenumi{\arabic{enumi}.}
\def\theenumi{\arabic{enumi}}
\def\labelenumii{(\alph{enumii})}
\def\theenumii{\alph{enumii}}
\def\p@enumii{\theenumi}
\def\labelenumiii{\roman{enumiii}.}
\def\theenumiii{\roman{enumiii}}
\def\p@enumiii{\theenumi(\theenumii)}
\def\labelenumiv{\Alph{enumiv}.}
\def\theenumiv{\Alph{enumiv}}
\def\p@enumiv{\p@enumiii\theenumiii}

%=============================================================================
% ITEMIZE
% Itemization is controlled by four commands: \labelitemi, \labelitemii,
% \labelitemiii, and \labelitemiv, which define the labels of the various
% itemization levels.
%=============================================================================
\def\labelitemi{$\bullet$}
\def\labelitemii{\bface --}
\def\labelitemiii{$\ast$}
\def\labelitemiv{$\cdot$}

%=============================================================================
% VERSE
%   The verse environment is defined by making clever use of the
%   list environment's parameters.  The user types \\ to end a line.
%   This is implemented by \let'in \\ equal \@centercr.
%=============================================================================
\def\verse{\let\\=\@centercr
  \list{}{\itemsep\z@ \itemindent -1.5em\listparindent \itemindent
          \rightmargin\leftmargin\advance\leftmargin 1.5em}\item[]}
\let\endverse=\endlist

%=============================================================================
% QUOTATION
%   Fills lines; Indents paragraph
%=============================================================================
\def\quotation{\list{}{\listparindent 1.5em
  \itemindent\listparindent
  \rightmargin\leftmargin\parsep 0pt plus 1pt}\item[]}
\let\endquotation=\endlist

%=============================================================================
% QUOTE -- same as quotation except no paragraph indentation,
%=============================================================================
\def\quote{\list{}{\rightmargin\leftmargin}\item[]}
\let\endquote=\endlist

%=============================================================================
% DESCRIPTION
%   To change the formatting of the label, redefine \descriptionlabel.
%=============================================================================
\def\descriptionlabel#1{\hspace\labelsep \bface #1}
\def\description{\list{}{\labelwidth\z@ \itemindent-\leftmargin
  \let\makelabel\descriptionlabel}}
\let\enddescription\endlist
\newdimen\descriptionmargin
\descriptionmargin=3em

%=============================================================================
% OTHER ENVIRONMENTS
%=============================================================================

%=============================================================================
% THEOREM
% \@begintheorem ... \@endtheorem are the commands executed at the
% beginning and end of a (user-defined) theorem-like environment.
% Except \@opargbegintheorem is executed when an optional argument is
% given.  Cf. LATEX.TEX.
%=============================================================================
\def\@begintheorem#1#2{\trivlist \item[\hskip \labelsep{\bface #1\ #2}]}
\def\@opargbegintheorem#1#2#3{\trivlist
  \item[\hskip \labelsep{\bface #1\ #2\ (#3)}]}
\def\@endtheorem{\endtrivlist}





%=============================================================================
% TITLEPAGE
%  In the normal environments, the titlepage environment does nothing but
%  start and end a page, and inhibit page numbers.  It also resets the page
%  number to zero.  In two-column style, it still makes a one-column page.
%=============================================================================
\newenvironment{titlepage}
 {\@restonecolfalse\if@twocolumn\@restonecoltrue\onecolumn
  \else \newpage \fi \thispagestyle{empty}
% \c@page\z@ -- deleted: count title page in thesis
}{\if@restonecol\twocolumn \else \newpage \fi}

%=============================================================================
% ARRAY AND TABULAR
%=============================================================================
\arraycolsep 5pt     % Half the space between columns in an array environment.
\tabcolsep 6pt       % Half the space between columns in a tabular environment.
\arrayrulewidth .4pt % Width of rules in array and tabular environment.
\doublerulesep 2pt   % Space between adjacent rules in array or tabular env.

%=============================================================================
% TABBING
%=============================================================================
\tabbingsep \labelsep   % Space used by the \' command.  (See LaTeX manual.)

%=============================================================================
% MINIPAGE
%  \@minipagerestore is called upon entry to a minipage environment to
%  set up things that are to be handled differently inside a minipage
%  environment. In the current styles, it does nothing.
%
% \skip\@mpfootins : plays same role for footnotes in a minipage as
%                    \skip\footins does for ordinary footnotes
%=============================================================================
\skip\@mpfootins = \skip\footins

%=============================================================================
% FRAMEBOX
%=============================================================================
\fboxsep = 3pt    % Space left between box and text by \fbox and \framebox.
\fboxrule = .4pt  % Width of rules in box made by \fbox and \framebox.

%=============================================================================
% CHAPTERS AND SECTIONS
%=============================================================================
% DEFINE COUNTERS:
% \newcounter{NEWCTR}[OLDCTR] : Defines NEWCTR to be a counter, which is
%                               reset to zero when counter OLDCTR is stepped.
%                               Counter OLDCTR must already be defined.
%=============================================================================
\newcounter {part}
\newcounter {chapter}
\newcounter {section}[chapter]
\newcounter {subsection}[section]
\newcounter {subsubsection}[subsection]
\newcounter {paragraph}[subsubsection]
\newcounter {subparagraph}[paragraph]
\newcounter {tocpage}
\newcounter {lofpage}
\newcounter {lotpage}
\newcounter {listofheading}

%=============================================================================
% For any counter CTR, \theCTR is a macro that defines the printed version
% of counter CTR.  It is defined in terms of the following macros:
%
%  \arabic{COUNTER} : The value of COUNTER printed as an arabic numeral.
%  \roman{COUNTER}  : Its value printed as a lower-case roman numberal.
%  \Roman{COUNTER}  : Its value printed as an upper-case roman numberal.
%  \alph{COUNTER}   : Value of COUNTER printed as a lower-case letter:
%                         1 = a, 2 = b, etc.
%  \Alph{COUNTER}   : Value of COUNTER printed as an upper-case letter:
%                           1 = A, 2 = B, etc.
%=============================================================================
\def\thepart          {\Roman{part}}
\def\thechapter       {\arabic{chapter}}
\def\thesection       {\thechapter.\arabic{section}}
\def\thesubsection    {\thesection.\arabic{subsection}}
\def\thesubsubsection {\thesubsection.\arabic{subsubsection}}
\def\theparagraph     {\thesubsubsection.\arabic{paragraph}}
\def\thesubparagraph  {\theparagraph.\arabic{subparagraph}}

%=============================================================================
% TABLE OF CONTENTS, ETC.
%=============================================================================
% A \subsection command writes a
%       \contentsline{subsection}{TITLE}{PAGE}
% command on the .toc file, where TITLE contains the contents of the
% entry and PAGE is the page number.  If subsections are being numbered,
% then TITLE will be of the form
%       \numberline{NUM}{HEADING}
% where NUM is the number produced by \thesubsection.  Other sectioning
% commands work similarly.
%
% A \caption command in a 'figure' environment writes
%    \contentsline{figure}{\numberline{NUM}{CAPTION}}{PAGE}
% on the .lof file, where NUM is the number produced by \thefigure and
% CAPTION is the figure caption.  It works similarly for a 'table' environment.
%
% The command \contentsline{NAME} expands to \l@NAME.  So, to specify
% the table of contents, we must define \l@chapter, \l@section,
% \l@subsection, ... ; to specify the list of figures, we must define
% \l@figure; and so on.  Most of these can be defined with the
% \@dottedtocline command, which works as follows.
%
% \@dottedtocline{LEVEL}{INDENT}{NUMWIDTH}{TITLE}{PAGE}
%    LEVEL    : An entry is produced only if LEVEL < or = value of
%               'tocdepth' counter.  Note, \chapter is level 0, \section
%               is level 1, etc.
%    INDENT   : The indentation from the outer left margin of the start of
%               the contents line.
%    NUMWIDTH : The width of a box in which the section number is to go,
%               if TITLE includes a \numberline command.
%
% This command uses the following three parameters, which are set
% with a \def (so em's can be used to make them depend upon the font).
%   \@pnumwidth : The width of a box in which the page number is put.
%   \@tocrmarg  : The right margin for multiple line entries.  One
%                 wants \@tocrmarg > or = \@pnumwidth
%   \@dotsep    : Separation between dots, in mu units.  Should be \def'd to
%                 a number like 2 or 1.7
%=============================================================================
\def\@pnumwidth{1.55em}
\def\@tocrmarg {2.55em}
\def\@dotsep{4.5}
\setcounter{tocdepth}{2}

%=============================================================================
% The value of \@tocchapterskip when the table of contents is
% formatted is the only value that matters.  Changing this value on
% the fly probably won't do what you want.  Re-defining \l@chapter has
% the same problem.  The only way around the problem I've thought of
% is to write the spacing commands directly into the toc file from the
% chapter command, as is done by \appendix, for instance.
%=============================================================================
\def\@tocchapterskip{1.0em plus 1pt}

\def\l@part#1#2{\addpenalty{-\@highpenalty}
  \addvspace{2.25em plus 1pt}  % space above part line
  \begingroup
  \@tempdima 3em         % width of box holding part number, used by
    \parindent \z@ \rightskip \@pnumwidth             %% \numberline
    \parfillskip -\@pnumwidth
    {\large \bface       % set line in \large boldface
    \leavevmode          % TeX command to enter horizontal mode.
    #1\hfil \hbox to\@pnumwidth{\hss #2}}\par
    \nobreak             % Never break after part entry
  \endgroup}

\def\l@chapter#1#2{\pagebreak[3]
  \vskip\@tocchapterskip % space above chapter line
  \@dottedtocline{0}{0pt}{1.5em}{{\bface #1}}{#2}}

\def\l@section{\@dottedtocline{1}{1.5em}{2.3em}}
\def\l@subsection{\@dottedtocline{2}{3.8em}{3.2em}}
\def\l@subsubsection{\@dottedtocline{3}{7.0em}{4.1em}}
\def\l@paragraph{\@dottedtocline{4}{10em}{5em}}
\def\l@subparagraph{\@dottedtocline{5}{12em}{6em}}
\def\l@figure{\vskip 1.0em plus 1pt \@dottedtocline{0}{0em}{2.5em}}
\let\l@table\l@figure

%=============================================================================
% TABLE OF CONTENTS; LIST OF FIGURES; LIST OF TABLES
%=============================================================================
% In report style, \tableofcontents, \listoffigures, etc. are always
% set in single-column style.  @restonecol is used to keep track of
% whether we need to switch back to double column style after the toc.
%
% The only known problem now is that the first page with the new
% layout is too long.  The problem seems to be that the change to
% textheight doesn't take place on the first page.  Even if it's the
% first line in the table of contents macro.  Presumably the same
% problem also occurs in the lof and lot.
%
% I'm taking a shot at fixing the problem by dropping in a throw-away
% page between the change to the height parameters and the start of
% the chapter.  Isn't elegance wonderful?
%
%=============================================================================
\def\tableofcontents{\@tableof{TABLE OF CONTENTS}{}{tocpage}{toc}{}}

\def\listoffigures{
  \@tableof{LIST OF FIGURES}{Figure}{lofpage}{lof}
  {\protect\addcontentsline{toc}{chapter}{LIST OF FIGURES}}}

\def\listoftables{
  \@tableof{LIST OF TABLES}{Table}{lotpage}{lot}
  {\protect\addcontentsline{toc}{chapter}{LIST OF TABLES}}}

\def\@tableof#1#2#3#4#5{
{ % limit scope of following declarations!!
  \@restonecolfalse\if@twocolumn\@restonecoltrue\onecolumn\fi
  \addtolength{\textheight}{-40pt}       % -24-16
  \addtolength{\majorheadskip}{-40pt}    % -24-16
  \addtolength{\headheight}{52pt}        %  36+16
  \addtolength{\headsep}{-12pt}          % -12
  \separatorpage{DISCARD THIS PAGE}
  \chapter*{#1}
  #5
  \relax\markboth{#1}{#1}
  \hbox to \hsize{#2 \hfil Page}
  \singlespace
  \setcounter{#3}{0}
  \setcounter{listofheading}{1}  % change from 0 to 1 by mccauley, 14may93
  \def\@oddhead{\vbox to \headheight{\vspace{4pt}
    \hbox to \hsize{\hfil\rm\thepage} \vfil
    \ifnum\value{#3}=1
      \ifnum\value{listofheading}=2
        \hbox to \hsize{Appendix\hfil} \vspace{4pt} \fi
      \ifnum\value{listofheading}=1
        \stepcounter{listofheading} \fi
      \hbox to \hsize{#2 \hfil Page}
    \else
      \setcounter{#3}{1}
    \fi}}
  \def\@evenhead{\vbox to \headheight{\vspace{4pt}
    \hbox to \hsize{\rm\thepage\hfil} \vfil
    \ifnum\value{#3}=1
      \ifnum\value{listofheading}=2
        \hbox to \hsize{Appendix\hfil} \vspace{4pt} \fi
      \ifnum\value{listofheading}=1
        \stepcounter{listofheading} \fi
      \hbox to \hsize{#2 \hfil Page}
    \else
      \setcounter{#3}{1}
    \fi}}
  \@starttoc{#4}  \if@restonecol\twocolumn\fi
  \newpage
}}

%=============================================================================
% BIBLIOGRAPHY
%=============================================================================
% The thebibliography environment executes the following commands:
%
%  o start a new 'chapter' with BIBLIOGRAPHY as the heading
%  o produce a separator page for the bibliography
%
%  \def\newblock{\hskip .11em plus .33em minus -.07em} --
%      Defines the `closed' format, where the blocks (major units of
%      information) of an entry run together.
%
%  \sloppy  -- Used because it's rather hard to do line breaks in
%      bibliographies,
%
%  \sfcode`\.=1000\relax --
%      Causes a `.' (period) not to produce an end-of-sentence space.
%=============================================================================
% \altbibtitle
%   The default title for the References chapter is ``LIST OF REFERENCES''
%   Since some people prefer ``BIBLIOGRAPHY'', the command
%   \altbibtitle has been added to change the chapter title.
%   This command does nothing more than change REFERENCES to BIBLIOGRAPHY
%============================================================================
\def\@bibchaptitle{LIST OF REFERENCES}
\def\altbibtitle{\def\@bibchaptitle{BIBLIOGRAPHY}}
\def\thebibliography#1{
  %\separatorpage{\@bibchaptitle}
  \global\@bibpresenttrue
  \chapter*{\@bibchaptitle\markboth{\@bibchaptitle}{\@bibchaptitle}}
  \addcontentsline{toc}{chapter}{\@bibchaptitle}
  \vspace{0.375in}    % added to match 4 line requirement
  \interlinepenalty=10000 % added to prevent breaking of bib entries
  \singlespace\list
  {[\arabic{enumi}]}{\settowidth\labelwidth{[#1]}\leftmargin\labelwidth
    \advance\leftmargin\labelsep \usecounter{enumi}}
  \def\newblock{\hskip .11em plus .33em minus -.07em}
  \sloppy
  \sfcode`\.=1000\relax}
\let\endthebibliography=\endlist

%=============================================================================
% \def\@biblabel#1{[#1]\hfill}  % Produces the label for \bibitem[...]
% \def\@cite#1{[#1]}            % Produces the output of the \cite command.
%=============================================================================

%=============================================================================
% THEINDEX ENVIRONMENT
%   Produces double column format, with each paragraph a separate entry.
%   The user commands \item, \subitem and \subsubitem are used to
%   produce the entries, and \indexspace adds an extra vertical space
%   that's the right size to put above the first entry with a new letter
%   of the alphabet.
%=============================================================================
\newif\if@restonecol
\def\theindex{\@restonecoltrue\if@twocolumn\@restonecolfalse\fi
  \columnseprule \z@
  \columnsep 35pt\twocolumn[\@makeschapterhead{Index}]
  \markboth{INDEX}{INDEX}\thispagestyle{plain}\parindent\z@
  \parskip\z@ plus .3pt\relax\let\item\@idxitem}
\def\@idxitem{\par\hangindent 40pt}
\def\subitem{\par\hangindent 40pt \hspace*{20pt}}
\def\subsubitem{\par\hangindent 40pt \hspace*{30pt}}
\def\endtheindex{\if@restonecol\onecolumn\else\clearpage\fi}
\def\indexspace{\par \vskip 10pt plus 5pt minus 3pt\relax}

%=============================================================================
% FOOTNOTES
%=============================================================================
% \footnoterule is a macro to draw the rule separating the footnotes from
% the text.  It should take zero vertical space, so it needs a negative
% skip to compensate for any positive space taken by the rule.  (See
% PLAIN.TEX.)
%=============================================================================
\def\footnoterule{\kern-3\p@
  \hrule width .4\columnwidth
  \kern 2.6\p@}                 % The \hrule has default height of .4pt .
% \newcounter{footnote}
\@addtoreset{footnote}{chapter}  % Numbers footnotes within chapters

%=============================================================================
% \@makefntext{NOTE} :
%   Must produce the actual footnote, using \@thefnmark as the mark
%   of the footnote and NOTE as the text.  It is called when effectively
%   inside a \parbox of width \columnwidth (i.e., with \hsize =
%   \columnwidth).
%
%   The following macro indents all lines of the footnote by 10pt, and
%   indents the first line of a new paragraph by 1em.  To change these
%   dimensions, just substitute the desired value for '10pt' [in both
%   places] or '1em'.  The mark is flushright against the footnote.
%     \long\def\@makefntext#1{\@setpar{\@@par\@tempdima \hsize
%        \advance\@tempdima-10pt\parshape \@ne 10pt \@tempdima}\par
%        \parindent 1em\noindent \hbox to \z@{\hss$^{\@thefnmark}$}#1}
%
%   A simpler macro is used, in which the footnote text is
%   set like an ordinary text paragraph, with no indentation except
%   on the first line of a paragraph, and the first line of the
%   footnote.  Thus, all the macro must do is set \parindent
%   to the appropriate value for succeeding paragraphs and put the
%   proper indentation before mark.
%=============================================================================
\long\def\@makefntext#1{\parindent 1em\noindent
  \hbox to 1.8em{\hss$^{\@thefnmark}$}\singlespace\footnotesize#1}

%=============================================================================
% \@makefnmark :
%   Macro to generate the footnote marker that goes in the text. Default used.
%=============================================================================

%=============================================================================
% FIGURES AND TABLES
%=============================================================================
% Float placement parameters.  See LaTeX manual for their definition.
%
% Note: some really flaky hiccups can happen because of interactions between
% these numbers.  For instance, if \textfraction and \floatpagefraction
% don't have any overlap figures can be drawn that are too large to fit
% on a text page and too small to fit on a float page.  This problem causes
% all of the figures to go to the end of the chapter.
%=============================================================================
\setcounter{topnumber}{2}
\def\topfraction{.5}
\setcounter{bottomnumber}{1}
\def\bottomfraction{.3}
\setcounter{totalnumber}{3}
\def\textfraction{.5}     % previously .55
\def\floatpagefraction{.3}
\setcounter{dbltopnumber}{2}
\def\dbltopfraction{.7}
\def\dblfloatpagefraction{.5}

%=============================================================================
% \@makecaption{NUMBER}{TEXT} : Macro to make a figure or table caption.
%      NUMBER : Figure or table number--e.g., 'Figure 3.2'
%      TEXT   : The caption text.
%  Macro should be called inside a \parbox of right width, with \normalsize.
%=============================================================================
\long\def\@makecaption#1#2{
  \vskip 10pt
  \singlespace\centering
  #1\ \ #2}

%=============================================================================
% ORIGINAL \@makecaption - Centering OR Flushleft if long.
% Thesis office doesn't really like this version.
%=============================================================================
% \long\def\@makecaption#1#2{
%   \vskip 10pt
%   \setbox\@tempboxa\hbox{#1\ \ #2}
%   \ifdim \wd\@tempboxa >\hsize           % IF   longer than one line:
%     {\singlespace \unhbox\@tempboxa\par} % THEN set as ordinary paragraph.
%   \else                                  % ELSE center.
%        \hbox to\hsize{\hfil\box\@tempboxa\hfil}
%   \fi}
%=============================================================================

%=============================================================================
% NEW \@makecaption - Thesis Format may not like the new format
%=============================================================================
% \newlength\@capnumlen
% \newlength\@captionlen
% \long\def\@makecaption#1#2{
%   \vskip 10pt
%   \setbox\@tempboxa\hbox{#1\ \ #2}
%   \ifdim \wd\@tempboxa >\hsize           % IF   longer than one line:
%   { \parindent 0pt
%     \singlespace\normalsize
%     \setlength{\@captionlen}{\hsize}
%     \settowidth{\@capnumlen}{#1\ \ }
%     \addtolength{\@captionlen}{-\@capnumlen}
%     #1\ \ \parbox[t]{\@captionlen}{\noindent#2}
%     \par\nobreak}                        % TeX penalty to prevent page break.
%   \else                                  % ELSE center.
%        \hbox to\hsize{\hfil\box\@tempboxa\hfil}
%   \fi}
%=============================================================================

%=============================================================================
% To define a float of type TYPE (e.g., TYPE = figure), the document style
% must define the following.
%
%  \fps@TYPE   : The default placement specifier for floats of type TYPE.
%
%  \ftype@TYPE : The type number for floats of type TYPE.  Each TYPE has
%                associated a unique positive TYPE NUMBER, which is a power
%                of two.  E.g., figures might have type number 1, tables type
%                number 2, programs type number 4, etc.
%
%  \ext@TYPE   : The file extension indicating the file on which the
%                contents list for float type TYPE is stored.  For example,
%                \ext@figure = 'lof'.
%
%  \fnum@TYPE  : A macro to generate the figure number for a caption.
%                For example, \fnum@TYPE == Figure \thefigure.
%
%  The actual float-making environment commands--e.g., the commands
%  \figure and \endfigure--are defined in terms of the macros \@float
%  and \end@float, which are described below.
%
%  \@float{TYPE}[PLACEMENT] : Macro to begin a float environment for a
%     single-column float of type TYPE with PLACEMENT as the placement
%     specifier.  The default value of PLACEMENT is defined by \fps@TYPE.
%     The environment is ended by \end@float.
%     E.g., \figure == \@float{figure}, \endfigure == \end@float.
%=============================================================================

%=============================================================================
% FIGURE
%=============================================================================
\newcounter{figure}[chapter]
\def\thefigure{\thechapter.\@arabic\c@figure}
\def\fps@figure{tbp}
\def\ftype@figure{1}
\def\ext@figure{lof}
\def\fnum@figure{Figure \thefigure}
\def\figure{\@float{figure}}
\let\endfigure\end@float
\@namedef{figure*}{\@dblfloat{figure}}
\@namedef{endfigure*}{\end@dblfloat}

%=============================================================================
% TABLE
%=============================================================================
\newcounter{table}[chapter]
\def\thetable{\thechapter.\@arabic\c@table}
\def\fps@table{tbp}
\def\ftype@table{2}
\def\ext@table{lot}
\def\fnum@table{Table \thetable}
\def\table{\@float{table}}
\let\endtable\end@float
\@namedef{table*}{\@dblfloat{table}}
\@namedef{endtable*}{\end@dblfloat}

%=============================================================================
% TITLE PAGE
%   - define \title{} \author{} \date{}
%   - The default degree is ``Doctor of Philosophy''
%     Degree can be changed using the command \degree{}
%   - for a masters project report, specify \project
%   - for a preliminary report, specify \prelim
%   - The default department is ``Electrical Engineering''
%     The department can be changed using the command \department{}
%   - once the above are defined, use \maketitle to generate the titlepage
%=============================================================================
\newcommand\@thesistitlemedskip{0.25in}
\newcommand\@thesistitlebigskip{0.6in}
\newcommand{\degree}[1]{\gdef\@degree{#1}}
\newcommand{\project}{\gdef\@doctype{A masters project report}}
\newcommand{\prelim}{\gdef\@doctype{A preliminary report}}
\newcommand{\thesis}{\gdef\@doctype{A thesis}}
\newcommand{\dissertation}{\gdef\@doctype{A dissertation}}
\newcommand{\department}[1]{\gdef\@department{(#1)}}

\if@msmargin    % Set Defaults based on Margin definition
   \gdef\@degree{Master of Science}    % Default is MS
   \gdef\@doctype{A thesis}            % Default is Thesis
\else
   \gdef\@degree{Doctor of Philosophy}    %Default is PhD
   \gdef\@doctype{A dissertation}         %Default is dissertation
\fi

\gdef\@department{(Electrical Engineering)} % Default is Electical Engineering

\newcommand{\maketitle}{%
  \begin{titlepage}
%-----------------------------------------------------------------------------
% -- The thesis office doesn't like thanks on title page.  Put it in
% -- the acknowledgments.  This is here so you don't have to change
% -- your titlepage when converting from report style. -> from Purdue, but I
%        left it here since it seems compatible with UW-Madison, Eric
%-----------------------------------------------------------------------------
    \def\thanks##1{\typeout{Warning: `thanks' deleted from thesis titlepage.}}
    \let\footnotesize\small \let\footnoterule\relax \setcounter{page}{1}
    \vspace*{0.1in}
    \begin{center}
      {\bf\expandafter\uppercase\expandafter{\@title}} \\[\@thesistitlebigskip]
       by \\[\@thesistitlemedskip]
      \@author \\[\@thesistitlebigskip]
      \@doctype\ submitted in partial fulfillment of \\
      the requirements for the degree of\\[\@thesistitlebigskip]
      \@degree \\[\@thesistitlemedskip]
      \@department \\[\@thesistitlebigskip]
      at the \\[\@thesistitlebigskip]
      UNIVERSITY OF WISCONSIN--MADISON\\[\@thesistitlebigskip]
      \@date
    \end{center}
  \end{titlepage}
  \setcounter{footnote}{0}
  \setcounter{page}{1} %title page is NOT counted
  \let\thanks\relax
  \let\maketitle\relax \let\degree\relax \let\project\relax \let\prelim\relax
  \let\department\relax
  \gdef\@thanks{}\gdef\@degree{}\gdef\@doctype{}
  \gdef\@department{}
  %\gdef\@author{}\gdef\@title{}
}

%=============================================================================
% ABSTRACT
%=============================================================================
% The abstract should begin with two single-spaced lines describing
% the author and title in a standard format.  After these lines comes
% the standard abstract.
%=============================================================================
\def\abstract{
  \chapter*{ABSTRACT}
  \addcontentsline{toc}{chapter}{ABSTRACT}
  \relax\markboth{ABSTRACT}{ABSTRACT}}
\def\endabstract{\par\newpage}


%=============================================================================
% UMI ABSTRACT
%=============================================================================
% The UMI abstract should begin with the author and title in a standard format.
% After the author comes the advisor and university. After these lines comes
% a bunch of double spaced text to make up the standard abstract.
% After the abstract, the advisor's approval signature follows.
% This page is not numbered and is delivered seperately to the thesis office.
%=============================================================================

\def\advisortitle#1{\gdef\@advisortitle{#1}}
\def\advisorname#1{\gdef\@advisorname{#1}}
\gdef\@advisortitle{Professor}
\gdef\@advisorname{Cheer E.\ Place}

\def\umiabstract{
             \thispagestyle{empty}
                  \addtocounter{page}{-1}
                \begin{center}
                  {\bf\expandafter\uppercase\expandafter{\@title}}\\
                  \vspace{12pt}
                  \@author \\
                  \vspace{12pt}
                  Under the supervision of \@advisortitle\ \@advisorname\\
                  At the University of Wisconsin-Madison
                \end{center}
}

\def\endumiabstract{\vfill \hfill\@advisorname\par\newpage}



%=============================================================================
% PAGE STYLES
%=============================================================================
% The page style 'foo' is defined by defining the command \ps@foo.  This
% command should make only local definitions.  There should be no stray
% spaces in the definition, since they could lead to mysterious extra
% spaces in the output.
%
% The \ps@... command defines the macros \@oddhead, \@oddfoot,
% \@evenhead, and \@evenfoot to define the running heads and
% feet---e.g., \@oddhead is the macro to produce the contents of the
% heading box for odd-numbered pages.  It is called inside an \hbox of
% width \textwidth.
%
% To make headings determined by the sectioning commands, the page style
% defines the commands \chaptermark, \sectionmark, ... , where
% \chaptermark{TEXT} is called by \chapter to set a mark, and so on.
% The \...mark commands and the \...head macros are defined with the
% help of the following macros.  (All the \...mark commands should be
% initialized to no-ops.)
%
% MARKING CONVENTIONS:
% LaTeX extends TeX's \mark facility by producing two kinds of marks
% a 'left' and a 'right' mark, using the following commands:
%     \markboth{LEFT}{RIGHT} : Adds both marks.
%     \markright{RIGHT}      : Adds a 'right' mark.
%     \leftmark  : Used in the \@oddhead, \@oddfoot, \@evenhead or \@evenfoot
%                  macro, gets the current 'left'  mark.  Works like TeX's
%                  \botmark command.
%     \rightmark : Used in the \@oddhead, \@oddfoot, \@evenhead or \@evenfoot
%                  macro, gets the current 'right'  mark.  Works like TeX's
%                  \firstmark command.
% The marking commands work reasonably well for right marks 'numbered
% within' left marks--e.g., the left mark is changed by a \chapter command and
% the right mark is changed by a \section command.  However, it does
% produce somewhat anomalous results if two \bothmark's occur on the same page.
%=============================================================================
\mark{{}{}}   % Initializes TeX's marks

%=============================================================================
% Definition of 'headings' page style
%  Note use of ##1 for parameter of \def\chaptermark inside \def\ps@headings.
%=============================================================================
\if@twoside                                                  % If two-sided
  \def\ps@headings{
    \def\@oddfoot{}\def\@evenfoot{}         % No feet.
    \def\@evenhead{\rm \thepage\hfil \sl \leftmark}          % Left heading.
    \def\@oddhead{\hbox{}\sl \rightmark \hfil \rm\thepage}   % Right heading.
    \def\chaptermark##1{\markboth {\uppercase{\ifnum \c@secnumdepth >\m@ne
      \@chapapp\ \thechapter. \ \fi ##1}}{}}%
    \def\sectionmark##1{\markright {\uppercase{\ifnum \c@secnumdepth >\z@
      \thesection. \ \fi ##1}}}}
\else                                                        % If one-sided
  \def\ps@headings{
    \def\@oddfoot{}\def\@evenfoot{}                          % No feet.
    \def\@oddhead{\hbox {}\sl \rightmark \hfil \rm\thepage}  % Heading.
    \def\chaptermark##1{\markright {\uppercase{\ifnum \c@secnumdepth >\m@ne
      \@chapapp\ \thechapter. \ \fi ##1}}}}
\fi

%=============================================================================
% Definition of 'thesis' page style -- riedl
%  -- no footer
%  -- header is just page number, in upper right corner
%  -- chaptermark and sectionmark are not needed
%  Note use of ##1 for parameter of \def\chaptermark inside \def\ps@headings.
%=============================================================================
\if@twoside                                                  % If two-sided
  \def\ps@thesis{
    \def\@oddfoot{}\def\@evenfoot{}                          % No feet.
    \def\@evenhead{\rm \thepage\hfil}                        % Left heading.
    \def\@oddhead{\hbox{}\hfil\rm\thepage}                   % Right heading.
    \def\chaptermark##1{}
    \def\sectionmark##1{}}
\else                                                        % If one-sided
  \def\ps@thesis{
    \def\@oddfoot{}\def\@evenfoot{}                          % No feet.
    \def\@oddhead{\hbox{}\hfil\rm\thepage}                   % Heading.
    \def\chaptermark##1{}}
\fi

%=============================================================================
% The following code sets \@currentipfile to current include file.
% taken from chapterbib.sty by
% Niel Kempson (JANET: rmcs-tex@uk.ac.cranfield.cdvc)

\let\@debugoutput=\@gobble

%-------------------------------------------------------------------------
% Keep track of files currently opened.
% 1. The counter @ipfilectr keeps track of the currently open I/P file,
%    and @includefilectr keeps track of the \include files.
% 2. @ipfilectr is set to @includefilectr, except when processing the root
%    file. An arbitrary large (number greater than the maximum number of
%    include files) is used to denote the root file.
%-------------------------------------------------------------------------
\newcounter{@ipfilectr}                 % current I/P file,
\def\the@ipfilectr{\roman{@ipfilectr}}  % displayed as roman numerals
\newcounter{@includefilectr}            % current \include file
\def\@rootfilevalue{1988}               % arbitrary value for root file
\def\savedjobname{\jobname}             % save root file name as a macro

%
%-----------------------------------------------------------------------
% The name of the current I/P file is stored in macro \@currentipfile
% and the global macro ipfile@the@ipfilectr is set to this name whenever
% \@currentipfile changes.
%-----------------------------------------------------------------------
\def\@currentipfile{\jobname}           % initialize to root file
\setcounter{@ipfilectr}{\@rootfilevalue}        % ditto
\global\@namedef{ipfile@\the@ipfilectr}{\@currentipfile}

%
%-----------------------------------------------------------------------
% Redefine the \include macro (taken from LATEX.TEX of 25-JAN-1988) so
% that when a file is \include'd
% 1. \@currentipfile is updated and \includefilectr incremented
% 2. ipfilectr set to includefilectr
% 3. global macro ipfile@\the@ipfilectr set to \@currentfile
%
% When the \include'd file is finished with
% 1. reset \@currentipfile to the root file name (\jobname)
% 2. reset ipfilectr to \@rootfilevalue
% 3. global macro ipfile@\the@ipfilectr set to \@currentfile
%-----------------------------------------------------------------------
\def\include#1{\clearpage
\def\@currentipfile{#1}
\@debugoutput{\string\@currentipfile = `\@currentipfile'}
\addtocounter{@includefilectr}{1}
\setcounter{@ipfilectr}{\value{@includefilectr}}
\global\@namedef{ipfile@\the@ipfilectr}{\@currentipfile}
%
\if@filesw \immediate\write\@mainaux{\string\@input{#1.aux}}\fi
\@tempswatrue\if@partsw \@tempswafalse\def\@tempb{#1}\@for
\@tempa:=\@partlist\do{\ifx\@tempa\@tempb\@tempswatrue\fi}\fi
\if@tempswa \if@filesw \let\@auxout=\@partaux
\immediate\openout\@partaux #1.aux
\immediate\write\@partaux{\relax}\fi\@input{#1.tex}\clearpage
\@writeckpt{#1}\if@filesw\immediate\closeout\@partaux \fi
\let\@auxout=\@mainaux\else\@nameuse{cp@#1}\fi
%
\def\@currentipfile{\jobname}
\@debugoutput{\string\@currentipfile = `\@currentipfile'}
\setcounter{@ipfilectr}{\@rootfilevalue}
\global\@namedef{ipfile@\the@ipfilectr}{\@currentipfile}}
%=============================================================================

%=============================================================================
% Definition of 'thesisdraft' page style -- riedl --dfk
%  -- header is just page number, in upper right corner
%  -- chaptermark and sectionmark are not needed
%  -- footer is set to identify this as a draft version (timestamped)
%  -- footer may be changed by changing \@draftfoot
%  Note use of ##1 for parameter of \def\chaptermark inside \def\ps@headings.
%=============================================================================
\newcounter{currenthour} \newcounter{currentminute}
\newcommand{\now}{
  \setcounter{currenthour}{\time}\divide \c@currenthour by 60
  \setcounter{currentminute}{\time}
    \multiply \c@currenthour by 60
    \addtocounter{currentminute}{-\c@currenthour}
    \divide \c@currenthour by 60
  \ifnum \c@currenthour=0 0 \fi    % force at least one digit in hour
  \thecurrenthour:%
  \ifnum \c@currentminute<10 0\fi % force two digits in minute
  \thecurrentminute
}

% changed by mccauley to print name of currently included file.
\def\@draftfooter{
  {\rm DRAFT: Do Not Distribute}\hfil{\now\ \today}
    \hfil{\tt \@currentipfile.tex}}

\if@twoside                                                  % If two-sided
  \def\ps@thesisdraft{
    \def\@oddfoot {\@draftfooter}                            % Footer
    \def\@evenfoot{\@draftfooter}
    \def\@evenhead{\rm \thepage\hfil}                        % Left heading.
    \def\@oddhead{\hbox{}\hfil\rm\thepage}                   % Right heading.
    \def\chaptermark##1{}
    \def\sectionmark##1{}}
\else                                                        % If one-sided
  \def\ps@thesisdraft{
    \def\@oddfoot {\@draftfooter}                            % Footer
    \def\@evenfoot{}
    \def\@oddhead{\hbox{}\hfil\rm\thepage}                   % Heading.
    \def\chaptermark##1{}}
\fi

%=============================================================================
% Definition of 'myheadings' page style.
%=============================================================================
\def\ps@myheadings{
  \def\@oddhead{\hbox{}\sl\rightmark \hfil \rm\thepage}
  \def\@oddfoot{}
  \def\@evenhead{\rm \thepage\hfil\sl\leftmark\hbox {}}
  \def\@evenfoot{}
  \def\sectionmark##1{}
  \def\subsectionmark##1{}}

%=============================================================================
% MISCELLANEOUS
%=============================================================================
%\def\mybox{\fbox{\hbox to 429bp {\vbox to 630bp {\null}}}}

%\def\pagebox{ \setlength{\unitlength}{1.0bp}
%             \begin{picture}(0,0)(36,558) \put(0,0){\mybox} \end{picture}
%              \begin{picture}(0,0)(18,523.25) \put(0,0){\mybox} \end{picture}
%              \begin{picture}(0,0)(18,500.5) \put(0,0){\mybox} \end{picture}
%           }


\def\draftmargins{
  \if@msmargin % Draw box for 1.25" left margins
      \special{!userdict begin /eop-hook{
            gsave 0.5 setlinewidth newpath
            90 72 moveto 540 72 lineto 540 720 lineto 90 720 lineto
            90 72 lineto
            stroke grestore
%        Cross at 0.5" down and 1" from right edge
%           gsave 0.5 setlinewidth newpath
%           522 756 moveto 558 756 lineto
%           540 738 moveto 540 774 lineto
            stroke grestore
           }def end
   }
  \else % Draw box for 1" margins
      \special{!userdict begin /eop-hook{
            gsave 0.5 setlinewidth newpath
            72 72 moveto 540 72 lineto 540 720 lineto 72 720 lineto
            72 72 lineto
            stroke grestore
%        Cross at 0.5" down and 1" from right edge
%           gsave 0.5 setlinewidth newpath
%           522 756 moveto 558 756 lineto
%           540 738 moveto 540 774 lineto
            stroke grestore
           }def end
   }

  \fi
}

\def\draftscreen{
   \special{!userdict begin /bop-hook{gsave
            260 185 translate
            65 rotate /Times-Roman findfont 144 scalefont setfont
            0 0 moveto 0.9 setgray (DRAFT) show grestore
            }def end
   }
}



%============================================================================
% VERBATIMFILE
%============================================================================
% \verbatimfile{<filename>}    for verbatim inclusion of a file
% - Note that the precise layout of line breaks in this file is important!
% - added the \singlespace - EB
%============================================================================
\def\verbatimfile#1{\begingroup \singlespace
                    \@verbatim \frenchspacing \@vobeyspaces
                    \input#1 \endgroup
}


%=============================================================================
% SEPARATOR Pages
%   Creates a blank page with a text centered horizontally and vertically.
%   The page is neither counted nor numbered.
%   These pages are required in the thesis format before sections such
%   as appendices, vita, bibliography, etc.
%=============================================================================
\def\separatorpage#1{
  \newpage
  \thispagestyle{empty}
  \addtocounter{page}{-1}
  \null
  \vfil\vfil
  \begin{center}
    {\bface #1}
  \end{center}
  \vfil\vfil
  \newpage}

%=============================================================================
% \@chapapp is initially defined to be 'Chapter'.  The \appendix
% command redefines it to be 'Appendix'.
%=============================================================================
\def\@chapapp{CHAPTER}

%=============================================================================
% APPENDIX
%=============================================================================
% The \appendix command must do the following:
%    -- reset the chapter counter to zero
%    -- set \@chapapp to Appendix (for messages)
%    -- redefine the chapter counter to produce appendix numbers
%    -- reset the section counter to zero
%    -- redefine the \chapter command if appendix titles and headings are
%       to look different from chapter titles and headings.
%    -- produce special toc, lot and lof entries
%=============================================================================
\def\appendix{
  \@appendix{Appendix}
  \def\@chapstyle{appendix}
  \def\@chapnumber{APPENDIX}
  \let\@chapterhead\@appendixchapterhead
  \let\@schapterhead\@appendixchapterhead}
\def\endappendix{
  \def\@chaptername##1{\uppercase {##1}}
  \def\@chapstyle{chapter}
  \def\@chapnumber{\thechapter.}
  \let\@chapterhead\@makechapterhead
  \let\@schapterhead\@makeschapterhead}

\def\l@appendix#1#2{\pagebreak[3]
  \vskip\@tocchapterskip % space above chapter line
  \@dottedtocline{0}{0pt}{7.3em}{{\bface #1}}{#2}}

\def\appendices{
  \addtocontents{toc}{
    \protect\addvspace{\@tocchapterskip}
    \hbox to \hsize{\bface APPENDICES \hfil}
    \protect\addvspace{\@tocchapterskip}}
  \@appendix{Appendices}
  \def\@chaptername##1{##1}
  \def\@chapstyle{appendices}
  \def\@chapnumber{Appendix\ \thechapter:}
  \let\@chapterhead\@appendiceschapterhead
  \let\@schapterhead\@appendiceschapterhead
  \if@bibpresent\relax\else\separatorpage{DISCARD THIS PAGE}\fi % This
                     % is a dirty fix.  For some reason, the TOC isn't
             % properly written unless the simple chapter with
             % sections is written.  The bibliography fixes this
             % problem when its present.  If its not, then the
             % seperator page is inserted to fix things.  EB
}

\let\endappendices\endappendix

\def\l@appendices#1#2{\@dottedtocline{0}{1.5em}{6.5em}{#1}{#2}}

\def\noappendixtables{\def\@lotheading{}}
\def\noappendixfigures{\def\@lofheading{}}
\def\@lotheading{
  \addtocontents{lot}{
    \protect\stepcounter{listofheading}
    \protect\addvspace{\@tocchapterskip}
    \hbox to \hsize{Appendix \hfil}
    \hbox to \hsize{Table \hfil}}}
\def\@lofheading{
  \addtocontents{lof}{
    \protect\stepcounter{listofheading}
    \protect\addvspace{\@tocchapterskip}
    \hbox to \hsize{Appendix \hfil}
    \hbox to \hsize{Figure \hfil}}}

\def\@appendix#1{\par
  \addtocontents{toc}{{\protect\setcounter{tocdepth}{0}}}
  \setcounter{chapter}{0}
  \setcounter{section}{0}
  \def\@chapapp{APPENDIX}
  \def\thechapter{\Alph{chapter}}
  \@lotheading
  \@lofheading
  %\separatorpage{\uppercase{#1}} % UW doesn't require a seperator page - EB
}

%=============================================================================
% COPYRIGHTPAGE
%=============================================================================
% The copyright must do the following:
% - start a new page with no number
% - place the copyright text centered at the bottom.
%=============================================================================
\def\copyrightpage{
  \newpage
  \thispagestyle{empty}    % No page number
  \addtocounter{page}{-1}
  \chapter*{}            % Required for \vfill to work
  \begin{center}
   \vfill
   \copyright\ Copyright by \@author\ \@date\\
   All Rights Reserved
  \end{center}}


%=============================================================================
% GLOSSARY
%=============================================================================
% The glossary environment must do the following:
% - produce the table of contents entry for the glossary
% - start a new page with GLOSSARY centered two inches from the top
%=============================================================================
\def\glossary{
  \chapter*{GLOSSARY}
  \addcontentsline{toc}{chapter}{GLOSSARY}}
\def\endglossary{\par\newpage}

%=============================================================================
% NOMENCLATURE
%=============================================================================
% The nomenclature environment must do the following:
% - produce the table of contents entry for the nomenclature section
% - start a new page with NOMENCLATURE centered two inches from the top
%=============================================================================
\def\nomenclature{\separatorpage{DISCARD THIS PAGE}
  \chapter*{NOMENCLATURE}
  \addcontentsline{toc}{chapter}{NOMENCLATURE}}
\def\endnomenclature{\par\newpage}

%=============================================================================
% LIST OF SYMBOLS
%=============================================================================
% The list of symbols environment must do the following:
% - produce the table of contents entry for the list of symbols section
% - start a new page with LIST OF SYMBOLS centered two inches from the top
%=============================================================================
\def\listofsymbols{\separatorpage{DISCARD THIS PAGE}
  \chapter*{LIST OF SYMBOLS}
  \addcontentsline{toc}{chapter}{LIST OF SYMBOLS}}
\def\endlistofsymbols{\par\newpage}

%=============================================================================
% VITA
%=============================================================================
% The vita environment must do the following:
% - produce a separator page with the word vita centered
% - produce the table of contents entry for the vita
% - start a new page with VITA centered two inches from the top
%=============================================================================
\def\vita{
%  \separatorpage{VITA}         % UW doesn't require this EB
  \chapter*{VITA}
  \addcontentsline{toc}{chapter}{VITA}}
\def\endvita{\par\newpage}

%=============================================================================
% ACKNOWLEDGMENTS
%=============================================================================
% The acknowledgments environment must do the following:
% - start a new page with ACKNOWLEDGMENTS centered two inches from the top
%=============================================================================
\def\acknowledgments{
  \chapter*{ACKNOWLEDGMENTS}
  \doublespace}
\def\endacknowledgments{\par\newpage}

%=============================================================================
% DEDICATION
%=============================================================================
% The dedication environment must do the following:
% - start a new page
% - center the text vertically
% - include the text in a center environment
%=============================================================================
\def\dedication{
  \newpage
  \null\vfil
  \begin{center}}
\def\enddedication{\end{center}\par\vfil\newpage}

%=============================================================================
% DATE
%=============================================================================
%\def\today{\ifcase\month\or
  %January\or February\or March\or April\or May\or June\or
  %July\or August\or September\or October\or November\or December\fi
  %\space\number\day, \number\year}
\newcount\@testday
\def\today{\@testday=\day
  \ifnum\@testday>30 \advance\@testday by -30
  \else\ifnum\@testday>20 \advance\@testday by -20
  \fi\fi
  \number\day\ \
  \ifcase\month\or
    January \or February \or March \or April \or May \or June \or
    July \or August \or September \or October \or November \or December
    \fi\ \number\year
}


%  Single counter for theorems and theorem-like environments:
\newtheorem{theorem}{Theorem}[chapter]
\newtheorem{assertion}[theorem]{Assertion}
\newtheorem{claim}[theorem]{Claim}
\newtheorem{conjecture}[theorem]{Conjecture}
\newtheorem{corollary}[theorem]{Corollary}
\newtheorem{definition}[theorem]{Definition}
\newtheorem{example}[theorem]{Example}
\newtheorem{figger}[theorem]{Figure}
\newtheorem{lemma}[theorem]{Lemma}
\newtheorem{prop}[theorem]{Proposition}
\newtheorem{remark}[theorem]{Remark}

%=============================================================================
% EQUATION and EQNARRAY -- put here because it must follow \chapter definition
%
% \newcounter{equation}
%=============================================================================
% \jot = 3pt      % Extra space added between lines of an eqnarray environment
\@addtoreset{equation}{chapter}   % Makes \chapter reset 'equation' counter.
\def\theequation{{\rm \thechapter}.\arabic{equation}}

%=============================================================================
% The macro \@eqnnum defines how equation numbers are to appear in equations.
%=============================================================================
\def\@eqnnum{[\theequation]}

%=============================================================================
% INITIALIZATION   -  Default initializations
%=============================================================================
\ps@thesis                  % 'plain' page style
\pagenumbering{arabic}      % Arabic page numbers
\onecolumn                  % Single-column.
\raggedbottom               % Ragged bottom
%\if@twoside\else\raggedbottom\fi % Ragged bottom unless twoside option.

\global\newif\if@bibpresent   % Flag for the appendicies fix.  If bib present
                              % then the seperator page isn't needed...

%-----------------------------------------------------------------------------
% Define \bface to be \bf so that all default boldface text can be
% disabled by redefining \bface to be \null.  -dfk
%-----------------------------------------------------------------------------
\newcommand{\bface}{\bf}

%-----------------------------------------------------------------------------
% The following penalties are for conforming with Purdue's
% requirements for the thesis format.
%-----------------------------------------------------------------------------
\brokenpenalty=10000
\clubpenalty=10000
\widowpenalty=10000

%=============================================================================
% END FILE
%=============================================================================
