(module comma)

;Main programs: comma, rational, multiply, and readfile:
;(comma file) where file is a string.  If file is "screen" output goes to
;screen, otherwise to that file.  
;Input:  a1 a2 ... an, b1 b2 ... bm, ... d1 d2 ... dp
;is interpreted so all the a's are multiplied (space between each), all the
;b's are multiplied, etc.  Each group (of a's, b's, etc.) is separated by
;a comma and then a space, and the entire input is terminated with a hit
;of the <return> key.
;(rational file m n) where file works as usual.  Converts rational number
; m/n to the appropriate knot representation.
;(multiply file a1 a2 ... an) where file works as usual.  Multiplies variable
;length list of a's (where each a is an integer).
;All of the above programs connect the north strands together and the south
;strands together before returning the output.
;(readfile file directory) will sift through the text file file which must
;be of the following format.  Each line contains the Conway representation
;(using commas and spaces) of a single knot.  The program then generates
;files named "knot" + the appropriate numbers of the knot name, which are
;placed in directory.  file and directory must be strings, and if file is
;"screen" output will NOT be directed to the screen; instead the file
;"screen" will be looked for and read in as the input file.  file must be
;terminated with "end" and a final carriage return.  Here is a sample file:
;(Ignore the semicolons.)
;
;2, 3, 4
;1 2
;1 2, 2
;end
;
;At the end of this file are the routines used to create the libraries of
;all algebraic knots and all rational knots, along with routines that make
;up readfile.

;The following are useful helper functions.

(define call/cc call-with-current-continuation)

(define newlist
  (lambda (l)
    (cond
     ((null? l) '())
     ((pair? (car l)) (cons (newlist (car l)) (newlist (cdr l))))
     (else (cons (car l) (newlist (cdr l)))))))

(define carof
  (lambda (proc)
    (lambda (x y)
      (car (proc x y)))))

(define add1
  (lambda (n) (+ 1 n)))

(define sub1
  (lambda (n) (- n 1)))

;converts integer to rep.
(define int->rep
  (lambda (n)
    (if (zero? n)
	(list (list 'ne 'junk)
	      (list 'junk 'junk)
	      (list 'se 'junk)
	      (list 'junk 'junk))
	(let ((direction 'junk)
	      (final 'junk)
	      (initial 'junk))
	  (begin
	    (if (positive? n)
		(begin
		  (set! direction 'r)
		  (set! final (list (list 'r 'ne 'junk 'se 'junk)))
		  (set! initial (list (list 1 'd) (list 'junk 'junk)
				      (list 1 'u) (list 'junk 'junk))))
		(begin
		  (set! direction 'l)
		  (set! final (list (list 'l 'se 'junk 'ne 'junk)))
		  (set! initial (list (list 1 'u) (list 'junk 'junk)
				      (list 1 'd) (list 'junk 'junk)))))
	    (letrec ((int->rep-helper
		      (lambda (m)
			(if (= m 1)
			    '()
			    (append (int->rep-helper (sub1 m))
				    (list (list direction m 'd m 'u)))))))
	      (append initial
		      (append (int->rep-helper (abs n)) final))))))))
      

;selects row n from rep (which describes crossing #n)
(define row
  (lambda (n rep)
    (if (= n 1)
	(cadddr (cdr rep))
	(row (sub1 n) (cdr rep)))))

;the following extractor functions take n for the knot number and return the
;appropriate value from the rep matrix.

(define direction-cell
  (lambda (n rep)
    (row n rep)))
(define direction (carof direction-cell))

(define over-knot-cell
  (lambda (n rep)
    (cdr (row n rep))))
(define over-knot (carof over-knot-cell))

(define over-pos-cell
  (lambda (n rep)
    (cddr (row n rep))))
(define over-pos (carof over-pos-cell))

(define under-knot-cell
  (lambda (n rep)
    (cdddr (row n rep))))
(define under-knot (carof under-knot-cell))

(define under-pos-cell
  (lambda (n rep)
    (cdddr (cdr (row n rep)))))
(define under-pos (carof under-pos-cell))

;The following extract the origin pairs (first four pairs of rep)

(define compose
  (lambda (f g)
    (lambda (x) (f (g x)))))
(define nw-knot caar)
(define nw-pos cadar)
(define ne-knot caadr)
(define ne-pos cadadr)
(define sw-knot caaddr)
(define sw-pos (compose cadadr cdr))
(define se-knot (compose caaddr cdr))
(define se-pos (compose cadadr cddr))
(define select-origin
  (lambda (corner rep)
    ((cond
      ((eq? corner 'nw) car)
      ((eq? corner 'ne) cadr)
      ((eq? corner 'sw) caddr)
      ((eq? corner 'se) cadddr))
     rep)))
     
;Multiplies rep by 0.
(define mult0
  (lambda (rep)
    (letrec ((mult0-helper (lambda (l)
			     (if (null? l) 
				 '()
				 (cons
				  (cond
				   ((eq? (car l) 'ne) 'sw)
				   ((eq? (car l) 'sw) 'ne)
				   ((eq? (car l) 'r) 'l)
				   ((eq? (car l) 'l) 'r)
				   (else (car l)))
				  (mult0-helper (cdr l))))))
	     (mult0-helper2 (lambda (ll)
			      (if (null? ll)
				  '()
				  (cons (mult0-helper (car ll))
					(mult0-helper2 (cdr ll)))))))
      (append (list (car rep) (caddr rep)
		    (cadr rep) (cadddr rep))
	      (mult0-helper2 (cddddr rep))))))

;Returns all of rep, except for the first row.
(define recur
  (lambda (rep)
    (append (list (car rep) (cadr rep)
		  (caddr rep) (cadddr rep))
	    (cdr (cddddr rep)))))

;renumber takes n (a number of knots in first tangle) and a rep.
; It renumbers the knots in rep to begin with n+1.
(define renumber
  (lambda (n rep)
    (letrec ((renumber-helper
	      (lambda (rep)
		(if (null? (cddddr rep))
		    '()
		    (cons
		     (list
		      (direction 1 rep)
		      (let ((k (over-knot 1 rep)))
			(if (number? k)
			    (+ n k)
			    k))
		      (over-pos 1 rep)
		      (let ((k (under-knot 1 rep)))
			(if (number? k)
			    (+ n k)
			    k))
		      (under-pos 1 rep))
		     (renumber-helper (recur rep))))))
	     (rh2 (lambda (pair)
		    (list (let ((k (car pair)))
			    (if (number? k)
				(+ n k)
				k))
			  (cadr pair)))))
      (append (list (rh2 (car rep)) (rh2 (cadr rep))
		    (rh2 (caddr rep)) (rh2 (cadddr rep)))
	      (renumber-helper rep)))))

;Given a corner to start with, a path, and a rep, construct! adds
;information to the rep determined by the path.  The last pair in
;rep remains unchanged, as well as each direction.
(define construct!
  (lambda (corner path rep)
    (letrec ((construct-helper
	      (lambda (path rep from pos)
		(if (null? path)
		    rep
		    (begin
		      (if (eq? pos 'u)
			  (begin
			    (set-car! (over-knot-cell from rep)
				      (caar path))
			    (set-car! (over-pos-cell from rep)
				      (cadar path)))
			  (begin
			    (set-car! (under-knot-cell from rep)
				      (caar path))
			    (set-car! (under-pos-cell from rep)
				      (cadar path))))
		      (construct-helper
		       (cdr path)
		       rep
		       (caar path)
		       (cadar path)))))))
      (construct-helper (cdr path)
			(cond
			 ((eq? corner 'nw)
			  (cons (car path) (cdr rep)))
			 ((eq? corner 'ne)
			  (cons (car rep)
				(cons (car path) (cddr rep))))
			 ((eq? corner 'sw)
			  (append (list (car rep) (cadr rep))
				  (cons (car path) (cdddr rep))))
			 ((eq? corner 'se)
			  (append (list (car rep) (cadr rep)
					(caddr rep))
				  (cons (car path) (cddddr rep)))))
			(caar path)
			(cadar path)))))

;make-path creates a path based on a given corner and rep.
(define make-path
  (lambda (corner rep)
    (call/cc
     (lambda (abort)
       (letrec ((make-path-helper
		 (lambda (rep from pos)
		   (cond
		    ((eq? from 'junk) (abort '()))
		    ((symbol? from) '())
		    (else
		     (let ((knot (if (eq? pos 'u)
				     (over-knot from rep)
				     (under-knot from rep)))
			   (newpos (if (eq? pos 'u)
				       (over-pos from rep)
				      (under-pos from rep))))
		       (cons (list knot newpos)
			     (make-path-helper rep knot newpos))))))))
	 (cond
	  ((eq? corner 'nw)
	   (cons (car rep)
		 (make-path-helper rep (nw-knot rep) (nw-pos rep))))
	  ((eq? corner 'ne)
	   (cons (cadr rep)
		 (make-path-helper rep (ne-knot rep) (ne-pos rep))))
	  ((eq? corner 'sw)
	   (cons (caddr rep)
		 (make-path-helper rep (sw-knot rep) (sw-pos rep))))
	  ((eq? corner 'se)
	   (cons (cadddr rep)
		 (make-path-helper rep (se-knot rep) (se-pos rep))))))))))

;reverse-path does NOT return a path.  It returns a list with car nw, ne, sw,
;or se (paired with 'junk), depending on where the new strand ORIGINATES,
;and the cdr is the actual path.
(define reverse-path
  (lambda (start path)
    (append (reverse path) (list (list start 'junk)))))

;Adds two reps together.
;The replace functions replace occurrences of ne&se in the first drep
;(replace 1 ...) or occurrences of nw&sw in the second drep (replace 2 ...)
;by where the strand will go to or come from after the tangles are summed.
;The append at the bottom takes drep1, uses its nw&sw origins, but replaces
;its ne&se origins by those of drep2 (after they have been (replace 2 ...)'d).
;This "new" drep1 is then (replace 1 ...)'d.  Finally, drep2 (minus its
;origins) is (replace 2 ...)'d.
(define sum
  (lambda (rep1 rep2)
    (letrec ((replace (lambda (n drep1 drep2)
			(cond
			 ((null? drep1) '())
			 ((even? (length (car drep1)))
			  (cons
			   (single-replace n (car drep1) drep2)
			   (replace n (cdr drep1) drep2)))
			 (else (cons 
				(cons (caar drep1)
				      (single-replace n (cdar drep1) drep2))
				(replace n (cdr drep1) drep2))))))
	     (single-replace (lambda (n single drep)
			       (if (= n 1)
				   (if (null? single)
				       '()
				       (append
					(cond
					 ((eq? (car single) 'ne) (car drep))
					 ((eq? (car single) 'se) (caddr drep))
					 (else (list (car single) (cadr single))))
					(single-replace n (cddr single) drep)))
				   (if (null? single)
				       '()
				       (append
					(cond
					 ((eq? (car single) 'nw) (cadr drep))
					 ((eq? (car single) 'sw) (cadddr drep))
					 (else (list (car single) (cadr single))))
					(single-replace n (cddr single) drep)))))))
      (let* ((directed-reps (direct-reps rep1 rep2))
	     (d-rep1 (car directed-reps))
	     (d-rep2 (cadr directed-reps))
	     (renum-d-rep2 (renumber (- (length rep1) 4) d-rep2)))
	(append (replace 1
			 (append (list (car d-rep1)
				       (single-replace 2 (cadr renum-d-rep2)
						       d-rep1)
				       (caddr d-rep1)
				       (single-replace 2 (cadddr renum-d-rep2)
						       d-rep1))
				 (cddddr d-rep1))
			 renum-d-rep2)
		(replace 2 (cddddr renum-d-rep2) d-rep1))))))

;Master program to take two reps, and redirect one of them so they may be
;joined.  It branches off into other programs depending on cases.
;1nc and 2nc mean one-not-connected and two-not-connected, respectively.
(define direct-reps
  (lambda (rep1 rep2)
    (cond
     ((and (connected? rep1) (connected? rep2))
      (direct-connected-reps rep1 rep2))
     ((connected? rep1) (direct-2nc-reps rep1 rep2))
     (else (direct-1nc-reps rep1 rep2)))))

;Checks if both east strands (and thus both west strands) are connected.
(define connected?
  (lambda (rep)
    (let ((ne (make-path 'ne rep))
	  (se (make-path 'se rep)))
      (cond
       ((and (not ne) (not se)) '())
       (ne (eq? (caar (reverse ne)) 'se))
       (else (eq? (caar (reverse se)) 'ne))))))

;If the ne and nw strands have opposite direction, then the se and sw strands
;have opposite direction (i.e., into or out of their respective tangles), so
;simply return both reps.  Otherwise, reverse a strand in rep2.
(define direct-connected-reps
  (lambda (rep1 rep2)
    (if (not (boolean=? (make-path 'ne rep1)
			(make-path 'nw rep2)))
	(list rep1 rep2)
	(list rep1 (reverse-strand 'nw rep2)))))

;If the top strands "collide", reverse top of rep1 but also check if bottom of
;rep1 needs to be reversed also.  If they don't collide, check bottom strands,
;and reverse bottom of rep1 if necessary.
(define direct-1nc-reps
  (lambda (rep1 rep2)
    (list (if (boolean=? (make-path 'nw rep2)
			 (make-path 'ne rep1))
	      (if (boolean=? (make-path 'sw rep2)
			     (make-path 'se rep1))
		  (reverse-strand 'ne (reverse-strand 'se rep1))
		  (reverse-strand 'ne rep1))
	      (if (boolean=? (make-path 'sw rep2)
			     (make-path 'se rep1))
		  (reverse-strand 'se rep1)
		  rep1))
	  rep2)))

;Same as above, but only reverse strands of rep2 if necessary.
(define direct-2nc-reps
  (lambda (rep1 rep2)
    (list rep1 (if (boolean=? (make-path 'nw rep2)
			      (make-path 'ne rep1))
		   (if (boolean=? (make-path 'sw rep2)
				  (make-path 'se rep1))
		       (reverse-strand 'nw (reverse-strand 'sw rep2))
		       (reverse-strand 'nw rep2))
		   (if (boolean=? (make-path 'sw rep2)
				  (make-path 'se rep1))
		       (reverse-strand 'sw rep2)
		       rep2)))))

;Finds opposite end of a strand beginning (or ending) at corner.
(define opposite-end
  (lambda (corner rep)
    (letrec ((oe-helper (lambda (corner rep directions)
			  (if (null? directions)
			      (caar (reverse (make-path corner rep)))
			      (let ((path (make-path (car directions) rep)))
				(if (or (null? path)
					(not (eq? (caar (reverse path))
						  corner)))
				    (oe-helper corner rep (cdr directions))
				    (car directions)))))))
      (oe-helper corner rep (delq corner (list 'nw 'ne 'sw 'se))))))

;Reverses the direction of a strand starting or ending at corner and returns
;the revised rep.
(define reverse-strand
  (lambda (corner rep)
    (let* ((answer (newlist rep))
	   (start (if (make-path corner answer)
		      corner
		      (opposite-end corner answer)))
	   (temp (reverse-path start (make-path start answer)))
	   (revpath (cdr temp))
	   (newcorner (opposite-end corner answer)))
      (begin
	(construct! newcorner revpath answer)
	(reverse-directions! revpath answer)
	(if (eq? corner start)
	    (change-origin newcorner (car revpath)
			   (junkate corner answer))
	    (change-origin corner (car revpath)
			   (junkate newcorner answer)))))))

;Changes origin of corner in rep with the pair new (e.g. (5 u)).
(define change-origin
  (lambda (corner new rep)
    (letrec ((make-list-of-directions
	      (lambda (l)
		(if (eq? (car l) corner)
		    (cons 'junk (cdr l))
		    (cons (car l) (make-list-of-directions (cdr l))))))
	     (construct
	      (lambda (lod origins)
		(cond
		 ((null? lod) '())
		 ((eq? (car lod) 'junk) (cons new
					      (construct (cdr lod)
							 (cdr origins))))
		 (else (cons (car origins) (construct (cdr lod)
						      (cdr origins))))))))
      (append (construct (make-list-of-directions (list 'nw 'ne 'sw 'se))
			 (list-head rep 4))
	      (cddddr rep)))))

;Special case of change-origin, makes the corner be (junk junk) (i.e., make
;it directed out of the knot).
(define junkate
  (lambda (corner rep)
    (change-origin corner (list 'junk 'junk) rep)))

;Used by reverse-strand, changes r or l to the other if the path passes
;through the particular crossing exactly once--actually alters rep.
(define reverse-directions!
  (lambda (path rep)
    (if (or (symbol? (car path)) (symbol? (caar path)))
	rep
	(begin
	  (set-car! (row (caar path) rep)
		    (if (eq? (car (row (caar path) rep)) 'r)
			'l
			'r))
	  (reverse-directions! (cdr path) rep)))))

;Multiplies variable numbers of integers.
(define mult
  (lambda nums
    (letrec ((mult-helper (lambda (l)
			    (cond
			     ((= 2 (length l))
			      (sum (mult0 (int->rep (car l)))
				   (int->rep (cadr l))))
			     ((= 1 (length l))
			      (int->rep (car l)))
			     (else(sum (mult0 (mult-helper
					       (list-head l (sub1 (length l)))))
				       (int->rep (car (reverse l)))))))))
      (mult-helper nums))))

;Exands m/n to list of integers forming the continued fraction equal to m/n
(define expand
  (lambda (m n)
    (cond
     ((= m 1) (list n))
     ((= n 1) (list m))
     (else (cons (quotient m n)
		 (expand n (remainder m n)))))))

;Returns the rep associated with m/n
(define rat->rep
  (lambda (m n)
    (apply mult (reverse (expand m n)))))

;Returns the rep associated with variable-length comma-list s
(define series
  (lambda s
    (letrec ((series-helper
	      (lambda (seq)
		(cond
		 ((null? seq) (int->rep 0))
		 ((pair? (car seq)) (sum (mult0 (apply mult (car seq)))
					 (series-helper (cdr seq))))
		 (else (sum (mult0 (int->rep (car seq)))
			    (series-helper (cdr seq))))))))
      (series-helper s))))

;Connects north strands and south strands of rep
(define connect
  (lambda (rep)
    (letrec ((join
	      (lambda (corn1 corn2 rep)
		(let ((newrep
		       (if (boolean=? (make-path corn1 rep)
				      (make-path corn2 rep))
			   (reverse-strand corn1 rep)
			   rep)))
		  (replace corn1 corn2
			   (replace corn2 corn1 newrep newrep)
			   newrep))))
	     (replace
	      (lambda (c1 c2 nrep fullnrep)
		(cond
		 ((null? nrep) '())
		 ((even? (length (car nrep)))
		  (cons (single-replace c1 c2 (car nrep) fullnrep)
			(replace c1 c2 (cdr nrep) fullnrep)))
		 (else (cons
			(cons (caar nrep)
			      (single-replace c1 c2 (cdar nrep) fullnrep))
			(replace c1 c2 (cdr nrep) fullnrep))))))
	     (single-replace
	      (lambda (c1 c2 single fullnrep)
		(if (null? single)
		    '()
		    (append
		     (if (eq? (car single) c1)
			 (select-origin c2 fullnrep)
			 (list (car single) (cadr single)))
		     (single-replace c1 c2 (cddr single) fullnrep))))))
      (cddddr (join 'nw 'ne (join 'sw 'se rep))))))

;Alternative to connect, prepare fixes up a rep if it is NOT to be connected.
;nw,ne,sw,se change to 1, 2, 3, 4, respectively, crossing numbers suitably
;moved up, and more!
(define prepare
  (lambda (rep)
     (replace-text 'nw 1
        (replace-text 'ne 2
	   (replace-text 'sw 3
	      (replace-text 'se 4
		 (replace-text 'junk 0
		    (flesh-out
		     (renumber 4 rep)))))))))

(define flesh-out
  (lambda (rep)
    (cond
     ((null? rep) '())
     ((= (length (car rep)) 2) (cons (append (list 0) (car rep) (list 0 0))
				     (flesh-out (cdr rep))))
     (else (cons (car rep) (flesh-out (cdr rep)))))))

(define replace-text
  (lambda (from to l)
    (cond
     ((null? l) '())
     ((pair? (car l)) (cons (replace-text from to (car l))
			    (replace-text from to (cdr l))))
     ((eq? (car l) from) (cons to (replace-text from to (cdr l))))
     (else (cons (car l) (replace-text from to (cdr l)))))))
  
		
    

;The following programs are the main ones, and work as described at the
;top of the page.
;mode: 1=comma, 2=comma-t, 3=multiply, 4=multiply-t
(define comma
  (lambda (input mode)
    (cond
     ((= mode 1) (output
		  (connect (apply series (seq->list (string-append
						     input ", "))))))
     ((= mode 2) (output
		  (prepare (apply series (seq->list (string-append
						     input ", "))))))
     (else (let* ((l (seq->list (string-append input ", ")))
		  (l2 (if (pair? (car l))
			  (car l)
			  l)))
	     (if (= mode 3)
		 (output (connect (apply mult l2)))
		 (output (prepare (apply mult l2)))))))))

(define comma-t
  (lambda (input)
    (let* ((thunk (lambda ()
		    (output
		     (prepare
		      (apply series (seq->list (string-append
						input ", "))))))))
      (thunk))))

;Outputs the rep in standard form
(define output
  (lambda (rep)
    (letrec ((output-helper
	      (lambda (rep)
		(if (null? rep)
		    (list->string '(#\newline))
		    (string-append
		     (single-output-helper (car rep))
		     (output-helper (cdr rep))))))
	     (single-output-helper
	      (lambda (single)
		(if (null? single)
		    (list->string '(#\newline))
		    (string-append
		     (let ((cs (car single)))
		       (if (symbol? cs)
			   (symbol->string cs)
			   (number->string cs)))
		     " "
		     (single-output-helper (cdr single)))))))
      (string-append
       (number->string (length rep))
       " "
       (output-helper rep)))))


;The following are string helper functions used by comma.


;Converts comma'd sequence into list.
(define seq->list
  (lambda (s)
    (if (no-more-nums? s)
	'()
	(let ((lfe (length-first-element s #\,)))
	  (if (no-spaces? s)
	      (cons (string->num (substring s 0 lfe))
		    (seq->list (string-tail s (+ 2 lfe))))
	      (cons (seq-with-spaces->list (string-append
					    (substring s 0 lfe) " "))
		    (seq->list (string-tail s (+ 2 lfe)))))))))

;Checks if spaces exist in the first term (terms separated by commas) in s.
(define no-spaces?
  (lambda (s)
    (cond
     ((or (string-null? s) (string=? (string-head s 1) ",")) #t)
     ((string=? (string-head s 1) " ") '())
     (else (no-spaces? (string-tail s 1))))))

;Converts sequence separated by spaces (considered one item in the overall
;sequence) into list
(define seq-with-spaces->list
  (lambda (s)
    (if (no-more-nums? s)
	'()
	(let ((lfe (length-first-element s #\space)))
	  (cons (string->num (substring s 0 lfe))
		(seq-with-spaces->list (string-tail s (add1 lfe))))))))

;Checks if another number appears in a string.
(define no-more-nums?
  (lambda (s)
    (cond
     ((string-null? s) #t)
     ((char->digit (string-ref s 0)) '())
     (else (no-more-nums? (string-tail s 1))))))

;Returns number of characters in a string before first appearance of char.
(define length-first-element
  (lambda (s char)
    (if (or (string-null? s)
	    (char=? (string-ref s 0) char))
	0
	(add1 (length-first-element (string-tail s 1) char)))))

;Converts a string of digits to the base 10 numeral.
(define string->num
  (lambda (s)
    (if (string-null? s)
	0
	(string->number s))))

;Reverses the order of occurrences of all characters in a string.
(define reverse-string
  (lambda (s)
    (if (string-null? s)
	""
	(string-append (reverse-string
			(string-tail s 1))
		       (string-head s 1)))))

(define done '())


(define add-parens
  (lambda (l)
    (cond
     ((null? l) '())
     ((pair? (car l)) (cons (car l) (add-parens (cdr l))))
     (else (cons (list (car l)) (add-parens (cdr l)))))))

(define sublist
  (lambda (l s e)
    (cond
     ((zero? s) 
      (if (= e 1)
	  (list (car l))
	  (cons (car l)
		(sublist (cdr l) 0 (sub1 e)))))
     (else (sublist (cdr l) (sub1 s) (sub1 e))))))

(define list-head
  (lambda (l k)
    (sublist l 0 k)))

(define delq
  (lambda (e l)
    (cond
     ((null? l) '())
     ((eq? (car l) e) (delq e (cdr l)))
     (else (cons (car l) (delq e (cdr l)))))))

(define boolean=?
  (lambda (a b)
    (if (or (equal? a #f) (equal? a '()))
	(if (or (equal? b #f) (equal? b '()))
	    #t
	    #f)
	(if (or (equal? b #f) (equal? b '()))
	    #f
	    #t))))

(define string-null?
  (lambda (s)
    (string-ci=? s "")))

(define string-tail
  (lambda (s n)
    (substring s n (string-length s))))

(define string-head
  (lambda (s n)
    (substring s 0 n)))

(define char->digit
  (lambda (c)
    (- (char->integer c) 48)))

"The programs are now installed.  Enjoy!!"

