/* lib.d file word.c */
#include <stdio.h> 
#include <ctype.h> 
#include "defs.h"
#include "list.h"
#include "word.h"
#include "input.h"
int num_gens=0;
gen * inv_of=0;
word * user_gen_name=0;
int gen_array_size=53;
static void word_expand PARMS((word * wp));
static void insert_gen PARMS((gen h, word * wp));

elt_fntab
WORD_fntab = {  
	word_sgn_dp,  
	word_cpy_dp,  
	(V2DP)word_create,  
	word_kill_dp,  
	word_print_dp, 
		0, /* these two spaces are for heap index functions */
		0
}; 

elt_fntab
GENWT_WORD_fntab = {  
	genwt_word_sgn_dp,  
	word_cpy_dp,  
	(V2DP)word_create,  
	word_kill_dp,  
	word_print_dp, 
		0, /* these two spaces are for heap index functions */
		0
}; 

elt_fntab
GEN_fntab = {  
		gen_sgn_dp,  
		gen_cpy_dp,  
		(V2DP)gen_create,  
		Free_dp,  
		gen_print_dp, 
		0, /* these two spaces are for heap index functions */
		0
}; 

static void word_stretch_reset PARMS((word * wp, int N, int a));
static void word_double PARMS((word * wp));

void
word_init(wp) 
	word *wp; 
{ 
	assert(wp);
	   	wp->g =valloc2(gen,16); 
		wp->first = 0;
		wp->last = -1;
		wp->space = 16;
} 

void
word_centreinit(wp) 
	word *wp; 
{ 
	assert(wp);
	   	wp->g =valloc2(gen,16); 
		wp->first = 8;
		wp->last = 7;
		wp->space = 16;
} 

void
word_longinit(wp,n) 
	word *wp; 
	int n;
{ 
	assert(wp);
	   	wp->g =valloc2(gen,n); 
		wp->first = 0;
		wp->last = -1;
		wp->space = n;
} 
/* Clear space allocated to a word.
The space assigned to a word (as a pointer to generators) is cleared.
*/
void word_clear(wp) 
		word * wp; 
{ 
		assert(wp); 
		Free_dp((dp)wp->g); wp->g = 0; 
} 

void word_reset(wp) 
	word *wp; 
{ 
	wp->last = wp->first - 1;
} 

void word_centrereset(wp) 
	word *wp; 
{ 
	wp->first = (wp->space)/2;
	wp->last = wp->first - 1;
} 

/* 
The word |*wp| has certainly been initialized, and may already contain a
non-trivial word. But the information it contains, if any, is redundant, and
can be thrown away. We just require the space for new information but there
may not be enough of it. If the word currently contains space for less than
n generators we increase the amount of space to this. We set the position
for the first word entry at a, and the last at a-1 (since at this stage the
word is empty).  
*/
static void 
word_stretch_reset(wp,N,a) 
	word *wp;
	int N;
	int a;
{
	if (wp->space < N) {
		Free_dp((dp) wp->g); wp->g = 0; 
		wp->g = valloc2(gen,N); 
		wp->space = N;
	}
		wp->first = a;
		wp->last = a - 1;
}


/* Initialize a |word_traverser| to make it traverse a particular
word, whose address is given in the second argument.
*/
void 
word_traverser_init(wtp,wp)
	word_traverser *wtp; 
	word *wp; 
{ 
	wtp->wp = wp; 
	wtp->posn = wp->first;
} 

/* |word_get_first()| gets hold of the first generator in a word without
changing it. It returns |TRUE| provided the word is non-trivial, |FALSE| 
otherwise.
The  pointer |gp| is set to point to the first generator in the word pointed
to by |wp|.
*/
boolean
word_get_first(wp,gp) 
	word *wp; 
	gen *gp; 
{ 
	assert(wp); 
	assert(wp->first >=0);
	if (wp->first <= wp->last) {
		*gp = (wp->g)[wp->first];
		assert(*gp != INVALID_GEN);
		return TRUE;
	}
	else {
		*gp = INVALID_GEN;
		return FALSE; 
	}
} 

boolean
word_halve(wp,w1p,w2p)
	word * wp;
	word * w1p;
	word * w2p;
{
	int n=word_length(wp);
	if (n>=2){
		word_cpy(wp,w1p);
		word_cpy(wp,w2p);
		w1p->last=w1p->first+n-n/2-1;
		w2p->first=w2p->last-n/2+1;
		return TRUE;
	}
	else{
		word_cpy(wp,w1p);
		word_reset(w2p);
		return FALSE;
	}
}
		

/* |word_next()| is used when traversing a word, generator by generator. It
returns |TRUE| while the |list_traverser| points to a generator, |FALSE| 
once the last generator in the word has been passed.
The  pointer |gp| is set to point to the generator at which the
|word_traverser| is currently positioned..
The procedure does not change the word being traversed.
*/
boolean
word_next(wtp,gp) 
	word_traverser *wtp; 
	gen *gp; 
{ 
	assert(wtp->posn >= wtp->wp->first);
	if (wtp->posn <= wtp->wp->last) {
		*gp = (wtp->wp->g)[wtp->posn]; 
		assert(*gp != INVALID_GEN);
		wtp->posn++;
		return TRUE; 
	} 
	else {
		*gp = INVALID_GEN;
		return FALSE; 
	}
} 


/* Reduce a word.
The word pointed to by |in| is reduced by the cancellation of adjacent
inverse generators to give the word pointed to by |out|. The function 
returns a boolean value, |TRUE| if a reduction has been made so that the
word pointed to by |out| is actually shorter than the word pointed to by
|in|, |FALSE| if no reduction is possible.
Before the function is applied |*out| could be a freshly initialized word or
an actual word already in use. Space is assigned or reassigned as necessary
within this function (via |word_cpy|).
*/
boolean
word_reduce(in,out) 
	word *in, *out; 
{ 
	int i=0; 
	boolean ans = FALSE; 
	gen * outgp=0;

	assert(in); 
	assert(out); 

	outgp = valloc2(gen,in->space); /* we have to do this in case in and
														out are identical */

	out->space = in->space;
	out->first = in->first;
	out->last = in->last;
	for (i=out->first;i<=out->last;i++)
		outgp[i] = in->g[i];
	
	i = out->first;
	while (i<out->last ) { 
		if (outgp[i] == inv(outgp[i+1])) {
		/* the generators  in positions
|i| and |i+1| are mutually inverse */
			int j;
			ans = TRUE; 
			out->last -=2;
			for ( j = i; j <= out->last; j++) 
				outgp[j] = outgp[j+2]; 
			if (i>out->first)  
				i--; 
		} 
		else 
			i++; 
	}
	Free_dp((dp)out->g); out->g = 0;
	out->g = outgp;
	return ans; 
} 


/* Cyclically reduce a word.
The boolean valued function |word_creduce()| operates just like
|word_reduce()| except that it also cancels inverse pairs of generators at
the two ends of the word. The function operates in two stages. First the
word is reduced using |word_reduce()|, then the inverse generators at the
two ends are cancelled.
*/
boolean word_creduce(in, out) 
 
		word *in, * out; 
{ 
	boolean ans; 

	ans = word_reduce(in,out); 
	while (out->first < out->last
			&& out->g[out->first]==inv(out->g[out->last])) {
		out->first++;
		out->last--;			
		ans = TRUE; 
	} 
	return ans;
}


/* Construct the inverse of a word.
If a word is read as a sequence of generators the inverse word consists of
the string of the inverses of these generators, but in the reverse order.
Before the function is applied |*inverse| could be a freshly initialized word or
an actual word already in use. It might be the same word as |*given|.
 Space is assigned or reassigned as necessary
within this function.
*/
void word_inv(given, inverse)
	word *given, *inverse; 
{
	int i;
	int j;
	gen * gp=0;
	gp = valloc2(gen,given->space);
	for (i=given->first,j=given->last;i<=given->last;i++,j--)
		gp[i] = inv(given->g[j]);	
	inverse->space=given->space;
	inverse->first=given->first;
	inverse->last = given->last;
	Free_dp((dp)inverse->g);inverse->g=0;
	inverse->g = gp;
}

boolean word_eqinv(w1p,w2p)
	word * w1p;
	word * w2p;
{
	int m;
	if ((m=((w1p->last) - (w1p->first))) != ((w2p->last) - (w2p->first)))
		return FALSE;
	else if (m>=0){
		int i;
		gen * g1p = w1p->g + w1p->first;
		gen * g2p = w2p->g + w2p->last;
		for (i=0;i<=m;i++)
			if (g1p[i]!=inv(g2p[-i]))
				return FALSE;
	}
	return TRUE;
}

/* Find the length of a word.
The length of a word is the number of non-trivial generators in the string
that makes up the word. The trivial word (the identity) has length 0.
*/
#if 0
int
word_length(wp) 
	word *wp; 
{ 
	int n;
	n = (wp->last) + 1 - (wp->first);
	return n;
} 
#endif

		
/* Adding on a generator to the right hand end of a word (right
multiplication).
*/
void
word_put_last(wp,g) 
	word *wp; 
	gen g; 
{ 
	int i;
	int n;
	n=wp->space;
	if (wp->last == n - 1){ /* there's an entry in the rightmost
piece of space */
		if (wp->first>n/2) {
			int k;
			k=n/4;
			wp->first -=k;
			wp->last -=k;
			for (i=wp->first;i<=wp->last;++i)
				wp->g[i]=wp->g[i+k];
		}
		else
			word_double(wp);
	}
	(wp->last)++;
	wp->g[wp->last] = g;
} 	

/*Adding on a generator to the left hand end of a word (left
multiplication).
*/
void
word_put_first(wp,g) 
	word *wp; 
	gen g; 
{ 
	if (wp->first == 0) {
		int i;
		int n;	
		n = wp->space;
		if (wp->last<=n/2) {
			int k;
			k=n/4;
			wp->first +=k;
			wp->last +=k;
			for (i=wp->last;i>=wp->first;--i)
				wp->g[i]=wp->g[i-k];
		}
		else	
			word_double(wp);
	}
	(wp->first)--;
	wp->g[wp->first] = g;
} 	

/* Double the amount of space available to a word, while preserving the
information stored in the word.
*/
static void
word_double(wp)
	word * wp;
{
	int n;
	int k;
	int i;
	gen * gp;
	n = wp->space;
	k = n/2;
	gp = valloc2(gen,2*n);
	for (i=k;i<3*k;++i) 
		gp[i] = (wp->g)[i-k];
	Free_dp((dp)wp->g); wp->g=0;
	wp->g = gp;
	wp->space *= 2 ;
	wp->first += k;
	wp->last += k; 
}


/*Get the first generator out of a word, and at the same time delete it from 
that word. 
The function returns |FALSE| if the word is trivial, |TRUE| otherwise.
*/
boolean
word_delget_first(wp,gp) 
	word *wp; 
	gen *gp; 
{ 
	assert(wp); 
	if (wp->first > wp->last) {
		*gp = INVALID_GEN;
		return FALSE;
	}
	else {
		*gp =  wp->g[wp->first++];
		return TRUE; 
	} 
} 
		
/*  Delete the first generator in a word.
The function returns |FALSE| if the word is trivial, |TRUE| otherwise.
*/
boolean
word_del_first(wp) 
	word *wp; 
{ 
	assert(wp); 
	if (wp->first > wp->last) 
		return FALSE;
	else  {
		wp->first++;
		return TRUE; 
	}
} 
		
/* Delete the last generator in a word.
The function returns |FALSE| if the word is trivial, |TRUE| otherwise.
*/
boolean
word_del_last(wp) 
	word *wp; 
{
	assert(wp); 
	if (wp->first > wp->last) 
		return FALSE;
	else  {
		wp->last--;
		return TRUE; 
	}
} 

/* Concatenate two words.
|wp1|, |wp2| and |wp3| are pointers to initialized words. |*wp1| and |*wp2| are
not changed by this procedure (provided that they are both distinct from
|*wp3|). At the end |*wp3| contains the concatenation
of |*wp1| and |*wp2|.
*/
void
word_concat(wp1,wp2,wp3) 
	word *wp1, *wp2, *wp3; 
{ 
	int n1,n2,n; 
	int N;
	int i;
	gen *factor,*concat; 

	n1 = word_length(wp1);
	n2 = word_length(wp2);
	n = n1 + n2;
	if (n<=0){
		wp3->last = wp3->first -1;
		return;
	} /* avoid malloc()ing 0 bytes */
	concat = valloc2(gen,n); 
	factor = wp1->g + wp1->first; 
	for (i=0;i<n1;i++)
		concat[i] = factor[i];
	factor = wp2->g + wp2->first; 
	for (i=n1;i<n;i++)
		concat[i] = factor[i-n1]; 
	N = max(wp1->space,wp2->space);
	while (N<2*n)
		N *=2;
	word_stretch_reset(wp3,N,n/2);
	for (i=0;i<n;++i)
		wp3->g[wp3->first + i] = concat[i];
	wp3->last = wp3->first + n - 1;
	Free_dp((dp)concat); concat = 0;
} 	

void
word_append(wp1,wp2)
	word * wp1, * wp2;
{
	int N = wp1->space;
	int last = wp1->last;
	int n2 = (wp2->last) - (wp2->first) + 1;
	int i;
	gen * genp1, * genp2, * genp3;
	while (N - 1 - last <= n2)
		N *= 2;
	if (N > wp1->space){
		gen * genp=vzalloc2(gen,N);
		for (i=wp1->first;i<=last;i++)
			genp[i]=(wp1->g)[i];
		Free_dp((dp)(wp1->g));
		wp1->g = genp;
		wp1->space = N;
	}
	genp1 = (wp1->g) + last;
	genp3 = genp1 + n2;
	genp2 = (wp2->g) + (wp2->first);
	while (++genp1<=genp3) *genp1 = *(genp2++);
	wp1->last += n2;
}

void
word_invappend(wp1,wp2)
	word * wp1, * wp2;
{
	int N= wp1->space;
	int last = wp1->last;
	int n2 = (wp2->last) - (wp2->first) + 1;
	int i;
	gen * genp1, * genp2, * genp3;
	while (N - 1 - last <= n2)
		N *= 2;
	if (N > wp1->space){
		gen * genp=vzalloc2(gen,N);
		for (i=wp1->first;i<=wp1->last;i++)
			genp[i]=(wp1->g)[i];
		Free_dp((dp)(wp1->g));
		wp1->g = genp;
		wp1->space = N;
	}
	genp1 = (wp1->g) + last;
	genp3 = genp1 + n2;
	genp2 = (wp2->g) + (wp2->last);
	while (++genp1<=genp3) *genp1 = inv(*(genp2--));
	wp1->last += n2;
}

void
crepeat_subword(wp,repeat,lp,limit)
	word * wp;
	word * repeat;
	int * lp;
	int limit;
{
	int m = word_length(wp);
	if (limit==0)
		limit = (m+1)/2;
	else
		limit = min(limit,(m+1)/2);
	if (*lp < limit) {
		gen * twice;
		gen * buf1;
		gen * buf2;
		gen * newgp;
		int l;
		int i=0;
		newgp = vzalloc2(gen,limit);
	/* we need to initialize the space to zero here so that we can tell
whether or not it changes */
		twice = valloc2(gen,2*m);
		for (l=0;l<m;++l) {
			twice[l]=wp->g[wp->first+l];
			twice[m+l]=twice[l];
		}
		while (i<m && *lp<limit) {
			int j;
			j = i+(*lp)+1;
			buf1 = twice + i;
			while (j<m && j<m+i-*lp && *lp<limit){
				int h=0;
				if (*(lp)==j-i) {
					j++;
					continue;
				}
				buf2 = twice + j; /*points to the j-th entry */
				while (*lp< limit && i+h<j
					&& j+h<m+i && buf1[h]==buf2[h]) {
					/* matching strings of length h+1 */
					if (h==*lp) 
						(*lp)++;
					h++;
				}
				if (h==*lp) 
					for (l=0;l<*lp;l++)
						newgp[l]=buf1[l];	
				j++;
			}
			i++;
		}
		Free_dp((dp)twice);
		twice=0;
		if (newgp[0]!=0) {
			int N;
			N = repeat->space;
			while (N < 2*(*lp))
				N *=2;
			word_stretch_reset(repeat,N,N/4);
			for (l=0;l<*lp;l++)
				repeat->g[l+repeat->first]=newgp[l];
			repeat->last=repeat->first+*lp-1;
		}
		Free_dp((dp)newgp);
		newgp=0;
	}
}


void
cinv_repeat_subword(wp,repeat,lp,limit)
	word *wp;
	word *repeat;
	int *lp;
	int limit;
{
	int m;
	m = word_length(wp);
	if (limit==0)
		limit = (m+1)/2;
	else
		limit = min(limit,(m+1)/2);
	if (*lp<limit){
		int l;
		int i=0;
		gen * newgp;
		gen * twice;
		gen * buf1;
		gen * buf2;
		newgp = vzalloc2(gen,limit);
/* we need to initialize the space to zero here, so that we can tell whether
or not a change has been made */
		twice = valloc2(gen,2*m);
		for (l=0;l<m;++l) {
			twice[l]=wp->g[wp->first+l];
			twice[m+l]=twice[l];
		}
		while (i<m && *lp<limit) {
			int h;
			int j=i+1+2*(*lp);
			buf1 = twice + i;
			while (j<m+(*lp) && j<m+i && *lp<limit){
				h=0;
				buf2 = twice + j;
				while (*lp<limit && i+h<j-h
						&& (buf1[h]==buf2[0])) {
					if (h==(*lp)) 
						(*lp)++;
					h++;
					buf2=buf2-1;
				}
				if (h==*lp)
					for(l=0;l<*lp;l++)
						newgp[l]=buf1[l];
				j++;
			}
			i++;
		}
		Free_dp((dp)twice);
		twice=0;
		if (newgp[0]!=0) {
			int N;
			N = repeat->space;
			while (N < 2*(*lp))
				N *=2;
			word_stretch_reset(repeat,N,N/4);
			for (l=0;l<*lp;l++)
				repeat->g[l+repeat->first]=newgp[l];
			repeat->last=repeat->first+*lp-1;
		}
		Free_dp((dp)newgp);
		newgp=0;
	}
}

void
cmatch_subword(wp1,wp2,repeat,lp,limit)
	word * wp1;
	word * wp2;
	word* repeat;
	int *lp;
	int limit;
{
	int m,n;
	m = word_length(wp1);
	n = word_length(wp2);
	assert(m<=n);
	if (limit==0)
		limit = m;
	else
		limit = min(m,limit);
	if (*lp<limit){
		int l;
		int i=0;
		gen * newgp;
		gen * twice1;
		gen * twice2;
		gen * buf1;
		gen * buf2;
		newgp = vzalloc2(gen,limit);
/* we need to initialize this space to zero, so that we can tell whether or
not a change has been made */
		twice1 = valloc2(gen,2*m);
		for (l=0;l<m;++l) {
			twice1[l]=wp1->g[wp1->first+l];
			twice1[m+l]=twice1[l];
		}
		twice2 = valloc2(gen,2*n);
		for (l=0;l<n;++l) {
			twice2[l]=wp2->g[wp2->first+l];
			twice2[n+l]=twice2[l];
		}
		while (i<m && *lp<limit) {
			int j=0;
			int h;
			buf1 = twice1 + i;
			while (j<n && *lp<limit){
				h=0;
				buf2 = twice2 + j;
				while (*lp<limit  && (buf1[h]==buf2[h])) {
					if (h==(*lp)) 
						(*lp)++;
					h++;
				}
				if (h==(*lp))
					for (l=0;l<*lp;l++)
						newgp[l]=buf1[l];
				j++;
			}
			i++;
		}
		Free_dp((dp)twice1);
		twice1=0;
		Free_dp((dp)twice2);
		twice2=0;
		if (newgp[0]!=0) {
			int N;
			N = repeat->space;
			while (N < 2*(*lp))
				N *=2;
			word_stretch_reset(repeat,N,N/4);
			for (l=0;l<*lp;l++)
				repeat->g[l+repeat->first]=newgp[l];
			repeat->last=repeat->first+*lp-1;
		}
		Free_dp((dp)newgp);
		newgp=0;
	}
}
		
void
word_csubst(wp,subwp,x)
	word * wp;
	word * subwp;
	gen x;
{
	gen* buf;
	gen * subgp;
	gen * begin;
	gen * newgp;
	int posn=0;
	int first;
	int last;
	int m,n,i,j;
	subgp = subwp->g + subwp->first;
	buf = wp->g + wp->first;
	begin = wp->g + wp->first;
	newgp = valloc2(gen,wp->space);
	first= wp->first;
	last = first-1; 
	m = word_length(wp);
	n = word_length(subwp);
	while (posn < m) {
		i=0;
		while (i<n && posn + i < m && buf[i]==subgp[i]) {
			i++;
			if (posn + i == m) {
				j=0;
				while (i<n && begin[j] == subgp[i]) {
					i++; j++;
				}
			}
		}
		if (i==n) {
			last++;
			newgp[last]=x;
			posn += n;
			if (posn>m) /* we've substituted for a substring
that wraps back round to the begininning of the word, so we need to delete the
generators in that string from the beginning of the word */
				first = first+j;
			buf += n;
		}
		else { /* no substitution was possible */
				last++;
				newgp[last]=buf[0];
				buf++;
				posn++;
		}
	}
	wp->first = first;
	wp->last = last;
	Free_dp((dp)wp->g);wp->g=0;
	wp->g = newgp;
}
			
	
	



/* The signature of an ordering of two words. 
Words are ordered by length and lexicographically (that is, w1 is
earlier than w2 in the ordering if it is shorter than w2 or if the two words
have the same length but in the first position where they differ the
generator in w1 is lower valued than the generator in w2). This function returns
1 if the two words are given with the earlier ordered one first, -1 if the
earlier ordered comes second, 0 if the two words are equal.
The arguments of the function must be datapointers, since this is a function
in the function table for lists of words; another version of the function
exists which uses pointers to words as its arguments. 
 We compare the two words lexicographically first, calculating the
difference between them in the first position where they differ. Then we
compare their lengths. Lexicographic comparison is only relevant for two words
of the same length.
*/
int
word_sgn_dp(dtp1,dtp2) 
	dp dtp1, dtp2; 
{ 
	int diff=0; 
	int ans = 0; 
	int n1, n2;
	int i=0;
	word * wp1, *wp2;
	gen * gp1, * gp2; 

	wp1 = (word *)dtp1;
	wp2 = (word *)dtp2;

	n1 = word_length(wp1);
	n2 = word_length(wp2);
	gp1 = wp1->g + wp1->first; 
	gp2 = wp2->g + wp2->first; 
	if (n1 < n2)
		ans = 1;
	else if (n1 > n2)
		ans = -1;
	else  { /* the two words have the same length */
		while(diff == 0 && i < n1) { 
			diff = (int)gp1[i] - (int)gp2[i]; 
			i++;
		} 
		if (diff > 0) 
			ans = -1;  
		else if (diff < 0) 
			ans = 1; 
	}
	return ans; 
} 

int 
genwt_word_sgn_dp(dp1,dp2)
	dp dp1, dp2; 
{ 
	int ans = 0; 
	word * wp1, *wp2;
	int m1=0;
	int m2=0;
	word_traverser wt;
	gen g;

	wp1 = (word *)dp1;
	wp2 = (word *)dp2;

	word_traverser_init(&wt,wp1);
	while (word_next(&wt,&g))
		if (g>m1)
			m1=g;	
	word_traverser_clear(&wt);
	word_traverser_init(&wt,wp2);
	while (word_next(&wt,&g))
		if (g>m2)
			m2=g;	
	word_traverser_clear(&wt);
	if (m1 < m2)
		ans = 1;
	else if (m1 > m2)
		ans = -1;
	else  
		ans=word_sgn(wp1,wp2);
	return ans; 
} 

/* Copy the word pointed at by |oldp| (a data pointer) into the address
specified by |newp| (also a data pointer).
Before the application of the function |newp| can point either to a
freshly initialized (empty) word or to an actual word of any length.
We need to interpret the data pointers as pointers to  words,  |oldwp| and
|newwp|. Once we have reallocated space (according to the length of the word
|*oldwp|) to the pointer |newwp| we copy the word across one generator at a 
time. 
*/
void word_cpy_dp(oldp,newp) 
	dp oldp, newp; 
{ 
		int m;
		int n;
		int i;
		word *newwp, *oldwp; 
		gen *newgp, *oldgp; 
		
		assert(newp); 
		assert(oldp);
		if (newp == oldp) return; /* nothing to do */ 
		newwp = (word*)newp; 
		oldwp = (word*)oldp;
		m = word_length(oldwp); 
		if ((n=newwp->space) < 2*m) {
			while (n<2*m)
				n *= 2;
			Free_dp((dp)newwp->g); newwp->g = 0; 
			newwp->g = valloc2(gen,n); 
			newwp->space = n;
		}
		newwp->first = m/2;
		newwp->last = newwp->first+m-1;
		newgp = newwp->g + newwp->first; 
		oldgp = oldwp->g + oldwp->first; 
		for (i=0;i<m;++i)
			newgp[i]=oldgp[i];
} 


/* Create initialized space for a word.
The function returns a pointer of type |word *| assigned initialized space equal
in size to the size of a word.
*/
word *  word_create()

{ 
		word * wp; 
		wp = vzalloc1(word); 
		word_init(wp); 
		return (wp); 
} 

/* Retrieve the space occupied by a word.
Both the space the word itself occupies and the space assigned to it as a
pointer to generators must be reclaimed. 
*/
void word_kill_dp(dtp)
	dp dtp; 
{ 
	word_clear((word *) dtp); 
	Free_dp(dtp); 
} 

/* Print a word.
This prints out a word  currently stored as a string of generators in the
program's notation for generators as a string of generators 
in the notation used by the user.
This is a function table function, so accepts a datapointer argument. It
also exists coerced to accept a |word *| argument.
*/
void word_print_dp(wfile,dtp) 
	FILE * wfile;
	dp dtp; 
{ 
		word * wp; 
		word user_word;
		word_init(&user_word);
		wp = (word *)dtp; /* coerce to type |word *| */ 
		word2user_name(wp,&user_word);
		user_word_print(wfile,&user_word);
		word_clear(&user_word);
} 

/* Print out a word. This is the same as the above except that it takes a
|word *| rather than a |dp| argument.
*/
void word_print(wfile,wp) 
	FILE * wfile;
	word * wp; 
{ 
		word user_word;
		word_init(&user_word);
		word2user_name(wp,&user_word);
		user_word_print(wfile,&user_word);
		word_clear(&user_word);
} 

/* Print a word.
This prints out a word as a string of generators interpreted as alphabet
characters.
*/
void user_word_print(wfile,wp) 
	FILE * wfile;
	word * wp; 
{ 
		gen * gp; 
		int i;
		gp = wp->g; 
		assert(gp);
		for (i=wp->first;i<=wp->last;i++) {
			assert(isascii(gp[i]) && isprint(gp[i]));
			if (gp[i]=='$')
				fprintf(wfile,"$");
			else
				fprintf(wfile,"%c",gp[i]); 
				
		} 
		if (word_length(wp)==0)
			fprintf(wfile,"epsilon");
} 

boolean read_next_word(wp,file)
	word * wp;
	FILE * file;
{
	boolean ans=TRUE;
	int c;
	word_reset(wp);
	while ((c=getc(file))!=EOF && !(isalpha(c)) &&!(isdigit(c))&& c!='\('
		&& c!='['){
		if (c=='$') 
			return TRUE;
		if (c=='}') {
	/* a '}' is used to terminate the input of words */
			ungetc('}',file);
			ans = FALSE;
			break;
		}
		else if (c=='#') /* comment symbol, skip to the next newline */
			while ((c=getc(file))!=EOF && c!='\n')
				;
	}
	if (ans==TRUE){
		gen separator = '*'; 
		int bracket_level=0;
		do {
			if (c==EOF)
				break;
			if (c=='\(')
				bracket_level++;
			if (c=='\)')
				bracket_level--;	
			if (bracket_level>=0)
				word_put_last(wp,c);
			else
				break;
	/* word reading is terminated by an unmatched right bracket */
			if (c=='*'){
				 do {
					c=getc(file);
					if (c==EOF)
						bad_data();
					} while ((c==' '||c=='\t'||c=='\n'));
	/* we can put spaces etc. into a word after *'s. */
			}
			else 
				c = getc(file);
			if (c==EOF)
				bad_data();
		} while (isalpha(c) ||isdigit(c)
			||c=='^'||c=='-'||c=='('||c==')'||c=='['||c==']'||c==','||c=='*');
		ungetc(c,file);
	}
	if (c==EOF)
		bad_data();
	else
		word_expand(wp);
	return ans;
}

static void
word_expand(wp)
{
	word expansion;
	word buf;
	word temp;
	word temp2;
	gen g;
	int separator= '*';
	int bracket_level=0;
	int commutator_level=0;
	word_init(&expansion);
	word_init(&buf);
	word_init(&temp);
	word_init(&temp2);
	while (word_delget_first(wp,&g)){
		if (isalpha(g) || isdigit(g))
			word_put_last(&buf,g);
		else if (g=='*'){
			word_append(&expansion,&buf);
			word_put_last(&expansion,g);
			word_reset(&buf);
		}
		else if (g=='^'){
/*	read the exponent that follows and replace whatever's currently in
the buf by the approriate number of copies of it or its inverse */ 
			int exponent=0;
			int sign=1;
			int i;
			(void)word_delget_first(wp,&g);
			if (g=='-'){
				sign = -1;
				(void)word_delget_first(wp,&g);
			}
			if (isdigit(g)){
				int exponent=0;
			/*Now g must be a digit */
				exponent= g - '0' + 10*exponent;
				while (word_get_first(wp,&g) && isdigit(g)){
						exponent= g - '0' + 10*exponent;
						(void) word_delget_first(wp,&g);
				}
				if (sign == -1) {
					gen h;
					word2prog_word(&buf,&temp);
					if (word_length(&temp)==1 && word_get_last(&temp,&h) &&
							 (inv_of==0 || inv_of[h]==0)){
/* in this case we're reading the name of a new generator */
						word_put_last(&buf,'^');
						word_put_last(&buf,'-');
						word_put_last(&buf,'1');
					}
					else {
						word_inv(&temp,&temp);
						word2user_name(&temp,&buf);
					}
				}
				word_cpy(&buf,&temp);
				word_reset(&buf);
				for (i=1;i<=exponent;i++){
					word_append(&buf,&temp);
					if (i<exponent)
						word_put_last(&buf,separator);
				}
				word_reset(&temp);
			}
			else {
				word exponent;
				bracket_level=0;
				commutator_level=0;
				word_init(&exponent);
				do{
					if (commutator_level==0 && bracket_level==0 &&
					   (g=='*'||g=='^')){
						word_put_first(wp,g);
						break;
					}
					if (g=='\(')
						bracket_level++;
					else if (g=='\)')
						bracket_level--;
					if (g=='[')
						commutator_level++;
					else if (g==']')
						commutator_level--;
					word_put_last(&exponent,g);
				} while (word_delget_first(wp,&g));
				word_expand(&exponent);
				word2prog_word(&exponent,&temp);
				word_inv(&temp,&temp);
				word2user_name(&temp,&temp2);
				word_put_last(&temp2,separator);
				word_concat(&temp2,&buf,&buf);
				word_put_last(&buf,separator);
				word_append(&buf,&exponent);
				word_reset(&temp);
				word_reset(&temp2);
				word_clear(&exponent);
			}
		}
		else if (g=='\(') {
			bracket_level = 1;
			while (word_delget_first(wp,&g)){
				if (g=='\(')
					bracket_level++;
				else if (g=='\)')
					bracket_level--;
				if (bracket_level==0)
					break;
				word_put_last(&buf,g);
			}
			word_expand(&buf);
		}
		else if (g=='[') {
			word w1, w2;
			commutator_level = 1;
			word_init(&w1);
			while (word_delget_first(wp,&g)){
				if (g==',' && commutator_level==1)
					break;
				else if (g=='[')
					commutator_level++;
				else if (g==']')
					commutator_level--;
				word_put_last(&w1,g);
			}
			word_init(&w2);
			while (word_delget_first(wp,&g)){
				if (g=='[')
					commutator_level++;
				else if (g==']'){
					commutator_level--;
					if (commutator_level==0)
						break;
				}
				word_put_last(&w2,g);
			}
			word_expand(&w1);
			word_expand(&w2);
			word2prog_word(&w1,&temp);
			word_inv(&temp,&temp);
			word2user_name(&temp,&buf);
			word_put_last(&buf,separator);
			word2prog_word(&w2,&temp);
			word_inv(&temp,&temp);
			word2user_name(&temp,&temp2);
			word_append(&buf,&temp2);
			word_put_last(&buf,separator);
			word_append(&buf,&w1);
			word_put_last(&buf,separator);
			word_append(&buf,&w2);
			word_reset(&temp);
			word_reset(&temp2);	
			word_clear(&w1); word_clear(&w2); 
		}
	}
	word_append(&expansion,&buf);
/* there's no * at the end */
	word_cpy(&expansion,wp);
	word_clear(&expansion);
	word_clear(&temp);
	word_clear(&temp2);
	word_clear(&buf);
}


boolean
common_prefix_erased (w1,w2)
	word *w1;
	word *w2;
{
	boolean ans = FALSE;
	gen g1,g2;
	while (word_get_first(w1,&g1) && word_get_first(w2,&g2) && (g1 == g2)) {
			ans = TRUE;
			(void)word_del_first(w1);
			(void)word_del_first(w2);
	}
	return ans;
}

/*
\Pre both |w1| and |w2| point to initialized words.
\Post The words pointed to have had the longest common suffix
erased.
\Returns |TRUE| if a non-trivial subword has been erased.
*/
boolean
common_suffix_erased (w1,w2)
	word *w1;
	word *w2;
{
	boolean ans = FALSE;
	gen g1,g2;
	while (word_get_last(w1,&g1) && word_get_last(w2,&g2) && (g1 == g2)) {
			ans = TRUE;
			(void)word_del_last(w1);
			(void)word_del_last(w2);
	}
	return ans;
}

/*
\Pre |wp| points to an initialized word.
\Returns |TRUE| if |wp| points to an empty word, and |FALSE| otherwise.
*/
boolean
word_empty(wp)
	word *wp;
{
	assert(wp);
	if (wp->first > wp->last)
		return TRUE;
	else
		return FALSE;
}

/*
\Pre |wp| points to an initialized word.
\Post |gp| points to the last letter of |*wp|, |*wp| is unchanged.
word.
\Returns |TRUE| if the word pointed to by |wp| is non-empty, and |FALSE|
if it is empty.
*/
boolean
word_get_last(wp,gp)
	word *wp;
	gen *gp;
{
	boolean ans = TRUE;
	if (wp->first > wp->last) {
		*gp = INVALID_GEN;
		ans = FALSE;
	}
	else
		*gp = wp->g[wp->last];
	return ans;
}
	
/*
\Pre |wp| points to an initialized word.
\Post |gp| points to the last letter of |*wp|, which is deleted from that
word.
\Returns |TRUE| if the word pointed to by |wp| is non-empty, and |FALSE|
if it is empty.
*/
boolean
word_delget_last(wp,gp)
	word *wp;
	gen *gp;
{
	boolean ans = TRUE;
	if (wp->first > wp->last) {
		*gp = INVALID_GEN;
		ans = FALSE;
	}
	else {
		*gp = wp->g[wp->last];
		wp->last--;
	}
	return ans;
}
	

void
lex_first_word(wp,n)
	word * wp;
	int n;
{
	int i;
	int m;
	if ((m=wp->space) < 2*n) {
		while (m<2*n)
			m *= 2;
		Free_dp((dp)wp->g); wp->g = 0; 
		wp->g = valloc2(gen,m); 
		wp->space = m;
	}
	wp->first = n/2;
	wp->last = wp->first + n-1;	
	for (i=wp->first;i<=wp->last;++i)
		wp->g[i]=1;
}

/*The word |*wp| is replaced by the word of the same length next in the
lexicographic ordering. If |*wp| is the highest ordered word of that
length the function returns |FALSE|, and |*wp| is set equal to the
lowest ordered word of that length. Otherwise the function returns |TRUE|.
*/
boolean
lex_next_word(wp)
word * wp;
{
	boolean ans = FALSE;
	int i=wp->last;
	while (wp->g[i]==num_gens && i>=wp->first){
		wp->g[i]=1;
		i--;
	}
	if (i>=wp->first) {
		wp->g[i] += 1;
		ans = TRUE;
	}
	return ans;
}
	
void
gen_print(wfile,g)
	FILE * wfile;
	gen g;
{
	word w;
	word_init(&w);
	if (g==IDENTITY)
		fprintf(wfile,"\"the identity\"");
	else if (g==num_gens+1)
		fprintf(wfile,"$");
	else {
		gen2user_name(g,&w);
		user_word_print(wfile,&w);
	}
	word_clear(&w);
}
		
int
gen_sgn_dp(gp1,gp2)
	dp gp1, gp2; /*really these are pointers to |gen|'s*/
{
	boolean ans = 0;
	gen *g1, *g2;
	g1 = (gen *)gp1;
	g2 = (gen *)gp2;
	if (*g1 < * g2)
		ans = 1;
	else if (*g1 > *g2)
		ans = -1;
	return ans;
}

void
gen_cpy_dp(oldp,newp)
	dp oldp, newp; /*these are really |gen *|'s*/
{
	gen * newg = (gen *)newp;
	gen * oldg = (gen *)oldp;
	*newg = *oldg;
}
gen *
gen_create()
{
	return valloc1(gen);
}
void
gen_print_dp(wfile,gp)
	FILE * wfile;
	dp gp; /*really a |gen *|*/
{
	word w;
	word_init(&w);
	gen2user_name(*((gen *)gp),&w);
	user_word_print(wfile,&w);
	word_clear(&w);
}

void
genstring(string,g)
	char * string;
	gen g;
{ 
	if (g==0){
		string[0]='0';
		string[1]='\0';
	}
	else {
		word w;
		gen h;
		int length;
		int i=0;
		word_init(&w);
		gen2user_name(g,&w);
		length = word_length(&w);	
		while(word_delget_first(&w,&h))
		string[i++]=(char)h;
		string[length]= '\0';
		word_clear(&w);
	}
}

void
word2prog_gen(user_namep,prog_genp) 
	word* user_namep;  
	gen * prog_genp;
{ 

  gen g; 

	*prog_genp=INVALID_GEN;
  for (g=1;g<=num_gens;++g) 
       if (word_sgn(user_namep,user_gen_name+g)==0) { 
             *prog_genp = g; 
             break; 
       } 
}  

void
word2prog_word(user_wordp,prog_wordp)
	word * user_wordp;
	word * prog_wordp;
{
	word  user_gen;
	word_traverser wt;
	gen g;
	gen h;
	int i;
	int n=word_length(user_wordp);
	char* epsilon ="epsilon";
	word identity;
	word_init(&identity);
	for (i=0;i<=6;i++)
		word_put_last(&identity,(gen)epsilon[i]);
	i=0;
	word_reset(prog_wordp);
	if (word_sgn(user_wordp,&identity)!=0){
		word_init(&user_gen);
		word_traverser_init(&wt,user_wordp);
		while (word_next(&wt,&g)){
			i++;
			if (g!='*')
				word_put_last(&user_gen,g);
			if (g=='*'||i==n){
				word2prog_gen(&user_gen,&h);
				word_reset(&user_gen);
				word_put_last(prog_wordp,h);
			}
		}
		word_traverser_clear(&wt);
		word_clear(&user_gen);
	}
	word_clear(&identity);
}

void
gen2prog_gen(user_gen,prog_genp)
	gen user_gen;
	gen * prog_genp;
{
	word w;
	word_init(&w);
	word_put_last(&w,user_gen);
	word2prog_gen(&w,prog_genp);
	word_clear(&w);
}


void
gen2user_name(program_gen,user_wordp) 
	gen program_gen;  
	word * user_wordp; 
{ 

	word_reset(user_wordp);
	if (program_gen==0||(num_gens!=0 && program_gen>num_gens)){
		gen g='$';
		word_put_last(user_wordp,g);
	}
	else 
 		word_cpy(user_gen_name+program_gen,user_wordp); 

}  

void
word2user_name(prog_wordp,user_wordp)
	word * prog_wordp, * user_wordp;
{
	word w;
	gen separator='*';
	gen g=INVALID_GEN;
	word_traverser wt;
	word_traverser_init(&wt,prog_wordp);
	word_reset(user_wordp);
	word_init(&w);
	while (word_next(&wt,&g)){
		gen2user_name(g,&w);
		word_append(user_wordp,&w);
		word_put_last(user_wordp,separator);
		word_reset(&w);
	}
	word_traverser_clear(&wt);
	word_del_last(user_wordp);
	word_clear(&w);
}

void
read_gen_name_array(file)
	FILE * file;	
{
	word w;
	int i;
	num_gens=0;
	user_gen_name = vzalloc2(word,gen_array_size);
	for (i=0;i<gen_array_size;++i)
		word_init(user_gen_name+i);
	while (getc(file)!='\{')
		;
	word_init(&w);
	while (read_next_word(&w,file)){
		define_next_gen(&w);
		word_reset(&w);
	}
	word_clear(&w);
	while (getc(file)!='\}')
		;
}

void
define_next_gen(wp)
	word * wp;
{
	word * ucpy;
	int size = gen_array_size;
	num_gens++;
	if (num_gens==gen_array_size-1){ 
	/* need more space */
		int i;
		gen_array_size = 2*num_gens + 1;
		ucpy = vzalloc2(word,gen_array_size);
		for (i=0;i<gen_array_size;++i) 
			word_init(ucpy+i);
		for (i=0;i<num_gens;++i){ 
			word_cpy(user_gen_name+i,ucpy+i);
			word_clear(user_gen_name+i);
		}
		Free_dp((dp)user_gen_name); user_gen_name=0;
		user_gen_name = ucpy;
		ucpy=0;
	}
	word_cpy(wp,user_gen_name+num_gens);
}

void
read_inverse_array(file)
	FILE * file;
{
	int c;
	gen g,h,k;
	word w;
	if (inv_of){
		fprintf(stderr,"Inverses already set, about to be overwritten.\n");
		bad_data();
	}
	inv_of=vzalloc2(gen,gen_array_size);
	word_init(&w);
	while (getc(file)!='\{')
		;
	while ((c=getc(file))==' '||c=='\n'||c=='\t')	
		;
	if (c=='f'){ /* first letter of formal */
		for (g=1;g<=num_gens;g++){
			word_traverser wt;
			word_traverser_init(&wt,user_gen_name+g);
			while (word_next(&wt,&h)){
				if (h=='^')
					break;
				else
					word_put_last(&w,h);
			}
			word_traverser_clear(&wt);
			if (h!='^'){
				word_put_last(&w,'^');
				word_put_last(&w,'-');
				word_put_last(&w,'1');
			}
			word2prog_gen(&w,&k);
			if (k==INVALID_GEN){
/* There's no generator yet defined with user name w, so we have to slot one
in, just after g.  */
				insert_gen(g+1,&w);
				k = g+1;
			}
			inv_of[g]=k;
			inv_of[k]=g;
			word_reset(&w);
		}
	}
	else if (c=='i'){ /*first letter of inv */
		char * label;
		label=vzalloc2(char,4);
		ungetc(c,file);
		while (read_next_string(label,3,file)){
			if (strcmp(label,"inv")!=0)
				bad_data();
			while (getc(file)!='\(')
				;
			read_next_word(&w,file);
			word2prog_gen(&w,&g);
			while (getc(file)!='\=')
				;
			read_next_word(&w,file);
			word2prog_gen(&w,&k);
			if (k==INVALID_GEN){
/* There's no generator yet defined with user name w, so we have to slot one
in, just after g.  */
				insert_gen(g+1,&w);
				k = g+1;
			}
			inv_of[g]=k; inv_of[k]=g;
		}
		Free_dp((dp)label);
		default_inverse_array();
	}
	else  {
		ungetc(c,file);
		g=1;
		while (read_next_int(inv_of+g,file))
			g++;
		if (g==1) /* i.e. no entries */
			default_inverse_array();
	}
	word_clear(&w);
	while (getc(file)!='\}')
		;
}

static void
insert_gen(h,wp)
	gen h;
	word * wp;
{
	word * ucpy;
	gen * icpy;
	int i;
	int size = gen_array_size;
	num_gens++;
	if (num_gens==gen_array_size-1) 
	/* need more space */
		gen_array_size = 2*num_gens + 1;
	ucpy = vzalloc2(word,gen_array_size);
	icpy = vzalloc2(gen,gen_array_size);
	for (i=0;i<gen_array_size;++i) 
		word_init(ucpy+i);
	for (i=0;i<h;++i){ 
		word_cpy(user_gen_name+i,ucpy+i);
	}
	word_cpy(wp,ucpy+h);
	for (i=h;i<=num_gens-1;++i) 
		word_cpy(user_gen_name+i,ucpy+i+1);
	for (i=0;i<size;++i)
		word_clear(user_gen_name+i);
	Free_dp((dp)user_gen_name); user_gen_name=0;
	user_gen_name = ucpy;
	ucpy=0;
	for (i=0;i<h;++i){
		if (inv_of[i]>h) icpy[i] = inv_of[i]+1;
		else icpy[i]=inv_of[i];
	}
	for (i=h+1;i<num_gens;i++){
		if (inv_of[i]>h) icpy[i+1] = inv_of[i]+1;
		else icpy[i+1] = inv_of[i];
	}
	inv_of = icpy;
	icpy = 0;
}

void
delete_gen(h)
	gen h;
{
	word * ucpy;
	gen * icpy;
	int i;
	num_gens--;
	ucpy = vzalloc2(word,gen_array_size);
	icpy = vzalloc2(gen,gen_array_size);
	for (i=0;i<gen_array_size;++i) 
		word_init(ucpy+i);
	for (i=0;i<h;++i){ 
		word_cpy(user_gen_name+i,ucpy+i);
	}
	for (i=h+1;i<=num_gens+1;++i) 
		word_cpy(user_gen_name+i,ucpy+i-1);
	for (i=0;i<gen_array_size;++i)
		word_clear(user_gen_name+i);
	Free_dp((dp)user_gen_name); user_gen_name=0;
	user_gen_name = ucpy;
	ucpy=0;
	for (i=0;i<h;++i){
		if (inv_of[i]>h) icpy[i] = inv_of[i]-1;
		else icpy[i]=inv_of[i];
	}
	for (i=h+1;i<num_gens;i++){
		if (inv_of[i]>h) icpy[i-1] = inv_of[i]-1;
		else icpy[i-1] = inv_of[i];
	}
	inv_of = icpy;
	icpy = 0;
}

void
default_inverse_array()
{
	word w;
	word inverse;
	gen h,j,k;
	gen l=INVALID_GEN;
	word_init(&w);
	word_init(&inverse);
	if (inv_of==0)
		inv_of=vzalloc2(gen,gen_array_size);
	for (h = 1; h <= num_gens; ++h) { 
		if (inv_of[h]==0){
	/* we may have set some inverses already explicitly */
			gen2user_name(h,&w);
			if (word_length(&w)!=1){
				fprintf(stderr,
					"Default inverses are only defined for generators\n");
				fprintf(stderr,
							"which are single alphabet characters.\n"); 
				bad_data();
			}
			(void)word_delget_first(&w,&j);
			if (islower(j))
				k=toupper(j);
			else
				k=tolower(j);
			gen2prog_gen(k,&l);
			if (l==INVALID_GEN){
/* There's no generator yet defined with user name k, so we have to slot one
in, just after h.  */
				word_put_first(&inverse,k);
				l=h+1;
				insert_gen(l,&inverse,&inv_of);
				word_reset(&inverse);
			}
			inv_of[h] = l; inv_of[l] = h;
			word_reset(&w);
		}
	} 
	word_clear(&inverse);
	word_clear(&w);
} 

boolean
read_next_rel(relp,rfile)
	word * relp;
	FILE * rfile;
{
	boolean ans=FALSE;
	word uw1;
	word_traverser wt;
	gen g;
	word_init(&uw1);
	ans = read_next_word(&uw1,rfile);
	if (ans){
		int c;
		word2prog_word(&uw1,relp);
		while ((c=getc(rfile))==' '||c=='\t'||c=='\n');
		if (c=='=' || c=='>'){
			word uw2, pw2;
			word_init(&uw2);
			word_init(&pw2);
			read_next_word(&uw2,rfile);
			word2prog_word(&uw2,&pw2);
			word_inv(&pw2,&pw2);
			word_append(relp,&pw2);
			word_clear(&uw2);
			word_clear(&pw2);
		}
		else
			ungetc(c,rfile);
	}
	word_clear(&uw1);
	word_traverser_init(&wt,relp);
	while (word_next(&wt,&g)){
		if (g==INVALID_GEN){
			fprintf(stderr,"Undeclared generator in relator.\n");
			bad_data();
		}
	}
	word_traverser_clear();
	return ans;
}
