#include "defines.h"

/****************************************************************/
/*	Top-level Delaunay Triangulation Procedure
/****************************************************************/

triangulate()
/*	Allocate storage, construct triangulation, compute voronoi corners  */
{
    ef_alloc(n_sites);
    delete_all_edges();
    delete_all_faces();

    build_delaunay_triangulation( n_sites );
    consolidate_edges();
    find_vor_corners();
}

static SITE_PTR *sp;

build_delaunay_triangulation( size )
    /* builds delaunay triangulation.
       va is an array of vertices, from 0 to size-1.  Each vertex consists of
       a vector and a data pointer.   One_bndry_edge is a pointer to an
       edge on the convex hull of the constructed delaunay triangulation. */

int size;
{
    int i, rows;
    EDGE_PTR lefte, righte;
    double sqrt(), log();

    rows = (int)( 0.5 + sqrt( (double) size / log( (double) size )));

    my_alloc( sp, SITE_PTR, size );

    /*  Sort the pointers by  x-coordinate of site  */
    for ( i=0 ; i < size ; i++ )   sp[i] = i;
    spsortx( sp, 0, size-1 ); 

    build_delaunay( 0, size-1, &lefte, &righte, rows );
    One_bndry_edge = lefte;

    free( (char *) sp );
}

/****************************************************************/
/*	Recursive Delaunay Triangulation Procedure
/*	Contains modifications for axis-switching division.
/****************************************************************/

build_delaunay( lo, hi, le, re, rows )
int lo,hi,rows;
EDGE_PTR *le,*re;
{
    EDGE_PTR a,b,c,ldo,rdi,ldi,rdo;
    int split, lowrows;
    register int low, high, maxx, minx;
    SITE_PTR s1, s2, s3;
    low = lo; high = hi;

    if ( low < (high-2) ) {
        /* more than three elements; do recursion */
	minx = sp[low]; maxx = sp[high];
	if (rows == 1) {	/* time to switch axis of division */
	    spsorty( sp, low, high);
	    rows = 65536;
	}	  
	lowrows = rows/2;
	split = low - 1 + (int) (0.5 +
	    ((double)(high-low+1) * ((double)lowrows / (double)rows)));
	build_delaunay( low, split, &ldo, &ldi, lowrows );
	build_delaunay( split+1, high, &rdi, &rdo, (rows-lowrows) );
	do_merge(&ldo, ldi, rdi, &rdo);
	while (orig(ldo) != minx) ldo = rprev(ldo);
	while (orig(rdo) != maxx) rdo = lprev(rdo);
	*le = ldo;
	*re = rdo;
    }
    else if (low >= (high - 1)) {	/* two or one points */
	a = makeedge(sp[low], sp[high]);
	*le = a;
	*re = sym(a);
	if (low==high) {printf("ERROR: Only 1 point!\n"); fflush(stdout);}
    }
    else { /*  (low == (high - 2))  */	/* three points */
	/* 3 cases: triangles of 2 orientations, and 3 points on a line. */
	a = makeedge((s1 = sp[low]), (s2 = sp[low+1]));
	b = makeedge(s2, (s3 = sp[high]));
	splice(sym(a), b);
	c = connect_left(b, a);
	if (ccw(s1, s3, s2)) {
	    *le = sym(c);
	    *re = c;
	}
	else {
	    *le = a;
	    *re = sym(b);
	    if (!ccw(s1, s2, s3)) deleteedge(c);    /* colinear */
	}
    }
}
