//$PolyShape,PolySketcher,SplineStretcher$
#include "PolyShape.h"
#include "DrawView.h"

const int cMaxPoints = 50;

static short PolyImage[]= {
#   include  "images/PolyShape.im"
};

//---- PolyShape ---------------------------------------------------------------

MetaImpl(PolyShape, (I_I(npts), I_PV(pts, npts)));

PolyShape::PolyShape()
{
    type= ePolyDefault;
}
	
PolyShape::~PolyShape()
{
    SafeDelete(pts);
}

short *PolyShape::GetImage()
{
    return PolyImage;
}

void PolyShape::Close()
{
    Point *tp= pts;
    
    if (Abs(tp[0] - tp[npts-1]) < 3) {
	type|= ePolyClose;
	npts--;
    }
    pts= new Point[npts];
    for (int i= 0; i < npts; i++)
	pts[i]= tp[i];
    Shape::Init(bbox.NW(), bbox.SE());
    pattern= ePatWhite;
    delete tp;
}

void PolyShape::Open()
{
    pts= new Point[cMaxPoints];
    npts= 0;
    bbox= gRect0;
    pattern= ePatNone;
    penpattern= ePatBlack;
    pensize= 1;
}

bool PolyShape::AddPt(Point p)
{
    if (npts >= cMaxPoints)
	return TRUE;
    p-= bbox.origin;
    pts[npts++]= p;
    CalcBBox();
    Invalidate();
    return FALSE;
}

void PolyShape::CalcBBox()
{
    Point oldorigin= bbox.origin;
    bbox= BoundingBox(npts, pts);
    bbox.origin+= oldorigin;
}

void PolyShape::MovePoint(int handle, Point delta)
{
    Invalidate();
    pts[handle]+= delta;
    CalcBBox();
    Invalidate();
    Changed();
}

void PolyShape::SetProperty(ShapeProperties what, int st)
{
    if (what == eShapeSmooth) {
	type&= ~(ePolyBezier|ePolySpline2|ePolySpline3);
	switch (st) {
	case 1:
	    type |= ePolyBezier;
	    break;
	case 2:
	    type |= ePolySpline2;
	    break;
	case 3:
	    type |= ePolySpline3;
	    break;
	}
    } else
	Shape::SetProperty(what, st);
}

int PolyShape::GetProperty(ShapeProperties what)
{
    if (what == eShapeSmooth) {
	if (type & ePolyBezier)
	    return 1;
	if (type & ePolySpline2)
	    return 2;
	if (type & ePolySpline3)
	    return 3;
	return 0;
    }
    return Shape::GetProperty(what);
}

bool PolyShape::CanSplit()
{
    return !GetSplit();
}

Point *PolyShape::MakeHandles(int *n)
{
    if (GetSplit()) {
	if (type & ePolyClose)
	    *n= npts-1;
	else
	    *n= npts;
	return pts;
    }
    return Shape::MakeHandles(n);
}

void PolyShape::Outline(Point p1, Point p2)
{
    GrStrokePolygon(NormRect(p1, p2).origin, pts, npts, type);
}

void PolyShape::Draw(Rectangle, Point p)
{
    Rectangle rr= bbox+p;
    GrFillPolygon(rr.origin, pts, npts, type);
    GrStrokePolygon(rr.origin, pts, npts, type);
}

ShapeSketcher *PolyShape::NewSketcher(DrawView *dv, SketchModes m)
{
    return new PolySketcher(dv, this, m, cMaxPoints);
}

ShapeStretcher *PolyShape::NewStretcher(DrawView *dv, int handle)
{
    if (GetSplit())
	return new SplineStretcher(dv, this, handle);
    return LineShape::NewStretcher(dv, handle);
}

bool PolyShape::ContainsPoint(Point p)
{
    p-= bbox.origin;
    if (penpattern != ePatNone) {
	for (int i= 1; i < npts; i++)
	    if (this->LineShape::PointAtLine(p, pts[i-1], pts[i]))
		return TRUE;
    }
    if (pattern != ePatNone)
	return TRUE;
}

ostream& PolyShape::PrintOn(ostream& s)
{
    Shape::PrintOn(s);
    s << npts SP;
    for (int i= 0; i < npts; i++)
	s << pts[i] SP;
    return s;
}

istream& PolyShape::ReadFrom(istream& s)
{
    Shape::ReadFrom(s);
    s >> npts;
    SafeDelete(pts);
    pts= new Point[npts];
    for (int i= 0; i<npts; i++)
	s >> pts[i];
    return s;
}

void PolyShape::MakePoints(int handle, Point delta, Point *p, int &i)
{
    int j, m= GetProperty(eShapeSmooth);
    
    CalcBBox();
    Point o= bbox.origin;
    pts[handle]+= delta;
    
    switch(m) {
    case 0:
	if (handle > 0)
	    p[i++]= pts[handle-1];
	p[i++]= pts[handle];
	if (handle < npts-1)
	    p[i++]= pts[handle+1];
	break;
    case 1:
	for (j= 0; j < 4; j++)
	    p[i++]= pts[j];
	break;
    case 2:
    case 3:
	for (j= handle-m; j <= handle+m; j++) {
	    if (j < 0)
		p[i++]= pts[0];
	    else if (j >= npts)
		p[i++]= pts[npts-1];
	    else
		p[i++]= pts[j];
	}
	break;
    }
    pts[handle]-= delta;
    
    for (j= 0; j < i; j++)
	p[j]+= o;
}

//---- Stretcher Methods -------------------------------------------------------

SplineStretcher::SplineStretcher(DrawView *dv, Shape *sp, int h) : (dv, sp, h) 
{
    lsp= (PolyShape*) sp;
    npts= 0;
}
    
void SplineStretcher::TrackFeedback(Point, Point, bool)
{
    if (npts > 0)
	GrStrokePolygon(gPoint0, pts, npts, (lsp->type & 0x7) | ePolyPartial);
}

void SplineStretcher::TrackConstrain(Point, Point, Point*)
{
}

Command *SplineStretcher::TrackMouse(TrackPhase tp, Point ap, Point, Point np)
{
    delta= np-ap;
    
    // DrawCommand::TrackMouse(tp, ap, pp, np);
    switch(tp) {
    case eTrackMove:
	npts= 0;
	lsp->MakePoints(handle, delta, pts, npts);
	break;
	
    case eTrackRelease:
	if (delta == gPoint0)
	    return gNoChanges;
	break;
	
    default:
	break;
    }
    return this;
}

void SplineStretcher::DoIt()
{
    lsp->MovePoint(handle, delta);
}

void SplineStretcher::UndoIt()
{
    lsp->MovePoint(handle, -delta);
}

//---- Sketcher Methods --------------------------------------------------------

PolySketcher::PolySketcher(DrawView *dv, Shape *sp, SketchModes m, int n)
							: (dv, sp, m) 
{
    maxpts= n;
    SetFlag(eCmdMoveEvents);
    newshape= (Shape*) proto->Clone();
    newshape->SetView(view);
    Poly()->Open();
    view->GetShapes()->Insert(newshape);
}

void PolySketcher::TrackFeedback(Point, Point np, bool)
{
    GrLine(lp, np);
}
 
Command *PolySketcher::TrackMouse(TrackPhase tp, Point ap, Point pp, Point np)
{
    if (tp != eTrackRelease)
	DrawCommand::TrackMouse(tp, ap, pp, np);
    view->ShowInfo(tp, "x: %4d y: %4d", np.x, np.y);
    switch (tp) {
    case eTrackPress:
	lp= ap;
	break;
    case eTrackRelease:
	Poly()->AddPt(np);
	int n= Poly()->GetPtCnt();
	if (n > 3 && (n >= maxpts || Abs(lp - np) < 3 || Abs(ap - np) < 3)) {
	    Poly()->Close();
	    view->GetShapes()->RemovePtr(newshape);
	    ResetFlag(eCmdMoveEvents);
	} else
	    view->UpdateEvent();
	lp= np;
	break;
    }
    return this;
}
