//$Document$
#include "Document.h"
#include "Application.h"
#include "Window.h"
#include "Alert.h"
#include "Menu.h"
#include "Error.h"
#include "CmdNo.h"
#include "String.h"
#include "FileDialog.h"
#include "System.h"
#include "DialogItems.h"
#include "Icon.h"
#include "ObjList.h"
#include "ObjectTable.h"

static FileDialog *loadDialog, *saveDialog, *importDialog;

ONEXIT(Document)
{
    SafeDelete(loadDialog);
    SafeDelete(saveDialog);
    SafeDelete(importDialog);
}

AbstractMetaImpl(Document, (I_O(application), I_I(changeCount), I_I(uniqueId),
	I_CS(docName), I_CS(docType), I_O(lastCmd), I_O(menu), I_B(isUntitled), 
	I_B(isOpen), I_B(isConverted), I_CS(loadDir), I_O(window), I_O(icon),
	I_O(windows)));

Document::Document(char *dt)
{
    changeCount= 0;
    lastCmd= gNoChanges;
    menu= new Menu("edit");
    icon= 0;
    window= 0;
    windows= new ObjList;
    isOpen= TRUE;
    application= 0;
    docName= 0;
    loadDir= 0;
    docType= strsave(dt);
    isUntitled= TRUE;
    isConverted= FALSE;
    uniqueId= 0;
}

Document::~Document()
{
    if (lastCmd && lastCmd != gResetUndo)
	Error("~Document", "lastCmd != gResetUndo");
    SafeDelete(lastCmd);
    SafeDelete(menu);
    SafeDelete(window);
    SafeDelete(docName);
    SafeDelete(docType);
    SafeDelete(loadDir);
    SafeDelete(icon);
    SafeDelete(windows);
}

FileDialog *Document::MakeFileDialog(FileDialogType)
{
    return new FileDialog;
}

void Document::SetName(char *name)
{
    if ((docName == 0) || (strcmp(name, docName) != 0)) {
	strreplace(&docName, name);
	if (window)
	    window->SetTitle(docName);
    }
}

EvtHandler *Document::GetNextHandler()
{
    return application;
}

void Document::SetApplication(Application *app)
{
    application= app;
}

void Document::OpenWindows()
{
    Point lastDocPos= application->GetNewDocumentPos();
    if (window= DoMakeWindows())
	windows->AddFirst(window);
    if (window= Guard(windows->First(), Window)) {
	DoMakeViews();
	char *name= window->GetTitle();
	if (name == 0 || *name == '\0')
	    window->SetTitle(docName, FALSE);
	window->OpenAt(lastDocPos);
    }
}

Window *Document::DoMakeWindows()
{
    return 0;
}

void Document::AddWindow(BlankWin *win)
{
    windows->Add(win);
}

void Document::DoMakeViews()
{
}

void Document::Toggle()
{
    if (icon == 0)
	icon= DoMakeIcon(GetName());
    if (window) {
	bool isopen= !icon->IsOpen();
	icon->OpenAt(isopen, window->GetRect().origin);
	windows->ForEach(BlankWin,Open)(!isopen);
    }
}

Icon *Document::DoMakeIcon(char *name)
{
    return new Icon(this, new TextItem(name, new Font(eFontTimes, 9)));
}

//---- Menus --------------------------------------------------------------------

Menu *Document::GetMenu()
{
    return menu;
}

void Document::DoCreateMenu(Menu *menu)
{
    Menu *file= new Menu("file");

    EvtHandler::DoCreateMenu(menu);

    file->AppendItems(
		     "load ...",                 cLOAD,
		     "-",
		     "save",                     cSAVE,
		     "save as ...",              cSAVEAS,
		     "revert",                   cREVERT,
		     "close",                    cCLOSE,
		     "-",
		     "application window",       cSHOWAPPLWIN,
		     0);
    
    if (CanImportDocument(0))
	file->InsertItemAfter(cLOAD, "import ...", cIMPORT);
	
    menu->AppendItems("undo",       cUNDO,
		      "-",
		      "cut",         cCUT,
		      "copy",        cCOPY,
		      "paste",       cPASTE,
		      "-",
		     0);

    menu->AppendMenu(file, cFILEMENU);
}

void Document::DoSetupMenu(Menu *menu)
{
    EvtHandler::DoSetupMenu(menu);
    if (lastCmd != gNoChanges) {
	menu->ReplaceItem(cUNDO, lastCmd->GetUndoName());
	if (lastCmd->TestFlag(eCmdCanUndo))
	    menu->EnableItem(cUNDO);
    }
    menu->EnableItem(cCLOSE);
    if (Overridden(Document, DoRead))
	menu->EnableItem(cLOAD);
    if (Overridden(Document, DoWrite)) {
	if (Modified()) {
	    menu->EnableItem(cSAVE);
	    if (!isUntitled)
		menu->EnableItem(cREVERT);
	}
	menu->EnableItem(cSAVEAS);
    }
    menu->EnableItems(cSHOWAPPLWIN, cFILEMENU, cIMPORT, 0);
}

Command *Document::DoMenuCommand(int cmd)
{
    switch(cmd) {
    case cUNDO:
	Undo();
	break;

    case cSAVE:
	Save();
	break;

    case cSAVEAS:
	SaveAs();
	break;

    case cLOAD:
	Open();
	break;

    case cIMPORT:
	return Import();
	
    case cCLOSE:
	Close();
	break;

    case cREVERT:
	Revert();
	break;

    case cCOLLAPSE:
	Toggle();
	break;
	
    default:
	return EvtHandler::DoMenuCommand(cmd);
    }
    return gNoChanges;
}

void Document::Control(int id, int part, void *vp)
{
    if (id == cIdCloseBox)
	Toggle();
    EvtHandler::Control(id, part, vp);
}

void Document::PerformCommand(Command* cmd)
{
    Perform(cmd, &lastCmd, &changeCount);
    if (window)
	window->UpdateEvent();
}

//---- Menu Commands -----------------------------------------------------------

bool Document::Open()
{
    if (Modified()) {
	switch(CautionAlert.Show("Save changes to @B%s@P ?", docName)) {
	case cIdYes:
	    if (! Save())
		return FALSE;
	    break;
	case cIdNo:
	    break;
	case cIdCancel:
	    return FALSE;
	}
    }

    PerformCommand(gResetUndo);

    if (loadDialog == 0) {
	loadDialog= MakeFileDialog(eFDTypeRead);
	ObjectTableAddRoot(loadDialog);
    }

    if (loadDialog->ShowInWindow(eFDRead, window, this) == cIdOk) {
	char *filename= loadDialog->FileName();
	FileType *ft= loadDialog->GetDocType();
	if (CanLoadDocument(ft))
	    Load(filename, TRUE, ft);
	else
	    application->OpenDocument(filename);
	return TRUE;
    }
    return FALSE;
}

bool Document::Close()   // return TRUE if OK
{
    bool rcode= TRUE;

    if (! isOpen)
	return rcode;

    if (Modified()) {
	switch (CautionAlert.Show("Save changes to @B%s@P ?", docName)) {
	case cIdNo:
	    rcode= TRUE;
	    break;
	case cIdYes:
	    rcode= Save();
	    break;
	case cIdCancel:
	    rcode= FALSE;
	    break;
	}
    }

    if (rcode) {
	PerformCommand(gResetUndo);
	windows->ForEach(BlankWin,Open)(FALSE);
	if (application)
	    application->RemoveDocument(this);
	isOpen= FALSE;
    }
    return rcode;
}

bool Document::Save()
{
    if (Modified() && (isUntitled || isConverted))
	return SaveAs();

    if (! Modified()) {
	NoteAlert.Show("No changes since last save");
	return TRUE;
    }

    if (docName[0] == '/') 
	Store(docName, 0);
    else 
	Store(form("%s/%s", loadDir, docName), 0);
    return TRUE;
}

bool Document::SaveAs()
{
    if (saveDialog == 0) {
	saveDialog= MakeFileDialog(eFDTypeWrite);
	ObjectTableAddRoot(saveDialog);
    }

    if (saveDialog->ShowInWindow(eFDWrite, window, this) == cIdOk) {
	PerformCommand(gResetUndo);

	char *filename= saveDialog->FileName();
	SetName(filename);
	Store(filename, saveDialog->GetSaveOption());
	FType ft(filename);
	uniqueId= ft.UniqueId();
	isUntitled= isConverted= FALSE;
	strreplace(&loadDir, gSystem->WorkingDirectory());
	return TRUE;
    }
    return FALSE;
}

void Document::Revert()
{
    if (Modified() &&
	CautionAlert.Show("Discard all changes of @B%s@P ?", docName) == cIdYes){
	    FType ft(docName);
	    Load(docName, FALSE, ft.FileType());
    }            
}

Command *Document::Import()
{
    Command *cmd= gNoChanges;
	
    if (importDialog == 0) {
	importDialog= MakeFileDialog(eFDTypeImport);
	ObjectTableAddRoot(importDialog);
    }

    if (importDialog->ShowInWindow(eFDImport, window, this) == cIdOk) {
	filebuf infb;
	char *filename= importDialog->FileName();
	FileType *ft= importDialog->GetDocType();
	if (CanImportDocument(ft)) {
	    infb.open(filename, input);
	    istream from(&infb);
	    ClassReset();
	    cmd= DoImport(from, ft);
	    ClassReset();
	}
    }
    return cmd;
}

void Document::Undo()
{
    if (lastCmd)
	lastCmd->Undo();
}

//---- Load/Store Documents -----------------------------------------------------

bool Document::CanLoadDocument(FileType *file)
{
    bool ok= strcmp(GetDocumentType(), file->Type()) == 0;
    if (!ok)
	if (strcmp(GetDocumentType(), cDocTypeAscii) == 0 && file->IsAscii())
	    ok= TRUE;
    return ok;
}

bool Document::CanImportDocument(FileType *)
{
    return FALSE;
}

void Document::Load(char *name, bool unique, FileType *filetype)
{
    filebuf infb;
    bool rcode= TRUE, asuntitled= FALSE;
    int uid;
    char *newname;
	    
    if ((uid= filetype->UniqueId()) == 0) {
	NoteAlert.Show("something wrong with @B%s@P\n%s", name,
						    gSystem->GetErrorStr());
	return;
    }

    if (unique && application->FindDocument(uid)) {
	if (CautionAlert.Show("Document @B%s@P still open; opening as untitled",
						     name) != cIdYes)
	return;
	asuntitled= TRUE;
	newname= "untitled";
    } else
	newname= name;
    uniqueId= uid;

    infb.open(name, input);
    istream from(&infb);
    SetName(newname);
    strreplace(&loadDir, gSystem->WorkingDirectory());

    ClassReset();
    DoRead(from, filetype);
    if (ClassReset() == 0) {
	changeCount= 0;
	isUntitled= asuntitled;
	if ((strcmp(docType, cDocTypeAscii) == 0) && 
					     filetype->IsAscii())
	    isConverted= FALSE;
	else
	    isConverted= strcmp(filetype->Type(), docType) != 0;
	if (lastCmd && lastCmd != gNoChanges)
	    delete lastCmd;
	lastCmd= gNoChanges;
    }
}

void Document::Store(char *name, int option)
{
    filebuf outfb;
    outfb.open(name, output);
    ostream to(&outfb);

    ClassReset();
    gInPrintOn= TRUE;
    DoWrite(to, option);
    gInPrintOn= FALSE;
    ClassReset();

    to.flush();
    changeCount= 0;
}

void Document::DoRead(istream& from, FileType *)
{
    char c;
    //overread magic cookie
    while (from.get(c))
	if (c == '\n')
	    break;
}

void Document::DoWrite(ostream& to, int)
{
    to << cMagic << docType SP << application->ProgramName() NL;
}

Command *Document::DoImport(istream& , FileType *)
{
    return gNoChanges;
}

void Document::InspectorId(char *buf, int bufSize)
{
    if (docName)
	strn0cpy(buf, docName, bufSize);
    else
	EvtHandler::InspectorId(buf, bufSize);        
}

