//$PostScriptPort,PostScriptPrinter$
#include "PostScript.h"
#include "../Error.h"
#include "../String.h"
#include "../Bitmap.h"
#include "../PrintPort.h"
#include "../DialogItems.h"
#include "../VObject.h"
#include <osfcn.h>

//---- PostScriptPort ----------------------------------------------------------

class PostScriptPort: public PrintPort {    
    bool smooth, wysiwyg;
    float xscale, yscale;
    float xorigin, yorigin;
    bool tofile, expandlibs, dirty;
    FILE *pfp, *prfp;
    int fix, lastpat, lastpsz, fid, maxfont, bcnt, pagecnt, currpage;
    struct fts *fonts;
    char *PrintTmpFile;
    FontPtr lbfont, lfont;
    byte bbuf[MaxTextBatchCnt];
    
protected:
    void SetDirty();
    void DrawObject(char, GrPattern, GrMode, Rectangle*, Point, int, GrLineCap);
    void DrawPolygon(char, GrPattern, GrMode, Rectangle*, Point*, int, GrPolyType, int, GrLineCap);
    void Printf(char *fmt, ...);
    bool SetPattern(int pat, int mode);
    void SetPenSize(int pensize);
    int EnrollFont(FontPtr fd);
    void flushfonts();
    void downloadfont(FontPtr fd, char *fname, int *ia);

public:
    PostScriptPort(char *name, float res, bool prolog, bool smooth, bool wys);
	
    void DevDestroy();
    void DevClip(Rectangle, Point);
    void DevResetClip();
    void DevStrokeLine(GrPattern, GrMode, int, Rectangle*, GrLineCap, Point, Point);
    void DevShowBitmap(GrPattern, GrMode, Rectangle*, struct Bitmap*);
    bool DevShowChar(FontPtr fdp, Point pos, byte c, bool isnew, Point);
    void DevShowTextBatch(GrPattern, GrMode, Rectangle*, Point);
    void DevGiveHint(int, int, void*);

    void OpenPage(int pn);
    void ClosePage();
};

#define MaxFonts 50
#define CharsPerFont 256
#define BitsPerWord (sizeof(int)*8)

struct fts {
    short fid;
    short face;
    int cbits[CharsPerFont/BitsPerWord];
};

inline void SetBit(int *a, int i)
{
    a[i/BitsPerWord] |= 1 << (i % BitsPerWord);
}

inline bool TestBit(int *a, int i)
{
    return (a[i/BitsPerWord] & (1 << (i % BitsPerWord))) != 0;
}

extern char *tempnam(char*, char*);
extern char *LibName;

static void loadlib(FILE *ofp, char *name, bool expandlibs)
{
    char buf[200];
    FILE *fin;
    
    if (expandlibs) {
	sprintf(buf, "./%s.ps", name);
	if ((fin= fopen(buf, "r")) == NULL) {
	    sprintf(buf, "%s/postscript/%s.ps", LibName, name);
	    if ((fin= fopen(buf, "r")) == NULL) {
		cerr << "can't read postscript library file " << buf NL;
		return;
	    }
	}
	while (fgets(buf, 200, fin) != NULL)
	    fputs(buf, ofp);
    } else
	fprintf(ofp, "%%%%Include: %s\n", name);
}

PostScriptPort::PostScriptPort(char *name, float res, bool pro, bool sm,
							    bool wys) : (name)
{
    expandlibs= pro;
    xorigin= 50.0;
    yorigin= 820.0;
    xscale= yscale= res;
    smooth= sm;
    wysiwyg= wys;
	
    fonts= new fts[MaxFonts];
    tofile= (name != 0);
    fid= 4711;
    pagecnt= 0;

    if (tofile) {
	if ((prfp= fopen(name, "w")) == NULL) {
	    cerr << "can't open file " << name << "; trying ./postscript\n"; 
	    if ((prfp= fopen("postscript", "w")) == NULL) {
		cerr << "can't open file \"./postscript\"\n";
		exit(1);
	    }
	}
    } else
	prfp= popen("/usr/ucb/lpr -v", "w");
     
    PrintTmpFile= tempnam("/usr/tmp", "etprt");
    if (PrintTmpFile == NULL)
	Error("PostScriptPort", "can't create tmpname");
    if ((pfp= fopen(PrintTmpFile, "w")) == NULL)
	Error("PostScriptPort", "can't open tmpfile");
    
    fprintf(prfp, "%%!PS-Adobe-1.0\n");
    fprintf(prfp, "%%%%Creator: ET++\n");
    fprintf(prfp, "%%%%Timeout: 1000\n");
}

void PostScriptPort::OpenPage(int pn)
{
    lastpsz= lastpat= -1;
    lbfont= 0;
    dirty= FALSE;
    currpage= pn;
}

void PostScriptPort::ClosePage()
{
    if (dirty) {
	Printf("showpage\n");
	Printf("restore\n");
	pagecnt++;
    }
    dirty= FALSE;
}

void PostScriptPort::SetDirty()
{
    if (! dirty) {
	Printf("%%%%Page: %d %d\n", pagecnt+1, currpage);
	Printf("save\n");
	Printf("init\n");
	dirty= TRUE;
    }
}

void PostScriptPort::DevDestroy()
{
    register FILE *rpfp;
    char buf[200];
    float xs, ys;
    
    if (gDebug)
	DevStrokeRect(ePatBlack, eRopCopy, 0, &bbox);
    fclose(pfp);
    
    pfp= prfp;
    
    if ((rpfp= fopen(PrintTmpFile, "r")) == NULL) {
	cerr << "can't open file " << PrintTmpFile NL;
	exit(1);
    }
    
    Printf("%%%%Pages: %d\n", pagecnt);
    xs= xscale / 300.0 * 72.0;
    ys= yscale / 300.0 * 72.0;
    Printf("%%%%BoundingBox: %d %d %d %d\n",
	    (int)(xorigin + xs*bbox.origin.x - 0.5),
	    (int)(yorigin - ys*(bbox.origin.y + bbox.extent.y) - 0.5),
	    (int)(xorigin + xs*(bbox.origin.x + bbox.extent.x) + 0.5),
	    (int)(yorigin - ys*bbox.origin.y + 0.5));
    Printf("%%%%EndComments\n");
    if (smooth)
	loadlib(prfp, "smooth", expandlibs);
    loadlib(prfp, "et", expandlibs);
    Printf("ET++Dict begin\n");
    flushfonts();
    Printf("/init {\n");
    Printf("%f %f translate\n", xorigin, yorigin);
    Printf("%f 300 div 72 mul %f 300 div 72 mul neg scale\n", xscale, yscale);
    Printf("} def\n");
    Printf("%%%%EndProlog\n");

    while (fgets(buf, 200, rpfp) != NULL)
	fputs(buf, prfp);
    fclose(rpfp);
	
    unlink(PrintTmpFile);

    Printf("%%%%Trailer\n");
    Printf("end\n");
    
    if (tofile)
	fclose(prfp);
    else
	pclose(prfp);
    SafeDelete(fonts);
}

/*---- clipping ---------------------------------------------------------------*/

void PostScriptPort::DevResetClip()
{
    Printf("ResetClip\n");
    lastpsz= -1;
    lastpat= -1;
    lbfont= 0;
}

void PostScriptPort::DevClip(Rectangle r, Point)
{
    if (r.IsNotEmpty()) {
	SetDirty();
	Printf("%r SetClip\n", &r);
    }
}

void PostScriptPort::DevStrokeLine(GrPattern pat, GrMode mode, int pensize, Rectangle *r,
		GrLineCap cap, Point p1, Point p2)
{
    if (SetPattern(pat, mode))
	return;
    SetPenSize(pensize);
    
    Merge(r);
    Printf("%p %p %d StrokeArrow\n", p1, p2, cap);
}

//---- text batching -----------------------------------------------------------

bool PostScriptPort::DevShowChar(FontPtr fdp, Point delta, byte c, bool isnew, 
									    Point)
{
    if (isnew) {                // first
	bcnt= 0;
	lfont= fdp;
    } else if (fdp != lfont || delta != gPoint0)
	return TRUE;
    bbuf[bcnt++]= c;
    return FALSE;
}

void PostScriptPort::DevShowTextBatch(GrPattern pat, GrMode mode, Rectangle *r,
								      Point pos)
{
    int i, blanks= 0;
    register byte c;
    
    if (lbfont != lfont) {
	fix= EnrollFont(lfont);
	Printf("%s-%s %d scalesetfont\n", gFontManager->Name(lfont->Fid()), 
			 gFontManager->StyleString(lfont->Face()), lfont->Size());
	lbfont= lfont;
    }
    SetPattern(pat, mode);
    Merge(r);
    Printf("%p m(", pos);
    for (i= 0; i < bcnt; i++) {
	c= bbuf[i];
	SetBit(fonts[fix].cbits, c);
	Printf("%c", c);
	if (c == ' ')
	    blanks++;
    }
    //if (blanks > 0)
    //    Printf(") %d %d bsshow\n", r->extent.x, blanks);
    //else
    //    Printf(") %d sshow\n", r->extent.x);        
    Printf(") show\n");        
} 

void PostScriptPort::DevShowBitmap(GrPattern pat, GrMode mode, Rectangle *r,
								Bitmap *bmp)
{
    int bytes, y, by, lines, yy;
    byte bbp[300];
    Point e= bmp->Size();
    
    SetPattern(pat, mode);
    if (!gDebug && smooth) {
	Merge(r);
	bytes= bmp->ShortsPerLine() * 2;
	lines= 1250 / bytes;
	
	Printf("%d %d div %d %d div %p %d %d SmoothBitmapStart\n",
		   r->extent.x, e.x, r->extent.y, e.y, r->origin, bytes, e.x);
	for (by= 0, y= 0; y < e.y; y+= by) {
	    by= min(lines, e.y-y);
	    Printf("%d Band\n", by);
	    for(yy= y-2; yy < y+by+2; yy++) {
		bmp->GetLine(bbp, yy, bytes);
		Printf("%X\n", bbp, bytes);
	    }
	}
	Printf("SmoothBitmapEnd\n");
    } else {
	bytes= bmp->BytesPerLine(); /* postscript images are Byte aligned */
	Merge(r);
	Printf("%r %p Bitmap\n", r, e);
	for (y= 0; y < e.y; y++) {
	    bmp->GetLine(bbp, y, bytes);
	    Printf("%X\n", bbp, bytes);
	}
    }
}

void PostScriptPort::DevGiveHint(int code, int, void *vp)
{
    if (code == eHintPostScript) {
	SetDirty();
	Printf("%s\n", (char*) vp);
    }
}

void PostScriptPort::DrawObject(char type, GrPattern pat, GrMode mode,
			Rectangle *r, Point e, int psz, GrLineCap)
{
    char *st= "?";
    
    Merge(r);
    
    if (SetPattern(pat, mode))
	return;
    SetPenSize(psz);
    if (r)
	Printf("%r ", r);
    if (e != gPoint0)
	Printf("%p ", e);
    switch (type) {
    case 'b':
    case 'B':
	st= "Rect";
	break;
    case 'o':
    case 'O':
	st= "Oval";
	break;
    case 'r':
    case 'R':
	st= "RoundRect";
	break;
    case 'w':
    case 'W':
	st= "Wedge";
	break;
    }
    Printf("%s%s\n", psz >= 0 ? "Stroke" : "Fill", st);
}

void PostScriptPort::DrawPolygon(char, GrPattern pat, GrMode mode, Rectangle *r,
		Point *pts, int npts, GrPolyType t, int psz, GrLineCap)
{
    int tt, mm;
    Merge(r);
    
    if (SetPattern(pat, mode))
	return;
    SetPenSize(psz);
    if (t & ePolyBezier)
	tt= 1;
    else if (t & ePolySpline2)
	tt= 2;
    else if (t & ePolySpline3)
	tt= 3;
    else
	tt= 0;
	
    if (t & ePolyClose)
	mm= 1;
    else if (t & ePolyPartial)
	mm= 2;
    else 
	mm= 0;
	
    Printf("%r %d %d %p [\n", r, mm, tt, r->extent);

    for (int i= 0; i < npts; i++)
	Printf("%p ", pts[i]);
	
    Printf("\n] %sPoly\n", (psz >= 0) ? "Stroke" : "Fill");
}

//---- output functions ---------------------------------------------------------

#undef va_start
#undef va_end
#undef va_arg

typedef char *va_list1;
# define va_start(list,par) list = (char *) &par
# define va_end(list)
# define va_arg(list,mode) ((mode *)(list += sizeof(mode)))[-1]

/*
 %c  PostScript character       byte
 %f  floating point             float
 %d  integer                    integer
 %x  2-digit hex num            byte
 %X  hex string with n hex nums byte*, n
 %r  Rectangle                  Rectangle*
 %p  Point                      Point
 %s  0-terminated string        char*
*/

void PostScriptPort::Printf(char *fmt, ...)
{
    static char *t= "0123456789abcdef";
    register char c;
    unsigned int b;
    byte *bl;
    char *s;
    Point p;
    int i, n;
    double d;
    Rectangle *r;
    va_list1 ap;
    
    va_start(ap, fmt);
    fmt= va_arg(ap, char*);
    
    while (c= *fmt++) {
	switch (c) {
	case '%':
	    switch (c= *fmt++) {
	    case 'c':
		b= va_arg(ap, unsigned int);
		if (b >= ' ' && b <= '~') {
		    if (b == '\\' || b == '(' || b == ')')
			fputc('\\', pfp); 
		    fputc(b, pfp);
		} else
		    fprintf(pfp, "\\%03o", b);
		break;
	    case 'f':
		d= va_arg(ap, double);
		fprintf(pfp, "%.3f", d);
		break;
	    case 'd':
		i= va_arg(ap, int);
		fprintf(pfp, "%d", i);
		break;
	    case 'x':
		b= va_arg(ap, unsigned int);
		fputc(t[(b&0xff)>>4], pfp);
		fputc(t[b&0xf], pfp);
		break;
	    case 'X':
		bl= va_arg(ap, byte*);
		n= va_arg(ap, int);
		for (i= 0; i<n; i++) {
		    fputc(t[(*bl&0xff)>>4], pfp);
		    fputc(t[*bl++&0xf], pfp);
		}
		break;
	    case 's':
		s= va_arg(ap, char*);
		while (c= *s++)
		    fputc(c, pfp);
		break;
	    case 'r':
		r= va_arg(ap, Rectangle*);
		fprintf(pfp, "%d %d %d %d", r->origin.x, r->origin.y, r->extent.x,
								  r->extent.y);
		break;
	    case 'p':
		p= va_arg(ap, Point);
		fprintf(pfp, "%d %d", p.x, p.y);
		break;
	    default:
		fputc(c, pfp);
		break;
	    }
	    break;
	default:
	    fputc(c, pfp);
	    break;
	}
    }
    va_end(ap);
}

//---- misc functions ---------------------------------------------------------

bool PostScriptPort::SetPattern(int pat, int mode)
{
    if (pat == 0 || (pat == (int) ePatBlack && mode == (int) eRopXor))
	return TRUE;
    if (pat == lastpat)
	return FALSE;
    lastpat= pat;
    SetDirty();
    Printf("%d SetPattern\n", pat);
    return FALSE;    
}

void PostScriptPort::SetPenSize(int pensize)
{
    if (pensize < 0 || (pensize == lastpsz))
	return;
    lastpsz= pensize;
    Printf("%d SetPenSize\n", pensize);
}

int PostScriptPort::EnrollFont(FontPtr fd)
{
    register int i;
    
    for (i= 0; i<maxfont; i++)
	if (fd->Fid() == fonts[i].fid && fd->Face() == fonts[i].face)
	    return i;
    fonts[i].fid= fd->Fid();
    fonts[i].face= fd->Face();
    maxfont++;
    return i;
}

static bool Reencode(int family)
{
    switch (family) {
    case eFontTimes:
    case eFontHelvetica:
    case eFontCourier:
    case eFontAvantgarde:
    case eFontBookman:
    case eFontSchoolbook:
    case eFontNHelvetica:
    case eFontPalatino:
    case eFontChancery:
	return TRUE;
    default:
	return FALSE;
    }
}

void PostScriptPort::flushfonts()
{
    int face, family;
    char fbuf[40];
    
    for (int i= 0; i < maxfont; i++) {
	family= fonts[i].fid;
	face= fonts[i].face;
	
	sprintf(fbuf, "%s-%s", gFontManager->Name(family), 
					    gFontManager->StyleString(face));
	
	if (! gFontManager->IsPSFont(family)) {
	    downloadfont(new Font(family, 99, face), fbuf, fonts[i].cbits);
	    fprintf(pfp, "/%s /%s findfont def\n", fbuf, fbuf);
	} else if (Reencode(family))
	    fprintf(pfp, "/%s Reencode\n", fbuf);
    }
}

void PostScriptPort::downloadfont(FontPtr fd, char *fname, int *ia)
{
    int z, i, w, hx, hy, sx, sy, y, starty, endy, ps, llx, lly, urx, ury;
    byte bbp[100];
    
    llx= lly= 9999;
    urx= ury= 0;
    ps= fd->Size();

    Printf("/%s %d %d [\n", fname, fid++, ps);
    for (z= i= 0; i < 256; i++) {
	if (! TestBit(ia, i) || fd->PMetric(i,&w,&hx,&hy,&sx,&sy,&starty,&endy))
	    continue;
	z++;
       
	if (sy <= 0 || sx <= 0)
	    sx= sy= hx= hy= 0;
	    
	llx= min(llx, hx);
	lly= min(lly, hy);
	urx= max(urx, hx+sx);
	ury= max(ury, hy+sy);
	
	Printf("<%x%x%x%x%x%x", i, w, sx, sy, hx, hy-sy);
	
	if (smooth)
	    for (y= starty-2; y < endy+2; y++) {
		fd->GetLine(bbp, i, y, (((sx+2)-1)/16+1)*2);
		Printf("%X", bbp, (((sx+2)-1)/16+1)*2);
	    }
	else
	    for (y= starty; y < endy; y++) {
		fd->GetLine(bbp, i, y, (sx-1) / 8 + 1);
		Printf("%X", bbp, (sx-1) / 8 + 1);
	    }
		
	Printf(">\n");
    }
    Printf("] %d [ %f %f %f %f ] new%sbitmapfont\n", z,
			(float)(llx)/(float)(ps), (float)(lly)/(float)(ps),
			(float)(urx)/(float)(ps), (float)(ury)/(float)(ps),
			smooth ? "smoothed" : "");
}

//---- PostScriptPrinter -------------------------------------------------------

const int cIdName       =   cIdFirstUser + 101,
	  cIdOptions    =   cIdFirstUser + 102,
	  cIdResolution =   cIdFirstUser + 103,
	  cIdOrientation=   cIdFirstUser + 104;

PostScriptPrinter::PostScriptPrinter() : ("PostScript", TRUE)
{
    resolution= 1;
    prolog= smooth= wysiwyg= FALSE;
}

class PrintPort *PostScriptPrinter::MakePrintPort(char *name)
{
    return new PostScriptPort(name, resolution, prolog, smooth, wysiwyg);
}

void PostScriptPrinter::DoSetDefaults()
{
    resolutionCluster->Set(cIdResolution+3);
    orientationCluster->Set(cIdOrientation);
    eti->SetString("lw");
}

void PostScriptPrinter::DoSave()
{
    oldresolution= resolution;
    oldportrait= portrait;
}

void PostScriptPrinter::DoRestore() 
{
    resolutionCluster->Set(cIdResolution+oldresolution-1);
    orientationCluster->Set(cIdOrientation+portrait);
}

VObject *PostScriptPrinter::DoCreatePrintDialog()
{
    //---- dialog parts ----
    
    orientationCluster= new OneOfCluster(cIdOrientation, eVObjHLeft, 5,
	"Portrait",
	"Landscape",
	0);
	
    optionsCluster= new ManyOfCluster(cIdOptions, eVObjHLeft, 5,
	"smooth",
	"wysiwyg",
	"PS-prolog",
	0);
	
    resolutionCluster= new OneOfCluster(cIdResolution, eVObjHLeft, 5,
	"1/300 inch",
	"2/300 inch",
	"3/300 inch",
	"4/300 inch",
	0);
	
    //---- overall layout ----
    return new Cluster(cIdNone, eVObjHLeft|eVObjHExpand, 20,
		new Cluster(cIdNone, eVObjVTop, 20,
		    new BorderItem("Options", optionsCluster),
		    new BorderItem("Resolution", resolutionCluster),
		    new BorderItem("Orientation", orientationCluster),
		    0
		),
		new BorderItem("Printer name", 
		    eti= new EditTextItem(cIdName, "lw", 100)
		),
		0
	    );
}

void PostScriptPrinter::Control(int id, int p, void *v)
{
    switch (id) {
    
    case cIdOptions:
	switch (p) {
	case cIdOptions:
	    smooth= bool(v);
	    break;
	case cIdOptions+1:
	    wysiwyg= bool(v);
	    break;
	case cIdOptions+2:
	    prolog= bool(v);
	    break;
	}
	break;
	
    case cIdResolution:
	resolution= p-cIdResolution+1;
	break;
	
    case cIdOrientation:
	portrait= (p == cIdOrientation);
	return;
	
    case cIdName:
	Printer::Control(id, p, v);
	if (p == cPartChangedText && v == eti)
	    EnableItem(cIdYes, eti->GetTextSize() > 0);
	break;

    default:
	Printer::Control(id, p, v);
	break; 
    }
}

Point PostScriptPrinter::GetPageSize()
{
    Point ps= Printer::GetPageSize()*4/resolution;
    if (portrait)
	return ps;
    return Point(ps.y,ps.x);
}

Printer *NewPostScriptPrinter()
{
    return new PostScriptPrinter;
}
