//$SunSystem,SunDirectory,ZombieHandler$
#include "SunSystem.h"
#include "../Error.h"
#include "../String.h"
#include "../ObjList.h"
#include "SunFileType.h"
#include "SunOsPttyConnection.h"

#include "dynlink.h"
#include "sunsystem.h"
#include "sunstacktrace.h"

#include <sys/file.h>
#include <sys/ioctl.h>
#include <sgtty.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>

#include <osfcn.h>
#include <signal.h>


extern int select(int, fd_set*, fd_set*, fd_set*, struct timeval*);
extern char *getenv(char*);
extern char **environ;
extern void usleep(unsigned int);


//---- signalHandler -----------------------------------------------------------

static bool signals[32];

static int sigHandler(int sig, int, struct sigcontext*)
{
    signals[sig]= TRUE;
    if (gSystem)
	((SunSystem*)gSystem)->DispatchSignals(TRUE);
    return 0;
}

//---- ZombieHandler -----------------------------------------------------------

class ZombieHandler : public SysEvtHandler {
    ObjList *zombieHandler;    
public:
    ZombieHandler(ObjList *zh) : (SIGCHLD)
	{ zombieHandler= zh; }
    void Notify(SysEventCodes, int);
};

void ZombieHandler::Notify(SysEventCodes, int)
{
    int pid;

    while ((pid= waitchild()) > 0) {
	Iter next(zombieHandler);
	SysEvtHandler *re;
	while (re= (SysEvtHandler*) next()) {
	    if (re->GetResourceId() == pid) {
		re->Notify(eSysEvtZombie, pid);
		re->Remove();
	    }
	}
    }
}

//---- SunSystem ---------------------------------------------------------------

System *NewSunSystem()
{
    return new SunSystem();
}

SunSystem::SunSystem() : ("SUNOS")
{
}

bool SunSystem::Init()
{
    if (System::Init())
	return TRUE;

    TraceInit(0);

    AddSignalHandler(new ZombieHandler(zombieHandler));

    signal(SIGQUIT, sigHandler);
    signal(SIGINT,  sigHandler);
    signal(SIGWINCH,sigHandler);
    signal(SIGALRM, sigHandler);
    signal(SIGCHLD, sigHandler);
    signal(SIGURG,  sigHandler);

    return FALSE;
}

void SunSystem::SetUpMask(fd_set* m, ObjList *hl)
{
    Iter next(hl);
    register SysEvtHandler *re;

    while (re= (SysEvtHandler*) next()) 
	if (re->HasInterest())
	    FD_SET(re->GetResourceId(), m);
}

void SunSystem::DispatchReadWrite(fd_set* m, ObjList *hl)
{
    Iter next(hl);
    register SysEvtHandler *re;

    while (re= (SysEvtHandler*) next()) {
	int fd= re->GetResourceId();
	if (FD_ISSET(fd, m) && re->HasInterest())
	    re->Notify(eSysEvtRead, fd);
    }
}

void SunSystem::DispatchSignals(bool async)
{
    ObjList *shl= async ? asyncSignalHandler : signalHandler;
    Iter next(shl);
    register SysEvtHandler *re;
    int sig;

    while (re= (SysEvtHandler *)next()) {
	sig= re->GetResourceId();
	if (signals[sig] && re->HasInterest()) {
	    signals[sig]= FALSE;
	    re->Notify(eSysEvtSignal, sig);
	}
    }
}

void SunSystem::InnerLoop()
{

    struct timeval t;                 // timeval to poll for a rubout
    fd_set readready, writeready;
    int nfds;
    
    if (signalHandler->Size())
	DispatchSignals(FALSE);          // poll interrupts

    FD_ZERO(&readready);
    FD_ZERO(&writeready);
    if (fileInputHandler->Size())
	SetUpMask(&readready, fileInputHandler);
    if (fileOutputHandler->Size())
	SetUpMask(&writeready, fileOutputHandler);

    t.tv_sec = 0L;
    t.tv_usec= 300000L;

    if ((nfds= select(FD_SETSIZE, &readready, &writeready, 0, &t)) == -1) {
	if (errno != EINTR)     // error
	    SysError("SunSystem::InnerLoop", "select");
    } else if (nfds == 0) {     // timeout
	Iter next(timeoutHandler);
	register SysEvtHandler *eh;
    
	while (eh= (SysEvtHandler*) next())
	    if (eh->HasInterest())
		eh->Notify(eSysEvtTimeout, 0);
    } else {                    // read/write fd ready
	if (fileInputHandler->Size())
	    DispatchReadWrite(&readready, fileInputHandler);
	if (fileOutputHandler->Size())
	    DispatchReadWrite(&writeready, fileOutputHandler);
    }
}

bool SunSystem::AccessPathName(char *path, int mode)
{
    if (access(path, mode) == 0)
	return FALSE;
    errorstr= sys_errlist[errno];
    return TRUE;
}

bool SunSystem::ChangeDirectory(char *path)
{
    return chdir(path) == 0; 
}

char *SunSystem::WorkingDirectory()
{
    return workingdirectory();
}

Directory *SunSystem::MakeDirectory(char *name)
{
    return new SunDirectory(name);
}
    
FileType *SunSystem::GetFileType(char *path)
{
    return new SunFileType(path);
}

void SunSystem::GetTtyChars(char &backspace, char &rubout)
{
    struct sgttyb ttyBuf;
    struct tchars tcharBuf;
    struct ltchars ltcharBuf;
    int fd;

    if ((fd= open("/dev/tty", O_RDONLY)) < 0)
	SysError("GetTtyChars", "can't open control terminal");

    if (ioctl(fd, (int) TIOCGETP, (char*) &ttyBuf) == -1 ||
	    ioctl(fd, (int) TIOCGETC, (char*) &tcharBuf) == -1 ||
		ioctl(fd, (int) TIOCGLTC, (char*) &ltcharBuf) == -1) {
	close(fd);
	return;
    }    
    close(fd);

    backspace= ttyBuf.sg_erase;
    rubout= tcharBuf.t_intrc;
}

PttyConnection *SunSystem::MakePttyConnection(char *prog, char **args)
{
    return new SunOsPttyConnection(prog, args);
}

char *SunSystem::getenv(char *name)
{
    return ::getenv(name);
}

void SunSystem::Setenv(char *name, char *value)
{
    static bool alloced= FALSE;

    register len;
    register char **ap, **newv, *buf;

    len= strlen(name);
    buf= (char *) Alloc((unsigned)(len + strlen(value) + 2));
    if (buf == NULL) 
	return;
    sprintf(buf, "%s=%s", name, value);
    for (ap= environ; *ap; ap++)
	if (strncmp(*ap, buf, len+1) == 0) {
	    *ap= buf;
	    return;
	}
    len= ap - environ;
    newv= (char**) Alloc((unsigned)((len + 2) * sizeof(char *)));
    if (newv == NULL) 
	return;
    bcopy((char*) environ, (char*) newv, len * sizeof(char *));
    newv[len]= buf;
    newv[len + 1]= 0;
    if (alloced)
	free((char*)environ);
    alloced= TRUE;
    environ= newv;
}

void SunSystem::Unsetenv(char *name)
{
    register len;
    register char **ap, **newv;

    len= strlen(name);
    for (newv= ap= environ;  (*newv= *ap) != NULL;  ap++) {
	if (strncmp(*ap, name, len) == 0  &&  (*ap)[len] == '=') {
	    /* Memory leak bug: we cannot free(*ap) here, because we don't know
	     * whether *ap was created with putenv(). */
	     ;
	} else
	    newv++;
    }
}

void SunSystem::stacktrace(int level, bool abortafter)
{
    if (level == 0)
	level= 1000;
    ::stacktrace(0, level);
    if (abortafter)
	::abort();
}

//---- expand the metacharacters as in the shell ------------------------

bool SunSystem::ExpandPathName(char *patbuf, int buflen)
{
    errorstr= expandpathname(patbuf, buflen);
    return errorstr != 0;
}

void SunSystem::wait(unsigned int duration)
{
    if (duration > 0) 
	usleep(duration*1000);  // usleep expects microseconds
}

//---- SunDirectory ------------------------------------------------------------

SunDirectory::SunDirectory(char *name) : (name)
{
    dirp= opendir(name);
}

SunDirectory::~SunDirectory()
{
    if (dirp)
	closedir(dirp);
}

char *SunDirectory::operator()()
{
    if (dirp)
	return getdirentry(dirp);
    return 0;
}

//---- dynamic loading and linking ---------------------------------------------

void *Load2(char *progname, char *name, char *suf)
{
    char eentry[100];

    DynLinkInit(progname);
    sprintf(eentry, "__%s__ctor%s", name, suf);
    if (DynLookup(eentry))
       return (Object*) DynCall(eentry)(0, 0, 0, 0);
    DynLoad(name);
    return (Object*) DynCall(eentry)(0, 0, 0, 0);
}

void *Load(char *progname, char *name)
{
    return Load2(progname, name, "F_");
}
