//$WindowPort$
#include "WindowPort.h"
#include "Error.h"
#include "System.h"
#include "WindowSystem.h"
#include <math.h>

extern bool fullscreen;

WindowPort *focusport;

const int MaxInvals= 20;

//---- window system independent window grabbing -------------------------------

static WindowPort *grabStack[20];
static int grabIx= 0;

void PushGrab(WindowPort *p)
{
    if (grabIx <= 20) {
	grabStack[grabIx]= p;
	grabIx++;
    }
}

void PopGrab()
{
    if (grabIx > 0) {
	grabIx--;
	grabStack[grabIx]= 0;
    }
}

inline WindowPort *TopGrab()
{
    return (grabIx > 0) ? grabStack[grabIx-1] : 0;
}

//---- WindowPort --------------------------------------------------------------

WindowPort::WindowPort() : (ePortWindow)
{
}

WindowPort::WindowPort(InpHandlerFun ih, void *p1, void *p2, bool ov, bool bl) : (ePortWindow)
{
    if (this == 0)
	gWindowSystem->MakeWindow(&this, ih, p1, p2, ov, bl);
    else {
	Fatal("WindowPort::WindowPort", "no static instances");
	this= 0;    // can't reach
    }
}

void WindowPort::Init(InpHandlerFun ih, void *p1, void *p2, bool ov, bool bl)
{
    ihf= ih;
    privdata= p1;
    privdata2= p2;
    overlay= ov;
    block= bl;
    state= eWsHidden;
    
    done= FALSE;
    havepushbacktoken= FALSE;
    
    if (invalRects == 0)
	invalRects= new Rectangle[MaxInvals];
    inval= 0;
}

void WindowPort::DevDestroy()
{
    if (state == eWsClosed)
	return;
    state= eWsClosed;
    DevDestroy2();
    SafeDelete(invalRects);
}

void WindowPort::InvalidateRect(Rectangle r)
{
    if (inval == 0) {
	invalRects[inval++]= invalBounds= r;
	return;
    }
    invalBounds.Merge(r);
    for (int i= 0; i < inval; i++) {
	if (invalRects[i].ContainsRect(r))
	    return;
	if (r.ContainsRect(invalRects[i])) {
	    invalRects[i]= r;
	    return;
	} 
	if (r.Intersects(invalRects[i])) {
	    invalRects[i].Merge(r);
	    return;
	}
    }
    if (inval >= MaxInvals)
	invalRects[inval-1].Merge(r);
    else
	invalRects[inval++]= r;
}

//---- cursor stuff ------------------------------------------------------------

GrCursor WindowPort::SetCursor(GrCursor c)
{
    if (c != cursor) {
	GrCursor oldc= cursor;
	cursor= c;
	DevSetCursor(c);
	GiveHint(eHintFlush, 0, 0);
	return oldc;
    }
    return c;
}

GrCursor WindowPort::GetCursor()
{
    return cursor;
}

GrCursor WindowPort::SetWaitCursor(unsigned int, GrCursor c)
{
    return SetCursor(c);
}

//---- input handling ----------------------------------------------------------

void WindowPort::GetEvent(Token *t, int timeout, bool overread)
{
    if (havepushbacktoken) {
	*t= pushbacktoken;
	havepushbacktoken= FALSE;
	return;
    }
    DevGetEvent(t, timeout, overread);
};

static Token lastClick;
extern Point lastpos;
int Clicks;
int xyzabc= 1;

void WindowPort::Send(Token *t)
{
    if (state != eWsShown)
	return;
    
    if (t->Code != eEvtDamage)
	lastpos= t->Pos;
	
    if (xyzabc) {
	WindowPort *wp= TopGrab();

	if (t->IsKey() && t->Flags == (eFlgShiftKey|eFlgCntlKey|eFlgMetaKey)
						     && t->MapToAscii() == 'u')
	    Exit(1, FALSE);
	
	switch (t->Code) {
	case eEvtEnter:
	    focusport= this;
	    break;
	case eEvtExit:
	    focusport= 0;
	    break;
	}
	if (wp && (wp != this)) {
	    switch (t->Code) {
	    case eEvtIdle:
	    case eEvtNone:
	    case eEvtLocMove:
	    case eEvtLocMoveBut:
	    case eEvtLocStill:
	    case eEvtEnter:
		// ignore events
		return;
	    case eEvtExit:
	    case eEvtDamage:
		// allow damages
		break;
	    default:
		wp->Bell(50);
		return;
	    }  
	}
	switch (t->Code) {
	case eEvtDamage:
	    switch (t->Flags) {
	    case eFlgDamage1:
		inval= 0;
		windowRect= t->DamageRect();
		break;
	    case eFlgDamage2:
		InvalidateRect(t->DamageRect());
		break;
	    case eFlgDamage3:
		windowRect= t->DamageRect();
		return;
	    case eFlgDamage4:
		InvalidateRect(t->DamageRect());
		return;
	    case eFlgDamage5:
		inval= 0;
		windowRect= t->DamageRect();
		InvalidateRect(windowRect.extent);
		break;
	    }
	    break;
	default:
	    if (t->IsMouseButton()) {
		if (lastClick.DoubleClick(*t))
		    Clicks++;
		else
		    Clicks= 1;
		lastClick= *t;
	    }
	    break;
	}
    }
    ihf (privdata, privdata2, t);
}

void WindowPort::PushEvent(Token t)
{
    if (t.Code != eEvtNone) {
	pushbacktoken= t;
	havepushbacktoken= TRUE;
    }
}

void WindowPort::Damage(EventFlags f, Rectangle *r)
{
    Token dt(f, *r);
    Send(&dt);
}

//---- window management -------------------------------------------------------

void WindowPort::Hide()
{
    if (state != eWsShown)
	return;
    state= eWsHidden;
    
    inval= 0;
	
    FlushAnyText();
    
    if (TopGrab() || block)
	done= TRUE;
    else
	DevHide1();
};

void WindowPort::Show(WindowPort *father, Rectangle r)
{
    Point delta;
    Rectangle rr= r;
    
    if (state != eWsHidden)
	return;
    state= eWsShown;
    
    inval= 0;
    
    FlushAnyText();

    if (father)
	rr.origin+= father->windowRect.origin;
    else {
	if (r.origin == gPoint_1)
	    r.origin= (gScreenRect.extent - r.extent)/2;
    }
    delta= rr.AmountToTranslateWithin(gScreenRect);
    rr.origin+= delta;
    
    DevShow1(father, rr);
    
    if (TopGrab() || block) {
	PushGrab(this);
	if (delta != gPoint0)
	    MoveMousePos(delta);
	done= FALSE;
	if (havepushbacktoken) {
	    havepushbacktoken= FALSE;
	    Send(&pushbacktoken);
	}
	while (! done)
	    gSystem->InnerLoop();
	DevHide1();
	PopGrab();
    }
};

void WindowPort::DevHide1()
{
    state= eWsHidden;
    DevHide();
    GiveHint(eHintFlush, 0, 0);      
}

void WindowPort::DevShow1(WindowPort *father, Rectangle r)
{
    state= eWsShown;
    DevShow(father, r);
    GiveHint(eHintFlush, 0, 0);      
}

void WindowPort::Fullscreen(bool m)
{
    FlushAnyText();
    
    fullscreen= m;
    if (m && inclip) {   // make sure that clipping is turned off
	DevResetClip();
	pendingclip= inclip= FALSE;
    }
    DevFullscreen(m);
}

Rectangle WindowPort::GetRect()
{
    return windowRect;
}

void WindowPort::SetRect(Rectangle *r)
{
    inval= 0;
    DevSetRect(r);
}

//---- graphic functions -------------------------------------------------------

void WindowPort::ScrollRect(Rectangle r, Point delta)
{
    Rectangle r1, r2, rl[4];
    int n;
    
    FlushMyText();
    FlushClip();
    r1= Inter(cliprect, r+origin);
    r2= r1 + delta;
    n= Difference(rl, r1, r2);
    for (int i= 0; i<n; i++)
	InvalidateRect(rl[i]);
    DevScrollRect(Inter(r1, r2), delta);
}

//---- window system independent graphic methods -------------------------------

#define PI180 0.017453293
#define ATAN2(x, y) (atan2((float)(x), (float)(y)))
#define SQRT(x) (sqrt((float)x))

static double _Phi (int dx, int dy)
{
    int r;
    
    r= (int) (ATAN2(dx, dy) * 180.0 / M_PI);
    r= (450-(r % 360)) % 360;
    return r * PI180;
}

Point WindowPort::DrawArrow(GrPattern pat, GrMode mode, int psz, Point s, Point e)
{
    Point c, pts[3];
    float phi, hyp, rot, w, h;
    
    w= (float) max(4, psz*2);
    h= (float) max(9, psz*5);
    rot= ATAN2(w, h);
    hyp= SQRT(w*w + h*h);
    if (hyp < 5.0)
	return e;     /* too short, give up */
    phi= _Phi(s.x-e.x, s.y-e.y);
    
    pts[0]= e;
    pts[1].x= e.x + (int)(hyp*cos(phi+rot)+0.5);
    pts[1].y= e.y + (int)(hyp*sin(phi+rot)+0.5);
    pts[2].x= e.x + (int)(hyp*cos(phi-rot)+0.5);
    pts[2].y= e.y + (int)(hyp*sin(phi-rot)+0.5);
    c= (pts[1]+pts[2])/2;
    
    Rectangle rr= BoundingBox(3, pts);
    DevFillPolygon2(&rr, pat, mode, pts, 3, ePolyClose);
    
    return c;
}

void WindowPort::DevStrokeLine(GrPattern pat, GrMode mode, int psz, Rectangle *r,
					GrLineCap cap, Point start, Point end)
{
    if (cap & eStartArrow)
	start= DrawArrow(pat, mode, psz, end, start);
    if (cap & eEndArrow)
	end= DrawArrow(pat, mode, psz, start, end);
    DevStrokeLine2(pat, mode, psz, r, cap, start, end);
}

void WindowPort::DevStrokeRect(GrPattern pat, GrMode mode, int psz, Rectangle *r)
{
    int rpsz= (psz <= 0) ? 1 : psz;
	
    rpsz*= 2;
    if (rpsz >= r->extent.x || rpsz >= r->extent.y)
	DevFillRect(pat, mode, r);
    else
	DevStrokeRect2(pat, mode, psz, r);
}

//---- splines -----------------------------------------------------------------

static Point *gPp;

static void spline3(Point *p, int p0, int p1, int p2, int p3)
{
    register int dist, j, px, py, lastpx, lastpy;
    int ax, bx, cx, dx, ay, by, cy, dy, dist1, dist2;
    
    ax= -p[p0].x + 3*p[p1].x - 3*p[p2].x + p[p3].x;
    bx= (p[p0].x - 2*p[p1].x + p[p2].x)*3;
    cx= (-p[p0].x + p[p2].x)*3;
    dx= p[p0].x + 4*p[p1].x + p[p2].x + 3;
    
    ay= -p[p0].y + 3*p[p1].y - 3*p[p2].y + p[p3].y;
    by= (p[p0].y - 2*p[p1].y + p[p2].y)*3;
    cy= (-p[p0].y + p[p2].y)*3;
    dy= p[p0].y + 4*p[p1].y + p[p2].y + 3;
    
    dist1= ax+bx+cx - dx;
    dist2= ay+by+cy - dy;
    dist= (abs(dist1) + abs(dist2)) >> 6;
    lastpx= gPp->x= dx/6;
    lastpy= (gPp++)->y= dy/6;

    for (j= 1; j<dist; j++) {
	px= ((((ax*j/dist+bx)*j/dist)+cx)*j/dist+dx)/6;
	py= ((((ay*j/dist+by)*j/dist)+cy)*j/dist+dy)/6;
	if (lastpx != px || lastpy != py) {
	    lastpx= gPp->x= px;
	    lastpy= (gPp++)->y= py;
	}
    }
    gPp->x= (ax+bx+cx+dx)/6;
    (gPp++)->y= (ay+by+cy+dy)/6;
}

static void spline2(Point *p, int p0, int p1, int p2)
{
    register int j, dist, px, py, lastpx, lastpy;
    int ax, bx, cx, ay, by, cy, dist1, dist2;
     
    ax=    p[p0].x - 2*p[p1].x + p[p2].x;
    bx= -2*p[p0].x + 2*p[p1].x;
    cx=    p[p0].x +   p[p1].x              + 1;
    
    ay=    p[p0].y - 2*p[p1].y + p[p2].y;
    by= -2*p[p0].y + 2*p[p1].y;
    cy=    p[p0].y +   p[p1].y              + 1;
    
    dist1= ax+bx - cx;
    dist2= ay+by - cy;
    dist= (abs(dist1) + abs(dist2)) >> 4;
    
    lastpx= gPp->x= cx/2;
    lastpy= (gPp++)->y= cy/2;

    for (j= 1; j < dist; j++) {
	px= (j*(j*ax/dist + bx)/dist + cx)/2;
	py= (j*(j*ay/dist + by)/dist + cy)/2;
	if (lastpx != px || lastpy != py) {
	    lastpx= gPp->x= px;
	    lastpy= (gPp++)->y= py;
	}
    }
    gPp->x= (ax+bx+cx)/2;
    (gPp++)->y= (ay+by+cy)/2;
}

/*
static void bezier(Point *p, int p0, int p1, int p2, int p3)
{
    register int j, dist= 50, px, py, lastpx, lastpy;
    int ax, bx, cx, dx, ay, by, cy, dy;

    ax=   -p[p0].x + 3*p[p1].x - 3*p[p2].x + p[p3].x;
    bx=  3*p[p0].x - 6*p[p1].x + 3*p[p2].x;
    cx= -3*p[p0].x + 3*p[p1].x;
    dx=    p[p0].x;
    
    ay=   -p[p0].y + 3*p[p1].y - 3*p[p2].y + p[p3].y;
    by=  3*p[p0].y - 6*p[p1].y + 3*p[p2].y;
    cy= -3*p[p0].y + 3*p[p1].y;
    dy=    p[p0].y;
    
    lastpx= gPp->x= dx;
    lastpy= (gPp++)->y= dy;
    
    for (j= 1; j < dist; j++) {
	px= j*(j*(j*ax/dist+bx)/dist+cx)/dist+dx;
	py= j*(j*(j*ay/dist+by)/dist+cy)/dist+dy;
	if (lastpx != px || lastpy != py) {
	    lastpx= gPp->x= px;
	    lastpy= (gPp++)->y= py;
	}
    }
    gPp->x= ax+bx+cx+dx;
    (gPp++)->y= ay+by+cy+dy;
}
*/

/*
 * bez: Subdivide a Bezier spline, until it is thin enough to be
 *      considered a line. Store line point in static array points.
*/

#define FIXED(a) (((int) (a)) << 16)
#define INT(a) (((a) + (1 << 15)) >> 16 )

static void bez(int x0, int y0, int x1, int y1, int x2, int y2, int x3, int y3)
{
    register int maxx, minx, maxy, miny;
    
    /* find bounding box of 4 control points */
    maxx = x0;
    maxx = max(maxx, x1);
    maxx = max(maxx, x2);
    maxx = max(maxx, x3);

    maxy = y0;
    maxy = max(maxy, y1);
    maxy = max(maxy, y2);
    maxy = max(maxy, y3);

    minx = x0;
    minx = min(minx, x1);
    minx = min(minx, x2);
    minx = min(minx, x3);

    miny = y0;
    miny = min(miny, y1);
    miny = min(miny, y2);
    miny = min(miny, y3);

    if (((maxx - minx) < FIXED(2)) || ((maxy - miny) < FIXED(2))) { // Consider it a line segment
	gPp->x= INT(x3);
	(gPp++)->y= INT(y3);
    } else {    // Subdivide into 2 new beziers
	register int tx, ty;
	
	tx = (x0 >> 3) + 3 * (x1 >> 3) + 3 * (x2 >> 3) + (x3 >> 3);
	ty = (y0 >> 3) + 3 * (y1 >> 3) + 3 * (y2 >> 3) + (y3 >> 3);
	bez(x0, y0, (x0 >> 1) + (x1 >> 1), (y0 >> 1) + (y1 >> 1),
		    (x0 >> 2) + (x1 >> 1) + (x2 >> 2),
		    (y0 >> 2) + (y1 >> 1) + (y2 >> 2),
		    tx, ty);
	bez(tx, ty, (x3 >> 2) + (x2 >> 1) + (x1 >> 2),
		    (y3 >> 2) + (y2 >> 1) + (y1 >> 2),
		    (x3 >> 1) + (x2 >> 1), (y3 >> 1) + (y2 >> 1),
		    x3, y3);
    }
}

static int makespline(Point *pts, Point *p, int n, GrPolyType type)
{
    register int i, j, nnp;
    Point delta, lastdelta, p1, p2;

    gPp= pts;
    if (type & ePolySpline3) {
	if (type & ePolyClose) {
	    spline3(p, n-1, 0, 1, 2);
	    for (i= 0; i < n-3; i++)
		spline3(p, i, i+1, i+2, i+3);
	    spline3(p, n-3, n-2, n-1, 0);
	    spline3(p, n-2, n-1, 0, 1); 
	} else if (type & ePolyPartial) {
	    for (i= 0; i < n-3; i++)
		spline3(p, i, i+1, i+2, i+3);
	} else {
	    spline3(p, 0, 0, 0, 1);
	    spline3(p, 0, 0, 1, 2);
	    for (i= 0; i < n-3; i++)
		spline3(p, i, i+1, i+2, i+3);
	    spline3(p, n-3, n-2, n-1, n-1);
	    spline3(p, n-2, n-1, n-1, n-1);
	}
    } else if (type & ePolySpline2) {
	if (type & ePolyClose) {
	    for (i= 0; i < n-2; i++)
		spline2(p, i, i+1, i+2);
	    spline2(p, n-2, n-1, 0);
	    spline2(p, n-1, 0, 1); 
	} else if (type & ePolyPartial) {
	    for (i= 0; i < n-2; i++)
		spline2(p, i, i+1, i+2);
	} else {
	    spline2(p, 0, 0, 1);
	    for (i= 0; i < n-2; i++)
		spline2(p, i, i+1, i+2);
	    spline2(p, n-2, n-1, n-1);
	}
    } else {
	bez(FIXED(p[0].x), FIXED(p[0].y), FIXED(p[1].x), FIXED(p[1].y),
	FIXED(p[2%n].x), FIXED(p[2%n].y), FIXED(p[3%n].x), FIXED(p[3%n].y));
    }
	
    // collapse some points to lines
    nnp= gPp - pts;
    p1= pts[1];
    lastdelta= p1 - pts[0];
    for (i= 2, j= 1; i < nnp; i++) {
	p2= pts[i];
	delta= p2 - p1;
	if ((delta != lastdelta) && (abs(delta.x) > 1 || abs(delta.y) > 1))
	    pts[j++]= p1;
	p1= p2;
	lastdelta= delta;
    }
    pts[j++]= p2;

    return j;
}


void WindowPort::DevStrokePolygon(Rectangle *r, GrPattern pat, GrMode mode,
		Point *pts, int npts, GrPolyType t, int psz, GrLineCap cap)
{
    if (cap & eStartArrow)
	DrawArrow(pat, mode, psz, pts[1]+r->origin, pts[0]+r->origin);
    if (cap & eEndArrow)
	DrawArrow(pat, mode, psz, pts[npts-2]+r->origin, pts[npts-1]+r->origin);
    if (t & 0x7) {
	Point tmppts[10000];
	int nn;
    
	nn= makespline(tmppts, pts, npts, t);
	if (nn > 0) {
	    pts= tmppts;
	    npts= nn;
	}
    }
    DevStrokePolygon2(r, pat, mode, pts, npts, 0, psz, cap);
}

void WindowPort::DevFillPolygon(Rectangle *r, GrPattern pat, GrMode mode,
					    Point *pts, int npts, GrPolyType t)
{
    if (t & 0x7) {
	Point tmppts[10000];
	int nn;
    
	nn= makespline(tmppts, pts, npts, t);
	if (nn > 0) {
	    pts= tmppts;
	    npts= nn;
	}
    }
    DevFillPolygon2(r, pat, mode, pts, npts, 0);
}

static Point Angle2Point(Rectangle *r, int ang)
{
    Point p;
    float fa= (float)((450-(ang % 360)) % 360)*M_PI/180.0;
    p.x= (int)(((1.0 + cos(fa)) / 2.0) * (float)(r->extent.x-1) + 0.5) + r->origin.x;
    p.y= (int)(((1.0 - sin(fa)) / 2.0) * (float)(r->extent.y-1) + 0.5) + r->origin.y;
    return p;
}

void WindowPort::DevStrokeWedge(GrPattern pat, GrMode mode, int psz,
				    GrLineCap cap, Rectangle *r, int s, int e)
{
    int rpsz= (psz <= 0) ? 1 : psz;
    rpsz*= 2;
    if (rpsz >= r->extent.x || rpsz >= r->extent.y) {
	DevFillWedge(pat, mode, r, s, e);
	return;
    }
    if (r->extent.x < 4 || r->extent.y < 4) {
	DevStrokeRect(pat, mode, psz, r);
	return;
    }
    if (cap & eStartArrow) {
	DrawArrow(pat, mode, psz, Angle2Point(r, s+10), Angle2Point(r, s));
	// s+= 10;
    }
    if (cap & eEndArrow) {
	DrawArrow(pat, mode, psz, Angle2Point(r, e-10), Angle2Point(r, e));
	// e-= 10;
    }
    DevStrokeWedge2(pat, mode, psz, cap, r, s, e);
}

void WindowPort::DevFillWedge(GrPattern pat, GrMode mode, Rectangle *r,
								  int s, int e)
{
    if (r->extent.x < 4 || r->extent.y < 4) {
	DevFillRect(pat, mode, r);
	return;
    }
    DevFillWedge2(pat, mode, r, s, e);
}

void WindowPort::DevStrokeRRect(GrPattern pat, GrMode mode, int psz, Rectangle *r,
								     Point dia)
{
    int rpsz= (psz <= 0) ? 1 : psz;
	
    rpsz*= 2;
    if (rpsz >= r->extent.x || rpsz >= r->extent.y) {
	DevFillRRect(pat, mode, r, dia);
	return;
    }
    if (ODD(dia.x))    // force to be even
	dia.x--;
    if (ODD(dia.y))    // force to be even
	dia.y--;
	
    if (dia.x >= r->extent.x && dia.y >= r->extent.y)
	DevStrokeOval(pat, mode, psz, r);
    else
	DevStrokeRRect2(pat, mode, psz, r, Min(dia, r->extent));
}

void WindowPort::DevFillRRect(GrPattern pat, GrMode mode, Rectangle *r, Point dia)
{
    if (ODD(dia.x % 1))    // force to be even
	dia.x--;
    if (ODD(dia.y % 1))    // force to be even
	dia.y--;
	
    if (dia.x > r->extent.x && dia.y > r->extent.y)
	DevFillOval(pat, mode, r);
    else
	DevFillRRect2(pat, mode, r, Min(dia, r->extent));
}

void WindowPort::DevStrokeOval(GrPattern pat, GrMode mode, int psz, Rectangle *r)
{
    int rpsz= (psz <= 0) ? 1 : psz;
    
    rpsz*= 2;
    if (rpsz >= r->extent.x || rpsz >= r->extent.y)
	DevFillOval(pat, mode, r);
    else if (r->extent.x < 4 || r->extent.y < 4)
	DevStrokeRect(pat, mode, psz, r);
    else
	DevStrokeOval2(pat, mode, psz, r);
}

void WindowPort::DevFillOval(GrPattern pat, GrMode mode, Rectangle *r)
{
    if (r->extent.x < 4 || r->extent.y < 4)
	DevFillRect(pat, mode, r);
    else
	DevFillOval2(pat, mode, r);
}

//---- abstract methods --------------------------------------------------------

void WindowPort::DevDestroy2()
{
    AbstractMethod("DevDestroy2");
}

void WindowPort::DevStrokeRect2(GrPattern, GrMode, int, Rectangle*)
{
    AbstractMethod("DevStrokeRect2");
}

void WindowPort::DevStrokeRRect2(GrPattern, GrMode, int, Rectangle*, Point)
{
    AbstractMethod("DevStrokeRRect2");
}

void WindowPort::DevFillRRect2(GrPattern, GrMode, Rectangle*, Point)
{
    AbstractMethod("DevFillRRect2");
}

void WindowPort::DevStrokePolygon2(Rectangle*, GrPattern, GrMode, Point*, int, GrPolyType, int, GrLineCap)
{
    AbstractMethod("DevStrokePolygon2");
}

void WindowPort::DevFillPolygon2(Rectangle*, GrPattern, GrMode, Point*, int, GrPolyType)
{
    AbstractMethod("DevFillPolygon2");
}

void WindowPort::DevStrokeLine2(GrPattern, GrMode, int, Rectangle*,
							GrLineCap, Point, Point)
{
    AbstractMethod("DevStrokeLine2");
}

void WindowPort::DevStrokeWedge2(GrPattern, GrMode, int, GrLineCap, Rectangle*, int, int)
{
    AbstractMethod("DevStrokeWedge2");
}

void WindowPort::DevFillWedge2(GrPattern, GrMode, Rectangle*, int, int)
{
    AbstractMethod("DevFillWedge2");
}

void WindowPort::DevStrokeOval2(GrPattern pat, GrMode mode, int psz, Rectangle *r)
{
    DevStrokeRRect2(pat, mode, psz, r, r->extent);
}

void WindowPort::DevFillOval2(GrPattern pat, GrMode mode, Rectangle *r)
{
    DevFillRRect2(pat, mode, r, r->extent);
}

void WindowPort::DevSetCursor(GrCursor)
{
    AbstractMethod("DevSetCursor");
}

void WindowPort::DevScrollRect(Rectangle, Point)
{
    AbstractMethod("DevScrollRect");
}

void WindowPort::DevHide()
{
    AbstractMethod("DevHide");
}

void WindowPort::DevShow(WindowPort*, Rectangle)
{
    AbstractMethod("DevShow");
}

void WindowPort::DevGrab(bool)
{
    AbstractMethod("DevGrab");
}

void WindowPort::DevFullscreen(bool)
{
    AbstractMethod("DevFullScreen");
}

void WindowPort::DevTop(bool)
{
    AbstractMethod("DevTop");
};

void WindowPort::DevSetRect(Rectangle*)
{
    AbstractMethod("DevSetRect");
};

void WindowPort::DevGetEvent(Token*, int, bool)
{
    AbstractMethod("DevGetEvent");
};

void WindowPort::DevSetMousePos(Point)
{
    AbstractMethod("DevSetMousePos");
};

void WindowPort::DevMoveMousePos(Point)
{
    AbstractMethod("DevMoveMousePos");
}

void WindowPort::DevBell(long)
{
    AbstractMethod("DevBell");
}
