//$XWindowPort$
#include "XWindowPort.h"

#include "XWindowSystem.h"
#include "XFont.h"
#include "XBitmap.h"
#include "XClipBoard.h"
#include "../String.h"

#include <X11/keysym.h>
#include <X11/cursorfont.h>

static bool xfullscreen;

static XPixmap MakePixmap(XPixmap &pm, Point &oldsize, Point newsize)
{
    if (newsize.x > oldsize.x || newsize.y > oldsize.y) {
	if (pm)
	    XFreePixmap(display, pm);
	oldsize= newsize+Point(100);
	pm= XCreatePixmap(display, rootwin, oldsize.x, oldsize.y,
						XDefaultDepth(display, screen));
    }
    return pm;
}

//---- Patterns ----------------------------------------------------------------

const int cMaxPatterns= 30;

static char PatBits[cMaxPatterns][32]= {
#include "patterns"
};

static XPixmap PatMap[cMaxPatterns];

void XWindowPort::mapPat(GrPattern pat)
{
    lastpat= pat;
    switch (pat) {
    case ePatWhite:
	XSetForeground(display, gc, XWhitePixel(display, screen));
	XSetFillStyle(display, gc, FillSolid);
	break;
	
    case ePatBlack:
	XSetForeground(display, gc, XBlackPixel(display, screen));
	XSetFillStyle(display, gc, FillSolid);
	break;
	
    default:
	if (XDefaultDepth(display, screen) == 1) {
	    if (PatMap[pat] == 0) {
		PatMap[pat]= XCreatePixmapFromBitmapData(display, rootwin,
		    PatBits[pat], 16, 16, XBlackPixel(display, screen),
					    XWhitePixel(display, screen),
						XDefaultDepth(display, screen));
	    }
	    XSetTile(display, gc, PatMap[pat]);
	    XSetFillStyle(display, gc, FillTiled);
	} else {
	    XSetForeground(display, gc, colors[pat]);
	    XSetFillStyle(display, gc, FillSolid);
	}
	break;
    }
}

//---- Cursors -----------------------------------------------------------------

const int cMaxCursors= 30;
static XCursor CurMap[cMaxCursors];

XCursor UseCursor(GrCursor c)
{
    if (CurMap[c] == 0) {
	unsigned int xc;
	
	switch (c) {
	case eCrsCross:         xc= XC_crosshair; break;
	case eCrsBoldCross:     xc= XC_cross; break;
	case eCrsUpArrow:       xc= XC_sb_up_arrow; break;
	case eCrsDownArrow:     xc= XC_sb_down_arrow; break;
	case eCrsLeftArrow:     xc= XC_sb_left_arrow; break;
	case eCrsRightArrow:    xc= XC_sb_right_arrow; break;
	case eCrsHourglass:     xc= XC_watch; break;
	case eCrsPen:           xc= XC_pencil; break;
	case eCrsMoveHand:      xc= XC_hand2; break;
	case eCrsHand:          xc= XC_hand2; break;
	case eCrsCrossHair:     xc= XC_crosshair; break;
	case eCrsMoveStretch:   xc= XC_hand2; break;
	case eCrsIBeam:         xc= XC_xterm; break;
	case eCrsLeftRightArrow: xc= XC_sb_h_double_arrow; break;
	case eCrsUpDownArrow:   xc= XC_sb_v_double_arrow; break;
	default:
	    xc= XC_left_ptr;
	    break;
	}
	CurMap[c]= XCreateFontCursor(display, xc);
    }
    return CurMap[c];
}

void XWindowPort::DevSetCursor(GrCursor c)
{
    XDefineCursor(display, winid, UseCursor(c));
}

//---- rop mapping ------------------------------------------------------------


void XWindowPort::setMode(GrMode mode)
{
    static int ModeMap[]= {
	GXcopy, GXxor
    };
    
    XSetFunction(display, gc, ModeMap[lastmode= mode]);
}

//---- Event mapping -----------------------------------------------------------

static void StateToFlags(Token *t, unsigned int state)
{
    if (state & ShiftMask)
	t->Flags|= eFlgShiftKey;
    if (state & ControlMask)
	t->Flags|= eFlgCntlKey;
    if (state & Mod1Mask)
	t->Flags|= eFlgMetaKey;
}

bool XWindowPort::MapEvent(Token *tp, XEvent *xe)
{
    Rectangle newrect;
    Token t;
    char buffer[20];
    int ccnt, bufsize= 20;
    XKeySym keysym;
    XComposeStatus compose;
    
    t.Flags= 0;
    t.Code= eEvtNone;
    t.Pos= lastpos;
    
    switch (xe->type) {
    case GraphicsExpose:
    case Expose:
	newrect= Rectangle(xe->xexpose.x, xe->xexpose.y, xe->xexpose.width,
							xe->xexpose.height);
	if (xe->xexpose.count)
	    Damage(eFlgDamage4, &newrect);
	else
	    Damage(eFlgDamage2, &newrect);
	return FALSE;

    case ConfigureNotify:
	newrect= Rectangle(xe->xconfigure.x, xe->xconfigure.y,
				xe->xconfigure.width, xe->xconfigure.height);
			    
	if (firstdamage || windowRect.extent != newrect.extent) {
	    firstdamage= FALSE;
	    Damage(eFlgDamage1, &newrect);
	    return FALSE;
	}
	
	if (windowRect.origin != newrect.origin) {
	    Damage(eFlgDamage3, &newrect);    // origin changed
	    return FALSE;
	}
	return FALSE;
    
    case MapNotify:
	if (firstdamage)
	    Damage(eFlgDamage1, &windowRect);
	firstdamage= FALSE;
	return FALSE;

    case KeyPress:
	ccnt= XLookupString(*((XKeyEvent*)xe), buffer, bufsize, &keysym, &compose);

	switch ((int)keysym) {
	case XK_Left:
	    t.Code= eEvtCursorLeft;
	    break;
	case XK_Up:
	    t.Code= eEvtCursorUp;
	    break;
	case XK_Right:
	    t.Code= eEvtCursorRight;
	    break;
	case XK_Down:
	    t.Code= eEvtCursorDown;
	    break;
	case XK_Tab:
	    t.Code= '\t';
	    break;
	case XK_BackSpace:
	    t.Code= '\b';
	    break;
	case XK_Linefeed:
	    t.Code= '\n';
	    break;
	case XK_Return:
	    t.Code= '\r';
	    break;

	default:
	    if ((int)keysym > 0 && (int)keysym < 256)
		t.Code= (short)keysym;
	    else
		return FALSE;
	    break; 
	}
	StateToFlags(&t, xe->xkey.state);
	if (t.Flags & eFlgCntlKey) {
	    if (t.Flags & eFlgShiftKey)
		t.Code-= 0x40;
	    else
		t.Code-= 0x60;
	}
	if (t.Flags & eFlgMetaKey)
	    t.Code+= 0x80;
	t.Pos= Point(xe->xkey.x, xe->xkey.y) - inputOffset;
	t.At= xe->xkey.time;
	break;
	
    case ButtonPress:
    case ButtonRelease:
	t.Code= eEvtButtons + xe->xbutton.button - 1;
	if (xe->type == ButtonRelease)
	    t.Flags|= eFlgButDown;
	StateToFlags(&t, xe->xbutton.state);
	t.At= xe->xbutton.time;
	t.Pos= Point(xe->xbutton.x, xe->xbutton.y) - inputOffset;
	break;
	
    case MotionNotify:
	// check on butState: prior to the event
	if (xe->xmotion.state & (Button1Mask|Button2Mask|Button3Mask))
	    t.Code= eEvtLocMoveBut;
	else
	    t.Code = eEvtLocMove;
	t.At= xe->xmotion.time;
	t.Pos= Point(xe->xmotion.x, xe->xmotion.y) - inputOffset;
	StateToFlags(&t, xe->xmotion.state);
	break;
	
    case LeaveNotify:
    case EnterNotify:
	if (xfullscreen)
	    return FALSE;
	if (xe->type == EnterNotify)
	    t.Code= eEvtEnter;
	else
	    t.Code= eEvtExit;
	t.At= xe->xcrossing.time;
	t.Pos= Point(xe->xcrossing.x, xe->xcrossing.y) - inputOffset; 
	break;

    case SelectionClear:
	if (xclip)
	    xclip->GiveUpSelection();
	return FALSE;

    case SelectionRequest:
	if (xclip)
	    xclip->SendClipBoard(xe);
	return FALSE;
	
    case PropertyNotify:
	if (xclip)
	    xclip->PropertyChanged(xe);
	return FALSE;
	
    default:
	return FALSE;
    }
    
    lastpos= t.Pos;
    *tp= t;
    return TRUE;
}

//------------------------------------------------------------------------------

void XWindowPort::setLineWidth(int psz, GrMode)
{
    XGCValues gcv;
    
    lastpsz= psz;
    if (psz <= 1)
	psz= 0;
    gcv.line_width= psz;
    XChangeGC(display, gc, GCLineWidth, &gcv);
}

static void AdjustRect(int psz, Rectangle *r)
{
    if (psz > 1) {
	r->origin+= psz/2;
	r->extent-= psz;
    } else
	r->extent-= 1;
}

//---- window creation/destruction ---------------------------------------------

XWindowPort::XWindowPort(InpHandlerFun nf, void *priv1, void *priv2, bool ov, bool bl)
{
    XSetWindowAttributes setWAttr;
    int cwmask;
     
    Init(nf, priv1, priv2, ov, bl);
    
    fso= Point(0);
    lastpat= lastmode= lastpsz= -1;
    savesize= gPoint0;
    save= 0;
    evtMask= StructureNotifyMask | ExposureMask | KeyPressMask |
	ButtonPressMask | ButtonReleaseMask | EnterWindowMask | LeaveWindowMask |
	PointerMotionMask | ButtonMotionMask | PropertyChangeMask;
    
    mySaveUnder= overlay /* && ! XDoesSaveUnders(display, screen) */;
    
    if (mySaveUnder)
	winid= rootwin;
    else {
	setWAttr.background_pixmap= XNone;
	setWAttr.border_pixmap= XNone;
	setWAttr.cursor= UseCursor(eCrsBoldArrow);
	setWAttr.override_redirect= XTrue;
	cwmask= CWBackPixmap | CWBorderPixmap | CWCursor | CWOverrideRedirect;
	if (overlay) {
	    setWAttr.save_under= XTrue;
	    cwmask|= CWSaveUnder;
	}
	winid= XCreateWindow(display, XRootWindow(display, screen),
			0, 0, 1, 1, 0, XDefaultDepth(display, screen),
			    XInputOutput, XCopyFromParent, cwmask, &setWAttr);
	XSelectInput(display, winid, evtMask);
	if (awin == 0)  
	    awin= winid;
    }
    id= winid;
    
    XSaveContext(display, winid, context, this);
    gc= XCreateGC(display, winid, 0, NULL);
    XSetGraphicsExposures(display, gc, XTrue);
    if (mySaveUnder)
	XSetSubwindowMode(display, gc, IncludeInferiors);
}

void XWindowPort::DevDestroy2()
{
    XDeleteContext(display, winid, context);
    if (! mySaveUnder)
	XDestroyWindow(display, winid);
}

//---- window management -------------------------------------------------------

void XWindowPort::DevShow(WindowPort *fp, Rectangle r)
{
    if (mySaveUnder) {
	xfullscreen= TRUE;
	father= (XWindowPort*)fp;
	ovr= r;
	fso= inputOffset= r.origin;
	if (father->mySaveUnder) {
	    XSelectInput(display, winid, evtMask);
	    XDefineCursor(display, winid, UseCursor(cursor));
	} else {
	    XGrabServer(display);
	    XGrabPointer(display, winid, XFalse, evtMask|PointerMotionHintMask,
			GrabModeAsync, GrabModeAsync, XNone, UseCursor(cursor),
								XCurrentTime);
	}
	MakePixmap(save, savesize, ovr.extent);
	SetMode(eRopCopy);
	XCopyArea(display, winid, save, gc, ovr.origin.x, ovr.origin.y,
					ovr.extent.x, ovr.extent.y, 0, 0);
	Damage(eFlgDamage5, &r);
    } else {
	firstdamage= TRUE;
	if (windowRect != r)
	    XMoveResizeWindow(display, winid, r.origin.x, r.origin.y, r.extent.x, r.extent.y);
	XMapRaised(display, winid);
	XSync(display, 0);
	XEvent ev;
	Token t;
	do {
	    XWindowEvent(display, winid, evtMask, ev);
	    MapEvent(&t, &ev);
	} while (ev.type != Expose);
    }
}

void XWindowPort::DevHide()
{     
    if (mySaveUnder) {
	inputOffset= gPoint0;
	DevResetClip();
	SetMode(eRopCopy);
	XCopyArea(display, save, winid, gc, 0, 0, ovr.extent.x, ovr.extent.y,
						    ovr.origin.x, ovr.origin.y);
	if (father->mySaveUnder) {
	    XSelectInput(display, winid, father->evtMask);
	    XDefineCursor(display, winid, UseCursor(father->cursor));
	} else {
	    XUndefineCursor(display, winid);
	    XUngrabPointer(display, XCurrentTime);
	    XUngrabServer(display);
	}
	xfullscreen= FALSE;
    } else
	XUnmapWindow(display, winid);
}

void XWindowPort::DevSetRect(Rectangle *r)
{
    if (windowRect.extent != r->extent)
	firstdamage= TRUE;
    if (windowRect.origin != r->origin && windowRect.extent != r->extent)
	XMoveResizeWindow(display, winid, r->origin.x, r->origin.y, r->extent.x,
								r->extent.y);
    else if (windowRect.origin != r->origin)
	XMoveWindow(display, winid, r->origin.x, r->origin.y);
    else if (windowRect.extent != r->extent)
	XResizeWindow(display, winid, r->extent.x, r->extent.y);
}

void XWindowPort::DevTop(bool top)
{
    if (top) {  
	Token t;
	XEvent ev;
	
	XRaiseWindow(display, winid);
	XSync(display, 0);
	while (XCheckTypedWindowEvent(display, winid, Expose, ev))
	    MapEvent(&t, &ev);
    } else
	XLowerWindow(display, winid);
}

void XWindowPort::DevBell(long d)
{
    XKeyboardControl kb_ctrl;
    kb_ctrl.bell_duration= (int) d;
    XChangeKeyboardControl(display, KBBellDuration, kb_ctrl);
    XBell(display, 0);
}

//---- clipping ----------------------------------------------------------------

void XWindowPort::DevClip(Rectangle r, Point p)
{
    r+= inputOffset;
    XSetClipRectangles(display, gc, 0, 0, (XRectangle*)&r, 1, YXBanded);
    XSetTSOrigin(display, gc, p.x, p.y);
} 

void XWindowPort::DevResetClip()
{
    XSetClipMask(display, gc, XNone);
    XSetTSOrigin(display, gc, 0, 0);
}

//---- graphical primitives ----------------------------------------------------
       
void XWindowPort::DevStrokeLine2(GrPattern pat, GrMode mode, int psz,
    Rectangle*, GrLineCap, Point start, Point end)
{
    start+= fso;
    end+= fso;
    MapPat(pat);
    SetMode(mode);
    SetLineWidth(psz, mode);
    XDrawLine(display, id, gc, start.x, start.y, end.x, end.y);
}

void XWindowPort::DevStrokeRect2(GrPattern pat, GrMode mode, int psz,
    Rectangle *r)
{
    MapPat(pat);
    SetMode(mode);
    SetLineWidth(psz, mode);
    AdjustRect(psz, r);
    XDrawRectangle(display, id, gc, fso.x+r->origin.x, fso.y+r->origin.y,
	r->extent.x, r->extent.y);
}

void XWindowPort::DevFillRect(GrPattern pat, GrMode mode, Rectangle *r)
{
    MapPat(pat);
    SetMode(mode);
    XFillRectangle(display, id, gc, fso.x+r->origin.x, fso.y+r->origin.y,
	 r->extent.x, r->extent.y);
}

void XWindowPort::DevStrokeRRect2(GrPattern pat, GrMode mode, int psz,
							Rectangle *r, Point dia)
{
    XArc a[4];
    XSegment s[4];
    
    AdjustRect(psz, r);
    Point o= fso + r->origin;
    Point e= r->extent-dia;
    
    for (int i= 0; i < 4; i++) {
	a[i].angle1= (i*90) << 6;
	a[i].angle2= 90 << 6;
	a[i].width= dia.x;
	a[i].height= dia.y;
	a[i].x= o.x;
	a[i].y= o.y;
    }
    a[0].x+= e.x;
    a[3].x+= e.x;
    a[2].y+= e.y;
    a[3].y+= e.y;
    
    s[0].x1= s[3].x1= o.x+dia.x/2;
    s[0].x2= s[3].x2= o.x+r->extent.x-dia.x/2;
    s[1].x1= s[1].x2= o.x;
    s[2].x1= s[2].x2= o.x+r->extent.x;
    
    s[0].y1= s[0].y2= o.y;
    s[1].y1= s[2].y1= o.y+dia.y/2;
    s[1].y2= s[2].y2= o.y+r->extent.y-dia.y/2;
    s[3].y1= s[3].y2= o.y+r->extent.y;
    
    MapPat(pat);
    SetMode(mode);
    SetLineWidth(psz, mode);
    XDrawArcs(display, id, gc, a, 4);
    XDrawSegments(display, id, gc, s, 4);
}

void XWindowPort::DevFillRRect2(GrPattern pat, GrMode mode, Rectangle *r,
								    Point dia)
{
    XArc a[4];
    XRectangle rr[3];
    Point o= fso + r->origin;
    Point e= r->extent-dia;
    
    for (int i= 0; i < 4; i++) {
	a[i].angle1= (i*90) << 6;
	a[i].angle2= 90 << 6;
	a[i].width= dia.x;
	a[i].height= dia.y;
	a[i].x= o.x;
	a[i].y= o.y;
    }
    a[0].x+= e.x;
    a[3].x+= e.x;
    a[2].y+= e.y;
    a[3].y+= e.y;
    
    rr[0].x= rr[2].x= o.x + dia.x/2;
    rr[1].x= o.x;
    
    rr[0].y= o.y;
    rr[1].y= o.y+dia.y/2;
    rr[2].y= o.y+r->extent.y-dia.y/2;
    
    rr[0].width= rr[2].width= e.x;
    rr[1].width= r->extent.x;
    
    rr[0].height= rr[2].height= dia.y/2;
    rr[1].height= e.y;
    
    MapPat(pat);
    SetMode(mode);
    XFillArcs(display, id, gc, a, 4);
    XFillRectangles(display, id, gc, rr, 3);
}

void XWindowPort::DevStrokeOval2(GrPattern pat, GrMode mode, int psz, Rectangle *r)
{
    MapPat(pat);
    SetMode(mode);
    SetLineWidth(psz, mode);
    AdjustRect(psz, r);
    XDrawArc(display, id, gc, fso.x+r->origin.x, fso.y+r->origin.y,
	 r->extent.x, r->extent.y, 0, 360*64);
}

void XWindowPort::DevFillOval2(GrPattern pat, GrMode mode, Rectangle *r)
{
    MapPat(pat);
    SetMode(mode);
    XFillArc(display, id, gc, fso.x+r->origin.x, fso.y+r->origin.y,
	 r->extent.x, r->extent.y, 0, 360*64);
}

void XWindowPort::DevStrokeWedge2(GrPattern pat, GrMode mode, int psz,
    GrLineCap, Rectangle *r, int s, int e)
{
    MapPat(pat);
    SetMode(mode);
    SetLineWidth(psz, mode);
    AdjustRect(psz, r);
    if (s > e)
	 e+= 360;
    XDrawArc(display, id, gc, fso.x+r->origin.x, fso.y+r->origin.y,
	 r->extent.x, r->extent.y, (450-s)*64, (s-e)*64);
}

void XWindowPort::DevFillWedge2(GrPattern pat, GrMode mode, Rectangle *r,
    int s, int e)
{
    MapPat(pat);
    SetMode(mode);
    if (s > e)
	 e+= 360;
    XFillArc(display, id, gc, fso.x+r->origin.x, fso.y+r->origin.y,
	 r->extent.x, r->extent.y, (450-s)*64, (s-e)*64);
}


void XWindowPort::DevStrokePolygon2(Rectangle *r, GrPattern pat, GrMode mode,
			Point *pts, int npts, GrPolyType, int psz, GrLineCap)
{
    Point xpts[1000];
    
    MapPat(pat);
    SetMode(mode);
    SetLineWidth(psz, mode);
    Point at= r->origin+fso;
    for (int i= 0; i < npts; i++)
	xpts[i]= pts[i]+at;
    XDrawLines(display, id, gc, (XPoint*) xpts, npts, CoordModeOrigin);
}

void XWindowPort::DevFillPolygon2(Rectangle *r, GrPattern pat, GrMode mode,
					    Point *pts, int npts, GrPolyType)
{
    Point xpts[1000];
    
    MapPat(pat);
    SetMode(mode);
    Point at= r->origin+fso;
    for (int i= 0; i < npts; i++)
	xpts[i]= pts[i]+at;
    XFillPolygon(display, id, gc, (XPoint*) xpts, npts,
			    (npts <= 3) ? Convex : Complex, CoordModeOrigin);
}

void XWindowPort::DevShowBitmap(GrPattern, GrMode mode, Rectangle *r, Bitmap *bm)
{
    XPixmap pm= ((XBitmap*)bm)->GetPixmap();
    
    /*
    SetMode(mode);
    XSetBackground(display, gc, XWhitePixel(display, screen));
    XSetForeground(display, gc, XBlackPixel(display, screen));
    XSetStipple(display, gc, pm);
    XSetFillStyle(display, gc, FillOpaqueStippled);
    XSetTSOrigin(display, gc, fso.x+r->origin.x, fso.y+r->origin.y);
    XFillRectangle(display, id, gc, fso.x+r->origin.x, fso.y+r->origin.y,
						    bm->Size().x, bm->Size().y);
    */
    SetMode(mode);
    XCopyArea(display, pm, id, gc, 0, 0, bm->Size().x, bm->Size().y,
				    fso.x+r->origin.x, fso.y+r->origin.y);
}

//---- text batching -----------------------------------------------------------

static XTextItem ti[40], *tip= ti;
static byte tbc[MaxTextBatchCnt], *tbp= tbc;

bool XWindowPort::DevShowChar(FontPtr fdp, Point delta, byte c, bool isnew, Point)
{
    if (delta.y)            // X cannot handle multiple lines !!
	return TRUE;
    if (isnew) {                // first
	tip= ti;
	tip->delta= 0;
	tip->chars= tbp= tbc;
	tip->nchars= 0;
	tip->font= ((XServerFont*)fdp)->GetId();
	lastfont= fdp;
    } else if (fdp != lastfont || delta.x) {   // switch to next
	tip++;
	tip->delta= delta.x;
	tip->chars= tbp;
	tip->nchars= 0;
	tip->font= ((XServerFont*)fdp)->GetId();
	lastfont= fdp;
    }
    
    *tbp++= c;
    tip->nchars++;
    return FALSE;
}

void XWindowPort::DevShowTextBatch(GrPattern pat, GrMode mode, Rectangle*,
								     Point pos)
{
    MapPat(pat);
    SetMode(mode);
    XDrawText(display, id, gc, pos.x+fso.x, pos.y+fso.y, ti, (tip-ti)+1);
} 

//---- scrolling ---------------------------------------------------------------

void XWindowPort::DevScrollRect(Rectangle r, Point delta)
{
    r+= inputOffset;
    SetMode(eRopCopy);
    XCopyArea(display, id, id, gc, r.origin.x - delta.x, r.origin.y - delta.y,
			      r.extent.x, r.extent.y, r.origin.x, r.origin.y);
}

//---- input -------------------------------------------------------------------

void XWindowPort::DevGetEvent(Token *t, int timeout, bool)
{   
    XEvent          ev;
    unsigned int    state;
    Point           w;
    
    if (timeout < 0) {
	do
	    XWindowEvent(display, winid, evtMask, ev);
	while (!MapEvent(t, &ev));
	return;
    }
    
    t->Flags= 0;
    t->Code= eEvtNone;
    
    while (XCheckWindowEvent(display, winid, evtMask|PointerMotionHintMask, ev)) {
	if (ev.type == MotionNotify) {
	    while(XCheckMaskEvent(display, ButtonMotionMask, ev))
		;   // overread MotionNotifies
	    break;
	}
	if (MapEvent(t, &ev))    
	    return;
    }
    
    t->Pos= getMousePos(state);
    if (t->Pos == lastpos)
	return;
    lastpos= t->Pos;
    if (state & (Button1Mask | Button2Mask | Button3Mask))
	t->Code= eEvtLocMoveBut;
    else
	t->Code= eEvtLocMove;
    StateToFlags(t, state);
}

void XWindowPort::DevFullscreen(bool mode)
{
    if (overlay)
	return;
    if (mode) {
	id= rootwin;
	fso= windowRect.origin;
	XSetSubwindowMode(display, gc, IncludeInferiors);
	XGrabServer(display);
	XGrabPointer(display, winid, XFalse, evtMask|PointerMotionHintMask,
		    GrabModeAsync, GrabModeAsync, XNone, XNone, XCurrentTime);
    } else {
	id= winid;
	fso= Point(0);
	XUngrabPointer(display, XCurrentTime);
	XUngrabServer(display);
    }
}

void XWindowPort::DevGrab(bool mode)
{
    if (overlay)
	return;
    if (mode)
	XGrabPointer(display, winid, XFalse, evtMask|PointerMotionHintMask,
		    GrabModeAsync, GrabModeAsync, XNone, XNone, XCurrentTime);
    else
	XUngrabPointer(display, XCurrentTime);
}

//---- mouse position ----------------------------------------------------------

Point XWindowPort::getMousePos(unsigned int &state)
{
    XWindow rootW, childW;
    int     x0, y0, wx, wy;
    
    XQueryPointer(display, winid, rootW, childW, x0, y0, wx, wy, state);
    return Point(wx, wy) - inputOffset;
}

void XWindowPort::DevSetMousePos(Point p)
{
    XWarpPointer(display, XNone, winid, 0, 0, 0, 0, p.x, p.y);
}

void XWindowPort::DevMoveMousePos(Point p)
{
    p+= windowRect.origin;
    XWarpPointer(display, XNone, XNone, 0, 0, 0, 0, p.x, p.y);
}

//---- double buffering --------------------------------------------------------

static Point batchSize;
static XPixmap batchPixmap;
static Rectangle batchRect;
static bool inBatch;

void XWindowPort::DevGiveHint(int code, int, void *vp)
{
    switch (code) {    
    case eHintBatch:
	if (overlay || (id != winid))
	    return;
	inBatch= TRUE;
	batchRect= *((Rectangle*)vp);
	id= MakePixmap(batchPixmap, batchSize, batchRect.origin+batchRect.extent);
	break;
	
    case eHintUnbatch:
	if (!inBatch)
	    return;
	inBatch= FALSE;
	id= winid;
	SetMode(eRopCopy);
	XCopyArea(display, batchPixmap, id, gc,
	    batchRect.origin.x, batchRect.origin.y,
		batchRect.extent.x, batchRect.extent.y,
		    batchRect.origin.x, batchRect.origin.y);
	break;
    case eHintFlush:
	XFlush(display);
	break;
    }
}

void XCleanup()
{
    if (batchPixmap) {
	XFreePixmap(display, batchPixmap);
	batchPixmap= 0;
    }
    for (int i= 0; i < cMaxPatterns; i++)
	if (PatMap[i])
	    XFreePixmap(display, PatMap[i]);
}
