//$DrawView$
#include "DrawView.h"
#include "ObjList.h"
#include "Commands.h"
#include "Menu.h"
#include "Document.h"
#include "TextShape.h"
#include "TextCmd.h"
#include "Group.h"
#include "CmdNo.h"
#include "CollectionView.h"
#include "ClipBoard.h"
#include "System.h"

extern Clipper *gClipper;

const int cARRANGEMENU  = 1121,
	  cOPTIONEMENU  = 1122,
	  cPOLYGONMENU  = 1123,
	  cGridSize     = 8;
	  
//---- DrawView ----------------------------------------------------------------

MetaImpl(DrawView, (I_O(activeTextView), I_O(currShape), I_P(lastClick), I_I(grid),
	I_B(showGrid), I_O(shapes), I_O(selection)));

DrawView::DrawView(Document *dp, Point extent, ObjList *sl) : (dp, extent)
{
    showGrid= TRUE;
    grid= cGridSize;
    activeText= 0;
    activeTextView= 0;
    lastClick = Point(100);
    shapes= sl;
    currShape= 0;
    selection= new ObjList;
}

DrawView::~DrawView()
{
    SafeDelete(selection);
    if (shapes) {
	shapes->FreeAll();
	SafeDelete(shapes);
    }
}

//---- initializing ------------------------------------------------------------

void DrawView::SetShapes(class ObjList *sl)
{
    if (shapes) {
	shapes->FreeAll();
	SafeDelete(shapes);
    }
    shapes= sl;
    SetSelection(0);
    shapes->ForEach(Shape,SetView)(this);
    ForceRedraw(); 
}

//---- drawing -----------------------------------------------------------------

void DrawView::DoHighlightSelection(HighlightState st)
{
    if (selection) {
	GrSetMode(eRopXor);
	GrSetPattern(ePatBlack);
	Iter next(selection);
	register Shape *s;
    
	while (s= (Shape*)next())
	    if (! s->GetDeleted())
		s->Highlight(st);
    }
}

void DrawView::Invalidate(ObjList *ol)
{
    Rectangle bbox;
    Iter next(ol);
    Shape *s;
    
    while (s= (Shape*)next())
	bbox.Merge(s->InvalRect());
    InvalidateRect(bbox);
}

void DrawView::DrawBackground(Rectangle r)
{
    if (!gPrinting) {
	if (showGrid)
	    GrPaintRect(r, ePat15);
	View::DrawBackground(r);
    }
}

void DrawView::Draw(Rectangle r)
{
    RevIter previous(shapes);
    Shape *s;
    
    GrSetMode(eRopCopy);
    GrSetPenMode(eRopCopy);
    while (s= (Shape*) previous())
	s->DrawAll(r, gPoint0);
}

//---- shape list management ---------------------------------------------------
    
void DrawView::Insert(Shape *s)
{
    SetSelection(0);
    s->SetView(this);
    shapes->Insert(s);
    selection->Add(s);
    InvalidateRect(s->InvalRect());
    ShowSelection();
}

void DrawView::InsertShapes(ObjList *ol, bool tofront= TRUE)
{
    RevIter previous(ol);
    Shape *s;
    
    SetSelection(0);
    while (s= (Shape*) previous()) {
	if (! s->IsKindOf(Shape))
	    continue;
	s->SetView(this);
	if (tofront)
	    shapes->Insert(s);
	else
	    shapes->Add(s);
	selection->Add(s);
    }
    Invalidate(ol);
    ShowSelection();
}

Shape *DrawView::FindShape(Point p)
{
    Iter next(shapes);
    Shape *s;
    
    while (s= (Shape*)next())
	if (s->ContainsPoint1(p))
	    return s;
    return 0;
}

Shape *DrawView::FindHandle(Point p, int *nh)
{
    Iter next(selection);
    Shape *s;
    int handle;
    
    while (s= (Shape*)next())
	if ((handle= s->PointOnHandle(p)) >= 0) {
	    *nh= handle;
	    return s;
	}
    return 0;
}

//---- misc --------------------------------------------------------------------

void DrawView::Point2Grid(Point *np)  // constrain to viewsize and align to grid
{
    Point p= Min(GetExtent(), Max(gPoint3, *np));
    *np= ((p+grid/2) / grid) * grid;
}

void DrawView::ConstrainScroll(Point *p)
{
    if (activeTextView)
	activeTextView->ConstrainScroll(p);
}

void DrawView::RequestTool(int tix)
{
    Control(GetId(), 0, (void*) tix);
}

void DrawView::SetTool(Object *cs)
{
    Shape *oldShape= currShape;

    if (cs && cs->IsKindOf(Shape))
	currShape= (Shape*) cs;
    else
	currShape= 0;
	
    // enter text mode
    if ((oldShape == 0 || !oldShape->IsKindOf(TextShape)) &&
				currShape && currShape->IsKindOf(TextShape)) {
	PerformCommand(gResetUndo);
	SetSelection(0);
    }

    // leave text mode
    if ((currShape == 0 || !currShape->IsKindOf(TextShape)) &&
				oldShape && oldShape->IsKindOf(TextShape)) {
	PerformCommand(gResetUndo);
	SetSelection(0);
	SetActiveText(0);
    }
}

void DrawView::ShowInfo(TrackPhase tp, char *fmt, ...)
{
    if (tp == eTrackRelease)
	Control(GetId(), 123, "");
    else {
	char buf[100];
	va_list ap;
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	va_end(ap);
	Control(GetId(), 123, buf);
    }
}

//---- Selection ---------------------------------------------------------------

Shape *DrawView::OneSelected()
{
    if (Selected() != 1)
	return 0;
    return (Shape*)selection->First();
}

ObjList *DrawView::GetCopyOfSelection()
{
    //RemoveDeleted();
    return (ObjList*)selection->Clone();
}

ObjList *DrawView::GetDeepCopyOfSelection()
{
    //RemoveDeleted();
    return (ObjList*)selection->DeepClone();
}

ObjList *DrawView::SetSelection(ObjList *newsel)
{
    ObjList *oldsel= selection;
    
    if (selection->Size()) {
	selection->ForEach(Shape,SetSplit)(FALSE);
	Invalidate(selection);
	SafeDelete(selection);
	selection= 0;
	UpdateEvent();
    }
    if (newsel) {
	selection= (ObjList*) newsel->Clone();
	Invalidate(selection);
	ShowSelection();   
    } else
	selection= new ObjList;
    return oldsel;
}

void DrawView::SetDeleted(ObjList* ol, bool b)
{
    ol->ForEach(Shape,SetDeleted)(b);
    Invalidate(ol);
}

void DrawView::RemoveDeleted()
{
    Iter next(shapes);
    register Shape *s;

    while (s= (Shape*) next())
	if (s->IsGarbage())
	    delete shapes->Remove(s);
}

void DrawView::ShowSelection()
{
    Rectangle bbox= BoundingBox();
    RevealRect(bbox.Expand(8), bbox.extent/3);
}

void DrawView::SelectInRect(Rectangle r)
{
    Iter next(shapes);
    register Shape *s;
    
    while (s= (Shape*)next())
	if (!s->GetDeleted() && r.ContainsRect(s->bbox))
	    selection->Add(s);
    Invalidate(shapes);
}

Rectangle DrawView::BoundingBox()
{
    Rectangle bbox;
    Iter next(selection);
    Shape *s;
    
    while (s= (Shape*)next())
	bbox.Merge(s->bbox);
    return bbox;
}

bool DrawView::HasSelection()
{
    if (activeTextView || selection->Size() > 0)
	return TRUE;
    return View::HasSelection();
}

//---- text --------------------------------------------------------------------

void DrawView::SetActiveText(TextShape *tp)
{
    /*
    TextView *tp= 0;
    if (ts)
	tp= ts->GetTextView();
    */
    if (activeText) {
	selection->Remove(activeText);
	activeText->Invalidate();
	activeText= 0;
	activeTextView= 0;
	UpdateEvent();
    }
    activeText= tp;
    activeTextView= 0;
    if (activeText) {
	activeTextView= activeText->GetTextView();
	if (! selection->ContainsPtr(activeText))
	    selection->Add(activeText);
	activeText->Invalidate();
    }
}

//---- event handling ----------------------------------------------------------

Command *DrawView::DoLeftButtonDownCommand(Point p, Token t, int)
{
    Shape *ShapeUnderMouse, *s;
    int handle;
    SketchModes sm= eSMDefault;
    bool inselection;
    
    if (t.Flags & eFlgShiftKey)
	sm|= eSMSquare;
    if (t.Flags & eFlgCntlKey)
	sm|= eSMCenter;
       
    if (currShape == 0 || !currShape->IsKindOf(TextShape))  // exit text mode
	SetActiveText(0);
	
    ShapeUnderMouse= FindShape(p);
    lastClick= p;
    
    if (currShape) {    // not pointer mode
	if (currShape->IsKindOf(TextShape) && ShapeUnderMouse
			&& ShapeUnderMouse->IsKindOf(TextShape)) {  // Text Mode
	    SetActiveText((TextShape*) ShapeUnderMouse);
	    return activeTextView->DispatchEvents(p, t, gClipper);
	}
	SetSelection(0);
	return currShape->NewSketcher(this, sm);
    }
    
    if (s= FindHandle(p, &handle))
	return s->NewStretcher(this, handle);

    if (ShapeUnderMouse == 0) {
	SetSelection(0);
	return new ShapeSelector(this);
    } 
    
    inselection= selection->ContainsPtr(ShapeUnderMouse);
    if (! inselection) {
	if (! (t.Flags & eFlgShiftKey))
	    SetSelection(0);
	selection->Add(ShapeUnderMouse);
	ShapeUnderMouse->Invalidate();
    } else {
	if (t.Flags & eFlgShiftKey) {
	    selection->Remove(ShapeUnderMouse);
	    ShapeUnderMouse->Invalidate();
	}
    }
    if (selection->ContainsPtr(ShapeUnderMouse)) {
	if ((handle= ShapeUnderMouse->PointOnHandle(p)) >= 0)
	    return ShapeUnderMouse->NewStretcher(this, handle);
	return new ShapeDragger(this, ShapeUnderMouse);
    }
    return gNoChanges;
}

Command *DrawView::DoKeyCommand(int code, Point lp, Token t)
{
    TextShape *ts;
    
    if (activeTextView)
	return activeTextView->DispatchEvents(lp, t, gClipper);

    if (code == gBackspace)
	return new SCutCopyCommand(this, cDELETE, "delete");
	
    //----- enter text mode and create an attached TextShape
    Shape *chief= OneSelected();
    RequestTool(1);
    
    //---- on text shape and selected hence append typed character at end
    if (chief && chief->IsKindOf(TextShape)) { 
	SetSelection(0);
	SetActiveText((TextShape*)chief);
	activeTextView->SetSelection(cMaxInt, cMaxInt);
	return activeTextView->DispatchEvents(lp, t, gClipper);
	//return activeTextView->DoKeyCommand(code, lp, t);
    }
    
    //---- create new textshape
    SetSelection(0);
    ts= (TextShape*) currShape->Clone();
    ts->SetView(this);
    if (chief) {
	Rectangle textRect= chief->GetTextRect();
	ts->Init(textRect.NW(), textRect.SE());
	ts->MakeDependentOn(chief); // make the text object dependend if there is a chief
    } else
	ts->Init(lastClick, lastClick);
    Insert(ts);
    SetActiveText(ts);
    
    // return activeTextView->DispatchEvents(lp, t, gClipper);
    return activeTextView->DoKeyCommand(code, lp, t);
}

Command *DrawView::DoCursorKeyCommand(EvtCursorDir cd, Point p, Token t)
{
    if (activeTextView)  
	return activeTextView->DoCursorKeyCommand(cd, p, t);
	
    if (selection->Size() > 0) {
	Point delta= t.CursorPoint() * grid;
	Rectangle bb= BoundingBox() + delta;
	delta+= bb.AmountToTranslateWithin(GetExtent());
	return new CursorMoveCommand(this, delta);
    }
    return View::DoCursorKeyCommand(cd, p, t);
}

GrCursor DrawView::GetCursor(Point lp)
{
    if (currShape)
	return currShape->SketchCursor();
    return View::GetCursor(lp);
}

//---- menus -------------------------------------------------------------------

void DrawView::DoCreateMenu(Menu *menu)
{
    View::DoCreateMenu(menu);
    Menu *m;

    menu->InsertItemsAfter(cLASTEDIT, 
		    "delete",       cDELETE,
		    "-",
		    "dup",          cDUP,
		    "edit shape",   cSPLIT,
		    "connect",      cCONNECT,
		    0);

    m= new Menu("arrange");
    m->AppendItems(
		    "bring to front",   cTOFRONT,
		    "send to back",     cTOBACK,
		    "-",
		    "group",            cGROUP,
		    "ungroup",          cUNGROUP,
		    0);
    menu->AppendMenu(m, cARRANGEMENU);
    
    m= new Menu("options");
    m->AppendItems(
		    "grid off",     cGRID,
		    "hide grid",    cSHWGRID,
		    0);
    menu->AppendMenu(m, cOPTIONEMENU);
    
    m= new Menu("polygons");
    m->AppendItems(
		    "Polygon",      cFIRSTSPLINE,
		    "Bezier",       cFIRSTSPLINE + 1,
		    "Spline 2",     cFIRSTSPLINE + 2,
		    "Spline 3",     cFIRSTSPLINE + 3,
		    0);
    menu->AppendMenu(m, cPOLYGONMENU);
}    

void DrawView::DoSetupMenu(Menu *menu)
{
    if (activeTextView) {
	if (!activeTextView->Caret())
	    menu->EnableItem(cDELETE);
    } else {
	int n= selection->Size();
    
	if (n > 0) {
	    Shape *s;
    
	    menu->EnableItems(cDELETE, cDUP, cTOFRONT, cTOBACK, cARRANGEMENU,
			    cPOLYGONMENU, cFIRSTSPLINE,
			    cFIRSTSPLINE+1, cFIRSTSPLINE+2, cFIRSTSPLINE+3, 0);
	    if (s= OneSelected()) {
		if (s->IsKindOf(Group))
		    menu->EnableItem(cUNGROUP);
		if (s->CanSplit())
		    menu->EnableItem(cSPLIT);
	    } else if (n >= 2)
		menu->EnableItems(cCONNECT, cGROUP, 0);
	}
    }
    menu->ReplaceItem(cGRID, (grid == cGridSize) ? "grid off" : "grid on");
    menu->ReplaceItem(cSHWGRID, showGrid ? "hide grid" : "show grid");
    menu->EnableItems(cOPTIONEMENU, cGRID, cSHWGRID, 0);
    View::DoSetupMenu(menu);
}

Command *DrawView::DoMenuCommand(int cmd)
{
    Shape *p= OneSelected();

    switch(cmd) {
    case cTOFRONT:
	return new FrontBackCommand(this, cmd, "bring to front");

    case cTOBACK:
	return new FrontBackCommand(this, cmd, "bring to back");

    case cDELETE:
	if (activeTextView)
	    return new CutCopyCommand(activeTextView, cCUT, "delete text");
	return new SCutCopyCommand(this, cmd, "delete");

    case cCUT:
    case cCOPY:
	View::DoMenuCommand(cmd);
	if (activeTextView)
	    return activeTextView->DoMenuCommand(cmd);
	return new SCutCopyCommand(this, cmd);

    case cDUP:
	return new DupCommand(this);
	
    case cGROUP:
	return new GroupCommand(this);
	
    case cUNGROUP:
	return new UngroupCommand(this, (Group*)p);
	
    case cSPLIT:
	if (p)
	    p->SetSplit(TRUE);
	break;
	
    case cCONNECT:
	return new ConnectCommand(this);
	
    case cGRID:
	grid= (grid == 1) ? cGridSize : 1;
	break;
	
    case cSHWGRID:
	showGrid= ! showGrid;
	ForceRedraw();
	break;
		
    default:
	if (cmd >= cFIRSTSPLINE && cmd <= cLASTSPLINE)
	    return new PropertyCommand(this, eShapeSmooth, cmd-cFIRSTSPLINE, "set smooth");

	return View::DoMenuCommand(cmd);
    }
    return gNoChanges;
}

//---- clipboard ----------------------------------------------------------------

void DrawView::SelectionToClipboard(char *type, ostream &os)
{
    if (activeTextView)
	activeTextView->SelectionToClipboard(type, os);
    else if (strcmp(type, cClipEt) == 0)
	os << selection SP;
}

Command *DrawView::PasteData(char *type, istream &is)
{
    if (activeTextView)
	return activeTextView->PasteData(type, is);
    
    if (strcmp(type, cClipEt) == 0) {
	Object *op;
	is >> op;
	if (op && op->IsKindOf(ObjList))
	    return new SPasteCommand(this, (ObjList*)op, lastClick);
    }
    return gNoChanges;
}

bool DrawView::CanPaste(char *type)
{
    if (activeTextView)
	return activeTextView->CanPaste(type);
    return strismember(type, cClipEt, 0);
}
