#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <varargs.h>

#define BUFSIZE 2000

extern char *malloc(), *strcpy();

static void Request();
static void Puts();

static char *progname;
static int debug= 0;
static int first= 1;
static int nooutput= 0;
static int skip, lastc;
static char *look;

void ProcessFile();
static int line;
static char *filename;
static char left= '<', right= '>';

static struct Tab {
    char *name;
    char *value;
    void (*proc)();
} Tab[100];

static void Error(va_alist)
va_dcl
{
    va_list args;
    char *fmt;
    
    va_start(args);
    fmt= va_arg(args, char*);
    fprintf(stderr, "%s: error in file \"%s\" line %d: ", progname, filename, line);
    vfprintf(stderr, fmt, args);
    fputc('\n', stderr);
    va_end(args);
}

static char* strsave(s)
char *s;
{
    return strcpy(malloc(strlen(s)), s);
}

struct Tab *Lookup(name)
char *name;
{
    register struct Tab *tb= Tab;

    for (tb= Tab; tb->name; tb++)
	if (strcmp(tb->name, name) == 0)
	    break;              
    return tb;
}

static void Print(ofp, args, buf)
FILE *ofp;
char **args, *buf;
{
    fprintf(stderr, "%s\n", buf);
}

static void Include(ofp, args, buf)
FILE *ofp;
char **args, *buf;
{
    FILE *ifp;

    if ((ifp= fopen(args[1], "r")) == 0)
	Error("can't open include file %s", args[1]);
    else {
	ProcessFile(ifp, ofp, args[1]);
	fclose(ifp);
    }
}

static void Def(ofp, args, buf)
FILE *ofp;
char **args, *buf;
{
    struct Tab *tb= Lookup(args[1]);
    if (tb->name == 0)
	tb->name= strsave(args[1]);
    if (tb->value)
	free(tb->value);
    tb->value= strsave(args[2]);
    tb->proc= 0;
}

static void EnterProc(name, proc)
char *name;
void (*proc)();
{
    struct Tab *tb= Lookup(name);
    tb->name= name;
    tb->proc= proc;
    if (tb->value)
	free(tb->value);
    tb->value= 0;
}

static void sigHandler(sig)
int sig;
{
    extern char *sys_siglist[];
    Error(sys_siglist[sig]);
    exit(sig); 
}

static int Fgetc(ifp)
FILE *ifp;
{
    int c= fgetc(ifp);
    if (c == '\n')
	line++;
    return c;
}

static void Putf(c, ofp)
int c;
FILE *ofp;
{
    if (first) {
	if (isspace(c))
	    return;
	first= 0;
	if (skip) {
	    skip= 0;
	    Putf('\n', ofp);
	}
    }
    if(! nooutput) {
	lastc= c;
	fputc(c, ofp);
    }
}

static void Puts(s, ofp)
register char *s;
FILE *ofp;
{
    register int c;

    while (c= *s++)
	Putf(c, ofp);
}

static void Replace(ofp, value, args, buf)
FILE *ofp;
char *value, **args, *buf;
{
    register int c;

    while (c= *value++) {
	switch (c) {
	case '\\':
	    c= *value++;
	    if (c != '@')
		Putf('\\', ofp);
	    Putf(c, ofp);
	    break;
	case '@':
	    c= *value++;
	    switch (c) {
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
		Puts(args[c-'0'], ofp);
		break;
	    case '@':
		Puts(buf, ofp);
		break;
	    case 'n':
		Putf('\n', ofp);
		break;
	    case 'N':
		if (lastc != '\n')
		    Putf('\n', ofp);
		break;
	    case 'm':
		skip= first= 1;
		break;
	    }
	    break;
	default:
	    Putf(c, ofp);
	}
    }   
}

static void Eval(args, ofp, buf)
char **args, *buf;
FILE *ofp;
{
    struct Tab *tb;

    tb= Lookup(args[0]);
    if (tb->name) {
	if (tb->value)
	    Replace(ofp, tb->value, args, buf);
	else if (tb->proc)
	    (tb->proc)(ofp, args, buf);
	else
	    Error("oops!");
    } else {
	Putf(left, ofp);
	Puts(args[0], ofp);
	Putf(' ', ofp);
	Puts(buf, ofp);
	Putf(right, ofp);
	Error("unknown request \"%s\"", args[0]);
    }
}

static void Request(ofp, bp)
FILE *ofp;
char *bp;
{
    char *args[10], buf1[BUFSIZE], buf[BUFSIZE];
    register int c, i;
    
    strcpy(buf1, bp);

    if (debug)
	fprintf(stderr, "%s\n", bp);

    for (bp= buf1, i= 0; *bp && i<10; i++) {
	args[i]= bp;
	if (i == 1)
	    strcpy(buf, bp);
	while ((c= *bp++) && !isspace(c))
	    ;
	bp[-1]= '\0';
	while ((c= *bp++) && isspace(c))
	    ;
	bp--;
    }
    for (; i < 10; i++)
	args[i]= "";
    if (look && strcmp(look, args[0]) == 0)
	fprintf(stderr, "%s\n", buf);
    Eval(args, ofp, buf);
}

static void ProcessFile(ifp, ofp, name)
FILE *ifp, *ofp;
char *name;
{
    register int c, i;
    char buf[BUFSIZE];
    int lastline;
	
    filename= name;
    line= 1;

    while ((c= Fgetc(ifp)) != EOF) {
	if (c == left) {
	    lastline= line;
	    c= Fgetc(ifp);
	    if (!isascii(c) || !isalpha(c)) {
		Putf(left, ofp);
		Putf(c, ofp);
		continue;
	    }
	    for (i= 0; c != right && i < BUFSIZE-1; c= Fgetc(ifp)) {
		if (c == '\\') {
		    c= Fgetc(ifp);
		    if (c != right)
			buf[i++]= '\\';
		}
		buf[i++]= c;
	    }
	    if (i >= BUFSIZE-1)
		Error("unmatched %c (last %c in line %d)", right, left, lastline);
	    buf[i]= 0;
	    Request(ofp, buf);
	} else if (c == '\\') {
	    if ((c= Fgetc(ifp)) != left)
		Putf('\\', ofp);
	    Putf(c, ofp);
	} else
	    Putf(c, ofp);
    }
}

main(argc, argv)
int argc;
char *argv[];
{
    extern int optind;
    extern char *optarg;
    
    int c;

    progname= argv[0];
    
    signal(SIGQUIT, sigHandler);
    signal(SIGINT,  sigHandler);
    signal(SIGBUS,  sigHandler);
    signal(SIGSEGV, sigHandler);
    
    while ((c= getopt(argc, argv,"di:of:xX")) != EOF) {
	switch (c) {
	case 'd':
	    debug= !debug;
	    break;
	case '?':
	    Error("wrong option: %c", c);
	    break;
	case 'f':
	    look= optarg;
	    nooutput= 1;
	    break;
	case 'o':
	    nooutput= 1;
	    break;
	case 'x':
	case 'X':
	default:
	    printf("usage: %s { -i file } { file }\n", progname);
	    exit(1);
	}
    }

    EnterProc("def", Def);
    EnterProc("include", Include);
    EnterProc("print", Print);
    
    /* process files */
    if (optind >= argc)
	ProcessFile(stdin, stdout, "stdin");
    else
	for (;optind < argc; optind++)
	    Include(stdout, &argv[optind-1], "");
}
