//$OrdCollection,OrdCollectionIter,RevOrdCollectionIter$
#include "OrdCollection.h"
#include "Error.h"
#include "FixedSizeStorage.h"

//---- ordered collection ------------------------------------------------------

const int cOrdColShrinkFactor = 2,
	  cMinExpand = 8;

static void ObjNotFound(char *where, Object *which)
{
    Error(where, "object (0x%x) not found", (int) which);
}

MetaImpl(OrdCollection, I_O(cont));

OrdCollection::OrdCollection(int s)
{
    cont= new ObjArray(s);
    size= 0;
} 

OrdCollection::~OrdCollection()
{
    SafeDelete(cont);
}

void OrdCollection::InitNew()
{
    cont= new ObjArray(cCollectionInitCap);
    size= 0;
}

void OrdCollection::FreeAll()
{
    cont->FreeAll();
    size= 0;
}

ObjPtr OrdCollection::At(int index)
{
    if (index < 0 || index >= size)
	Error ("At", "illegal index %d (size= %d)", index, size);
    if (!AnyDeleted())
	return cont->UncheckedAt(index);

    // take care of deleted objects
    Object *op;
    int i,j;

    for (i= j= 0; i < size; i++) {
	op= cont->UncheckedAt(i);
	if (!op->IsDeleted()) {
	    if (j == index) 
		return op;
	    j++;
	}
    }
    return 0;
}

ObjPtr OrdCollection::AtAfter(ObjPtr op)
{
    if (op == 0) 
	return 0;
    int i= cont->ContainsAt(op);
    if (i == -1) { 
	ObjNotFound("AtAfter", op);
	return 0;
    }
    if (i == size-1)
	return 0;    
    return cont->At(i+1);
}

ObjPtr OrdCollection::AtBefore(ObjPtr op)
{
    if (op == 0) 
	return 0;
    int i= cont->ContainsAt (op);
    if (i == -1) { 
	ObjNotFound("AtBefore", op);
	return 0;
    }
    if (i == 0)
	return 0;    
    return cont->At(i-1);
}

ObjPtr OrdCollection::First()
{
    return Size() ? At(0): 0;
}

ObjPtr OrdCollection::Last()
{
    int i;
    Object *op;

    for (i= Size()-1; i >= 0; i--) {
	op= cont->At(i);
	if (!op->IsDeleted())
	    return op;
    }
    return 0;
}

int OrdCollection::Capacity()
{
    return cont->Size();
}

ObjPtr OrdCollection::Add(ObjPtr op)
{
    if (op == 0)
	return 0;
    CheckActiveIter("Add");
    if (size == cont->Size())
	cont->Expand (GrowBy(max(cont->Size(), cMinExpand)));
    return cont->AtPut(size++,op);
}

void OrdCollection::InsertAfter(ObjPtr after, ObjPtr op)
{
    if (op == 0 || cont->Size() <= 0)
	return;
    CheckActiveIter("InsertAfter");
    int i= cont->ContainsAt(after);
    if (i == -1) 
	ObjNotFound("InsertAfter", op);
    AddAt(i+1, op);
}

void OrdCollection::InsertBefore(ObjPtr before, ObjPtr op)
{
    if (op == 0 || cont->Size() <= 0)
	return;
    CheckActiveIter("InsertBefore");
    int i= cont->ContainsAt(before);
    if (i == -1)
	ObjNotFound("InsertBefore", op);
    AddAt(i, op);
}

ObjPtr OrdCollection::Remove(ObjPtr e)
{
    register ObjPtr t = 0;

    if (e == 0) 
	return 0;
    for (int i = 0; i < size; i++)
	if (e->IsEqual(cont->At(i)))
	    t= DoRemoveAt(i);
    return t;
}

ObjPtr OrdCollection::RemovePtr(ObjPtr e)
{
    register ObjPtr t= 0;

    if (e == 0) 
	return 0;
    for (int i = 0; i < size; i++)
	if (e == cont->At(i))
	    t= DoRemoveAt(i);
    return t;
}

void OrdCollection::AddAt(int at, ObjPtr op)
{
    register int j;

    if (op == NULL)
	return;
    CheckActiveIter("AddAt");
    if (at > size)
	Error("AddAt", "out of range");
    if (size == cont->Size())
	cont->Expand (GrowBy(max(cont->Size(), cMinExpand)));
    for (j= size-1; j >= at; j--)
	(*cont)[j+1]= (*cont)[j];

    (*cont)[at]= op;
    size++;
}

ObjPtr OrdCollection::RemoveAt(int i)
{
    if (!AnyDeleted())
	return DoRemoveAt(i);
    return RemovePtr(At(i));
}

ObjPtr OrdCollection::DoRemoveAt(int i)
{
    ObjPtr t;
    register int j;

    t= cont->RemoveAt(i);
    if (InIterator()) {
	cont->AtPut (i, new DeletedObject);
	AnnounceRemove();
	return t;
    }
    for (j= i; j < size -1; j++)
	cont->AtPut(j, cont->At(j+1));
    size--;
    if (LowWaterMark())
	cont->Expand (cont->Size() / cOrdColShrinkFactor);
    return t;
}

void OrdCollection::RemoveDeleted()
{
    Object *op;
    for (int j= 0; j < size;) {
	op= cont->At(j);
	if (op && op->IsDeleted()) {
	    DoRemoveAt(j);
	    SafeDelete(op);
	}
	else 
	    j++;
    }
}

void OrdCollection::Empty (int s)                                         
{
    CheckActiveIter("Empty");
    SafeDelete(cont);
    cont= new ObjArray(max(s,4)); 
    size= 0; 
}

Iterator *OrdCollection::GetIterator() 
{ 
    return new OrdCollectionIter(this); 
}

Iterator *OrdCollection::ReversedIterator () 
{ 
    return new RevOrdCollectionIter(this); 
}

void OrdCollection::Sort()
{
    CheckActiveIter("Sort");
    cont->Sort(size);
}

int OrdCollection::BinarySearch(ObjPtr op)
{
    return cont->BinarySearch(op, size);
}

//---- ordered collection iterator ---------------------------------------------

FIXED_STORAGE(OrdCollectionIter, 20);

OrdCollectionIter::OrdCollectionIter(Collection *s)
{ 
    FIXED_SIZE_ALLOC(OrdCollectionIter);
    cs= (OrdCollection *)s;
    ce= 0; 
}

OrdCollectionIter::~OrdCollectionIter()
{
    IteratorEnd();
    FIXED_SIZE_DEALLOC(OrdCollectionIter);    
}

void OrdCollectionIter::Reset(Collection *s)
{
    if (s == 0)
	s= Coll();
    cs= (OrdCollection *)s;
    ce= 0;
    Iterator::Reset(s);
}

Collection *OrdCollectionIter::Coll()
{
    return cs;
}

ObjPtr OrdCollectionIter::operator()()
{
    IteratorStart();
    // find next non deleted object
    for (;ce < cs->Size() && cs->cont->UncheckedAt(ce)->IsDeleted(); ce++);
    if (ce < cs->size) 
	return cs->cont->UncheckedAt(ce++);
    IteratorEnd();
    return 0;
}

//---- RevOrdCollectionIter ----------------------------------------------------

RevOrdCollectionIter::RevOrdCollectionIter(Collection *s):(s)
{
    ce = cs->size-1;
}

void RevOrdCollectionIter::Reset(Collection *s)
{
    OrdCollectionIter::Reset(s);
    ce= cs->size-1;
}

ObjPtr RevOrdCollectionIter::operator()()
{
    IteratorStart();
    // find next non deleted object
    for (; ce >= 0 && cs->cont->At(ce)->IsDeleted(); ce--);
    if (ce >= 0) 
	return cs->cont->UncheckedAt(ce--);
    IteratorEnd();
    return 0;
}
