//$BrowserDoc,ClassListView,ClassTreeView,DefaultItem,FihView, FihClassTable$
 
#include "Browser.h"
#include "CodeTextView.h"
#include "StyledText.h"
#include "Window.h"
#include "Application.h"
#include "Alert.h"
#include "Menu.h"
#include "DialogItems.h"
#include "Document.h"
#include "OrdCollection.h"
#include "ObjList.h"
#include "Scroller.h"
#include "TreeView.h"
#include "WindowSystem.h"
#include "PathLookup.h"
#include "GotoDialog.h"
#include "ClassItem.h"
#include "Expander.h"
#include "ProgEnv.h"
#include "CmdNo.h"

static PathLookup SrcPath(Getenv("ET_SRC_PATH"));

//---- BrowserDoc -------------------------------------------------------------

MetaImpl(BrowserDoc, (I_O(textview), I_O(listview), I_O(text), I_O(ccl), 
		       I_O(log), I_B(isCCode)));

BrowserDoc::BrowserDoc() : (cDocTypeAscii)
{
    char *p;

    Font *fd= gFixedFont;
    if (Getenv("ET_NO_STYLEDCODE") == 0)
	text= new StyledText(256, fd);
    else
	text= new GapText(256, fd);
    text->SetTabWidth(fd->Width(' ')*8);
    ccl= 0;
    definition= FALSE;
    log= new OrdCollection;
    isCCode= FALSE;
    // add ET_DIR/src to the src path
    p= Getenv("ET_DIR");
    if (!p)
	p= "/local/ET++";
    SrcPath.Add(form("%s/src", p));
}

BrowserDoc::~BrowserDoc()
{
    gProgEnv->Closed(this);
    SafeDelete(textview);
    SafeDelete(listview);
    SafeDelete(text);
    SafeDelete(log);
}

Window *BrowserDoc::DoMakeWindows()
{   
    textview= new CodeTextView(this, Rectangle(Point(2000,cFit)), text);
    listview= new ClassListView(this); 

    Scroller *sp= new Scroller(listview, Point(-1, 100));
    sp->SetFlag(eVObjHFixed);
    return new Window(this, Point(700, 400), eWinDefault,
	new Expander(eHor, gPoint2,
	    sp,
	    new Scroller(textview),
	0
	)
    );
}

bool BrowserDoc::Open()
{
    int rcode= Document::Open();
    if (rcode) {
	ccl= 0;
	Changed();
    }
    return rcode;
}

bool BrowserDoc::CanImportDocument(FileType *ft)
{
    if (ft == 0)
	return TRUE;
    return strismember(ft->Type(), cDocTypeAscii, 0);
}

Command *BrowserDoc::DoImport(istream& s, FileType *ft)
{
    GapText *t = new GapText(1024);
    Command *cmd;
    
    t->ReadFromAsPureText(s, ft->SizeHint());
    cmd= textview->InsertText(t);
    SafeDelete(t);
    return cmd;
}

void BrowserDoc::DoWrite(ostream &s, int)
{
    text->PrintOnAsPureText(s);
}

void BrowserDoc::DoRead(istream &s, FileType *ft)
{
    text->ReadFromAsPureText(s, ft->SizeHint());
    if (isCCode= ft->IsCCode())
	textview->FormatCode();
    else
	textview->SetDefaultStyle();
    textview->SetText(text);
}

void BrowserDoc::Log(Class *cl)
{
    if (cl != ccl)
	log->Add(cl);
}

void BrowserDoc::SetClass(Class *cl, bool what)
{
    Log(cl);
    DoSetClass(cl, what, TRUE);
}

void BrowserDoc::DoSetClass(Class *cl, bool decl, bool unique)
{
    char fname[1000];
    const char *path, *base, *bp;
    bool sameFile= (cl == ccl && decl == definition);

    if (!sameFile) {
	
	fname[0] = '\0';
	if (decl) {
	    base= cl->GetDeclFileName();
	    path= cl->GetDeclDir();
	} 
	else {
	    base= cl->GetImplFileName();
	    path= cl->GetImplDir();
	} 
	if (base[0] != '/') {
	    strcpy(fname,path);
	    strcat(fname,"/");
	}
	if (base[0] == '.' && base[1] == '/') {
	    bp= &base[2];
	    strcat (fname,bp);
	}
	else {
	    bp= base;
	    strcat (fname,base);
	}
	char *p= GetName();
	if (p == 0 || strcmp(p, fname) != 0) {   
	    if (Modified())
		switch (CautionAlert.Show("Save changes to \"%s\" ?",GetName())){
		case cIdYes:
		    Save();
		    break;
		case cIdCancel:
		    Changed();
		    return;
		}
	    if(gSystem->AccessPathName(fname, 4) != 0) {
		if (!SrcPath.Lookup(bp,fname)) {
		    NoteAlert.Show("Can't open document \"%s\" for reading\n%s",
						    base,gSystem->GetErrorStr());
		    return;
		}
	    }
	    FType ft(fname);
	    PerformCommand(gResetUndo);
	    Load(fname, unique, ft.FileType());
	    if (p == 0) 
		sameFile= TRUE; // force redraw of selection
	}
	else
	    sameFile= TRUE;
	definition= decl;
	ccl= cl;
	Changed();
    }
    SelectSourceLine(sameFile);
}

void BrowserDoc::SelectSourceLine(bool)
{
    int line;
    if (definition)
	line= ccl->GetDeclFileLine();
    else 
	line= ccl->GetImplFileLine();

    LineMark *lm= textview->MarkAtLine(line-1);
    textview->SetSelection(lm->Pos(),lm->End(),TRUE/*redraw*/);
    textview->RevealSelection();
}

void BrowserDoc::SetMode(bool what)
{
    SetClass(ccl, what);
}
    
void BrowserDoc::Spawn()
{    
    if (Modified() &&
	CautionAlert.Show("Save changes to \"%s\" ?", GetName()) == cIdYes)
	Save();
    BrowserDoc *browserDoc = new BrowserDoc;
    gApplication->AddDocument(browserDoc);
    browserDoc->OpenWindows();
    browserDoc->DoSetClass(ccl, definition, FALSE);
}

void BrowserDoc::EditSelectedClass()
{
    Class *clp;
    char *classname;

    classname= FirstWordOfSelection();
    clp= ClassFind(classname);
    if (clp)
	SetClass(clp,definition);
    else
	NoteAlert.Show("No Metaclass found for \"%s\" ?", classname);
}
 
void BrowserDoc::DoCreateMenu(Menu *menu)
{
    Document::DoCreateMenu(menu);
    
    Menu *browse= new Menu("browse");

    browse->AppendItems(
		      "other",                  cEDITOTHER,
		      "superclass",             cEDITSUPER,
		      "edit selected class",    cEDITSELECTED,
		      "spawn",                  cSPAWN,
		      "-",
		      "show in hierarchy",      cSHOWHIERARCHY,
		      "show inheritance path",  cSHOWINHPATH,
		      "go back",                cGOBACK,
		      "-",
		      "go to line...",          cGOTOLINE,
		      0);

    menu->AppendMenu(browse, cBROWSEMENU);
    menu->InsertItemAfter(cLASTEDIT, "reformat", cREFORMAT);
}

void BrowserDoc::DoSetupMenu(Menu *menu)
{
    char *current;
    if (definition)
	current= "implementation";
    else
	current= "definition";
	
    menu->ReplaceItem(cEDITOTHER, current);
    if (ccl)
	menu->EnableItems(cEDITOTHER, 
			  cSPAWN, 
			  cSHOWHIERARCHY,
			  cSHOWINHPATH, 
			  cGOTOLINE, 
			  0);
    if (isCCode && text->IsKindOf(StyledText))
	menu->EnableItem(cREFORMAT);
    menu->EnableItem(cGOTOLINE);
    if (ccl && ccl->Super())
	menu->EnableItem(cEDITSUPER);
    if (log->Size() > 1)
	menu->EnableItem(cGOBACK);
	
    menu->ReplaceItem(cEDITSELECTED, "selected class");
    if (!textview->Caret()) {
	char *q= FirstWordOfSelection();
	if (strlen(q) && ClassFind(q)) {
	    menu->ReplaceItem(cEDITSELECTED, form("class \"%s\"", q));
	    menu->EnableItem(cEDITSELECTED);
	}
    }
    menu->EnableItem(cBROWSEMENU);
    Document::DoSetupMenu(menu);
}

char *BrowserDoc::FirstWordOfSelection()
{
    static char buf[50];
    textview->SelectionAsString(buf, sizeof buf);
    for (char *q= buf; *q; q++)
	if (Isinword(*q)) break;
    for (char *p= q;*p; p++)
	if (!Isinword(*p)) break;
    *p= '\0';
    return q;
}

Command *BrowserDoc::DoMenuCommand(int cmd)
{
    switch (cmd) {

    case cEDITOTHER:
	SetMode(!definition);
	break;
    
    case cEDITSUPER:
	SetClass(ccl->Super(), definition);
	break;
	
    case cEDITSELECTED:
	EditSelectedClass();
	break;
	
    case cGOBACK:
	log->RemoveAt(log->Size()-1);
	DoSetClass((Class*)log->Last(), definition, TRUE);
	break;

    case cSPAWN:
	Spawn();
	break;
	
    case cSHOWHIERARCHY:
	gProgEnv->ShowInHierarchy(ccl);
	break;
	
    case cSHOWINHPATH:
	gProgEnv->ShowInheritancePath(ccl);
	break;

    case cGOTOLINE:
	GotoLine(textview);
	
    case cREFORMAT:
	textview->FormatCode();
	textview->ForceRedraw();
	return gNoChanges;
    default:
	break;
    }
    return Document::DoMenuCommand(cmd);
}

//---- ClassListView ---------------------------------------------------------------

MetaImpl(ClassListView, (I_O(ClassItems), I_O(browserDoc)));

ClassListView::ClassListView(BrowserDoc *sd) : (0, 0, eCVDontStuckToBorder)
{
    ClassItems= new OrdCollection(200); 
    LoadClasses();   
    browserDoc= sd;
    browserDoc->AddDependent(this);
    classTable->AddDependent(this);
}

ClassListView::~ClassListView()
{
    if (browserDoc)
	browserDoc->RemoveDependent(this);
    if (classTable)
	classTable->RemoveDependent(this);
}

void ClassListView::LoadClasses()
{
    Iter next(ClassIterator());
    int i= 0;
    Class *clp;
    
    while (clp= (Class*) next())
	ClassItems->Add(new ClassItem(i++,clp));
    SetCollection(ClassItems);
    Update();    
}

void ClassListView::DoSelect(Rectangle r)
{
    if (r.IsEmpty())
	return;
    ClassItem *ci= (ClassItem*) ClassItems->At(r.origin.y);
    if (ci == 0)
	return;     
    browserDoc->SetClass(ci->GetClass(), browserDoc->GetMode());    
}

void ClassListView::DoUpdate(Object* op,void*)
{
    if (op == browserDoc)
	SelectCurrentClass();
    else if (op == classTable) {
	ClassItems->Empty();
	LoadClasses();
	SelectCurrentClass();
    }   
}

void ClassListView::SelectCurrentClass()
{
    Rectangle r, sr;
    Class *ccl;
    
    ClassItem *ci= (ClassItem*) ClassItems->At(GetSelection().origin.y);
    ccl= browserDoc->GetClass();
    if (ci && ci->GetClass() == ccl)
	return;
    if (ccl == 0) {
	SetNoSelection();
	return;
    }
    ClassItem *clp= 0, searchItem(0,browserDoc->GetClass());

    int pos= ClassItems->BinarySearch(&searchItem);
    if (pos != -1)
	clp= (ClassItem*) ClassItems->At(pos);
    if (clp) {
	sr= Rectangle(0, clp->GetId(), 1, 1);
	SetSelection(sr);
	RevealAlign(ItemRect(sr));
    }  
}

//---- default Item for FihView ----------------------------------------------

class DefaultItem: public VObject {
public:
    MetaDef(DefaultItem);
    DefaultItem()
	{ Disable(FALSE); }
    void Draw(Rectangle )
	{ GrPaintRect(contentRect, ePatGrey25); }
};

MetaImpl0(DefaultItem);

//---- FihClassTable ----------------------------------------------------------

MetaImpl0(FihClassTable);

FihClassTable::FihClassTable()
{
    // find out the depth of the inheritance path
    Class *clp;
    int level; 
    nClasses= maxlevel= 0;
    
    Iter next(ClassIterator());
    while (clp= (Class*)next()) {
	nClasses++; 
	level= 1;      
	while (clp->Super() != 0) {
	    level++;
	    clp= clp->Super();
	}
	maxlevel= max(level, maxlevel);
    }
    // fill in table with inheritance path
    next.Reset(0);
    
    int i,j;
    table= new ObjArray((maxlevel+1)*(nClasses));
    for (i= 0; clp= (Class*)next(); i++) {
	AtPut(0, i, MakeCell(clp, TRUE));
	j= 1;
	while (clp->Super() != 0) {
	    clp= clp->Super();
	    AtPut(j++, i, MakeCell(clp));
	}
	while (j < maxlevel)
	    AtPut(j++, i, new DefaultItem());
    }
}

FihClassTable::~FihClassTable()
{   
    SafeDelete(table);
}

VObject *FihClassTable::MakeCell(Class *clp, bool bold= FALSE) 
{ 
    Font *fp= new Font(gFixedFont->Fid(), gFixedFont->Size(),
		  (clp->IsAbstract() ? eFaceItalic :eFacePlain) | 
						      (bold ? eFaceBold : 0));
    return new TextItem(clp->Name(), fp);
} 

//---- FihView ---------------------------------------------------------------

MetaImpl(FihView, (I_O(ft)));

void FihView::DoSelect(Rectangle r)
{
    VObject *gop= (VObject*) ft->At(r.origin.x, r.origin.y);
    Class *clp= ClassFind(gop->AsString());
    if (clp)
	gProgEnv->EditSourceOf(clp, TRUE);
    SetNoSelection();
}

void FihView::ShowClass(Class *cl)
{   
    char *name= cl->Name();
    for (int i= 0; i < ft->Rows(); i++) {
	VObject *vop= (VObject*)ft->At(0,i);
	if (vop && (strcmp(vop->AsString(), name) == 0)) {
	    RevealRect(vop->ContentRect().Expand(100), Point(0,100));
	    SetSelection(Rectangle(0, i, ft->Cols(), 1));      
	    GraphicDelay(500);
	    SetNoSelection();
	}
    }
}

//---- ClassTreeView ----------------------------------------------------------

MetaImpl0(ClassTreeView);

ClassTreeView::ClassTreeView() : ((Document*)0)
{
    ClassSetupSubclasses();
    Class *clp= ClassFind("Object");
    SetTree((TreeNode*)BuildClassTree(clp));
}

VObject *ClassTreeView::BuildClassTree(Class *cl)
{
    VObject *g= new ClassItem(0, cl);
    ObjList *list= new ObjList;
    list->Add(g);
    Iterator *it= cl->SubclassIterator();
    if (it) {
	Iter next(it);
	while (cl= (Class*)next()) {
	    VObject *newitem= BuildClassTree(cl);
	    if (!newitem->IsKindOf(TreeNode))
		newitem= new TreeNode(5, newitem, 0);
	    list->Add(newitem);
	}
    }
    g= new TreeNode(4, list);
    return g;
}

Command *ClassTreeView::NodeSelected(VObject *t, int)
{
    gProgEnv->EditSource(((ClassItem*)t)->GetClass());
    return gNoChanges;
}

void ClassTreeView::ShowClass(Class *cl)
{
    ClassItem dummy(99, cl);
    VObject *f= FindNode(&dummy);
    if (f) {
	RevealRect(f->ContentRect().Expand(100), Point(0,100));
	for (int i= 0; i < 2; i++) {
	    /*
	    f->Highlight(On);    
	    Wait(250);
	    f->Highlight(Off);    
	    Wait(150);
	    */
	}
    }
}
