//$Rectangle$
#include <math.h>
#include "Rectangle.h"
#include "Class.h"
#include "Error.h"

const Rectangle gRect0;

static Class RectangleClassMetaImpl0("Rectangle",
				  sizeof (Rectangle),
				  0,
				  __FILE__,
				  "./Rectangle.h",
				  __LINE__,
				  1,
				  __COMPILEDIR__,
				  __COMPILEDIR__);

Rectangle::Rectangle()
{
    origin.x= origin.y= extent.x= extent.y= 0;
}

Rectangle::Rectangle(short x, short y, short w, short h)
{
    origin.x= x;
    origin.y= y;
    extent.x= w;
    extent.y= h;
}

Rectangle::Rectangle(Point o, Point e)
{
    origin= o;
    extent= e;
}
    
Rectangle::Rectangle(Point e)
{
    origin= 0;
    extent= e;
}

Rectangle::Rectangle(short w, short h)
{   
    origin= 0;
    extent.x= w;
    extent.y= h;
}

Rectangle NormRect(Point p1, Point p2)
{
    Rectangle r;
    
    r.origin.x= min(p1.x, p2.x);
    r.origin.y= min(p1.y, p2.y);
    r.extent.x= max(p1.x, p2.x)+1-r.origin.x;
    r.extent.y= max(p1.y, p2.y)+1-r.origin.y;
    return r;
}

bool Rectangle::Intersects(Rectangle &r)
{
    return origin.x < r.origin.x + r.extent.x
		    && r.origin.x < origin.x + extent.x
		    && origin.y < r.origin.y + r.extent.y
		    && r.origin.y < origin.y + extent.y;
}

bool Rectangle::ContainsRect(Rectangle &r)
{
    Point c1= origin + extent, c2= r.origin + r.extent;
    
    return    origin.x <= r.origin.x && c1.x >= c2.x
	   && origin.y <= r.origin.y && c1.y >= c2.y;
}

Rectangle Rectangle::Merge(Rectangle &r)
{ 
    register int x1, x2, y1, y2;
    
    if (IsEmpty())
	return *this= r;
    if (r.IsEmpty())
	return *this;
    x1= origin.x + extent.x; 
    y1= origin.y + extent.y; 
    x2= r.origin.x + r.extent.x; 
    y2= r.origin.y + r.extent.y; 
    origin.x= min(origin.x, r.origin.x);
    origin.y= min(origin.y, r.origin.y);
    extent.x= max(x1, x2) - origin.x;
    extent.y= max(y1, y2) - origin.y;
    return *this;
}

Rectangle Union(Rectangle &r1, Rectangle &r2)
{
    Rectangle rr= r1;
    return rr.Merge(r2);
}

bool Rectangle::Clip(Rectangle &r)
{
    register int p1x, p1y, p2x, p2y;
    
    if (IsEmpty())
	return FALSE;
    if (r.IsEmpty()) {
	*this= gRect0;
	return FALSE;
    }
    p1x= origin.x + extent.x;
    p1y= origin.y + extent.y;
    p2x= r.origin.x + r.extent.x;
    p2y= r.origin.y + r.extent.y;
    origin.x= max(origin.x, r.origin.x);
    origin.y= max(origin.y, r.origin.y);
    extent.x= min(p1x, p2x) - origin.x;
    extent.y= min(p1y, p2y) - origin.y;
    return IsNotEmpty();
}

Rectangle Inter(Rectangle &r1, Rectangle &r2)
{
    Rectangle rr= r1;
    rr.Clip(r2);
    return rr;
}

Rectangle Rectangle::Intersect(Rectangle &r)
{
    Clip(r);
    return *this;
}

Rectangle BoundingBox(int npts, Point *pts)
{
    Point min= pts[0], max= pts[0];
    Rectangle r;
    register int i;
    
    for (i= 1; i<npts; i++) {
	min= Min(min, pts[i]);
	max= Max(max, pts[i]);
    }
    r.origin= min;
    r.extent= max-min+1;
    for (i= 0; i<npts; i++)
	pts[i]= pts[i]-min;
	
    return r;
}

int Difference(Rectangle *rp, Rectangle &r1, Rectangle &ra)
{
    register int p2x, p2y;
    Rectangle r2, dr[4];
    int n, i;

    r2= Inter(r1, ra);
    if (r2.IsNotEmpty()) {
	p2x= r2.origin.x+r2.extent.x;
	p2y= r2.origin.y+r2.extent.y;
	
	dr[0].origin= r1.origin;
	dr[0].extent.x= r1.extent.x;
	dr[0].extent.y= r2.origin.y-r1.origin.y;
	
	dr[1].origin.x= r1.origin.x;
	dr[1].origin.y= r2.origin.y;
	dr[1].extent.x= r2.origin.x-r1.origin.x;
	dr[1].extent.y= r2.extent.y;
	
	dr[2].origin.x= p2x;
	dr[2].origin.y= r2.origin.y;
	dr[2].extent.x= r1.origin.x+r1.extent.x-p2x;
	dr[2].extent.y= r2.extent.y;
	
	dr[3].origin.x= r1.origin.x;
	dr[3].origin.y= p2y;
	dr[3].extent.x= r1.extent.x;
	dr[3].extent.y= r1.origin.y+r1.extent.y-p2y;
    } else 
	dr[0]= r1;
    
    for (n= i= 0; i<4; i++)
	if (dr[i].IsNotEmpty())
	    rp[n++]= dr[i];
    return n;
}

Point ConstrainMoveRect(Rectangle &r1, Rectangle &r2, Point delta)
{
    Point p= r1.origin + r1.extent - r2.extent;
    return Max(r1.origin, Min(p, r2.origin+delta))-r2.origin;
}

Point Rectangle::AmountToTranslateWithin(Rectangle &r)
{
    Point delta(0), rUL= NW(), rLR= SE(), vrUL= r.NW(), vrLR= r.SE();
    
    for (int v= 0; v <= 1; v++) {
	if (rUL[v] < vrUL[v] && rLR[v] > vrLR[v])
	    continue;
	if (rLR[v] - vrUL[v] < extent[v])
	    delta[v]= vrUL[v] - rUL[v];
	if (vrLR[v] - rUL[v] < extent[v])
	    delta[v]= vrLR[v] - rLR[v];
    }
    return delta;
}

ostream& operator<<(ostream& s, Rectangle &r)
{
    return s << "<" << r.origin << "," << r.extent << ">";
}                  

istream& operator>> (istream& s, Rectangle &r)
{
    Point o, e;
    char c= 0;
    
    s >> c;
    if (c == '<') {
	s >> o >> c;
	if (c == ',') {
	    s >> e >> c;
	    if (c != '>')
		s.clear(_bad);
	} else
	    s.clear(_bad);
    } else
	s.clear(_bad);
    if (s)
	r= Rectangle(o,e);
    return s;
}

int Rectangle::PointToAngle(Point p)
{
    float x= (float)(p.x-origin.x) - (float)extent.x/2.0,
	  y= (float)(p.y-origin.y) - (float)extent.y/2.0;
    float d= (float)extent.x / (float)extent.y;
    int a= (int)((atan2(x, -y*d) * 180.0 / PI) + 0.5);
    if (a < 0)
	return a + 360;
    else
	return a;
}

#define TAN(x) (sin((float)(x))/cos((float)(x)))
#define COT(x) (cos((float)(x))/sin((float)(x)))
#define PI180 0.017453293

Point Rectangle::AngleToPoint(int a)
{
    int b;
    Point p;
    float d;
    
    if (a < 0)
	b= (abs(a % 360) + 90) % 360;
    else
	b= (450-(a % 360)) % 360;
    d= PI180 * (float) b;
    
    if (a > 45 && a <= 135) {
	p.x= extent.x-1;
	p.y= (int)((1.0-TAN(d))/2.0 * float(extent.y-1)+0.5);
    } else if (a > 135 && a <= 225) {
	p.x= (int)((1.0-COT(d))/2.0*float(extent.x-1)+0.5);
	p.y= extent.y-1;
    } else if (a > 225 && a <= 315) {
	p.x= 0;
	p.y= (int)((1.0+TAN(d))/2.0*float(extent.y-1)+0.5);
    } else {
	p.x= (int)((1.0+COT(d))/2.0*float(extent.x-1)+0.5);
	p.y= 0;
    }
    return origin + p;
}

// Corners:
//         7|0|1
//        -------
//         6| |2
//        -------
//         5|4|3

int Rectangle::PointToCorner(Point p)
{
    return ((PointToAngle(p)+22)/45) % 8;
}

Point Rectangle::CornerToPoint(int n)
{
    return AngleToPoint((n % 8)*45);
}

char *Rectangle::AsString()
{
    return form("x: %d y: %d w: %d h: %d", origin.x, origin.y, extent.x, extent.y);
}
