#include <stdio.h>
#include <a.out.h>
#include <sys/file.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <ctype.h>

typedef void* ((*func_ptr)());

typedef struct {
    struct nlist **array; 
    int size;           /* Number of slots in table.. power of 2 */
    int mask;           /* always = size-1, for calculating (num mod size)*/
    int entries;        /* number of entries in AssocTable */
} AssocTable;

typedef struct Module {
    char *name, *name1;
    int fd;
    struct Module *next;
    struct nlist *symtab, *endp;
    long text;
    func_ptr dtor;
    int undefs;
    struct exec header;
};

#define bad_format 2
#define ok 0

/* The hash value is a function of the table size. */
#define hash(h,ss,m) { register char *_s=ss; for(h=0; *_s;) h+=((h<<1)+*_s++); h&=m; }

#define REHASH(hash, table) (((hash+1)*3) & (table)->mask)

#define INIT_TABLE_SIZE 8192            /* Must be a power of two */

extern char *getenv();
extern char *strsave();

static AssocTable *mainhashtable= 0;
static struct Module mainmodule;
static struct Module *firstModule= 0;
static long new_common= 0;

static struct Module *FindModule(name)
char *name;
{
    register struct Module *mp;
    
    for (mp= firstModule; mp; mp= mp->next)
	if (strcmp(name, mp->name) == 0)
	    return mp;
    return 0;
}

static struct Module *AddModule(name, name1)
char *name, *name1;
{
    struct Module *mp;
    
    mp= (struct Module *) Malloc(sizeof(struct Module));
    mp->next= firstModule;
    mp->undefs= 0;
    mp->fd= -1;
    mp->name= strsave(name, -1);
    mp->name1= strsave(name1, -1);
    mp->dtor= 0;
    
    return firstModule= mp;
}

static AssocTable *NewAssocTable()
{
    register AssocTable* table;

    table= (AssocTable*) Malloc(sizeof(AssocTable));

    table->size= INIT_TABLE_SIZE;
    table->mask= table->size-1;
    table->entries= 0;
    table->array= (struct nlist**) Malloc(table->size * sizeof(struct nlist*));

    return table;
}

static void ExpandTable(table)
register AssocTable* table;
{
    register struct nlist **old_table, **sym, **endp, *sp;
    register int rehash;
    int old_size;
 
    old_table= table->array;
    old_size= table->size;
    table->size*= 2;
    table->mask= table->size-1;
    table->array= (struct nlist**) Malloc(table->size*sizeof(struct nlist*));

    /* Move the members from the old small table to the new big one. */
    sym= old_table;
    endp= sym + old_size;
    for (; sym < endp; sym++) {
	if (sp= *sym) {
	    hash(rehash, sp->n_un.n_name, table->mask);
	    while (table->array[rehash])
		rehash= REHASH(rehash, table);
	    table->array[rehash]= sp;
	}
    }
    Free(old_table);
}

static void Assoc(table, sym)
register AssocTable* table;
struct nlist *sym;
{
    register int rehash;
    struct nlist *s;

    if (table->entries >= (table->size/2) - 1)
	ExpandTable(table);

    hash(rehash, sym->n_un.n_name, table->mask);
    while (s= table->array[rehash]) {
	if (strcmp(s->n_un.n_name, sym->n_un.n_name) == 0)
	    return;
	rehash= REHASH(rehash, table);
    }
    table->array[rehash]= sym;
    table->entries++;
}

static struct nlist *Lookup(table, name)
register AssocTable* table;
char *name;
{
    register struct nlist *syms;
    register int rehash;

    hash(rehash, name, table->mask);
    while (syms= table->array[rehash]) {
	if (strcmp(syms->n_un.n_name, name) == 0)
	    return syms;
	rehash= REHASH(rehash, table);
    }
    return 0;
}

static char *which(search, file)
char *search, *file;
{
    static char name[MAXPATHLEN];
    register char *ptr, *next;
    struct stat buffer;

    if (file[0] == '/') {
	if (stat(file, &buffer) != -1)
	    return file;
	return 0;
    }

    if (search == 0)
	search= ".";

    for (ptr= search; *ptr;) {
	for (next= name; *ptr && *ptr != ':'; )
	    *next++= *ptr++;
	*next= '\0';
	   
	strcat(name, "/");
	strcat(name, file);
	   
	if (stat(name, &buffer) != -1)
	    return name;
	if (*ptr)
	    ptr++;
    }
    return 0;
}

static void GetHeader(mp, disp)
struct Module *mp;
long disp;
{
    lseek(mp->fd, disp, 0);
    if (read(mp->fd, &mp->header, sizeof(struct exec)) != sizeof(struct exec)) {
	perror("GetHeader: read error");
	_exit(1);
    }
}

static void GetSymbols(mp, disp)
struct Module *mp;
long disp;
{
    register struct nlist *buffer, *sym;
    unsigned long size;
    long displ;

    lseek(mp->fd, disp + N_SYMOFF(mp->header) + mp->header.a_syms, 0);
    if (read(mp->fd, &size, 4) != 4) {
	perror("GetSymbols: read error");
	return;
    }

    buffer= (struct nlist*) Malloc(mp->header.a_syms + size);
  
    lseek(mp->fd, disp + N_SYMOFF(mp->header), 0);
    if (read(mp->fd, buffer, mp->header.a_syms + size) != mp->header.a_syms + size) {
	Free(buffer);
	perror("GetSymbols: read error");
	return;
    }
  
    mp->symtab= buffer;
    mp->endp= buffer + (mp->header.a_syms/sizeof(struct nlist));
    
    displ= (long)buffer + (long)(mp->header.a_syms);

    for (sym= buffer; sym < mp->endp; sym++)
	sym->n_un.n_name= (char*) sym->n_un.n_strx + displ;
}

static struct relocation_info *GetRelocInfo(mp, disp)
struct Module *mp;
long disp;
{
    struct relocation_info *buffer;
    int size;
  
    lseek(mp->fd, disp + N_TXTOFF(mp->header) + mp->header.a_text + mp->header.a_data, 0);
  
    size= mp->header.a_trsize + mp->header.a_drsize;

    if (size == 0) {
	Error("GetRelocInfo", "size == 0");
	return 0;
    }
    
    buffer= (struct relocation_info*) Malloc(size);

    if (read(mp->fd, buffer, size) != size) {
	SysError("GetRelocInfo", "read");
	Free(buffer);
	return 0;
    }
  
    return buffer;
}

static void GetTextAndData(mp, disp)
struct Module *mp;
long disp;
{
    int size, rsize;
    
    lseek(mp->fd, disp + N_TXTOFF(mp->header), 0);
  
    rsize= mp->header.a_text + mp->header.a_data;
    size= rsize + mp->header.a_bss;

    mp->text= (long) Malloc(size);
  
    if (read(mp->fd, mp->text, rsize) !=  rsize) {
	perror("GetTextAndData: read error");
	Free(mp->text);
	mp->text= 0;
	return;
    }
    
    bzero(mp->text + mp->header.a_text + mp->header.a_data, mp->header.a_bss);
}

/*----------------------------------------------------------------------------*/

static void FindAll(mp, hashtable)
struct Module *mp;
AssocTable* hashtable;
{
    register struct nlist *symp= mp->symtab, *symbol;
    int value;

    mp->undefs= 0;
    for (; symp < mp->endp; symp++) {
	if (symp->n_type & N_EXT) {
	    symbol= Lookup(hashtable, symp->n_un.n_name);

	    if ((symp->n_type == N_EXT + N_UNDF) && (symbol == 0)) {
		value= symp->n_value;
		if (value > 0) {    /* is common */
		    int rnd= value >= sizeof(double) ? sizeof(double) - 1
			       : value >= sizeof(long) ? sizeof(long) - 1
			       : sizeof(short) - 1;

		    symp->n_type= N_COMM;
		    new_common+= rnd;
		    new_common&= ~(long)rnd;
		    symp->n_value= new_common;
		    new_common+= value;
		    Assoc(hashtable, symp);
		} else {
		    mp->undefs++;
		    try(symp->n_un.n_name);
		}
	    } else if (symbol == 0) /* is new */
		Assoc(hashtable, symp);
	}
    }
}

int Testload(name)
char* name;
{
    char buf[100], *name2;
    struct nlist *symbol;
    int code;
    struct Module *mp;
    struct stat buffer;
    
    sprintf(buf, "%s.o", name);
    
    name2= which(getenv("ET_DYN_PATH"), buf);
    if (name2 == 0)
	return;

    if (FindModule(name2))
	return;
	
    mp= AddModule(name2, name);
    
    if ((mp->fd= open(name2, O_RDONLY)) < 0)
	return;
	
    GetHeader(mp, 0L);
    GetSymbols(mp, 0L);

    close(mp->fd);
}

int try(s)
char *s;
{
    int code= 0;

    if (s[0] == '_' && s[1] == '_' /* && isupper(s[2]) */) {
	char *st= &s[2];
	s+= 2;
	while (*s++ != '_')
	    ;
	if (*s++ == '_') {
	    if (s[0] == 'c' && s[1] == 't' && s[2] == 'o' && s[3] == 'r') {
		char buf[200];
		int l= s-2-st;
		strncpy(buf, st, l);
		buf[l]= '\0';
		Testload(buf);
		code= 1;
	    }
	}
    }
    return code;
}

/*----------------------------------------------------------------------------*/

static void Resolve(mp, hashtable)
struct Module *mp;
AssocTable* hashtable;
{
    register struct nlist *symbol, *symp;
    int value;

    mp->undefs= 0;
    for (symp= mp->symtab; symp < mp->endp; symp++) {
	if (symp->n_type == N_EXT + N_UNDF) {   /* is not defined here yet. */
	    if (symbol= Lookup(hashtable, symp->n_un.n_name)) {
		if (symbol->n_type == N_COMM)
		    symp->n_type= N_COMM;
		else
		    symp->n_type= N_EXT + N_COMM;
		symp->n_value= symbol->n_value;
	    } else {
		if (value= symp->n_value) {    /* is common */
		    int rnd= value >= sizeof(double) ? sizeof(double) - 1
			       : value >= sizeof(long) ? sizeof(long) - 1
			       : sizeof(short) - 1;

		    symp->n_type= N_COMM;
		    new_common+= rnd;
		    new_common&= ~(long)rnd;
		    symp->n_value= new_common;
		    new_common+= value;
		} else {  /* is extern */
		    Error("Resolve", "undef %s", symp->n_un.n_name);
		    mp->undefs++;
		}
	    }
	}
    }
}

static void UpdateSymtab(mp, hashtable)
register struct Module *mp;
AssocTable* hashtable;
{
    register struct nlist *symp;

    for (symp= mp->symtab; symp < mp->endp; symp++)
	if (symp->n_type == N_TEXT+N_EXT || symp->n_type == N_DATA+N_EXT)
	    symp->n_value+= mp->text;
}

static int Relocate(mp, common)
register struct Module *mp;
long common;
{
    struct relocation_info *reloc;
    register char *address;
    register long datum;
    register struct relocation_info *rel, *first_data, *endp;
    struct nlist *symbols= mp->symtab;

    reloc= GetRelocInfo(mp, 0L);
    if (reloc == 0)
	return 0;
	
    /* text relocation */
	
    first_data= reloc + (mp->header.a_trsize  / sizeof(struct relocation_info));
    endp= reloc + (mp->header.a_trsize+mp->header.a_drsize )/ sizeof(struct relocation_info);

    for (rel= reloc; rel < endp; rel++) {
	address= (char*) (rel->r_address + mp->text);

	if (rel >= first_data)
	    address+= mp->header.a_text;
      
	switch (rel->r_length) {
	case 0:               /* byte */
	    datum= *address;
	    break;
	case 1:               /* word */
	    datum= *(short*) address;
	    break;
	case 2:               /* long */
	    datum= *(long*) address;
	    break;
	default:
	    return bad_format;
	}
	  
	if (rel->r_extern) {    /* Look it up in symbol-table */
	    struct nlist *symbol= &symbols[rel->r_symbolnum];
	    
	    switch (symbol->n_type) {
	    case N_EXT + N_COMM:    /* old common or external */
		datum+= symbol->n_value;
		break;
	    case N_COMM:            /* new common */
		/*
		datum+= mp->text + mp->header.a_text + mp->header.a_data;
		*/
		datum+= common;
		break;
	    default:
		return bad_format;
	    }
	} else {                /* is static */
	    switch (rel->r_symbolnum & N_TYPE) {
	    case N_TEXT:
	    case N_DATA:
		datum+= mp->text;
		break;
	    case N_BSS:
		datum+= mp->text /* + new_common */;
		break;
	    case N_ABS:
		break;
	    }
	}
	
	if (rel->r_pcrel)
	    datum-= mp->text;

	switch (rel->r_length) {
	case 0:             /* byte */
	    if (datum < -128 || datum > 127)
		return bad_format;
	    *address= datum;
	    break;
	case 1:             /* word */
	    if (datum < -32768 || datum > 32767) 
		return bad_format;
	    *(short*) address= datum;
	    break;
	case 2:             /* long */
	    *(long*) address= datum;
	    break;
	}
    }
    
    Free(reloc);

    return ok;
}

int DynLoad(name)
char *name;
{
    char buf[100];
    int allundefs, lastundefs, code= 0;
    struct Module *mp;
    long common= 0;
    struct nlist *symbol, *symp, *endp;
    func_ptr ctor;
    
    Testload(name);
    
    lastundefs= 0;
    for (;;) {
	allundefs= 0;
	for (mp= firstModule; mp; mp= mp->next) {
	    FindAll(mp, mainhashtable);
	    allundefs+= mp->undefs;
	}
	if (allundefs == lastundefs)
	    break;
	lastundefs= allundefs;
    }
    
    if (allundefs > 0) {
	for (mp= firstModule; mp; mp= mp->next) {
	    if (mp->undefs <= 0)
		continue;
	    
	    Error("DynLoad", "%s",  mp->name);
	    
	    for (symp= mp->symtab; symp < mp->endp; symp++)
		if ((symp->n_type & N_EXT) && Lookup(mainhashtable, symp->n_un.n_name) == 0)
		    Error("DynLoad", "%s", symp->n_un.n_name);
	}
	_exit(1);
    }
    
    if (new_common > 0)
	common= (long) Malloc(new_common);
    new_common= 0;
    
    for (mp= firstModule; mp; mp= mp->next) {
    
	if ((mp->fd= open(mp->name, O_RDONLY)) < 0)
	    perror("open");
	    
	Resolve(mp, mainhashtable);
	allundefs+= mp->undefs;
	lastundefs= allundefs;
	GetTextAndData(mp, 0L);
	Relocate(mp, common);
	UpdateSymtab(mp, mainhashtable);
	close(mp->fd);
    }
    
    for (mp= firstModule; mp; mp= mp->next) {
	sprintf(buf, "__STI%s_C_", mp->name1);
	symbol= Lookup(mainhashtable, buf);
	if (symbol == 0) {
	    sprintf(buf, "__STI%s_c_", mp->name1);
	    symbol= Lookup(mainhashtable, buf);
	}
	if (symbol) {
	    ctor= (func_ptr) symbol->n_value;
	    if (ctor)
		(*ctor)();              /* call static constructors */
	}
	
	sprintf(buf, "__STD%s_C_", mp->name1);
	symbol= Lookup(mainhashtable, buf);
	if (symbol == 0) {
	    sprintf(buf, "__STD%s_c_", mp->name1);
	    symbol= Lookup(mainhashtable, buf);
	}
	if (symbol)
	    mp->dtor= (func_ptr) symbol->n_value;
    }
    firstModule= 0;
    return code;  
}

static struct nlist *Copy(sym)
struct nlist *sym;
{
    struct nlist *tmp;
    
    tmp= (struct nlist*) Malloc(sizeof(struct nlist));
    *tmp= *sym;
    tmp->n_un.n_name= strsave(sym->n_un.n_name, -1);
    return tmp;
}

void *DynLinkInit(me)
char *me;
{
    int fd;
    register struct nlist *symp;
    char *name;
    
    if (mainhashtable)
	return;

    name= which(getenv("PATH"), me);
    mainmodule.fd= open(name, O_RDONLY);
    GetHeader(&mainmodule, 0L);
    GetSymbols(&mainmodule, 0L);
    close(mainmodule.fd);
    
    mainhashtable= NewAssocTable();
    
    for (symp= mainmodule.symtab; symp < mainmodule.endp; symp++)
	if ((symp->n_type & N_EXT) && strcmp(symp->n_un.n_name, "_main"))
	    Assoc(mainhashtable, Copy(symp));
    Free(mainmodule.symtab);
}

char *DynLookup(name)
char *name;
{
    struct nlist *symbol;
    
    if (symbol= Lookup(mainhashtable, name))
	return (char*) symbol->n_value;
    return 0;
}

static void CallDtors(mp)
struct Module *mp;
{
    if (mp) {
	CallDtors(mp->next);
	if (mp->dtor)
	    (*mp->dtor)();              /* call static destructors */
    }
}

void DynCleanup()
{
    if (firstModule) {
	CallDtors(firstModule);
	firstModule= 0;
    }
}

static void *func_error()
{
    return 0;
}

func_ptr DynCall(name)
char *name;
{
    struct nlist *symbol;
    
    if (symbol= Lookup(mainhashtable, name))
	return (func_ptr) symbol->n_value;
    Error("DynCall", "can't find %s", name);
    return func_error;
}
