#ifndef StaticTextView_First
#define StaticTextView_First

#include "Text.h"
#include "Mark.h"
#include "ObjArray.h"
#include "View.h"

enum eTextJust { 
    eLeft, 
    eRight, 
    eCenter, 
    eJustified  
};

enum eSpacing  { 
    eOne        = 2, 
    eOneHalf    = 3, 
    eTwo        = 4 
};

enum TextViewFlags {
    eTextViewReadOnly   =   BIT(eVObjLast+1),   // disable text modifications
    eTextViewVisMode    =   BIT(eVObjLast+2),   // show new line characters
    eTextFormPreempt    =   BIT(eVObjLast+3),   // is formating process preemptive
    eTextFixedOrigin    =   BIT(eVObjLast+4),   // if contentRect.width is set to
						// cFit and the justification is eRight or
						// eCenter, should the origin of the text 
						// remain fixed
    eTextViewNoBatch    =   BIT(eVObjLast+5),   // do not batch characters in DoKeyCommand
    eTextFixedSpacing   =   BIT(eVObjLast+6),   // use fixed line spaceing
    eTextNoFind         =   BIT(eVObjLast+7),   // do not add menu items
    eTextLast           =   eVObjLast+7,     
    eTextViewDefault    =   eVObjDefault
};

extern Rectangle gFitRect;

//---- constants passed to DoUpdate to the dependents of a StaticTextView

enum { 
    eOrigin,           // offset of text changed
    eExtent,           // extent of text changed
    eReplacedText      // text replaced
};

const int cFit = -1;
const int cMaxFormat = 3;      // maximum number of consecutive lines formatted 
			       // without interrupt

//---- line mark ------------------------------------------------------------

class LineMark: public Mark {
public:
    static lineChanged;    // height or base of line changed 
    LineDesc ld;

public:
    MetaDef(LineMark);

    LineMark(LineDesc ldesc,int pos = 0, int len= 0, eMarkState s= 0);
    ~LineMark();
    void ChangeMark (int pos, int len, LineDesc ldesc, eMarkState s = 0)
	{ if (!lineChanged) 
	      lineChanged = !ld.IsEqual(ldesc);
	  Mark::ChangeMark(pos,len,s); 
	  ld = ldesc;
	}
    ostream& DisplayOn (ostream&s);
};

//---- global functions ----------------------------------------------------

int TextViewLineHeight(FontPtr fd = gSysFont, eSpacing sp = eOne); 
    // return the height of a textline

inline int TextViewlh(eSpacing sp, int h)
{
    return sp == eOne ? h: ((int)sp)*h/2;
}

//---- TextView -------------------------------------------------------------

class StaticTextView: public View {
protected:
    Text *text;                 // shown text
    eTextJust just;             // justification
    eSpacing spacing;           // spacing between two lines
    bool wrap;                  // break line at view boundary
    bool drawViewBorder;        
    bool vertExtend,horExtend;  // determine vertExtend, horExtend depending
				// on text
    Point border;               // border around the text

    int dirtyAt;                // first line where text is dirty (needs reformatting)
    bool interruptable;         // is formatting process interruptable ?
    int nLines;                 // number of lines
    ObjArray *lines;            // array of line marks, grows dynamically
    MarkList *marks;            // notice: this set of marks is managed by
				// members of the class StaticTextView not by
				// an instance of Text   
    class TextIter *nextc;      // iterator of currently loaded text
    LineDesc fixedSpacing;      // line spacing to use with fixed line spaceing

    int FirstCharPos (int start,int end); 
	// position of first char on a line
    bool MarkLine(int line, int start, int end, LineDesc *ld); 
	// add a line mark and return if the line changed
    void MarkLineAsChanged(int line);
    void BreakWord(int lWidth,int *start,int *line,int *end,int *width,LineDesc*);
    void Init(Rectangle, Text*, eTextJust, eSpacing, bool wrap, TextViewFlags,Point);
    bool IsJustified(int endOfLine);
    virtual int OutOfRangeLineMetric(bool lineHeight);   
    void highlight3(void *n, void *rv);

public:
    MetaDef(StaticTextView);

    StaticTextView(EvtHandler *eh, Rectangle r, Text *t, eTextJust m= eLeft,
	     eSpacing= eOne, bool wrap= TRUE, TextViewFlags= eVObjDefault, 
	     Point border= gBorder, int id = -1);    
	    // contentRect.width/height can be set to cFit, see above

    ~StaticTextView();

    void Draw(Rectangle);
    int SelectionRects(Rectangle *, int from, Point fp, int to, Point tp);
    void Invert (int from,Point fp,int to,Point tp);
    virtual void DrawLine (Point p,int line,Rectangle lineRect, Rectangle clip);
    virtual int Format(int startLine, int startCh, int &numLines, 
					   int upto = cMaxInt, int minUpto = 0);       
	// returns last formated line
    virtual int NoWrapFormat(int startLine, int startCh, int &numLines, 
					   int upto = cMaxInt, int minUpto = 0); 
    void ResumeFormat(bool interruptable = FALSE);
	// resume interrupted formating
    virtual void Reformat();                  // reformat and redisplay whole text
    virtual void ChangedAt(int line, int ch = 0, bool redrawAll = FALSE, int minUpto = 0);
	// format text starting at 'line' and formats at least to the line 'upto'
    virtual void InvalidateRange(int from, int to);

    virtual Text *SetText(Text*);     // returns old text       
    virtual void SetString(byte *str, int len= -1);
    Text *GetText()
	{ return text; }

    //---- acessing
    void SetSpacing (eSpacing);
    eSpacing GetSpacing()
	{ return spacing; }
    void SetFont(FontPtr fd);
    FontPtr GetFont()
	{ return text->GetFont(); }
    eTextJust GetJust() 
	{ return just; }
    void SetJust(eTextJust);
    void SetVisibleMode(bool);
    bool GetVisibleMode();
    void SetNoBatch(bool);
    bool GetNoBatch();
    void SetWordWrap(bool m);
    bool GetWordWrap();
    void SetReadOnly(bool);
    bool GetReadOnly();
    bool Empty()
	{ return text->Size() == 0; }
    int NumberOfLines();

    //---- mapping
    virtual void PointToPos(Point p,Point *viewPos,int *lineNo,int *CharNo); // viewPos.y ???
    virtual void CharToPos (int charNo,int *lineNo,Point *viewPos, bool relative = TRUE);
    virtual Point LineToPoint (int line, bool basePoint = FALSE, bool relative = TRUE);
    virtual int PointToLine (Point p);
    virtual int CharToLine(int ch);   // return line number of character

    //---- line structure access
    LineMark *MarkAtLine(int i)
	{ return (LineMark*)(*lines)[i]; }
    int StartLine(int l)
	{ return MarkAtLine(l)->pos; }
    int EndLine(int l)
	{ return MarkAtLine(l)->pos + MarkAtLine(l)->len; }
    int LineHeight(int l){
	    return TestFlag(eTextFixedSpacing) ? 
		 TextViewlh(spacing, fixedSpacing.lnHeight) :
		 l >= nLines ? 
		      OutOfRangeLineMetric(TRUE) :
		      TextViewlh(spacing, MarkAtLine(l)->ld.lnHeight);
	}
    int BaseHeight(int l){
	    return TestFlag(eTextFixedSpacing) ? 
		 fixedSpacing.lnAscent :
		 l >=nLines ? 
		     OutOfRangeLineMetric(FALSE) : 
		     MarkAtLine(l)->ld.lnAscent; 
	}

    //---- size managment
    virtual void Fit();     
	// adapt view size to the extension of text if wrap is set to TRUE only 
	// the height is changed
    void SetExtent(Point p); // (reformats text)
    Metric GetMinSize();
    int Base();
    Point GetInnerOrigin()  // 
	{ return contentRect.origin+border; }
    Point GetInnerExtent()  
	{ return contentRect.extent-2*border; }

    //----- client marks    
    void AddMark(Mark *);
    Mark *RemoveMark(Mark *);
    Iterator *GetMarkIter(); 
    MarkList *GetMarkList();    

    //----- debugging
    virtual void Dump();
    void InspectorId(char *buf, int sz);

    //----- generic methods
    char *AsString();
    ostream& PrintOn (ostream&s);
    istream& ReadFrom(istream &s);
};

#endif StaticTextView_First     
