########################################################################
##
## Copyright (C) 2020-2025 The Octave Project Developers
##
## See the file COPYRIGHT.md in the top-level directory of this
## distribution or <https://octave.org/copyright/>.
##
## This file is part of Octave.
##
## Octave is free software: you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## Octave is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with Octave; see the file COPYING.  If not, see
## <https://www.gnu.org/licenses/>.
##
########################################################################

## -*- texinfo -*-
## @deftypefn  {} {@var{angle} =} xtickangle ()
## @deftypefnx {} {@var{angle} =} xtickangle (@var{hax})
## @deftypefnx {} {} xtickangle (@var{angle})
## @deftypefnx {} {} xtickangle (@var{hax}, @var{angle})
## Query or set the rotation angle of the tick labels on the x-axis of the
## current axes.
##
## When called without an argument, return the rotation angle in degrees of the
## tick labels as specified in the axes property @qcode{"XTickLabelRotation"}.
## When called with a numeric scalar @var{angle}, rotate the tick labels
## counterclockwise to @var{angle} degrees.
##
## If the first argument @var{hax} is an axes handle, then operate on this axes
## rather than the current axes returned by @code{gca}.
##
## Programming Notes:
## @enumerate
## @item
## The @qcode{"XTickLabelRotation"} property is currently unimplemented in
## Octave.  The property can be set and queried, but has no effect on the plot.
##
## @item
## Requesting a return value while also setting a specified
## rotation will result in an error.
## @end enumerate
##
## @seealso{ytickangle, ztickangle, get, set}
## @end deftypefn

function angle = xtickangle (hax, angle)

  switch (nargin)
    case 0
      angle = __tickangle__ (mfilename ());

    case 1
      if (nargout > 0)
        angle = __tickangle__ (mfilename (), hax);
      else
        __tickangle__ (mfilename (), hax);
      endif

    case 2
      if (nargout > 0)
        angle = __tickangle__ (mfilename (), hax, angle);
      else
        __tickangle__ (mfilename (), hax, angle);
      endif

  endswitch

endfunction


%!test
%! hf = figure ("visible", "off");
%! hax = axes (hf);
%! unwind_protect
%!   xtickangle (45);
%!   assert (xtickangle (), 45);
%!   xtickangle (hax, 90);
%!   a1 = xtickangle ();
%!   a2 = xtickangle (hax);
%!   assert (a1, a2);
%!   assert (a1, 90);
%! unwind_protect_cleanup
%!   close (hf);
%! end_unwind_protect

## Test input validation
%!error <HAX must be a handle to an axes object> xtickangle (0, 45)
%!error <ANGLE must be .* scalar> xtickangle (eye (2))
%!error <ANGLE must be .* numeric> xtickangle ({90})
%!error <ANGLE must be .* finite> xtickangle (Inf)
%!error <called with output query and input set value> ang = xtickangle (45)
