/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_nsid.c	1.17 96/04/09 Sun Microsystems"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_proto.h>
#include <skip_types.h>
#include <skip_keymgrmsgs.h>
#include <skip_crypt.h>
#include <skip_key.h>
#include <skip_ioctl.h>

/*
 * Name space identifiers (NSID) decoders
 */
unsigned char	skip_nsid[] =
		{
			/* "Not present" */ 		 0,
			/* "IPv4 Address" */ 		 4,
			/* "POSIX/XOPEN User ID" */	 4,
			/* "IPv6 Address" */		16,
			/* "MD5 (DNS name)" */		16,
			/* "MD5 (ASN.1 DN)" */		16,
			/* "MD5 (Social Sec. #)" */	16,
			/* "802.x Address" */		 6,
			/* "MD5 (DH Pub.Value)" */	16,
			/* "MD5 (Email Address)" */	16,
			/* "MD5 (Bank Account #)" */	16,
			/* "MD5 (NIS Name)" */		16
		};

	
/*
 * Array of lists of local Master Key IDs per NSID
 */
static skip_keyid_t	*skip_local_keys[SKIP_MAX_NSID] = { 0 };

int skip_max_keyids = SKIP_MAX_LOCAL_KEYIDS;

/*
 * skip_local_keyid_init()
 *
 * Initiliaze local Master Keys ID and NSID
 */
void
skip_local_keyid_init()
{
	BZERO(&skip_local_keys, sizeof(skip_local_keys));
}

/*
 * skip_local_keyid_uninit()
 *
 * Cleanup for all local Master Keys ID
 */
void
skip_local_keyid_uninit()
{

	skip_keyid_t	*p_mkeyid;
	skip_keyid_t	*p_next;
	int		nsid;
	
	for (nsid = 0; nsid < SKIP_MAX_NSID; nsid++) {

		p_mkeyid = skip_local_keys[nsid];

		for (; p_mkeyid; p_mkeyid = p_next) {
			p_next = p_mkeyid->next;
			SYSMEM_FREE(p_mkeyid, sizeof(skip_keyid_t));
		}

		skip_local_keys[nsid] = (skip_keyid_t *) NULL;
	}

	BZERO(&skip_local_keys, sizeof(skip_local_keys));
}

/*
 * skip_check_local_keyid()
 *
 * Check for a local Master Key ID validity
 */
int
skip_check_local_keyid(unsigned char nsid, skip_key_var_t *mkeyid)
{
	skip_keyid_t	*p_mkeyid = skip_local_keys[nsid];

	for (; p_mkeyid; p_mkeyid = p_mkeyid->next) {

		if (KEYVAREQUAL(p_mkeyid->mkeyid, *mkeyid)) {
			/*
			 * This local Master Key ID exits
			 */
			return(0);
		}
	}

	/*
	 * This local Master Key ID is not supported by the key manager
	 */
	return(1);
}

/*
 * skip_check_keyid()
 *
 * Check for a NSID/Master Key ID validity
 */
int
skip_check_keyid(unsigned char nsid, skip_key_var_t *mkeyid, boolean_t is_local)
{

	if ((nsid == 0) && (mkeyid->len != 0)) {
		/*
		 * The Master key ID length should be null
		 * if no NSID is specified
		 */
		return(-1);
	}

	if ((int) nsid >= SKIP_MAX_NSID) {
		/*
		 * The NSID is out of range
		 */
		return(-1);
	}

	if (mkeyid->len != skip_nsid[nsid]) {
		/*
		 * The master Key ID length is invalid
		 */
		return(-1);
	}

	if ((nsid != 0) && (is_local == B_TRUE)) {
		/*
		 * Check if this local Master Key ID is supported
		 */
		if (skip_check_local_keyid(nsid, mkeyid)) {
			/*
			 * Key manager does not support this local Master Key ID
			 */
			return(-1);
		}
	}

	/*
	 * NSID and Master Key Id are OK...
	 */
	return(0);
}

/*
 * skip_set_local_keyid()
 *
 * Set a local Master Key ID (Sender) for a given NSID
 */
void
skip_set_local_keyid(unsigned char nsid, skip_key_var_t *mkeyid)
{

	skip_keyid_t	*p_mkeyid;
	skip_keyid_t	*p_new;
	skip_keyid_t	*p_last = (skip_keyid_t *) NULL;
	int		key_count = 0;

	/*
	 * Check the NSID range
	 */
	if ((int) nsid >= SKIP_MAX_NSID) {
		return;
	}

	/*
	 * Check the Master Key ID length
	 */
	if (mkeyid->len != skip_nsid[nsid]) {
		return;
	}

	p_mkeyid = skip_local_keys[nsid];

	for (; p_mkeyid; p_mkeyid = p_mkeyid->next) {

		/*
 	 	 * Check for Master Key ID unicity
		 */
		if (KEYVAREQUAL(p_mkeyid->mkeyid, *mkeyid)) {
			/*
			 * This Master Key already exits - drop the request
			 */
			return;
		}

		key_count++;

		/*
		 * Keep track of the end of the list
		 */
		p_last = p_mkeyid;
	}

	if (key_count < skip_max_keyids) {
		/*
		 * We can add this new local Master Key ID
		 */
		p_new = (skip_keyid_t *) SYSMEM_ALLOC(sizeof(skip_keyid_t));

		if (p_new == NULL) {
			return;
		}

		/*
		 * Set new Local Key ID value
		 */
		KEYVARSET(p_new->mkeyid, *mkeyid);
		p_new->next = (skip_keyid_t *) NULL;

		/*
		 * Add this local Master Key ID into the list
		 */
		if (p_last) {
			p_last->next = p_new;
		} else {
			skip_local_keys[nsid] = p_new;
		}
	}

}

/*
 * skip_get_default_keyi(d)
 *
 * Get the first local Master Key for a given NSID.
 */
int
skip_get_default_keyid(unsigned char nsid, unsigned char *p_buf)
{
	skip_keyid_t *p_mkeyid;

	if (((int) nsid >= SKIP_MAX_NSID) || ((int) nsid <= 0)) {
		/*
		 * The NSID is out of range
		 */
		return(0);
	}

	p_mkeyid = skip_local_keys[nsid];

	if (p_mkeyid) {

		KEYTOBUF(p_mkeyid->mkeyid, p_buf);

		return(p_mkeyid->mkeyid.len);
	}

	/*
	 * The key manager does not support this NSID
	 */
	return(0);
}

/*
 * skip_get_keyid_list()
 *
 * Fill in the local master key ID list Ioctl
 */
void
skip_get_keyid_list(ioctl_keyid_t *p_ioctl)
{
	skip_keyid_list_t	*p_keyid_lst;
	skip_keyid_t		*p_mkeyid;
	int			nsid;

	BZERO(p_ioctl, sizeof(ioctl_keyid_t));

	p_keyid_lst = p_ioctl->keyid_lst;

	for (nsid = 0; nsid < SKIP_MAX_NSID; nsid++) {

		p_mkeyid = skip_local_keys[nsid];

		for (; p_mkeyid; p_mkeyid = p_mkeyid->next) {
			
			KEYVARSET(p_keyid_lst->mkeyid[p_keyid_lst->count],
							p_mkeyid->mkeyid);
			p_keyid_lst->count++;

			if (p_keyid_lst->count >= SKIP_MAX_LOCAL_KEYIDS) {
				
				break;
			}
		}

		p_keyid_lst++;
	}
}
