/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)cdp_queue.C	1.11 96/09/06"

#include <skip_os.h>
#include <skip_proto.h>
#include "skip_log.h"

#include "AuthPolicy.h"
#include "cdp_queue.h"
#include "cdp.h"
#include "skip_certbase.h"

extern void process_cert_put(cdp_rec *p, cdp* reply);

LL<resolver> resolvers;

static int advance_cdp_requestor(cdp_queue_entry *p);
static int advance_cdp_entry(cdp_queue_entry *p);


//  Find resolver N in resolvers list and return a pointer to it
//  Indexed from 1

resolver* find_resolver(cdp_queue_entry *ent)
{
	resolver *p;
	int n = ent->who_sent;

	for (p = resolvers.first(); p; p = resolvers.next()) {
		n--;
		if (n <= 0) {
			if (p->kind == RESOLVER_PING)
				p->server = ent->remote.ip_addr;
			else if (p->kind == RESOLVER_TUNNEL)
				p->server = ent->remote.tunnel_addr;
			return p;
		}
	}

	return NULL;
}

void add_cdp_resolver(int kind, char *arg) 
{
	resolver r;

	r.kind = kind;

	switch (kind) {
	case RESOLVER_PING:
		skip_log(SKIP_NOTICE, "add_cdp_resolver(@) (bilateral)");
		break;

	case RESOLVER_TUNNEL:
		skip_log(SKIP_NOTICE, "add_cdp_resolver(=) (tunnel)");
		break;
		
	case RESOLVER_HOST:
		if (parse_ip_address(&r.server, arg))
			return;
		skip_log(SKIP_NOTICE, "add_cdp_resolver(%s)", arg);
		break;

	default:
		skip_log(SKIP_ERROR, "add_cdp_resolver: internal error");
		return;
	}

	resolvers.append(r);
}

cdp_queue_entry*
cdp_queue::first()
{
	return l.first();
}

cdp_queue_entry*
cdp_queue::next()
{
	return l.next();
}

void cdp_queue::add_entry(struct remote_info *remote, char *buf, int size)
{
	cdp_queue_entry n;
	cdp_queue_entry *p;

	if (l.length() >= skip_conf.max_cdp_queue) {
		skip_log(SKIP_NOTICE, "cdp work queue full, request dropped");
		return;
	}

//  If we're already doing a lookup for this (nsid,mkid),
//  don't start another one

	for (p = first(); p; p = next()) {
		if (remote->nsid == p->remote.nsid &&
		    remote->keyid == p->remote.keyid)
			return;
	}

	n.remote = *remote;
	n.orig_request = malloc(size);
	if (n.orig_request == 0) {
		skip_log(SKIP_ERROR, "add_cdp_queue: out of memory\n");
		return;
	}

	bcopy(buf, n.orig_request, size);
	n.orig_request_size = size;

	n.n_try = skip_conf.cdp_max_retry;
	n.time_sent = 0;
	n.who_sent = 0;
	n.request_fd = -1;

	if (advance_cdp_entry(&n)) {
		skip_log(SKIP_ERROR,
			"CDP initiation failed for nsid=%d mkid=%s",
				remote->nsid, remote->keyid.get_info_str());
		return;
	}

	l.append(n);
}

void cdp_queue::delete_current()
{
	cdp_queue_entry *p;

	p = l.current();

	if (p) {
		if (p->orig_request)
			free(p->orig_request);

		if (p->request_fd >= 0)
			close(p->request_fd);

		l.delete_current();
	}
}

static int decode_cdp_response(int sock, cdp& response)
{

	Bstream bufstr = read_sock(sock);

	if (bufstr.getlength() == 0)
		return -1;

	if (response.decode(bufstr)) {
		skip_log(SKIP_ERROR,
			"decode_cdp_response: mangled answer: %s",
			sock_info_str);
		return -1;
	}

	if (response.action != CDP_RESPONSE) {
		skip_log(SKIP_ERROR,
		"decode_cdp_response: unexpected response action %d: %s",
			response.action, sock_info_str);
		return -1;
	}

	return 0;
}

static int collect_received_certs(cdp_queue_entry *p, cdp& response,
						CertInfoList& cert_l)
{
	int ncerts = 0;
	cdp_rec *q;
	CertInfo *t;
	int err_print = 1;

	for (q = response.recs.first(); q; q = response.recs.next()) {
		switch (q->action) {
		case CDP_REC_GETOK:
			if (q->ci.cert == NULL) {
				skip_log(SKIP_ERROR, "%s %s: no certificate "
					"in GET-OK response", sock_info_str,
					q->get_info_str());
				err_print = 0;
				continue;
			}

			ncerts++;

			if (q->ci.nsid != p->remote.nsid ||
			    q->ci.mkid != p->remote.keyid) {
				skip_log(SKIP_ERROR, "%s %s: reply contains "
					"certificate not requested",
					sock_info_str, q->get_info_str());
				skip_log(SKIP_ERROR,
					"  requested nsid=%d, mkid=%s",
					p->remote.nsid,
					p->remote.keyid.get_info_str());
				err_print = 0;
				continue;
			}

			cert_l.append(q->ci);
			break;

		case CDP_REC_GETFAIL:
			skip_log(SKIP_NOTICE, "%s %s: response=getfail",
				sock_info_str, q->get_info_str());
			err_print = 0;
			break;

		case CDP_REC_PUT:
			process_cert_put(q, NULL);
			break;

		default:
			skip_log(SKIP_ERROR,
				"unknown action %d in CDP response",
				q->action);
		}
	}

	if (ncerts == 0) {
		if (err_print)
			skip_log(SKIP_NOTICE,
			"%s: no certificates present in CDP response",
					sock_info_str);
		return -1;
	}

// Choose a valid certificate from the ones returned and
// send the answer back to the kernel

	t = cert_l.find_valid_cert(auth);

	if (t == 0) {
		skip_log(SKIP_NOTICE,
			"%s: no valid certs in CDP response, ncerts=%d",
					sock_info_str, ncerts);

		int n = 0;
		char *err_s;

		for (CertInfo *s = cert_l.first(); s; s = cert_l.next()) {
			n++;
			if ((err_s = s->check_valid(auth)) != NULL)
				skip_log(SKIP_NOTICE, "  reason %d = %s",
					n, err_s);
		}

		return -1;
	}

// Save certificate in DB

	db_put_certinfo(*t);

	return 0;
}

int cdp_queue::check_answer_entry(cdp_queue_entry *p, int sock, int kernfd)
{
	cdp response;
	CertInfoList cert_l;

// Decode CDP answer

	if (decode_cdp_response(sock, response))
		return TRUE;

// See if one of the certificates obtained from CDP is valid
// if so, store it in the DB

	if (collect_received_certs(p, response, cert_l))
		return TRUE;

// Send response to the kernel
//
// the certificate we just received will be obtained from the DB
// by a callee of make_response

	void *new_msg = malloc(MGTCSIZE);
	int size;

	if (new_msg == 0) {
		skip_log(SKIP_ERROR, "check_answer_entry: out of memory");
		return FALSE;
	}

	make_response(p->orig_request, new_msg, &size, &p->remote);
	put_kernel_message(kernfd, new_msg, size);
	free(new_msg);

// we're done, delete this cdp queue entry

	return FALSE;
}

//  See if any entry in the cdp queue is waiting on sock for an answer
//  If so, process it:
//
//	find socket data was received on in the cdp queue
//	  if there was something wrong with the answer
//	    try to move to the next cdp server
//	  otherwise we're done

int cdp_queue::check_answer_received(int sock, int kernfd)
{
	cdp_queue_entry *p;

	for (p = first(); p; p = next())
		if (p->request_fd == sock) {
			if (check_answer_entry(p, sock, kernfd)) {
				if (!advance_cdp_requestor(p))
					return TRUE;
			}

			delete_current();
			return TRUE;
		}

	return FALSE;
}

static int advance_cdp_requestor(cdp_queue_entry *p)
{

	if (p->request_fd >= 0) {
		close(p->request_fd);
		p->request_fd = -1;
	}

	p->n_try = 0;
	p->who_sent++;
	p->cur_resolver = find_resolver(p);

	if (p->cur_resolver == 0)
		return -1;

	p->request_fd = open_socket(CDP_PORT_INITIATOR, 1);
	if (p->request_fd < 0)
		p->request_fd = open_socket(0);

	if (p->request_fd < 0) {
		skip_log(SKIP_NOTICE,
				"advance_cdp_requestor: open_socket failed");
		return -1;
	}

	return 0;
}

static int advance_cdp_entry(cdp_queue_entry *p)
{
	time_t now = time(NULL);

	if (now - p->time_sent < skip_conf.cdp_retry_time)
		return FALSE;		// just sent a request, wait more

	if (p->n_try >= skip_conf.cdp_max_retry)
		if (advance_cdp_requestor(p)) {
			skip_log(SKIP_NOTICE,
				"cdp failed for nsid=%d mkid=%s",
					p->remote.nsid,
					p->remote.keyid.get_info_str());

			return TRUE;	// CDP failed, we're done
		}

//  Send CDP request

	if (p->n_try == 0)
		skip_log(SKIP_NOTICE,
			"sending CDP request for nsid=%d mkid=%s to %s",
				p->remote.nsid,
				p->remote.keyid.get_info_str(),
				get_ip_str(&(p->cur_resolver->server)));

	cdp request(CDP_INITIATE);
	request.cdp_get(p->remote.nsid, p->remote.keyid);
	Bstream cdp_s = request.encode();

	set_sock_partner(&(p->cur_resolver->server), CDP_PORT_RESPONDER);
	write_sock(p->request_fd, cdp_s);

	p->n_try++;
	p->time_sent = now;

	return FALSE;
}

void cdp_queue::heartbeat()
{
	cdp_queue_entry *p;

	for (p = first(); p; p = next()) {
		if (advance_cdp_entry(p))
			delete_current();
	}
}

