/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#include <skip_os.h>

#include <skip_conf.h>

#include "AuthPolicy.h"
#include "X509skip.h"
#include "HashCert.h"
#include "SkipCert.h"
#include "LL.h"
#include "manualcrl.h"
#include "db.h"

#define CA_PATH	SKIP_DIR"/CAs"
AuthPolicy auth;

void
usage(char *s){
	fprintf(stderr, "%s [add|extract|init|list|revoke|rm|unrevoke]\n",s);
	exit(1);
}

void
init_usage(char *s)
{
	fprintf(stderr,"%s init [-rq]\n",s);
	exit(1);
}

void
list_usage(char *s)
{
	fprintf(stderr,"%s list [-Vve]\n",s);
	exit(1);
}

void
revoke_usage(char *s)
{
	fprintf(stderr,"%s revoke -s slot-name serial-number\n",s);
	exit(1);
}

void
unrevoke_usage(char *s)
{
	fprintf(stderr,"%s unrevoke -s slot-name serial-number\n",s);
	exit(1);
}

void
add_usage(char *s)
{
	fprintf(stderr,"%s add filename\n",s);
	exit(1);
}

void
extract_usage(char *s)
{
	fprintf(stderr,"%s extract slot-number\n",s);
	exit(1);
}

void
del_usage(char *s)
{
	fprintf(stderr,"%s rm slot-number\n",s);
	exit(1);
}

void
revlist(char *name, int machprt)
{
	char buf[1024], buf1[255];
	int gotone = 0;
	struct stat s;
	ManualCRL revlist;

	sprintf(buf,"%s/%s", CA_PATH, name);
	if (stat(buf, &s)) {
		fprintf(stderr,"Error: No Certificate present in slot %s\n",
				name);
		exit(1);
	}
	strcpy(buf1, name);
	buf1[strlen(buf1)-5] = (char)0;
	sprintf(buf,"%s/%s.mcrl", CA_PATH, buf1);

	revlist.load(buf);
	if (machprt) 
		printf(" revoked=[");

	for (Bigint *p=revlist.serialnos.first(); p!=NULL; p=revlist.serialnos.next()) {
		String temp;
		temp = p->getnumstr();
		if (!gotone && !machprt) {
			gotone = 1;
			printf("\tRevoked Certificates:\n");
		}		

		if (machprt) 
			printf("%s ",(const char *)temp);
		else 
			printf("\t\tserial number = %s\n",(const char *)temp);
	
	}
	if (machprt) 
		printf("]\n");
}	

void
skip_list_ca(int nargc, char *nargv[])
{
	char *name;
	struct dirent *e;
	DIR *d;
	int c;
	int verbose = 0, machprt = 0, expiration =0;
	int revoked = 0;

	while ((c = getopt(nargc-1, &nargv[1],"vVer")) != EOF)
		switch (c) {
			case 'v':
				verbose = 1;
				break;
			case 'e':
				expiration = 1;
				break;
			case 'V':
				machprt = 1;
				break;
			case 'r':
				revoked = 1;
				break;
			default:
				fprintf(stderr, "Invalid flag %c\n",c);
				list_usage(nargv[0]);
				break;
		}

	optind++;
        if (optind >= nargc) {
		name = NULL;
	} else {
		name = nargv[optind];
		if (name == NULL) {
		fprintf(stderr,"Invalid Name\n");
		exit(1);
		}
	}

	d=opendir(CA_PATH);
	if (verbose)
		expiration = 0;

	if (d == NULL) {
		fprintf(stderr,"Error: Permission denied or uninitalized system\n");
		exit(1);
	}
	while ((e = readdir(d))!=NULL) {
		int len=strlen(e->d_name);
		char buf[1024];
		if (len < 5)
			continue;
		if (strcmp(e->d_name + len - 5, ".x509") != 0)
			continue;
		strncpy(buf, e->d_name, len - 5);

		buf[len-5] = '\0';
		if (name != NULL && strcmp(name, buf))
			continue;
		printf("ca-slot=%s ", buf);
		if (!machprt)
			printf("\n");

		X509Cert cert;
		Bstream certstr;
		sprintf(buf, "%s/%s",CA_PATH, e->d_name);
		certstr = File_to_Bstr(buf);
		if (certstr.getlength() == 0) {
			printf("error = bad-decode\n");
			continue;
		}
		if (asn1_der_decode_cert(certstr, cert) != SUCCESS) {
			printf("error = bad-decode1\n");
			continue;
		}
		if (verbose) {
			if (machprt)  {
				cert.machprint();
				printf("%%%%%%%%%%\n");
			} else
				cert.print();
			continue;
		}
		if (expiration) {
			if (machprt) {
				printf("invalidbefore=%lu ",
					cert.getnotbefore().ntptime());
				printf("invalidafter=%lu ",
					cert.getnotafter().ntptime());
			} else {
				printf("\tNot Valid Before: "); 
				cert.getnotbefore().print();
				printf("\n\tNot Valid After : "); 
				cert.getnotafter().print();
				printf("\n");
			}
		}

		if (machprt) {
			printf(" Subject=");
			cert.getsubject().print();
		} else {
			printf("\tSubject: ");
			cert.getsubject().print();
			printf("\n");
		}	
		if (machprt) {
			printf(" Issuer=");
			cert.getissuer().print();
		} else {
			printf("\tIssuer: ");
			cert.getissuer().print();
			printf("\n");
		}

		if (revoked) 
			revlist(e->d_name, machprt);
		printf("\n");
		fflush(stdout);
	}
	closedir(d);
}

void
skip_init_ca(int nargc, char **nargv)
{
	struct stat s;
	int quiet = 0;
	int c;
	char buf[1024];
	int overwrite = 0;

	while ((c = getopt(nargc-1, &nargv[1],"qr")) != EOF)
		switch(c) {
			case 'q':
				quiet = 1;
				break;
			case 'r':
				overwrite = 1;
				break;
			default:
				init_usage(nargv[0]);
		}
	if (stat(CA_PATH, &s)) {
		if (!mkdir(CA_PATH, 0700))
			exit(0);
		else {
			fprintf(stderr,"Couldn't create Directory\n");
			exit(2);
		}
	}
	if (overwrite) {
		int ret;
		sprintf(buf,"/usr/bin/rm -rf %s", CA_PATH);
		ret = system(buf);
		if (ret) {
			fprintf(stderr,"Unable to Clean old directory %s \n",
					CA_PATH);
			exit(2);
		}
		if (!mkdir(CA_PATH, 0700))
			exit(0);
		else {
			fprintf(stderr,"Unable to Created New Direcotry %s\n",
					CA_PATH);
			exit(2);
		}
	} else {
		if (!quiet)
			fprintf(stderr, "Directory Already exists\n");
		exit(0);
	}
}

void
skip_del_ca(int argc, char **argv)
{
	char buf[1024];

	if (argc != 3)
		del_usage(*argv);
	argv += 2;
	argc -= 2;
	sprintf(buf,"%s/%s.x509",CA_PATH, *argv);
	if (unlink(buf)){
		fprintf(stderr,"Couldn't delete CA slot %s - %s\n",
				*argv,strerror(errno));
		exit(1);
	}
	fprintf(stderr,"Successfully deleted CA slot %s\n", *argv);
	exit(0);
}

void
skip_extract_ca(int argc, char **argv)
{
	char buf[1024];
	Bstream certstr;

	if (argc != 3)
		extract_usage(*argv);
	argv += 2;
	argc -= 2;
	sprintf(buf,"%s/%s.x509",CA_PATH, *argv);

	certstr = File_to_Bstr(buf);
	if (certstr.getlength() == 0) {
		fprintf(stderr,"CA not found or permission denied\n");
		exit(1);
	}
	fwrite(certstr.getdatap(), certstr.getlength(), 1, stdout);
	exit(0);
}


void
skip_add_ca(int argc, char **argv)
{
	Bstream certstr1, certstr2;
	Bstream temp;
	X509Cert cert1, cert2;
	struct dirent *e;
	char buf[1024];
	DIR *d;
	FILE *fp;	
	
	if (argc != 3)
		add_usage(*argv);
	argv+=2;
	argc-=2;
	certstr1 = File_to_Bstr(*argv);
	temp = certstr1;
	if (asn1_der_decode_cert(temp, cert1) != SUCCESS) {
		fprintf(stderr, "Unable to decode %s\n", *argv);
		exit(1);
	}
	d=opendir(CA_PATH);
	if (d == NULL) {
		fprintf(stderr,"Error: Permission denied or uninitalized system\n");
		exit(1);
	}
	while ((e = readdir(d))!=NULL) {
		int len=strlen(e->d_name);

		if (len < 5)
			continue;
		if (strcmp(e->d_name + len - 5, ".x509") != 0)
			continue;


		sprintf(buf,"%s/%s",CA_PATH, e->d_name);
		certstr2 = File_to_Bstr(buf);

		if (certstr2.getlength() ==0)		// XXX
			continue; 

		if (asn1_der_decode_cert(certstr2, cert2) != SUCCESS)
			continue;

		if (cert2 == cert1) {
			fprintf(stderr, "Certificate already exists!\n");
			exit(2);
		}	
	}
	closedir(d);
	int filecount = 0;
	do {
		sprintf(buf,"%s/%d.x509", CA_PATH, filecount);
		fp=fopen(buf,"r");
		if (fp == NULL) {
			if (certstr1.store(buf)){
				fprintf(stderr, 
			 	      "Added CA certificate as ca-slot %d\n", 
					filecount);

				exit(0);
			} else {
				fprintf(stderr,	
					"Unable to save certificate!\n");
				exit(1);
			}
		}
		fclose(fp);
		filecount++;
	} while(1);
}

void
skip_ca_revoke(int nargc, char *nargv[])
{
	char *slot=NULL, *serialno;
	int c;
	while ((c = getopt(nargc-1, &nargv[1],"s:")) != EOF)
		switch(c) {
			case 's':
				slot = optarg;
				break;
			default:
				fprintf(stderr, "Invalid option %c\n", c);
				exit(1);
		}
	if (slot == NULL) {
		fprintf(stderr,"Error: Slot arguement required\n");
		revoke_usage(nargv[0]);
	}

	char buf[1024];
	struct stat s;
	ManualCRL revlist;

	sprintf(buf,"%s/%s.x509", CA_PATH, slot);
	if (stat(buf, &s)) {
		fprintf(stderr,"Error: No Certificate present in slot %s\n",
				slot);
		exit(1);
	}
	sprintf(buf,"%s/%s.mcrl", CA_PATH, slot);


	optind++;
        if (optind >= nargc) {
		serialno = NULL;
	} else {
		serialno = nargv[optind];
		if (serialno == NULL) {
			fprintf(stderr,"Invalid Name\n");
			revoke_usage(nargv[0]);
			exit(1);
		}
	}
	revlist.load(buf);
	Bigint serno(serialno);
	revlist.add(serno);
	revlist.store(buf);
}	

void
skip_ca_unrevoke(int nargc, char *nargv[])
{
	char *slot=NULL, *serialno;
	int c;
	while ((c = getopt(nargc-1, &nargv[1],"s:")) != EOF)
		switch(c) {
			case 's':
				slot = optarg;
				break;
			default:
				fprintf(stderr, "Invalid option %c\n", c);
				exit(1);
		}
	if (slot == NULL) {
		fprintf(stderr,"Error: Slot arguement required\n");
		revoke_usage(nargv[0]);
	}

	char buf[1024];
	struct stat s;
	ManualCRL revlist;

	sprintf(buf,"%s/%s.x509", CA_PATH, slot);
	if (stat(buf, &s)) {
		fprintf(stderr,"Error: No Certificate present in slot %s\n",
				slot);
		exit(1);
	}
	sprintf(buf,"%s/%s.mcrl", CA_PATH, slot);


	optind++;
        if (optind >= nargc) {
		serialno = NULL;
	} else {
		serialno = nargv[optind];
		if (serialno == NULL) {
			fprintf(stderr,"Invalid Name\n");
			unrevoke_usage(nargv[0]);
			exit(1);
		}
	}
	revlist.load(buf);
	Bigint serno(serialno);
	if (!revlist.ispresent(serno)){
		fprintf(stderr,"Serial number not present in manual revocation list");
		exit(1);
	}
	revlist.del(serno);
	revlist.store(buf);
}	

main(int argc, char *argv[])
{

	if (argc < 2) 
		usage(argv[0]);
	if (strcmp(argv[1],"add")==0)
		skip_add_ca(argc, argv);

	else if (strcmp(argv[1],"rm")==0 || strcmp(argv[1],"del")==0)
		skip_del_ca(argc, argv);

	else if (strcmp(argv[1],"list")==0)
		skip_list_ca(argc, argv);
	else if (strcmp(argv[1],"ls")==0)
		skip_list_ca(argc, argv);

	else if (strcmp(argv[1],"init")==0)
		skip_init_ca(argc, argv);

	else if (strcmp(argv[1],"extract")==0)
		skip_extract_ca(argc, argv);
	else if (strcmp(argv[1],"revoke")==0)
		skip_ca_revoke(argc, argv);
	else if (strcmp(argv[1],"unrevoke")==0)
		skip_ca_unrevoke(argc, argv);
	else 
		usage(argv[0]);
	exit(0);
}
