/*
 * Copyright (C) 1995, 1996 Systemics Ltd (http://www.systemics.com/)
 * All rights reserved.
 *
 * This library and applications are FREE FOR COMMERCIAL AND NON-COMMERCIAL USE
 * as long as the conditions within the COPYRIGHT file are adhered to.
 *
 */

package cryptix.pgp;

import java.io.IOException;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import cryptix.crypt.MessageHash;
import cryptix.crypt.MD5;
import cryptix.crypt.HashMD5;
import cryptix.crypt.SHA;
import cryptix.crypt.HashSHA;
import cryptix.util.*;

/**
 * static class that reads an write message hashes.
 */
public final class HashFactory
{
	private static final int NONE_IDENT = 0;
	private static final int MD5_IDENT = 1;
	private static final int SHA_IDENT = 2;

	private HashFactory()
	{
	}
	
	public static final MessageHash
	load( byte buf[] )
	throws IOException
	{
		return read( new DataInputStream( new ByteArrayInputStream( buf ) ) );
	}

	public static final byte[]
	save( MessageHash mh )
	throws IOException
	{
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		write( new DataOutputStream( out ), mh );
		return out.toByteArray();
	}
	
	public static final MessageHash
	read( DataInput in )
	throws IOException
	{
		int type;
		byte buf[];
		switch( type = in.readByte() & 0xFF )
		{
		case NONE_IDENT:
			return null;
		case MD5_IDENT:
			buf = new byte[MD5.HASH_LENGTH];
			in.readFully( buf );
			return new HashMD5( buf );
		case SHA_IDENT:
			buf = new byte[SHA.HASH_LENGTH];
			in.readFully( buf );
			return new HashSHA( buf );
		default:
			throw new IOException( "Hash type (" + type + ") not recognized" );
		}
	}

	public static final void
	write( DataOutput out, MessageHash mh )
	throws IOException
	{
		if ( mh == null )
			out.writeByte( NONE_IDENT );
		else
		{
			if ( mh instanceof HashMD5 )
				out.writeByte( MD5_IDENT );
			else if ( mh instanceof HashSHA )
				out.writeByte( SHA_IDENT );
			else
				throw new IOException( "Hash type (" + mh + ") not recognized" );
			out.write( mh.toByteArray() );
		}
	}
	
	public static final String
	toString( MessageHash mh )
	{
		if ( mh == null )
			return "NONE:";
		return mh.toString();
	}
	
	public static final MessageHash
	fromString( String str )
	{
		if ( str.startsWith( "MD5" ) )
			return new HashMD5( ByteArrayFactory.fromHexString( str.substring( 4 ) ) );
		else if( str.startsWith( "SHA" ) )
			return new HashSHA( ByteArrayFactory.fromHexString( str.substring( 4 ) ) );
		else if( str.startsWith( "NONE" ) )
			return null;
		else
			throw new RuntimeException( "string is not a hash" + str );
	}

	//
	//	Hack - can't have colons in DOS, remember?
	//
	public static final String
	toFilename( MessageHash mh )
	{
		if ( mh == null )
			return "MT";
		return mh.toString().replace( ':', '_' );
	}
}
