/* rsaglue.c - These functions wrap and unwrap message digests (MDs) and
 * data encryption keys (DEKs) in padding and RSA-encrypt them into
 * multi-precision integers.  This layer was introduced to allow the
 * transparent use of RSAREF for the encryption (in regions where it is
 * legally available - the U.S. and treaty partners), or Philip Zimmermann's
 * mpi library (where permitted by patent law - outside the U.S.).
 *
 * Compile with -DRSAREF or -DUSA_ONLY to use the RSAREF library.
 */

#include <string.h> 	/* for mem*() */
#include <assert.h>
#include "mpilib.h"
#include "mpiio.h"
#include "pgp.h"
#include "rsaglue.h"
#include "random.h"	/* for cryptRandByte() */
#include "language.h"	/* for _LANG() */

#ifdef USA_ONLY
#define RSAREF
#endif

#ifdef RSAREF
char signon_legalese[] =
_LANG("Distributed by the Massachusetts Institute of Technology.  \
Uses RSAREF.\n");
#else
char signon_legalese[] =
_LANG("International version - not for use in the USA.  \
Does not use RSAREF.\n");
#endif

/* These functions hide all the internal details of RSA-encrypted
 * keys and digests.  They owe a lot of their heritage to
 * the preblock() and postunblock() routines in mpiio.c.
 */

#ifdef RSAREF

#include <global.h>
#include <rsaref.h>
#include <rsa.h>
/*
 * The functions we call in rsa.h are:
 *
 * int RSAPublicEncrypt PROTO_LIST
 *   ((unsigned char *, unsigned int *, unsigned char *, unsigned int,
 *     R_RSA_PUBLIC_KEY *, R_RANDOM_STRUCT *));
 * int RSAPrivateEncrypt PROTO_LIST
 *   ((unsigned char *, unsigned int *, unsigned char *, unsigned int,
 *     R_RSA_PRIVATE_KEY *));
 * int RSAPublicDecrypt PROTO_LIST
 *   ((unsigned char *, unsigned int *, unsigned char *, unsigned int,
 *     R_RSA_PUBLIC_KEY *));
 * int RSAPrivateDecrypt PROTO_LIST
 *   ((unsigned char *, unsigned int *, unsigned char *, unsigned int,
 *     R_RSA_PRIVATE_KEY *));
 */

/* Functions to convert to and from RSAREF's bignum formats */

void
rsaref2reg (unitptr to, byte *from, int frombytes)
/* Convert an RSAREF-style MSB-first array of bytes to an mpi-style
 * native-byte-order integer.  (global_precision units long.)
 */
{
	int tobytes;

	tobytes = units2bytes (global_precision);
	if (tobytes > frombytes) {
		memset(to, 0, tobytes - frombytes);
		memcpy((byte *)to + tobytes - frombytes, from, frombytes);
	} else {
		memcpy((byte *)to, from + frombytes - tobytes, tobytes);
	}
#ifndef HIGHFIRST
	hiloswap((byte *)to, tobytes);
#endif
} /* rsaref2reg */

void
reg2rsaref (byte *to, int tobytes, unitptr from)
/* Convert the other way, mpi format to an array of bytes. */
{
	int frombytes;

	frombytes = units2bytes(global_precision);

#ifdef HIGHFIRST
	if (tobytes > frombytes) {
		memset(to, 0, tobytes-frombytes);
		memcpy(to + tobytes - frombytes, (byte *)from, frombytes);
	} else {
		memcpy(to, (byte *)from + frombytes - tobytes, tobytes);
	}
#else
	if (tobytes > frombytes) {
		memcpy(to, (byte *)from, frombytes);
		memset(to + frombytes, 0, tobytes-frombytes);
	} else {
		memcpy(to, (byte *)from, tobytes);
	}
	hiloswap(to, tobytes);
#endif
} /* reg2rsaref */

int
make_RSA_PUBLIC_KEY(R_RSA_PUBLIC_KEY *rpk, unitptr e, unitptr n)
/* Given mpi's e and n, fill in an R_RSA_PUBLIC_KEY structure.
 * Returns -3 on error (key too big), 0 on success
 */
{
	rpk->bits = countbits(n);

	if (rpk->bits > MAX_RSA_MODULUS_BITS)
		return -3;

	reg2rsaref(rpk->modulus, MAX_RSA_MODULUS_LEN, n);
	reg2rsaref(rpk->exponent, MAX_RSA_MODULUS_LEN, e);
	return 0;
} /* make_RSA_PUBLIC_KEY */

/* Returns -1 on error, 0 on success */
int
make_RSA_PRIVATE_KEY(R_RSA_PRIVATE_KEY *rpk, unitptr e, unitptr d, unitptr p,
		     unitptr q, unitptr dp, unitptr dq, unitptr u, unitptr n)
/* Given a number of necessary mpi's, fill in an R_RSA_PRIVATE_KEY structure.
 * Returns -3 on error (key too big), 0 on success
 */
{
	rpk->bits = countbits(n);

	if (rpk->bits > MAX_RSA_MODULUS_BITS ||
	    countbits(p) > MAX_RSA_PRIME_BITS ||
	    countbits(q) > MAX_RSA_PRIME_BITS)
		return -3;

	reg2rsaref(rpk->modulus, MAX_RSA_MODULUS_LEN, n);
	reg2rsaref(rpk->publicExponent, MAX_RSA_MODULUS_LEN, e);
	reg2rsaref(rpk->exponent, MAX_RSA_MODULUS_LEN, d);
	/* The larger prime (p) first */
	reg2rsaref(rpk->prime[0], MAX_RSA_PRIME_LEN, q);
	reg2rsaref(rpk->prime[1], MAX_RSA_PRIME_LEN, p);
	/* d mod (p-1) and d mod (q-1) */
	reg2rsaref(rpk->primeExponent[0], MAX_RSA_PRIME_LEN, dq);
	reg2rsaref(rpk->primeExponent[1], MAX_RSA_PRIME_LEN, dp);
	/* 1/q mod p */
	reg2rsaref(rpk->coefficient, MAX_RSA_PRIME_LEN, u);
	return 0;
} /* make_RSA_PRIVATE_KEY */

/*
 * These functions hide all the internal details of RSA-encrypted
 * keys and digests.
 */

/* Abstract Syntax Notation One (ASN.1) Distinguished Encoding Rules (DER)
   encoding for RSA/MD5, used in PKCS-format signatures. */
static byte asn_array[] = {	/* PKCS 01 block type 01 data */
	0x30,0x20,0x30,0x0c,0x06,0x08,0x2a,0x86,0x48,0x86,0xf7,0x0d,
	0x02,0x05,0x05,0x00,0x04,0x10 };
/* This many bytes from the end, there's a zero byte */
#define ASN_ZERO_END 3

int
rsa_public_encrypt(unitptr outbuf, byteptr inbuf, short bytes,
	 unitptr E, unitptr N)
/* Encrypt a DEK with a public key.  Returns 0 on success.
 * <0 means there was an error.
 * -1: Generic error
 * -3: Key too big
 * -4: Key too small
 */
{
	unit temp[MAX_UNIT_PRECISION];
	unsigned int blocksize;
	int i;	/* Temporary, and holds error codes */
	R_RSA_PUBLIC_KEY PubKey;
	R_RANDOM_STRUCT Random;

	/* Fill in the R_RSA_PUBLIC_KEY structure as needed later. */
	i = make_RSA_PUBLIC_KEY(&PubKey, E, N);
	if (i < 0)
		return i;

	/* The RSAREF routines have their own random number generator
	 * to generate random padding.  The following code seeds it
	 * from PGP's random number generator.
	 */
	R_RandomInit(&Random);
	for (;;) {
		/* Bytes needed is an unsigned int */
		R_GetRandomBytesNeeded(&blocksize, &Random);
		if (!blocksize)
			break;
		if (blocksize > sizeof(temp))
			blocksize = sizeof(temp);
		for (i = 0; i < blocksize; i++)
			((byte *)temp)[i] = cryptRandByte();
		R_RandomUpdate(&Random, (byte *)temp, blocksize);

	}
	/* Pad and encrypt */
	i = RSAPublicEncrypt((byte *)temp, &blocksize,
			     inbuf, bytes, &PubKey, &Random);
	R_RandomFinal(&Random);	/* Clean up RSAREF's RNG */
	burn(Random);		/* Just to be sure */

	if (i)
		i = (i == RE_LEN) ? -4 : -1;

	rsaref2reg(outbuf, (byte *)temp, blocksize);

Cleanup:
	mp_burn(temp);
	burn(PubKey);
	return i < 0 ? i : 0;
} /* rsa_public_encrypt */

int
rsa_private_encrypt(unitptr outbuf, byteptr inbuf, short bytes,
	 unitptr E, unitptr D, unitptr P, unitptr Q, unitptr U, unitptr N)
/* Encrypt a message digest with a private key.
 * Returns <0 on error:
 * -1: generic error
 * -3: Key too big
 * -4: Key too small
 */
{
	unit temp[MAX_UNIT_PRECISION];
	unit DP[MAX_UNIT_PRECISION], DQ[MAX_UNIT_PRECISION];
	R_RSA_PRIVATE_KEY PrivKey;
	byte *p;
	int i;
	unsigned int blocksize;

	/* PGP doesn't store these coefficents, so we need to compute them. */
	mp_move(temp,P);
	mp_dec(temp);
	mp_mod(DP,D,temp);
	mp_move(temp,Q);
	mp_dec(temp);
	mp_mod(DQ,D,temp);

	p = (byte *)temp;

	i = make_RSA_PRIVATE_KEY(&PrivKey, E, D, P, Q, DP, DQ, U, N);
	if (i < 0)
		goto Cleanup;
	memcpy(p, asn_array, sizeof(asn_array)); /* ASN data */
	p += sizeof(asn_array);
	memcpy(p, inbuf, bytes);	/* User data */
	/* Pad and encrypt */
	i = RSAPrivateEncrypt((byte *)temp, &blocksize,
	                      (byte *)temp, bytes+sizeof(asn_array), &PrivKey);
	burn(PrivKey);
	if (i)
		i = (i == RE_LEN) ? -4 : -1;

	rsaref2reg(outbuf, (byte *)temp, blocksize);

Cleanup:
	burn(temp);

	return i;
} /* rsa_private_encrypt */

/* Remove a signature packet from an MPI */
/* Thus, we expect constant padding and the MIC ASN sequence */
int
rsa_public_decrypt(byteptr outbuf, unitptr inbuf,
	unitptr E, unitptr N)
/* Decrypt a message digest using a public key.  Returns the number of bytes
 * extracted, or <0 on error.
 * -1: Corrupted packet.
 * -3: Key too big
 * -4: Key too small
 * -5: Maybe malformed RSA packet
 * -7: Unknown conventional algorithm
 * -9: Malformed RSA packet
 */
{
	R_RSA_PUBLIC_KEY PubKey;
	unit temp[MAX_UNIT_PRECISION];
	unsigned int blocksize;
	int i;
	byte *front, *back;

	i = make_RSA_PUBLIC_KEY(&PubKey, E, N);
	if (i < 0)
		return i;
	blocksize = countbytes(inbuf);
	reg2rsaref((byte *)temp, blocksize, inbuf);

	i = RSAPublicDecrypt((byte *)temp, &blocksize,
			     (byte *)temp, blocksize, &PubKey);
	burn(PubKey);
	if (i) {
		mp_burn(temp);
		if (i == RE_LEN)
			return -4;
		if (i == RE_DATA)
			return -5;
		return -1;
	}
	front = (byte *)temp;
	back = front+blocksize;

	if (memcmp(front, asn_array, sizeof(asn_array))) {
		mp_burn(temp);
		return -7;
	}
	front += sizeof(asn_array);

/* We're done - copy user data to outbuf */
	if (back < front)
		goto ErrorReturn;
	blocksize = back-front;
	memcpy(outbuf, front, blocksize);
	mp_burn(temp);
	return blocksize;
ErrorReturn:
	mp_burn(temp);
	return -9;
} /* rsa_public_decrypt */

/* We expect to find random padding and an encryption key */
int
rsa_private_decrypt(byteptr outbuf, unitptr inbuf,
	 unitptr E, unitptr D, unitptr P, unitptr Q, unitptr U, unitptr N)
/* Decrypt an encryption key using a private key.  Returns the number of bytes
 * extracted, or <0 on error.
 * -1: Generic error
 * -3: Key too big
 * -4: Key too small
 * -5: Maybe malformed RSA
 * -7: Unknown conventional algorithm
 * -9: Malformed RSA packet
 */
{
	R_RSA_PRIVATE_KEY PrivKey;
	byte *front;
	unsigned int blocksize;
	unit temp[MAX_UNIT_PRECISION];
	unit DP[MAX_UNIT_PRECISION], DQ[MAX_UNIT_PRECISION];
	int i;

	/* PGP doesn't store (d mod p-1) and (d mod q-1), so compute 'em */
	mp_move(temp,P);
	mp_dec(temp);
	mp_mod(DP,D,temp);
	mp_move(temp,Q);
	mp_dec(temp);
	mp_mod(DQ,D,temp);

	i = make_RSA_PRIVATE_KEY(&PrivKey, E, D, P, Q, DP, DQ, U, N);
	mp_burn(DP);
	mp_burn(DQ);
	mp_burn(temp);
	
	if (i < 0)
		return i;

	blocksize = countbytes(inbuf);
	reg2rsaref((byte *)temp, blocksize, inbuf);
	i = RSAPrivateDecrypt((byte *)temp, &blocksize,
			      (byte *)temp, blocksize, &PrivKey);
	burn(PrivKey);
	if (i) {
		if (i == RE_LEN)
			return -4;
		if (i == RE_DATA)
			return -5;
		return -1;
	}
	front = (byte *)temp;			/* Start of block */

	memcpy(outbuf, front, blocksize);
	mp_burn(temp);
	return blocksize;

Corrupted:
	mp_burn(temp);
	return -9;
} /* rsa_private_decrypt */

/*
 * Stub to replace RSAREF's NN_ModExp with a call to the mpilib's
 * faster mp_modexp.  (A bit over 3x faster on an IBM PC.)
 * It's too bad that RSAREF's NN routines are pretty clean, while the
 * mpilib is a hard-to-follow kludge.  But the mpilib is pretty fast,
 * especially on 16-bit machines, so the kludginess is forgivable.
 * Note that we are still using RSAREF, but we are just using a faster
 * modulo exponentiation routine.
 * If you comment out the following block of code, you get a (much slower)
 * pure RSAREF version.
 */
#ifdef USEMPILIB

/*
 * The mpilib keeps numbers in native byte order, in arrays global_precision
 * "units" long.  RSAREF keeps numbers in little-endian arrays of 32-bit
 * "digits".
 */
static void
nn2mpi(unit *mpi, word32 *nn, unsigned nndigits)
{
	/* nndigits must be <= global_precision */
	unsigned i;
	word32 *p;

	assert((units2bytes(global_precision) & 3) == 0);
	i = units2bytes(global_precision) >> 2;
	if (nndigits > i)
		nndigits = i;
	i -= nndigits;

#ifdef HIGHFIRST
	p = (word32 *)(mpi+global_precision);
	while (nndigits--)
		*--p = *nn++;
	while (i--)
		*--p = 0;
#else
	p = (word32 *)mpi;
	while (nndigits--)
		*p++ = *nn++;
	while (i--)
		*p++ = 0;
#endif
}

static void
mpi2nn(word32 *nn, unsigned nndigits, unit *mpi)
{
	/* nndigits must be >= global_precision */
	unsigned i;
	word32 *p;

	assert((units2bytes(global_precision) & 3) == 0);
	i = units2bytes(global_precision) >> 2;
	if (i > nndigits)
		i = nndigits;
	nndigits -= i;

#ifdef HIGHFIRST
	p = (word32 *)(mpi+global_precision);
	while (i--)
		*nn++ = *--p;
#else
	p = (word32 *)mpi;
	while (i--)
		*nn++ = *p++;
#endif
	while (nndigits--)
		*nn++ = 0;
}

void
NN_ModExp(word32 *result, word32 *base, word32 *exponent, unsigned expdigits,
	  word32 *modulus, unsigned moddigits)
{
	unit a[MAX_UNIT_PRECISION], b[MAX_UNIT_PRECISION];
	unit c[MAX_UNIT_PRECISION], d[MAX_UNIT_PRECISION];
	int i;
	unsigned oldprecision;

	oldprecision = global_precision;
	set_precision(MAX_UNIT_PRECISION);

	nn2mpi(b, base, moddigits);
	nn2mpi(c, exponent, expdigits);
	nn2mpi(d, modulus, moddigits);

	i = mp_modexp(a, b, c, d);
	assert(i == 0);
	mpi2nn(result, moddigits, a);
	set_precision(oldprecision);
}

#endif /* USEMPILIB */

#else /* !RSAREF */

/* Abstract Syntax Notation One (ASN.1) Distinguished Encoding Rules (DER)
   encoding for RSA/MD5, used in PKCS-format signatures. */
static byte asn_array[] = {	/* PKCS 01 block type 01 data */
	0x30,0x20,0x30,0x0c,0x06,0x08,0x2a,0x86,0x48,0x86,0xf7,0x0d,
	0x02,0x05,0x05,0x00,0x04,0x10 };
/* This many bytes from the end, there's a zero byte */
#define ASN_ZERO_END 3

int
rsa_public_encrypt(unitptr outbuf, byteptr inbuf, short bytes,
	 unitptr E, unitptr N)
/* Encrypt a DEK with a public key.  Returns 0 on success.
 * <0 means there was an error
 */
{
	unit temp[MAX_UNIT_PRECISION];
	unsigned int blocksize;
	byte *p;
	int i;	/* Temporary, and holds error codes */

	blocksize = countbytes(N) - 1;	/* Bytes available for user data */

	p = (byte *)temp;

	for (i = units2bytes(global_precision) - blocksize; i > 0; --i)
		*p++ = 0;

	/* Both the PKCS and PGP 2.0 key formats add a type byte, and a
	 * a framing byte of 0 to the user data.  The remaining space
	 * is filled with random padding.  (PKCS requires that there be
	 * at least 1 byte of padding.)
	 */
	i = blocksize - 2 - bytes;

	if (pkcs_compat) {
		if (i < 1)		/* Less than minimum padding? */
			return -1;
		*p++ = CK_ENCRYPTED_BYTE;	/* Type byte */
		while (i)			/* Non-zero random padding */
			if ((*p = cryptRandByte()))
				++p, --i;
		*p++ = 0;			/* Framing byte */
		memcpy(p, inbuf, bytes);	/* User data */
	} else {	/* !pkcs_compat */
		if (i < 0)
			return -1;
		memcpy(p, inbuf, bytes);	/* User data */
		p += bytes;
		*p++ = 0;			/* Framing byte */
		while (i)			/* Non-zero random padding */
			if ((*p = cryptRandByte()))
				++p, --i;
		*p = CK_ENCRYPTED_BYTE;		/* Type byte */

	} /* !pkcs_compat */

	mp_convert_order((byte *)temp);		/* Convert buffer to MPI */
	i = mp_modexp(outbuf, temp, E, N);	/* Do the encryption */

Cleanup:
	mp_burn(temp);
	return i < 0 ? i : 0;
} /* rsa_public_encrypt */

int
rsa_private_encrypt(unitptr outbuf, byteptr inbuf, short bytes,
	 unitptr E, unitptr D, unitptr P, unitptr Q, unitptr U, unitptr N)
/* Encrypt a message digest with a private key.
 * Returns <0 on error.
 */
{
	unit temp[MAX_UNIT_PRECISION];
	unit DP[MAX_UNIT_PRECISION], DQ[MAX_UNIT_PRECISION];
	int byte_precision;
	unsigned int blocksize;
	byte *p;
	int i;

	/* PGP doesn't store these coefficents, so we need to compute them. */
	mp_move(temp,P);
	mp_dec(temp);
	mp_mod(DP,D,temp);
	mp_move(temp,Q);
	mp_dec(temp);
	mp_mod(DQ,D,temp);

	blocksize = countbytes(N) - 1;	/* Space available for data */

	p = (byte *)temp;

	for (i = units2bytes(global_precision) - blocksize; i > 0; --i)
		*p++ = 0;
	i = blocksize - 2 - bytes;		/* Padding needed */

	if (pkcs_compat) {
		i -= sizeof(asn_array);		/* Space for type encoding */
		if (i < 0) {
			i = -1;			/* Error code */
			goto Cleanup;
		}
		*p++ = MD_ENCRYPTED_BYTE;	/* Type byte */
		memset(p, ~0, i);		/* All 1's padding */
		p += i;
		*p++ = 0;			/* Zero framing byte */
		memcpy(p, asn_array, sizeof(asn_array)); /* ASN data */
		p += sizeof(asn_array);
		memcpy(p, inbuf, bytes);	/* User data */
	} else {	/* Not pkcs_compat */
		--i;				/* Space for type byte */
		if (i < 0) {
			i = -1;			/* Error code */
			goto Cleanup;
		}
		*p++ = MD5_ALGORITHM_BYTE;	/* Algorithm type byte */
		memcpy(p, inbuf, bytes);	/* User data */
		p += bytes;
		*p++ = 0;			/* Framing byte of 0 */
		memset(p, ~0, i);		/* All 1's padding */
		p += i;
		*p = MD_ENCRYPTED_BYTE;		/* Type byte */


	} /* !pkcs_compat */

	mp_convert_order((byte *)temp);
	i = mp_modexp_crt(outbuf, temp, P, Q, DP, DQ, U);	/* Encrypt */

Cleanup:
	mp_burn(temp);
	return i < 0 ? i : 0;
} /* rsa_private_encrypt */

/* Remove a signature packet from an MPI */
/* Thus, we expect constant padding and the MIC ASN sequence */
int
rsa_public_decrypt(byteptr outbuf, unitptr inbuf,
	unitptr E, unitptr N)
/* Decrypt a message digest using a public key.  Returns the number of bytes
 * extracted, or <0 on error.
 * -1: Corrupted packet.
 * -2: Unrecognized message digest algorithm.
 */
{
	unit temp[MAX_UNIT_PRECISION];
	unsigned int blocksize;
	int i;
	byte *front, *back;

	i = mp_modexp(temp, inbuf, E, N);
	if (i < 0) {
		mp_burn(temp);
		return i;
	}
	mp_convert_order((byte *)temp);
	blocksize = countbytes(N) - 1;
	front = (byte *)temp;			/* Points to start of block */
	i = units2bytes(global_precision);
	back = front + i;			/* Points to end of block */
	i -= countbytes(N) - 1;			/* Expected leading 0's */

	/* This could be stricter, but the length returned by the RSAREF code
	   is not documented too well. */
	if (i < 0)
		goto ErrorReturn;
	while (i--)		/* Any excess should be 0 */
		if (*front++)
			goto ErrorReturn;
	/* front now points to the data */

	/* How to distinguish old PGP from PKCS formats.
	 * The old PGP format ends in a trailing type byte, with
	 * all 1's padding before that.  The PKCS format ends in
	 * 16 bytes of message digest, preceded by an ASN string
	 * which is not all 1's.
	 */
	if (back[-1] == MD_ENCRYPTED_BYTE &&
	    back[-17] == 0xff && back[-18] == 0xff) {
		/* Old PGP format: Padding is at the end */
		if (*--back != MD_ENCRYPTED_BYTE)
			goto ErrorReturn;
		if (*front++ != MD5_ALGORITHM_BYTE) {
			mp_burn(temp);
			return -2;
		}
		while (*--back == 0xff)	/* Skip constant padding */
			;
		if (*back)		/* It should end with a zero */
			goto ErrorReturn;
	} else {
		/* PKCS format: padding at the beginning */
		if (*front++ != MD_ENCRYPTED_BYTE)
			goto ErrorReturn;
		while (*front++ == 0xff) /* Skip constant padding */
			;
		if (front[-1])	/* First non-FF byte should be 0 */
			goto ErrorReturn;
		/* Then comes the ASN header */
		if (memcmp(front, asn_array, sizeof(asn_array))) {
			mp_burn(temp);
			return -2;
		}
		front += sizeof(asn_array);
/* This is temporary - to be removed on release */
		if (back-front == 17 && *front == MD5_ALGORITHM_BYTE) {
			front++;
			fprintf(stderr, "PGP 2.2b signature bug!\n");
		}
	}
/* We're done - copy user data to outbuf */
	if (back < front)
		goto ErrorReturn;
	memcpy(outbuf, front, back-front);
	mp_burn(temp);
	return back-front;
ErrorReturn:
	mp_burn(temp);
	return -1;
} /* rsa_public_decrypt */

/* We expect to find random padding and an encryption key */
int
rsa_private_decrypt(byteptr outbuf, unitptr inbuf,
	 unitptr E, unitptr D, unitptr P, unitptr Q, unitptr U, unitptr N)
/* Decrypt an encryption key using a private key.  Returns the number of bytes
 * extracted, or <0 on error.
 * -1: Corrupted packet.
 */
{
	unsigned int blocksize;
	unit temp[MAX_UNIT_PRECISION];
	unit DP[MAX_UNIT_PRECISION], DQ[MAX_UNIT_PRECISION];
	byte *front, *back;
	int i;

	mp_move(temp,P);
	mp_dec(temp);
	mp_mod(DP,D,temp);
	mp_move(temp,Q);
	mp_dec(temp);
	mp_mod(DQ,D,temp);

	i = mp_modexp_crt(temp, inbuf, P, Q, DP, DQ, U);
	if (i < 0)
		goto ErrorReturn;
	mp_convert_order((byte *)temp);
	front = (byte *)temp;			/* Start of block */
	i = units2bytes(global_precision);
	back = (byte *)front + i;		/* End of block */
	blocksize = countbytes(N) - 1;
	i -= blocksize;				/* Expected # of leading 0's */
	if (i < 0)				/* This shouldn't happen */
		goto ErrorReturn;
	while (i--)				/* Extra bytes should be 0 */
		if (*front++)
			goto ErrorReturn;

	/* How to distinguish old PGP from PKCS formats.
	 * PGP packets have a trailing type byte (CK_ENCRYPTED_BYTE),
	 * while PKCS formats have it leading.
	 */
	if (front[0] != CK_ENCRYPTED_BYTE && back[-1] == CK_ENCRYPTED_BYTE) {
		/* PGP 2.0 format  - padding at the end */
		if (back[-1] != CK_ENCRYPTED_BYTE)
			goto ErrorReturn;
		while (*--back)	/* Skip non-zero random padding */
			;
	} else {
		/* PKCS format - padding at the beginning */
		if (*front++ != CK_ENCRYPTED_BYTE)
			goto ErrorReturn;
		while (*front++)	/* Skip non-zero random padding */
			;
	}
	if (back < front)
		goto ErrorReturn;
	memcpy(outbuf, front, back-front);
	mp_burn(temp);
	mp_burn(DP);
	mp_burn(DQ);
	return back-front;

ErrorReturn:
	mp_burn(temp);
	mp_burn(DP);
	mp_burn(DQ);
	return -1;
} /* rsa_private_decrypt */

#endif
