;-----------------------------------------------------------------------;
; COMPAND.ASM                                                           ;
; (C) 1993                                                              ;
; Walla Walla College                                                   ;
; By: Fred Christensen                                                  ;
;                                                                       ;
;                                                                       ;
; Hardware required:  TI TMS320 DSP Starter Kit (DSK)                   ;
;    Software files:  COEF.ASM (FIR filter coefficients)                ;
;                     EXP.ASM (cosine & sine lookup tables)             ;
;                     DSKA.EXE (Compiler)                               ;
;                     DSKD.EXE (Debugger interface)                     ;
;                     README (Operating/Hookup instructions)            ;
;                                                                       ;
; Program to input an audio signal using the AIC A/D                    ;
; converter, compand or decompand the frequencies, and                  ;
; output it to the AIC D/A converter.                                   ;
;-----------------------------------------------------------------------;
;
;           Memory Management
; ______________________________________________________________________________
; B2   | 0060h - 007Fh | Data Memory   | Used by DSKD program
; B0   | FA00h - FAFFh | Program Memory| Used by DSKD program
;      | FB00h - FBCFh | Program Memory| Compand program
;      | FBD0h - FBFFh | Program Memory| Filter1 coefficients
; B1   | FC00h - FC33h | Program Memory| Filter1 coefficients
;      | FC34h - FC97h | Program Memory| Filter2 transmit (real) coefficients
;      | FC98h - FCFBh | Program Memory| Filter2 transmit (imag) coefficients
;      | FCFCh - FD5Fh | Program Memory| Filter2 receive (real) coefficients
;      | FD60h - FDC3h | Program Memory| Filter2 receive (imag) coefficients
;      | FDC4h - FDFFh | Program Memory| Unused
; B3   | 0600h - 0609h | Data Memory   | Variables used by Compand program
;      | 06A0h - 06AFh | Data Memory   | Cosine lookup table
;      | 06B0h - 06BFh | Data Memory   | Sine lookup table
;      | 06C0h - 079Ah | Data Memory   | Unused
;      | 079Bh - 07FFh | Data Memory   | Sampling buffer for FIR filtering
;       
;-----------------------------------------------------------------------;
;    Define Global Constants                                            ;
;-----------------------------------------------------------------------;
YES             .set    1               ;
NO              .set    0               ;
TRANSMIT        .set    1               ;
RECEIVE         .set    0               ;
TR_DEFAULT      .set    TRANSMIT        ; Set transmit/receive flag
NUMTAPS         .set    99              ; Number of filter taps  - 1.
LUPSIZE         .set    16              ; Size of lookup table 

;-----------------------------------------------------------------------;
;    Setup Memory Mapped Registers (MMRs)                               ;
;-----------------------------------------------------------------------;
DRR             .set    0       ; Serial port receive register (from A/D)
DXR             .set    1       ; Serial port transmit register (to D/A)
IMR             .set    4       ; Interrupt Mask Register
GREG            .set    5       ; Global memory allocation register
	
;-----------------------------------------------------------------------;
;    Analog Interface Circuit (AIC) Constants                           ;
;                                                                       ;
;       These are used to configure the TLC32040 AIC for                ;
;       desired operations.                                             ;
;-----------------------------------------------------------------------;  

AIC_1           .set    03060h  ;TA =RA = 24 0011000001100000 = 03060h
AIC_2           .set    00205h  ;TA'=RA'= 1  0000001000000101 = 00205h
AIC_3           .set    02a56h  ;RB =TB = 21 0010101001010110 = 02a56h 10khz
AIC_CMD         .set    00007h  ;   COMMAND  0000000000000111 = 00007h Filter

;-----------------------------------------------------------------------;
;    Define Compander Data Memory Variables (All in block B3)           ;
;-----------------------------------------------------------------------;

;  Note:  The naming convention follows the pseudocode with the addition
;         of _R or _I  for real or imaginary variables, respectively.

ANSR1_R         .set    0600h   ; Real output of Filter #1, temporary variable
ANSR2_R         .set    0601h   ; Real output of Filter #2, temporary variable
ANSR2_I         .set    0602h   ; Imag output of Filter #2, temporary variable
TR_FLAG         .set    0603h   ; Transmit/Receive flag 
EXPPTR_R        .set    0604h   ; Pointer to elements in EXP_R array 
EXPPTR_I        .set    0605h   ; Pointer to elements in EXP_I array 
BUFFER          .set    079Bh   ; Buffer of previous inputs
BUFFER_END      .set    07FEh   ;   Oldest data sample in BUFFER

;-----------------------------------------------------------------------;
;    Secondary interrupt vector table located in B0 program RAM         ;
;-----------------------------------------------------------------------;
	.ps     0fa00h          ;     > USERCODE SHOULD NOT OVERWRITE DSKD  <
	b       START           ;RS   > VECTORS.  ON LOAD, INT2 IS RESTORED <
	b       START           ;INT0 > BY DSKD, BUT TRAP IS NOT
	b       START           ;INT1
	b       START           ;INT2  > DSKD LOAD IGNORES INT2 VECTOR
	b       START           ;TINT
	b       RINT            ;RINT  Branch to receive interrupt routine
	eint                    ;XINT  XINT is only for timing, so just return
	ret                     ;
      ; Begin TRAP/DSKD Kernal  ;DSKD load does not restore this code!

;-----------------------------------------------------------------------;
;    Application code is located after DSKD kernal                      ;
;-----------------------------------------------------------------------;
	.ps     0fb00h          ;
	.entry                  ;
START:  spm     0               ; Sets P shift mode to no shift
	ldpk    0               ; Set direct addressing to MMRs and B2
	fort    0               ; Serial port : 16 bit
	rtxm                    ;             : ext. FSX
	sfsm                    ;             ; burst mode
	lack    080h            ; AIC reset by pulsing /BR (Global Data)
	sach    DXR             ; send 0 to DXR (AIC)
	sacl    GREG            ; 256 * 100 nS /BR pulse
	lrlk    AR0,0FFFFh      ;
	rptk    255             ; read junk from address 0xFFFF
	lac     *,0,AR0         ;
	conf    2               ; B0,B3 as DRAM if direct bootload

				;  ***** INITIALIZE *****
	
	ldpk    12              ; Set addressing to compander data memory
	lack    TR_DEFAULT      ; Load & store transmit/receive status
	sacl    TR_FLAG         ; 
	lrlk    AR5,EXP_R       ; Load & store pointer cosine 
	sar     AR5,EXPPTR_R    ;    lookup table       
	lrlk    AR6,EXP_I       ; Load & store pointer to sine
	sar     AR6,EXPPTR_I    ;    lookup table       
	ldpk    0               ; Set direct addressing to MMRs and B2  
	sovm                    ; catch accumulator overflows

;-----------------------------------------------------------------------;       
;    Load the configuration commands into the AIC                       ;
;-----------------------------------------------------------------------;
AIC_RS  lack    024h            ; Turn on XINT
	sacl    IMR             ;
	idle                    ;
	lalk    AIC_1           ; Load each AIC configuration word
	call    AIC_2nd         ; and load it into the AIC
	lalk    AIC_2           ;
	call    AIC_2nd         ;
	lalk    AIC_3           ;
	call    AIC_2nd         ;
	lalk    AIC_CMD         ;
	call    AIC_2nd         ;
	ssxm                    ;
	lack    014h            ; AIC RINT
	sacl    IMR             ; where INT0 indicates EOC (End Of Conv)

;-----------------------------------------------------------------------;
;    Main program loop                                                  ;
;-----------------------------------------------------------------------;
LOOPING idle                    ; Infinite loop waiting for interrupts
	b       LOOPING  

;-----------------------------------------------------------------------;
;    RINT Interrupt handler                                             ;
;-----------------------------------------------------------------------;
RINT:   dint                    ; disable interrupts

				;  ***** INPUT *****

	ldpk    12              ; Set addressing to compander data memory
	lar     AR5,EXPPTR_R    ; Load cosine lookup table pointer
	lar     AR6,EXPPTR_I    ; Load sine lookup table pointer        
	ldpk    0               ; Set addressing to MMRs and B2
	lac     DRR,0           ; Load ACC with A/D from AIC
	ldpk    15              ; Set addressing to filter buffer
	sacl    BUFFER          ; Store data in first filter buffer location

				;  ***** FILTER #1 *****

	call    FIRINIT         ; FIR initialization 
	larp    AR7             ;
	rptk    NUMTAPS         ; Compute FIR output for filter #1 
	mac     COEF1_R,*-,AR7  ;
	apac                    ;
	ldpk    12              ; Set addressing to compander data memory 
	sach    ANSR1_R,0       ; Divide by 2^16 and store.

				;  ***** FILTER #2 *****

	lac     TR_FLAG         ; Load Receive/Transmit flag
	bz      RECV            ; Branch if receive mode (TR_FLAG=0)
TRANS   call    FIRINIT         ; FIR initialization
	rptk    NUMTAPS         ; Compute FIR output for filter #2 (real) 
	mac     COEF2T_R,*-,AR7 ;
	apac                    ;
	ldpk    12              ; Set addressing to compander data memory 
	sach    ANSR2_R,0       ; Divide by 2^16 and store.

	call    FIRINIT         ; FIR initialization
	rptk    NUMTAPS         ; Compute FIR output for filter #2 (imag)
	macd    COEF2T_I,*-     ;    with buffer shift
	apac                    ;
	ldpk    12              ; Set addressing to compander data memory
	sach    ANSR2_I,0       ; Divide by 2^16 and store.     
	b       SHIFT           ; Branch to frequency shifting routine

RECV    call    FIRINIT         ; FIR initialization
	rptk    NUMTAPS         ; Compute FIR output for filter #2 (real) 
	mac     COEF2R_R,*-,AR7 ;
	apac                    ;
	ldpk    12              ; Set addressing to compander data memory 
	sach    ANSR2_R,0       ; Divide by 2^16 and store.

	call    FIRINIT         ; FIR initialization
	rptk    NUMTAPS         ; Compute FIR output for filter #2 (imag)
	macd    COEF2R_I,*-     ;    with buffer shift
	apac                    ;
	ldpk    12              ; Set addressing to compander data memory
	sach    ANSR2_I,0       ; Divide by 2^16 and store

				;  ***** SHIFT *****

SHIFT   zac                     ; Zero accumulator
	lt      ANSR2_R,0       ; Multiply output (real part) of Filter #2      
	larp    AR5
	mpy     *+,AR6          ;    by cos(2*pi*fo*t)
	apac
	sach    ANSR2_R,0       ; Divide by 2^16 and store result
	zac                     ; Zero ACC
	lt      ANSR2_I,0       ; Multiply output (imag part) of Filter #2
	mpy     *+,AR0          ;    by sin(2*pi*fo*t)
	apac
	sach    ANSR2_I,0       ; Divide by 2^16 and store result
	
	lrlk    AR0,EXP_R       ; Check if lookup pointers are at end
	adrk    LUPSIZE         ;    of lookup tables.  If so, then 
	larp    AR5             ;    reload them to point to beginning
	cmpr    0               ;   
	bbz     SHIFTUD         ;
	lrlk    AR5,EXP_R       ;
	lrlk    AR6,EXP_I       ;

SHIFTUD lac     TR_FLAG         ; Test if Receive or Transmit
	bnz     SHIFTDN         ; If Transmit then branch
SHIFTUP lac     ANSR2_R,0       ; Shift band up
	add     ANSR2_I,0       ; 
	b       SUM
SHIFTDN lac     ANSR2_I,0       ; Shift band down 
	neg
	add     ANSR2_R,0       ; 

				;  ***** COMBINE *****

SUM     add     ANSR1_R,0       ; Add filt#1 to shifted filt#2 output

				;  ***** OUTPUT *****

OUT     ldpk    0               ; Set addressing to MMRs and B2
	andk    0fffch          ; Prepare word for transmit to AIC
	sacl    DXR,0           ; Output ACC to AIC for D/A
	ldpk    12              ; Set addressing to compander data memory
	sar     AR5,EXPPTR_R    ; Store cosine table lookup pointer
	sar     AR6,EXPPTR_I    ; Store sine table lookup pointer       
	eint                    ; enable interrupts
	ret                     ;




;-----------------------------------------------------------------------;
;    Routine which sends a signal to the AIC requesting permission      ;
;    to send the configuration commands to it.                          ;
;-----------------------------------------------------------------------;
AIC_2nd adlk    6,15            ; set ACCU_hi = 3 for secondary XMIT
	idle                    ; Wait for a XINT
	sach    DXR             ;
	idle                    ; ACCU_hi requests 2nd XMIT
	sacl    DXR             ;
	idle                    ; ACCU_lo sets up registers
	sacl    DXR,2           ; close command with LSB = 00
	idle                    ;
	eint                    ;
	ret                     ;

;-----------------------------------------------------------------------;
;    Routine to initialize setup for FIR computations                   ;
;-----------------------------------------------------------------------;
FIRINIT zac                     ; Zero Accumulator
	mpyk    0               ; Zero product register
	ldpk    15              ; Set addressing to filter buffer memory
	lrlk    AR7,BUFFER_END  ; Load AR7 with end of buffer location
	ret



;-----------------------------------------------------------------------;
;  The following filter coefficients were obtained from Matlab,         ;
;  multiplied by (2^15 - 1) and rounded to the nearest integer value.   ;
;  See Matlab file for an explanation on how to compute these.          ;
;-----------------------------------------------------------------------;
	.include        "COEF.ASM"  ; Filter coefficients


;-----------------------------------------------------------------------;
;  Include the EXP lookup tables.  Note that:                           ;
;      Re( exp(-j*2*pi*fo*t) )  =  cos(2*pi*fo*t)                       ;
;      Re( exp(-j*2*pi*fo*t) )  =  cos(2*pi*fo*t)                       ;
;  These were generated in Matlab and the number of elements in these   ; 
;  tables is a function of the sampling rate and frequency shift (fo).  ;
;-----------------------------------------------------------------------;
	.ds             06a0h 
	.include        "EXP.ASM"   ; Exponential lookup tables




