#ifndef lint
static char sccsid[] = "@(#)j1.c	1.1	(ucb.beef)	10/1/89";
#endif	/* !defined(lint) */
/* 
 *  This packet computes first-order Bessel functions of the first and
 *    second kind (j1 and y1), for real arguments x, where 0 < x <= XMAX
 *    for y1, and |x| <= XMAX for j1.  It contains three function-type
 *    subprograms, j1, y1 and caljy1.  The calling statements for the
 *    primary entries are:
 * 
 *            y = j1(x)
 *    and
 *            y = y1(x),
 * 
 *    where the entry points correspond to the functions j1(x) and y1(x),
 *    respectively.  The routine caljy1() is intended for internal packet
 *    use only, all computations within the packet being concentrated in
 *    this one routine.  The function subprograms invoke  caljy1  with
 *    the statement
 *
 *            result = caljy1(x,jint);
 *
 *    where the parameter usage is as follows:
 * 
 *       Function                  Parameters for caljy1
 *        call              x             result          jint
 * 
 *       j1(x)       |x| <= XMAX          j1(x)           0
 *       y1(x)     0 < x <= XMAX          y1(x)           1
 * 
 *    The main computation uses unpublished minimax rational
 *    approximations for x <= 8.0, and an approximation from the 
 *    book  Computer Approximations  by Hart, et. al., Wiley and Sons, 
 *    New York, 1968, for arguments larger than 8.0   Part of this
 *    transportable packet is patterned after the machine-dependent
 *    FUNPACK program for j1(x), but cannot match that version for
 *    efficiency or accuracy.  This version uses rational functions
 *    that are theoretically accurate to at least 18 significant decimal
 *    digits for x <= 8, and at least 18 decimal places for x > 8.  The
 *    accuracy achieved depends on the arithmetic system, the compiler,
 *    the intrinsic functions, and proper selection of the machine-
 *    dependent constants.
 * 
 *********************************************************************
 * 
 *  Explanation of machine-dependent constants
 * 
 *    XINF   = largest positive machine number
 *    XMAX   = largest acceptable argument.  The functions floor(), sin()
 *             and cos() must perform properly for  fabs(x) <= XMAX.
 *             We recommend that XMAX be a small integer multiple of
 *             sqrt(1/eps), where eps is the smallest positive number
 *             such that  1+eps > 1. 
 *    XSMALL = positive argument such that  1.0-(1/2)(x/2)**2 = 1.0
 *             to machine precision for all  fabs(x) <= XSMALL.
 *             We recommend that  XSMALL < sqrt(eps)/beta, where beta
 *             is the floating-point radix (usually 2 or 16).
 * 
 *      Approximate values for some important machines are
 * 
 *                           eps      XMAX     XSMALL      XINF  
 * 
 *   CDC 7600      (S.P.)  7.11E-15  1.34E+08  2.98E-08  1.26E+322
 *   CRAY-1        (S.P.)  7.11E-15  1.34E+08  2.98E-08  5.45E+2465
 *   IBM PC (8087) (S.P.)  5.96E-08  8.19E+03  1.22E-04  3.40E+38
 *   IBM PC (8087) (D.P.)  1.11e-16  2.68e08  3.72e-09  1.79e308
 *   IBM 195       (D.P.)  2.22e-16  6.87e09  9.09e-13  7.23e75
 *   UNIVAC 1108   (D.P.)  1.73e-18  4.30e09  2.33e-10  8.98e307
 *   VAX 11/780    (D.P.)  1.39e-17  1.07e09  9.31e-10  1.70e38
 * 
 *********************************************************************
 *********************************************************************
 * 
 *  Error Returns
 * 
 *   The program returns the value zero for  x > XMAX, and returns
 *     -XINF when BESLY1 is called with a negative or zero argument.
 * 
 * 
 *  Intrinsic functions required are:
 * 
 *      cos, fabs, floor, log, sin, sqrt
 * 
 * 
 *   Author: w. J. Cody
 *           Mathematics and Computer Science Division 
 *           Argonne National Laboratory
 *           Argonne, IL 60439
 * 
 *   Latest modification: November 10, 1987
 */

#include "fpumath.h"

					/* Machine-dependent constants */
#if defined(vax) || defined(tahoe)
#define	XMAX	(double)1.07e9
#define	XSMALL	(double)9.31e-10
#define	XINF	(double)1.7e38
#else	/* defined(vax) || defined(tahoe) */
#define	XMAX	(double)2.68e08
#define	XSMALL	(double)3.72e-09
#define	XINF	MAXFLOAT
#endif	/* defined(vax) || defined(tahoe) */
					/* Mathematical constants */
#define	EIGHT	(double)8
#define	FOUR	(double)4
#define	HALF	(double)0.5
#define	THROV8	(double)0.375
#define	PI2	(double)6.3661977236758134308e-1
#define	P17	(double)1.716e-1
#define	TWOPI	(double)6.2831853071795864769e0
#define	ZERO	(double)0
#define	TWOPI1	(double)6.28125
#define	TWOPI2	(double)1.9353071795864769253e-03
#define	TWO56	(double)256
#define	RTPI2	(double)7.9788456080286535588e-1
					/* Zeroes of Bessel functions */
#define	XJ0	(double)3.8317059702075123156e0
#define	XJ1	(double)7.0155866698156187535e0
#define	XY0	(double)2.1971413260310170351e0
#define	XY1	(double)5.4296810407941351328e0
#define	XJ01	(double)981
#define	XJ02	(double)(-3.2527979248768438556e-4)
#define	XJ11	(double)1796
#define	XJ12	(double)(-3.8330184381246462950e-5)
#define	XY01	(double)562
#define	XY02	(double)1.8288260310170351490e-3
#define	XY11	(double)1390
#define	XY12	(double)(-6.4592058648672279948e-6)

/*
 * Coefficients for rational approximation to ln(x/a)
 */
static double PLG[] = {
	-2.4562334077563243311e01,
	 2.3642701335621505212e02,
	-5.4989956895857911039e02,
	 3.5687548468071500413e02,
};
static double QLG[] = {
	-3.5553900764052419184e01,
	 1.9400230218539473193e02,
	-3.3442903192607538956e02,
	 1.7843774234035750207e02,
};

/*
 * Coefficients for rational approximation of
 * j1(x) / (x * (x**2 - XJ0**2)),  XSMALL  <  |x|  <=  4.0
 */
static double PJ0[] = {
	 9.8062904098958257677e05,
	-1.1548696764841276794e08,
	 6.6781041261492395835e09,
	-1.4258509801366645672e11,
	-4.4615792982775076130e03,
	 1.0650724020080236441e01,
	-1.0767857011487300348e-02,
};
static double QJ0[] = {
	5.9117614494174794095e05,
	2.0228375140097033958e08,
	4.2091902282580133541e10,
	4.1868604460820175290e12,
	1.0742272239517380498e03,
};

/*
 * Coefficients for rational approximation of
 * j1(x) / (x * (x**2 - XJ1**2)),  4.0  <  |x|  <=  8.0
 */
static double PJ1[] = {
	 4.6179191852758252280e00,
	-7.1329006872560947377e03,
	 4.5039658105749078904e06,
	-1.4437717718363239107e09,
	 2.3569285397217157313e11,
	-1.6324168293282543629e13,
	 1.1357022719979468624e14,
	 1.0051899717115285432e15,
};
static double QJ1[] = {
	1.1267125065029138050e06,
	6.4872502899596389593e08,
	2.7622777286244082666e11,
	8.4899346165481429307e13,
	1.7128800897135812012e16,
	1.7253905888447681194e18,
	1.3886978985861357615e03,
};

/*
 * Coefficients for rational approximation of
 *   (y1(x) - 2 LN(x/XY0) j1(x)) / (x**2 - XY0**2),
 *       XSMALL  <  |x|  <=  4.0
 */
static double PY0[] = {
	 2.2157953222280260820e05,
	-5.9157479997408395984e07,
	 7.2144548214502560419e09,
	-3.7595974497819597599e11,
	 5.4708611716525426053e12,
	 4.0535726612579544093e13,
	-3.1714424660046133456e02,
};
static double QY0[] = {
	8.2079908168393867438e02,
	3.8136470753052572164e05,
	1.2250435122182963220e08,
	2.7800352738690585613e10,
	4.1272286200406461981e12,
	3.0737873921079286084e14,
};

/*
 * Coefficients for rational approximation of
 *   (y1(x) - 2 LN(x/XY1) j1(x)) / (x**2 - XY1**2),
 *        .0  <  |x|  <=  8.0
 */
static double PY1[] = {
	 1.9153806858264202986e06,
	-1.1957961912070617006e09,
	 3.7453673962438488783e11,
	-5.9530713129741981618e13,
	 4.0686275289804744814e15,
	-2.3638408497043134724e16,
	-5.6808094574724204577e18,
	 1.1514276357909013326e19,
	-1.2337180442012953128e03,
};
static double QY1[] = {
	1.2855164849321609336e03,
	1.0453748201934079734e06,
	6.3550318087088919566e08,
	3.0221766852960403645e11,
	1.1187010065856971027e14,
	3.0837179548112881950e16,
	5.6968198822857178911e18,
	5.3321844313316185697e20,
};

/*
 * Coefficients for Hart,s approximation,  |x| > 8.0
 */
static double P0[] = {
	-1.0982405543459346727e05,
	-1.5235293511811373833e06,
	-6.6033732483649391093e06,
	-9.9422465050776411957e06,
	-4.4357578167941278571e06,
	-1.6116166443246101165e03,
};
static double Q0[] = {
	-1.0726385991103820119e05,
	-1.5118095066341608816e06,
	-6.5853394797230870728e06,
	-9.9341243899345856590e06,
	-4.4357578167941278568e06,
	-1.4550094401904961825e03,
};
static double P1[] = {
	1.7063754290207680021e03,
	1.8494262873223866797e04,
	6.6178836581270835179e04,
	8.5145160675335701966e04,
	3.3220913409857223519e04,
	3.5265133846636032186e01,
};
static double Q1[] = {
	3.7890229745772202641e04,
	4.0029443582266975117e05,
	1.4194606696037208929e06,
	1.8194580422439972989e06,
	7.0871281941028743574e05,
	8.6383677696049909675e02,
};

static double
#if defined(__STDC__) || defined(__GNUC__)
caljy1(double x, int jint)
#else
caljy1(x,jint)
double x;
int jint;
#endif
{
	int i;
	double ax,resj,down,up,xden,xnum,w,wsq,z,zsq;

	ax = fabs(x);		/* Check for error conditions */
	if (jint && (x <= ZERO || (x < HALF && ax*XINF < PI2)))
		return -XINF;
	else if (ax > XMAX)
		return ZERO;
/*
 * Calculate j1 or y1 for |x|  >  8.0
 */
	if (ax > EIGHT) {
		z = EIGHT/ax;
		w = floor(ax/TWOPI)+THROV8;
		w = (ax-w*TWOPI1)-w*TWOPI2;
		zsq = z*z;
		xnum = P0[5];
		xden = zsq+Q0[5];
		up = P1[5];
		down = zsq+Q1[5];
		for (i = 0; i <= 4; i++) {
			xnum = xnum*zsq+P0[i];
			xden = xden*zsq+Q0[i];
			up = up*zsq+P1[i];
			down = down*zsq+Q1[i];
		}
#define	r0 xnum
#define	r1 up
		r0 = xnum/xden;
		r1 = up/down;
		return RTPI2/sqrt(ax)*(jint ? r0*sin(w)+z*r1*cos(w) :
			(x < ZERO ? z*r1*sin(w)-r0*cos(w) :
			r0*cos(w)-z*r1*sin(w)));
#undef	r1
#undef	r0
	}
	else if (ax <= XSMALL)
		return jint ? -PI2/ax : x*HALF;
/*
 * Calculate j1 for appropriate interval, preserving
 *    accuracy near the zero of j1
 */
	zsq = ax*ax;
	if (ax <= FOUR) {
		xnum = (PJ0[6]*zsq+PJ0[5])*zsq+PJ0[4];
		xden = zsq+QJ0[4];
		for (i = 0; i <= 3; i++) {
			xnum = xnum*zsq+PJ0[i];
			xden = xden*zsq+QJ0[i];
		}
#define	prod resj
		prod = x*((ax-XJ01/TWO56)-XJ02)*(ax+XJ0);
	}
	else {
		xnum = PJ1[0];
		xden = (zsq+QJ1[6])*zsq+QJ1[0];
		for (i = 1; i <= 5; i++) {
			xnum = xnum*zsq+PJ1[i];
			xden = xden*zsq+QJ1[i];
		}
		xnum = xnum*(ax-EIGHT)*(ax+EIGHT)+PJ1[6];
		xnum = xnum*(ax-FOUR)*(ax+FOUR)+PJ1[7];
		prod = x*((ax-XJ11/TWO56)-XJ12)*(ax+XJ1);
	}
#define	result resj
	result = prod*(xnum/xden);
#undef	prod
	if (!jint)
		return result;
/*
 * Calculate y1.  First find  resj = pi/2 ln(x/xn) j1(x),
 *   where xn is a zero of y1
 */
#define	xy z
	if (ax <= FOUR) {
		up = (ax-XY01/TWO56)-XY02;
		xy = XY0;
	}
	else {
		up = (ax-XY11/TWO56)-XY12;
		xy = XY1;
	}
	down = ax+xy;
	if (fabs(up) < P17*down) {
		w = up/down;
		wsq = w*w;
		xnum = PLG[0];
		xden = wsq+QLG[0];
		for (i = 1; i <= 3; i++) {
			xnum = xnum*wsq+PLG[i];
			xden = xden*wsq+QLG[i];
		}
		resj = PI2*result*w*xnum/xden;
	}
	else
		resj = PI2*result*log(ax/xy);
#undef	xy
#undef	result
/*
 * Now calculate y1 for appropriate interval, preserving
 *    accuracy near the zero of y1
 */
	if (ax <= FOUR) {
		xnum = PY0[6]*zsq+PY0[0];
		xden = zsq+QY0[0];
		for (i = 1; i <= 5; i++) {
			xnum = xnum*zsq+PY0[i];
			xden = xden*zsq+QY0[i];
		}
	}
	else {
		xnum = PY1[8]*zsq+PY1[0];
		xden = zsq+QY1[0];
		for (i = 1; i <= 7; i++) {
			xnum = xnum*zsq+PY1[i];
			xden = xden*zsq+QY1[i];
		}
	}
	up *= down; up /= ax; up *= xnum; up /= xden; up += resj;
	return up;
}

/*
 * This subprogram computes approximate values for Bessel functions
 *   of the first kind of order zero for arguments  |x| <= XMAX
 *   (see comments heading caljy1).
 */
float
j1f(float x)
{
	return ((float)caljy1(x,0));
}

/*
 * This subprogram computes approximate values for Bessel functions
 *   of the second kind of order zero for arguments 0 < x <= XMAX
 *   (see comments heading caljy1).
 */
float
y1f(float x)
{
	return ((float)caljy1(x,1));
}
