#undef DEBUG
#define USE_DMA /* this controls whether reads/writes use pseudo-DMA */

#include <sys/types.h>
#include "aic6250.h"
#include "icu.h"
#include "aicscsi.h"
#include "saio.h"

#define MAX_TARG 	1
#define MAX_TCS		MAX_TARG
#define MAX_LUN  	1
#define MAX_LUS		MAX_LUN
#define MAX_WAIT	2000000

#define MAX_RETRY	3

#define SELECT_TIMEOUT_ERR -2
#define ERR		-1
#define OK		0

#define READ		0
#define WRITE		2
#define NO_DMA		0

#if defined(USE_DMA)
#define DMA_READ	READ|DC_DMA_EN
#define DMA_WRITE	WRITE|DC_DMA_EN
#else
#define DMA_READ	READ
#define DMA_WRITE	WRITE
#endif

volatile aic_t *aic = AIC6250POLLADDR;
int aic_inited = 0;

struct chip_data {
     unsigned char port;
     unsigned char val;
};

struct chip_data init_stuff[] = {
     CR1_REG, CR1_CLK_20|CR1_PORTB_IO,
     CR0_REG, CR0_PORTA_OUT|CR0_MASTER,
     IM0_REG, 0
	  -1, -1,
     };

struct chip_data cmd_stuff[] = {
     DC_REG_LB, 0,
     DC_REG_MB, 0,
     DC_REG_HB, 0,
     IM0_REG, IM0_SELOUT_INT|IM0_CMD_DONE_INT|IM0_ERROR_INT,
     OC_REG, 0,
     DC_REG, 0, /* depends on whether it's a read or write */
     IM1_REG, IM1_PH_MISM_INT|IM1_SCSI_REQ_INT,
     CR0_REG, CR0_MASTER|CR0_PORTA_OUT,
     CR1_REG, CR1_CLK_20|CR1_PORTB_IO,
     -1, -1,
};

int testrdycmd = 0;

#define ONE_SEC		1000		/* 1 sec */
#define SMALL_TIMEOUT	ONE_SEC
#define SELECT_TIMEOUT	(30 * ONE_SEC)
#define ONE_MINUTE	(60 * ONE_SEC)
#define THREE_MINUTES	(180 * ONE_SEC)

static int curr_c, curr_t, curr_l;
static int scsi_status;
static int scsi_message;
static int testreadycmd = 0;


int  boot_debug = 1;
int  scsi_debug = 1;
int autoboot;
int was_power_on = 1;

int scsi_inited	= 0;
int scsi_ctrl	= 0;	/* only one had is available: ctrlno = 0 */
int hd_is_inited[MAX_TCS] = {0/*,0,0,0,0,0,0,0*/};
int mt_is_inited[MAX_TCS] = {0/*,0,0,0,0,0,0,0*/};
int hd_is_open[MAX_TCS] = {0/*,0,0,0,0,0,0,0*/};
int mt_is_open[MAX_TCS] = {0/*,0,0,0,0,0,0,0*/};

unsigned char scsi_type[MAX_TCS][MAX_LUS];
unsigned char scsi_scsi2[MAX_TCS][MAX_LUS];

#define vtop(x) x
#define ptov(x) x
     
     do_scsi_init()
{
     if (!scsi_inited) {
	  scsi_init1();
	  scsi_inited++;
	  scsi_init2();
     }
}

scsi_init1()
{
     aic_init(scsi_ctrl);
}

scsi_init2()
{
     scsi_init_targets(scsi_ctrl);
}

check_sensedata(sensedata)
{
     if (sensedata != 0 && sensedata != 1) 
	  return(sensedata);
     if (sensedata == 0) 
	  printf("Warning: have no sense key but check condition status\n");
     if (scsi_debug && sensedata == 1)
	  printf("rsense: have an recovered error sensekey = 1\n");
     return OK;
}

drive_init(drivetype)
register int drivetype;
{
     register int res;
     
     if (testrdy() < 0) goto bad;
     
     testreadycmd = 1;
     if ((res = testrdy()) != OK && res != CHECK_CONDITION) {
	  testreadycmd = 0;
	  goto bad;
     }
     testreadycmd = 0;
     
     if (res == CHECK_CONDITION) {
	  res = rsense(drivetype);
	  if (res == NOT_READY && drivetype != DISKTYPE) {
	       if (check_sensedata(res) != OK)
		    return(res);
	  }
     }
     if (drivetype == TAPETYPE && scsi_scsi2[curr_t][curr_l] == 2) {
	  if ((res = mt_modesel()) != OK && res != CHECK_CONDITION) {
	       printf("Warning: error in modeselect command\n");
	       goto bad;
	  }
	  if (res == CHECK_CONDITION) {
	       res = rsense(drivetype);
	       if (check_sensedata(res) != OK)
		    printf("Warning: error in modeselect command\n");
	  }
     } else { /* disk drive */
	  if ((res = startunit()) != OK && res != CHECK_CONDITION) goto bad;
	  
	  if (res == CHECK_CONDITION) {
	       res = rsense(drivetype);
	       if (check_sensedata(res) != OK)
		    return(res);
	  }
     }
     if ((res = inquiry()) != OK && res != CHECK_CONDITION) goto bad;
     if (res == CHECK_CONDITION) {
	  res = rsense(drivetype);
	  if (check_sensedata(res) != OK)
	       return(res);
     }
     
     if (drivetype == DISKTYPE)
	  hd_is_inited[curr_t]++;
     else
	  mt_is_inited[curr_t]++;
     
     return OK;
 bad:
     return ERR;
}

hd_open(iobp)
register struct iob *iobp;
{
     return(drive_open(iobp, DISKTYPE));
}

mt_open(iobp)
register struct iob *iobp;
{
     return(drive_open(iobp, TAPETYPE));
}

drive_open(iobp, drivetype)
register struct iob *iobp;
register int drivetype;
{
     register int res;
     register int t,l,i;
     register int mark = iobp->i_bn;
     
     do_scsi_init();
     curr_c = iobp->i_cntrl;
     curr_t = iobp->i_target;
     curr_l = iobp->i_lun;
     if (curr_t == -1) { /* find first available drive */
	  for (t=0; t<MAX_TCS; t++)
	       for (l=0; l<MAX_LUS; l++)
		    if (scsi_type[t][l] == drivetype) {
			 curr_t = iobp->i_target = t;
			 curr_l = iobp->i_lun    = l;
			 goto found;
		    }
     }
 found:
     if (boot_debug)
	  printf("drive_open: c=%d, t=%d, l=%d scsitype=%x\n",
		 curr_c, curr_t, curr_l, scsi_type[curr_t][curr_l]);
     if (scsi_type[curr_t][curr_l] != drivetype) {
	  printf("No %s type (target: %d, type: %x)\n",
		 drive_type[drivetype], curr_t, scsi_type[curr_t][curr_l]);
	  return ERR;
     }
     if (drivetype != DISKTYPE && drivetype != TAPETYPE) {
	  printf("Unknown drive type (0x%x)\n", drivetype);
	  return ERR;
     }
     if (drivetype == DISKTYPE && !hd_is_inited[curr_t]) {
	  if (drive_init(DISKTYPE) != OK) {
	       printf("Could not initialize hard disk.\n");
	       return ERR;
	  }
     } else if (drivetype == TAPETYPE && !mt_is_inited[curr_t]) {
	  if (drive_init(TAPETYPE) != OK) {
	       printf("Could not initialize streamer drive.\n");
	       return ERR;
	  }
     }
     if (iobp->i_cntrl != 0) {
	  printf("Bad hostadapter (%d)\n", iobp->i_cntrl);
	  return ERR;
     }
     if (iobp->i_target >= MAX_TCS || iobp->i_target == CR0_MASTER) {
	  printf("Bad target (%d)\n", iobp->i_target);
	  return ERR;
     }
     if (iobp->i_lun >= MAX_LUS) {
	  printf("Bad lun (%d)\n", iobp->i_lun);
	  return ERR;
     }
     if ((res = testrdy()) != OK && res != CHECK_CONDITION) return ERR;
     
     if (res == CHECK_CONDITION) {
	  res = rsense(drivetype);
	  if (check_sensedata(res) != OK)
	       return(res);
     }
     
     if (drivetype == DISKTYPE) {
	  hd_is_open[curr_t]++;
     } else {
	  if ((res = rewind()) != OK && res != CHECK_CONDITION)
	       return ERR;
	  if (res == CHECK_CONDITION) {
	       res = rsense(drivetype);
	       if (check_sensedata(res) != OK)
		    return(res);
	  }
	  for (i=0; i<mark; i++) {
	       if ((res = mark1()) != OK && res != CHECK_CONDITION)
		    return ERR;
	       if (res == CHECK_CONDITION) {
		    res = rsense(drivetype);
		    if (check_sensedata(res) != OK)
			 return(res);
	       }
	  }
	  mt_is_open[curr_t]++;
     }
     
     return OK;
}

hd_strategy(iobp, func)
register struct iob *iobp;
register int func;
{
     return(drive_strategy(iobp, func, DISKTYPE));
}

mt_strategy(iobp, func)
register struct iob *iobp;
register int func;
{
     return(drive_strategy(iobp, func, TAPETYPE));
}

drive_strategy(iobp, func, drivetype)
register struct iob *iobp;
register int func;
register int drivetype;
{
     register int res;
     register int maxretry = MAX_RETRY;
     register daddr_t pblk = iobp->i_bn + iobp->i_boff;
     
 again:
     if (boot_debug)
	  printf("drivestra: cc=%d ma=0x%x bn=%d boff=%d  pblk=%d func=%s ",
		 iobp->i_cc, iobp->i_ma, iobp->i_bn, iobp->i_boff, pblk, func == WRITE ? "write" : "read");
     if (iobp->i_cc % 512) {
	  printf("count not a multiple of block size\n");
	  return ERR;
     }
     if (drivetype == DISKTYPE && !hd_is_inited[curr_t])
	  return ERR;
     if (drivetype == TAPETYPE && !mt_is_inited[curr_t])
	  return ERR;
     if (func == READ) {
	  if ((res = scsi_read(iobp->i_ma, pblk, iobp->i_cc, drivetype)) != OK &&
	      res != CHECK_CONDITION)
	       return ERR;
	  if (res == CHECK_CONDITION) {
	       res = rsense(drivetype);
	       if (check_sensedata(res) != OK) {
		    if (drivetype == DISKTYPE && maxretry-- >= 0) {
			 printf("Disk read error (retry).\n");
			 goto again;
		    }
		    return(res);
	       }
	  }
     } else {
	  if ((res = scsi_write(iobp->i_ma, pblk, iobp->i_cc, drivetype)) != OK &&
	      res != CHECK_CONDITION)
	       return ERR;
	  if (res == CHECK_CONDITION) {
	       res = rsense(drivetype);
	       if (check_sensedata(res) != OK)
		    return(res);
	  }
     }
     if (boot_debug)
	  printf("strategy returns %d for blck %d\n", iobp->i_cc, pblk);
     return iobp->i_cc;
}

hd_close()
{
     return(drive_close(DISKTYPE));
}

mt_close()
{
     return(drive_close(TAPETYPE));
}

drive_close(drivetype)
register int drivetype;
{
     if (boot_debug)
	  printf("close\n");
     if (drivetype == DISKTYPE && !hd_is_open[curr_t])
	  return ERR;
     else
	  hd_is_open[curr_t] = 0;
     if (drivetype == TAPETYPE && !mt_is_open[curr_t])
	  return ERR;
     else
	  mt_is_open[curr_t] = 0;
     return OK;
}

testrdy()
{
     unsigned char 	trdy_cdb[] = { HD_TRDY, 0, 0, 0, 0, 0 };
     
     if (boot_debug)
	  printf("testrdy target=%d\n", curr_t);
     return(scsi_cmd(curr_c, curr_t, curr_l, &trdy_cdb,
		     sizeof(trdy_cdb), (char *)0, 0, NO_DMA));
}

rsense(drivetype)
register int drivetype;
{
     register int 	res; 
     register int 	retry = 3; 
     unsigned char 	rsense_cdb[] = { HD_RSENSE, 0, 0, 0, 16, 0 };
     unsigned char	sense[16];
     
 again:	if (boot_debug)
      printf("rsense: target=%d, adr=%x\n", curr_t, sense);
     res = scsi_cmd(curr_c, curr_t, curr_l, &rsense_cdb,
		    sizeof(rsense_cdb), sense, 16, READ);
     if (boot_debug)
	  printf("res=%x, sensekey=%x\n", res, sense[2] & 0xff);
     if (res != OK && res != CHECK_CONDITION)
	  return res;
     if (res == CHECK_CONDITION && --retry > 0)
	  goto again;
     if (res != OK)
	  return res;
     sense[2] &= 0xf;
     if (boot_debug)
	  printf("rsense: sensekey = 0x%x (%s), errcode = 0x%x\n",
		 sense[2], sensekey[sense[2]], sense[12]);
     if (sense[2] && sense[2] != RECOVERED_ERR && sense[2] != UNITATTENTION) {
	  if (testreadycmd && sense[2] == NOT_READY)
	       goto ok;
	  printf("%s error: sensekey = 0x%x (%s), errcode = 0x%x\n",
		 drive_type[drivetype], sense[2], sensekey[sense[2]], sense[12]);
     }
 ok:	return(sense[2]);
}

inquiry()
{
     register int	res; 
     unsigned char 	inquiry_cdb[] = { MT_INQUIRY, 0, 0, 0, 36, 0 };
     unsigned char 	inquiry[36];
     
 again:	bzero(inquiry, 36);
     if (boot_debug)
	  printf("inquiry: target=%d, adr=%x\n", curr_t, inquiry);
     res = scsi_cmd(curr_c, curr_t, curr_l, &inquiry_cdb,
		    sizeof(inquiry_cdb), inquiry, 36, READ);
     if (boot_debug)
	  printf("%s\n", &inquiry[8]);
     return res;
}

startunit()
{
     register int res;
     unsigned char start_cdb[] = { HD_MSTART, 0, 0, 0, 1, 0 };
     
     if (boot_debug)
	  printf("startunit: target=%d\n", curr_t);
     res = scsi_cmd(curr_c, curr_t, curr_l, &start_cdb,
		    sizeof(start_cdb),(char *)0, 0, NO_DMA);
     /* pollsleepms(200); */
     return(res);
}

scsi_read(buf, baddr, bytes, drivetype)
register char		*buf;
register unsigned	baddr;
register int		bytes;
register int		drivetype;
{
     register int		res;
     register int		blocks;
     register int		cdb_len;
     unsigned char		read_cdb[10];
     
     bzero(read_cdb, sizeof(read_cdb));
     blocks = bytes >> 9;
     if (drivetype == DISKTYPE) {
	  read_cdb[0] = HD_READ;
	  read_cdb[1] = 0;
	  read_cdb[2] = (baddr >> 24) & 0xff;
	  read_cdb[3] = (baddr >> 16) & 0xff;
	  read_cdb[4] = (baddr >> 8) & 0xff;
	  read_cdb[5] = baddr & 0xff;
	  read_cdb[6] = 0;
	  read_cdb[7] = (blocks >> 8) & 0xff;
	  read_cdb[8] = blocks & 0xff;
	  read_cdb[9] = 0;
	  cdb_len = 10;
     } else if (drivetype == TAPETYPE) {
	  read_cdb[0] = MT_READ;
	  read_cdb[1] = 1;			/* FBM */
	  read_cdb[2] = (blocks >> 16) & 0xff;
	  read_cdb[3] = (blocks >> 8) & 0xff;
	  read_cdb[4] = blocks & 0xff;
	  read_cdb[5] = 0;
	  cdb_len = 6;
     } else {
	  printf("Unknown drive type (%x)\n", drivetype);
	  return ERR;
     }
     
     if (boot_debug)
	  printf("scsi_read: bytes=%d blocks=%d adr=%x\n", bytes, blocks, buf);
     res = scsi_cmd(curr_c, curr_t, curr_l, &read_cdb,
		    cdb_len, buf, bytes, DMA_READ);
     return res;
}

scsi_write(buf, baddr, bytes, drivetype)
register char		*buf;
register unsigned	baddr;
register int		bytes;
register int		drivetype;
{
     register int		blocks;
     register int		cdb_len;
     unsigned char		write_cdb[10];
     
     bzero(write_cdb, sizeof(write_cdb));
     blocks = bytes >> 9;
     if (drivetype == DISKTYPE) {
	  write_cdb[0] = HD_WRITE;
	  write_cdb[1] = 0;
	  write_cdb[2] = (baddr >> 24) & 0xff;
	  write_cdb[3] = (baddr >> 16) & 0xff;
	  write_cdb[4] = (baddr >> 8) & 0xff;
	  write_cdb[5] = baddr & 0xff;
	  write_cdb[6] = 0;
	  write_cdb[7] = (blocks >> 8) & 0xff;
	  write_cdb[8] = blocks & 0xff;
	  write_cdb[9] = 0;
	  cdb_len = 10;
     } else if (drivetype == TAPETYPE) {
	  write_cdb[0] = MT_WRITE;
	  write_cdb[1] = 1;			/* FBM */
	  write_cdb[2] = (blocks >> 16) & 0xff;
	  write_cdb[3] = (blocks >> 8) & 0xff;
	  write_cdb[4] = blocks & 0xff;
	  write_cdb[5] = 0;
	  cdb_len = 6;
     } else {
	  printf("Unknown drive type (%x)\n", drivetype);
	  return ERR;
     }
     if (boot_debug)
	  printf("scsi_write: bytes=%d blocks=%d adr=%x\n", bytes, blocks, buf);
     return(scsi_cmd(curr_c, curr_t, curr_l, &write_cdb,
		     cdb_len, buf, bytes, DMA_WRITE));
}

rewind()
{
     unsigned char	rewind_cdb[] = { MT_RWND, 0, 0, 0, 0, 0 };
     
     if (boot_debug)
	  printf("rewind target = %d\n", curr_t);
     return(scsi_cmd(curr_c, curr_t, curr_l, &rewind_cdb,
		     sizeof(rewind_cdb), (char *)0, 0, NO_DMA));
}

mark1()
{
     unsigned char	space_dcb[] = { DM_SPACE, 1, 0, 0, 1, 0x80 };
     
     if (boot_debug)
	  printf("mark1 target=%d\n", curr_t);
     return(scsi_cmd(curr_c, curr_t, curr_l, &space_dcb,
		     sizeof(space_dcb), (char *)0, 0, NO_DMA));
}

mt_modesel()
{
     unsigned char	modesel_cdb[] = { MT_MDSEL, 0x10, 0, 0, 12, 0 };
     char 		modesel[12] = {0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 2, 0};
     
     if (boot_debug)
	  printf("mt_modesel target=%d\n", curr_t);
     return(scsi_cmd(curr_c, curr_t, curr_l, &modesel_cdb,
		     sizeof(modesel_cdb), modesel, 12, WRITE));
}

scsi_cmd (c, t, l, cdb_p, cdb_len, data_p, data_len, rdwt)
int c, t, l;		/* HA Controller, target, lun */
unsigned char *cdb_p;	/* Command Descriptor Block pointer */
int cdb_len;		/* Command Descriptor Block length */
unsigned char *data_p;	/* data pointer */
int data_len;		/* data length */
int rdwt;		/* read or write */
{
     int res, i;
     
     SELSCSI(AIC6250);

     for(i=0; cmd_stuff[i].port != 0xff; i++) {
	  aic->sel_reg = cmd_stuff[i].port;
	  aic->acc_reg = cmd_stuff[i].val;
     }
     
     res = aic_select (t, 0);
     if (res < 0)
	  return (res);
     printf("call xferinf for CMD ");
     res = xferinf (cdb_p, cdb_len, PH_CMD, WRITE);
     if (res < 0)
	  return (ERR);
     printf("call xferinf for %s ", (((rdwt & WRITE) == WRITE) ? "PH_DO":"PH_DI"));
     if (data_p)
	  res = xferinf (data_p, data_len, (((rdwt & WRITE) == WRITE) ? PH_DO:PH_DI), rdwt);
     if (res < 0)
	  return (ERR);
     printf("call ccs\n");
     res = ccs();
     if (res < 0)
	  return (ERR);
     
     return (scsi_status);
     
}

aic_select(target, watn)
{
     int i, res, timo;
     unsigned char stat, stat1;
     unsigned char identify = 0x40;
     
     aic->sel_reg = SSRW_REG;
     stat = aic->acc_reg = 0;
     printf("0aic_select: the bus shows: %d\n", aic->acc_reg);
     aic->sel_reg = SIDD_REG;
     aic->acc_reg = (unsigned char)(1 << CR0_MASTER | 1 << target);
     aic->sel_reg = IM0_REG;
     aic->acc_reg = IM0_SELOUT_INT|IM0_CMD_DONE_INT|IM0_ERROR_INT|watn|IM0_ARB_SEL_STRT;
     aic->sel_reg = SR1_REG;
     for (i = 0;; ++i) {		/* wait for initiator to assert SEL */
	  if((stat = aic->acc_reg) & (SR1_SEL_OUT|SR1_CMD_DONE|SR1_ERROR))
	       break;
	  if (i > MAX_WAIT * 2) {			/* timeout */
	       printf ("SCSI: SELECT OUT bit never set\n");
	       aic->sel_reg = SSRR_REG;
	       printf("1the bus shows 0x%x\n", aic->acc_reg);
	       aic_unselect ();
	       return(ERR);
	  }
     }
     if(stat & SR1_ERROR) {
	  aic->sel_reg = SR0_REG;
	  printf("aic_select: SR0 0x%x\n",aic->acc_reg);
	  aic_unselect();
	  return(ERR);
     }
     for (i = 0;; ++i) {		/* wait for target to assert SEL */
	  if(aic->acc_reg & SR1_CMD_DONE) {
	       aic->sel_reg = IM0_REG;
	       stat = aic->acc_reg;
	       aic->sel_reg = IM0_REG;
	       stat &= ~IM0_CMD_DONE_INT; /* to clear the bit */
	       aic->acc_reg = stat;
	       break;
	  } /* else
	       printf("SR1_REG 0x%x\n",aic->acc_reg); */
	  if (i > MAX_WAIT * 2) {			/* timeout */
	       printf ("SCSI: SELECT timeout\n");
	       aic->sel_reg = SSRR_REG;
	       printf("2the bus shows 0x%x\n", aic->acc_reg);
	       aic_unselect();
	       return(SELECT_TIMEOUT_ERR);
	  }
     }
     aic->sel_reg = SSRR_REG;
     stat = aic->acc_reg;
     /* after power on the chip asserts ATN even tho it shouldn't */
     if (stat & SSRR_ATN_IN && !watn) {
	  res = xferinf (&identify, 1, PH_MO, WRITE);
	  timo = MAX_WAIT;
	  while (--timo)
	       ;
	  stat = aic->acc_reg;
	  if (res < 0)
	       return (ERR);
	  if ((stat & PH_MASK) == PH_MI) 
	       res = msgin();
	  if (res < 0)
	       return (res);
 	  ccs ();
	  return (ERR);
     } else if (stat & SSRR_ATN_IN) {
	  res = xferinf (&identify, 1, PH_MO, WRITE);
	  if (res < 0)
	       return (ERR);
     }
     return(OK);
}

aic_unselect()
{
     int i;
     unsigned char scsi_bus;
     
     aic->sel_reg = SIDD_REG;
     aic->acc_reg = 0;
     for(i=0; i < 30000; i++) ; /* supposed to wait 200us */
     aic->sel_reg = SSRR_REG;
     scsi_bus = aic->acc_reg;
     if(scsi_bus & SSRR_BSY_IN) {
	  aic_inited = 0;
	  /* aic_reset();  this is done in aic_init now */
	  aic_init();
     } else {
	  aic->sel_reg = IM0_REG;
	  scsi_bus = aic->acc_reg;
	  scsi_bus &= ~IM0_ARB_SEL_STRT;
	  aic->sel_reg = IM0_REG;
	  aic->acc_reg = scsi_bus;
     }
}

aic_reset()
{
     int i;
     
     aic->sel_reg = CR1_REG;
     aic->acc_reg |= CR1_SCSI_RST_O;
     for(i=0; i < 300000; i++) ;
     aic->sel_reg = CR1_REG;
     /* aic->acc_reg &= ~CR1_SCSI_RST_O; */
     aic->acc_reg = CR1_CLK_20|CR1_PORTB_IO;
}

ccs()
{
     int res;
     
     res = xferinf (&scsi_status, 1, PH_STA, READ);
     if (res < 0)
	  return (ERR);
     printf("status 0x%x ", scsi_status);
     
     res = xferinf (&scsi_message, 1, PH_MI, READ);
     if (res < 0)
	  return (ERR);
     printf("msg 0x%x\n", scsi_message);
     return (OK);
}

xferinf (ptr, len, phase, rdwt)
unsigned char *ptr;
int len;
int phase;
int rdwt;
{
     unsigned char ph, stat;
     unsigned int timo;
     int i = len;
     int j;
     
     /*     timo = MAX_WAIT;
	    while (--timo)
	    ; */
     if (rdwt & DC_DMA_EN) {
	  dma_setup (len, phase, rdwt);
	  return(do_dma (ptr, len, rdwt /*& ~DC_DMA_EN*/));
     } else {
	  aic->sel_reg = DC_REG_LB;
	  aic->acc_reg = 0;	/* REG 0 */
	  aic->acc_reg = 0;	/* REG 1 */
	  aic->acc_reg = 0;     /* REG2 */
	  aic->acc_reg = IM0_CMD_DONE_INT|IM0_ERROR_INT;    /* REG 3 */
	  aic->acc_reg = 0;
	  aic->acc_reg = rdwt;  /* REG 5 */
	  aic->acc_reg = IM1_PH_MISM_INT|IM1_SCSI_REQ_INT;  /* REG 6 */
	  aic->sel_reg = SSRW_REG;
	  aic->acc_reg = phase;
     }
     /* while-loop for AUTO PIO */
     while (i--) {
	  if (rdwt == WRITE) {
	       aic->sel_reg = IM0_REG;
	       aic->acc_reg = IM0_CMD_DONE_INT|IM0_ERROR_INT;
	       aic->sel_reg = SSRW_REG;
	       aic->acc_reg = phase;
	       aic->sel_reg = SIDD_REG;
	       /* for (j = 0; j < 40; j++) ; /* kill some time */
	       /* printf("write 0x%x ", *ptr); */
	       aic->acc_reg = *ptr++;
	       aic->sel_reg = CR1_REG;
	       aic->acc_reg |= CR1_AUTO_PIO;
	       timo = MAX_WAIT;
	       while (!((stat = aic->acc_reg) & (SR1_CMD_DONE|SR1_ERROR)) && --timo) ;
	       if (timo <= 0) {
		    printf("1xferinf write: timeout waiting for int\n");
		    return (ERR);
	       }
	       if (stat & SR1_CMD_DONE) {
		    aic->sel_reg = IM0_REG;
		    aic->acc_reg = 0;
		    continue;
	       }
	       if (stat & SR1_ERROR) {
		    aic->sel_reg = SR0_REG;
		    stat = aic->acc_reg;
		    if (stat & (SR0_PH_MISM_ERR|SR0_SCSI_PH_CHNG)) {
			 printf("xferinf write: phase change, wanted 0x%x", phase);
			 aic->sel_reg = SSRR_REG;
			 ph = aic->acc_reg;
			 printf(" got 0x%x len %d i %d\n", ph, len, i);
			 if ((ph & PH_MASK) == PH_STA) {
			      ccs();
			      return (ERR);
			 } else if ((ph & PH_MASK) == PH_MI) {
			      if (msgin() < 0)
				   return (ERR);
			      return (ERR);
			 }
			 
			 if (stat & SR0_SCSI_REQ_ON) {
			      aic->sel_reg = SR1_REG;
			      while (!((stat = aic->acc_reg) & SR1_CMD_DONE) && --timo)
				   ;
			      if (timo <= 0) {
				   printf("2xferinf write: timeout waiting for int\n");
				   return (ERR);
			      }
			      continue;
			 }
			 /* printf("xferinf write: ERROR SR0_REG 0x%x\n", stat);
			    return (ERR); */
		    }
	       }
	  } else { /* READ */
	       aic->sel_reg = IM0_REG;
	       aic->acc_reg = IM0_CMD_DONE_INT|IM0_ERROR_INT;
	       aic->sel_reg = SSRW_REG;
	       aic->acc_reg = phase;
	       timo = MAX_WAIT;
	       while (!(aic->acc_reg & SSRR_REQ_IN) && --timo)
		    ;
	       if (timo <= 0) {
		    printf("2xferinf timeout waiting for REQ\n");
		    ph = aic->acc_reg;
		    printf("wanted phase %x, got %x\n", phase, ph);
		    if (ph & PH_STA) {
			 ccs();
		    }
		    return (ERR);
	       }
#if 1
	       aic->sel_reg = SR1_REG;
	       stat = aic->acc_reg;
	       /* printf ("after REQ SR1_REG 0x%x\n", stat); */
	       if (stat & SR1_ERROR) {
		    aic->sel_reg = SR0_REG;
		    stat = aic->acc_reg;
		    if (stat & (SR0_PH_MISM_ERR|SR0_SCSI_PH_CHNG)) {
			 printf("1xferinf read: phase change, wanted 0x%x", phase);
			 aic->sel_reg = SSRR_REG;
			 ph = aic->acc_reg;
			 printf(" got 0x%x\n", ph);
			 return (ph & PH_MASK);
			 /* return (ERR); */
		    }
	       }
#endif
	       aic->sel_reg = SIDD_REG;
	       *ptr = aic->acc_reg;
	       /* printf("read 0x%x ", *ptr); */
	       ++ptr;
	       aic->sel_reg = CR1_REG;
	       aic->acc_reg |= CR1_AUTO_PIO;
	       timo = MAX_WAIT;
	       while (!((stat = aic->acc_reg) & (SR1_CMD_DONE|SR1_ERROR)) && --timo) ;
	       if (timo <= 0) {
		    printf("xferinf read: timeout waiting for int\n");
		    return (ERR);
	       }
	       if (stat & SR1_CMD_DONE) {
		    aic->sel_reg = IM0_REG;
		    aic->acc_reg = 0;
		    continue;
	       }
	       if (stat & SR1_ERROR) {
		    aic->sel_reg = SR0_REG;
		    stat = aic->acc_reg;
		    if (stat & (SR0_PH_MISM_ERR|SR0_SCSI_PH_CHNG)) {
			 printf("2xferinf read: phase change, wanted 0x%x", phase);
			 aic->sel_reg = SSRR_REG;
			 ph = aic->acc_reg;
			 printf(" got 0x%x\n", ph);
			 /* return (ph & PH_MASK); */
			 return (ERR);
			 if (stat & SR0_SCSI_REQ_ON)
			      continue;
			 /* printf("xferinf read: ERROR SR0_REG 0x%x\n", ph);
			    return (ERR); */
		    }
	       }
	  }
     }
     /* printf("\n"); */
     return (OK);
}

scsi_reset()
{
     printf("scsi_reset called\n");
}

scsi_resetck()
{
     printf("scsi_resetck called\n");
}

aic_init()
{
     unsigned int i;
     unsigned char stat;
     int timo;
     
     printf("enter aic_init ");
     SELSCSI(AIC6250);
	  
     printf("before init ");
     /*     i = 10; i = 20; i = 0; */
     aic->sel_reg = CR1_REG;
     aic->acc_reg = CR1_CLK_20|CR1_PORTB_IO;
     for(i=0; init_stuff[i].port != 0xff; i++) {
	  aic->sel_reg = (unsigned char)init_stuff[i].port;
	  aic->acc_reg = (unsigned char)init_stuff[i].val;
     }
     
     for(i=0; i < MAX_WAIT; i++) ;
     printf("after wait\n");
     
     aic->sel_reg = SR1_REG;
     stat = aic->acc_reg;
     printf("at start 0x%x\n",stat);
     
     if(stat & SR1_ERROR) {
	  aic->sel_reg = SR0_REG;
	  printf("aic_init: SR0 0x%x\n",aic->acc_reg);
	  return(ERR);
     }
     return (0);
}

msgin()
{
     unsigned char msgin[32];
     int i ,res;
     
     res = xferinf (&msgin[0], 1, PH_MI, READ);
     if (res < 0)
	  return (res);
     if (msgin[0] == 1) { /* extended message */
	  res = xferinf (&msgin[1], 1, PH_MI, READ);
	  if (res < 0)
	       return (res);
	  if (msgin[1]) {
	       for (i=0; i<msgin[1]; i++) {
		    res = xferinf (&msgin[2+i], 1, PH_MI, READ);
		    if (res < 0)
			 return (res);
		    printf("msgin[%d]: 0x%x ", 2+i, msgin[2+i]);
	       }
	  }
     }
     printf("\n");
     return (OK);
}

#define	POWERUP_TRY_COUNTER 	5
#define WAIT_ON_POWERUP		6

int try_counter;

scsi_init_targets(c)
int c;
{
     register int 	t,l;
     register int	res, sres; 
     unsigned char	mtc_inquiry[] = { MT_INQUIRY, 0, 0, 0, 36, 0 };
     unsigned char	inquiry[36];
     
     if (c != 0) {
	  printf("scsi_init_targets: Wrong hostadapter number (is %d, should 0)\n",c);
	  return ERR;
     }
     
     for (t=0; t<MAX_TCS; t++)
	  for (l=0; l<MAX_LUS; l++) {
	       scsi_type[t][l] = 0xff;
	       scsi_scsi2[t][l] = 1;
	  }
     
     for (t=0; t<MAX_TCS; t++) {
	  for (l=0; l<MAX_LUS; l++) {
	       /* we only use lun 0 */
	       if (l)
		    break;
	  again:
	       if (boot_debug)
		    printf("do_target_init: adr=%x\n", inquiry);
	       res = scsi_cmd(c, t, l, (caddr_t)&mtc_inquiry,
			      sizeof(mtc_inquiry), (caddr_t)inquiry,
			      36, READ);
	       if (res == SELECT_TIMEOUT) {
		    if (autoboot && was_power_on && try_counter-- > 0) {
			 if (scsi_debug)
			      printf("\
We have autoboot and poweron and SELECT_TIMOUT, wait 5 secs\n\
and try it again (trycounter=%d)\n",try_counter);
			 /* pollsleep(WAIT_ON_POWERUP); */
			 goto again;
		    }
	       }
	       if (res == CHECK_CONDITION) {
		    sres = rsense(inquiry[0] & 0x7f);
		    if (inquiry[0] & 0x7f == DISKTYPE && autoboot && was_power_on && 
			try_counter-- > 0 && sres == NOT_READY) {
			 if (scsi_debug)
			      printf("\
We have autoboot and poweron and checkcondition, wait 5 secs\n\
and try it again (trycounter=%d)\n",try_counter);
			 /* pollsleep(WAIT_ON_POWERUP); */
			 goto again;
		    }
	       }
	       if (res != OK)
		    res = scsi_cmd(c, t, l, (caddr_t)&mtc_inquiry,
				   sizeof(mtc_inquiry), (caddr_t)inquiry,
				   36, READ);
	       if (res == OK) {
		    scsi_type[t][l] = inquiry[0] & 0x7f;
		    scsi_scsi2[t][l] = inquiry[2] & 3;
		    if (boot_debug)
			 printf("OK %s removeable=%x type=%x\n", 
				&inquiry[8], inquiry[1] & 0xff, inquiry[0] % 0x7f);
	       }
	  }
     }
}


unsigned char *aic_dma_addr = (unsigned char *)0;

#define USE_LONGS 1
#undef USE_MOVSD 1
#define WAIT_FOR_REQ 1

do_dma (dma_addr, dma_len, rdwt)
unsigned char *dma_addr;
int dma_len, rdwt;
{
#if defined(USE_LONGS) || defined(USE_MOVSD)
     register int nlongs asm("r3");
#endif
     int nbytes;
     unsigned char stat, ph;
#if defined(USE_MOVSD)
     register unsigned char *src asm("r1"), *dest asm("r2");
     register int cnt asm("r0");
#else
     register unsigned char *src , *dest ;
#endif
     int leftover, timo;
     int times_thru = 0;

#if defined(USE_LONGS) || defined (USE_MOVSD)
     nbytes = dma_len & 3;
     nlongs = dma_len >> 2;
#else
     nbytes = dma_len;
#endif
     if ((rdwt & WRITE) == WRITE) {
	  src = dma_addr;
	  dest = aic_dma_addr;
     } else {
	  src = aic_dma_addr;
	  dest = dma_addr;
     }
  dma_again:
	++times_thru;

#if defined(WAIT_FOR_REQ)
     aic->sel_reg = SSRW_REG;
     timo = MAX_WAIT;
     while (!(aic->acc_reg & SSRR_REQ_IN) && --timo)
	  ;
     if (timo <= 0) {
	  printf("do_dma timeout waiting for REQ\n");
	  ph = aic->acc_reg;
	  printf(" got %x\n", ph);
	  if (ph & PH_STA) {
	       ccs();
	  }
	  return (ERR);
     }
#endif

#if defined(USE_LONGS)
     for (; nlongs; nlongs--)
	  *((long *)dest)++ = *((long *)src)++;
#endif

#if defined(USE_MOVSD)
	cnt = nlongs;
	if (nlongs) {
		asm("movsd");
	}
#endif

     /* nbytes = 1; */
     if (nbytes)
	  while(nbytes--)
	       *dest++ = *src++;

     /* return;
     /* NOTREACHED */
     for (leftover = 30000; leftover; leftover--) ;
     aic->sel_reg = FS_REG;
     stat = aic->acc_reg;
printf("FIFO 0x%x", stat);
     aic->sel_reg = SSRR_REG;
printf(" phase 0x%x\n", aic->acc_reg);
     aic->sel_reg = SR1_REG;
     stat = aic->acc_reg;
printf("1stat 0x%x\n", stat);
     if (stat & SR1_CMD_DONE) {
printf("CMD_DONE\n");
	  aic->sel_reg = IM0_REG;
	  aic->acc_reg = 0;
	  return (OK);
     }
     if (stat & SR1_ERROR) {
	  aic->sel_reg = SR0_REG;
	  stat = aic->acc_reg;
printf("2stat 0x%x\n", stat);
	  if (stat & (SR0_PH_MISM_ERR|SR0_SCSI_PH_CHNG)) {
	       aic->sel_reg = SSRR_REG;
	       stat = aic->acc_reg;
	       if ((stat & PH_MASK) == PH_STA)
		   return (OK);
	       printf("do_dma: phase change");
	       printf(" got 0x%x\n", stat);
	       printf("ERROR\n");
	       return (ERR);
	  }
     }
     leftover = 0;
     aic->sel_reg = DC_REG_LB;
     leftover = aic->acc_reg;
printf(" DMACNT %d  ",leftover);
     aic->sel_reg = DC_REG_MB;
     leftover += aic->acc_reg << 8;
printf(" DMACNT %d  ",leftover);
     aic->sel_reg = DC_REG_HB;
     leftover += aic->acc_reg << 16;
printf(" DMACNT %d  ",leftover);
     if (leftover > dma_len) {
	printf("the chip thinks that there's more left than was wanted !!!\n");
	return (ERR);
     }
     if (leftover) {
	  if ((rdwt & WRITE) == WRITE)
	       src -= leftover;
	  else
	       dest -= leftover;
#if defined(USE_LONGS) || defined(USE_MOVSD)
	if(times_thru > 5) {
		nlongs = 0;
		times_thru = 0;
	     nbytes = leftover;
	} else {
	     nlongs = leftover >> 2;
	     nbytes = leftover & 3;
	}
#else
	nbytes = leftover;
#endif
	     goto dma_again;
     } else
	     aic_dma_addr += dma_len;
     return (OK);
}

dma_setup(len, phase, rdwt)
{
     if(aic_dma_addr == (unsigned char *)0)
	  aic_dma_addr = (unsigned char *)AIC6250DMAADDR;
     else if (aic_dma_addr >= (unsigned char *)(AIC6250DMAADDR+MAX_CACHE))
          aic_dma_addr = (unsigned char *)AIC6250DMAADDR;
     aic->sel_reg = DC_REG_LB;
     aic->acc_reg = len & 0xff;	        /* REG 0 */
     aic->sel_reg = DC_REG_MB;
     aic->acc_reg = (len >> 8) & 0xff;	/* REG 1 */
     aic->sel_reg = DC_REG_HB;
     aic->acc_reg = (len >> 16) & 0xff;    /* REG 2 */
     aic->sel_reg = IM0_REG;
     aic->acc_reg = IM0_CMD_DONE_INT|IM0_ERROR_INT;    /* REG 3 */
     aic->sel_reg = OC_REG;
     aic->acc_reg = 0;      /* REG 4, assumes non-synchronous */
     aic->sel_reg = IM1_REG;
     aic->acc_reg = IM1_PH_MISM_INT|IM1_SCSI_REQ_INT;  /* REG 6 */
     aic->sel_reg = CR0_REG;
     aic->acc_reg = CR0_MASTER|CR0_PORTA_OUT;
     aic->sel_reg = CR1_REG;
     aic->acc_reg = CR1_CLK_20|CR1_PORTB_IO;
     aic->sel_reg = SSRW_REG;
     aic->acc_reg = phase;
     aic->sel_reg = DC_REG;
     aic->acc_reg = rdwt;   /* REG 5 */
}
