/*
 * include file for the AIC-6250 SCSI Protocol Controller
 */

typedef struct {
	unsigned char sel_reg; /* select a register */
	unsigned char acc_reg; /* access it */
} aic_t;

#define AIC6250POLLADDR	((aic_t *)0x30000000);
#define AIC6250DMAADDR	0x38000000
#define MAX_CACHE	0x10000

/*
 * The AIC-6250 is set up so that, for registers 0-7, the register number
 * in sel_reg is automatically incremented. This allows accesses to these
 * registers in ascending order without having to reload sel_reg. For the
 * remaining registers the number in sel_reg is not automatically
 * incremented.
 *
 * The registers are as follows:
 * 0, 1 and 2 are for setting the DMA count (write only)
 * 3 is EN INT MSK REG 0 (write only)
 * 4 is OFFSET CONTROL (write only)
 * 5 is FIFO STATUS (read only)
 * 6 is REV CNTL (read only) or EN INT MSK REG 1 (write only)
 * 7 is STATUS REG 0 (read only) or CONTROL REG 0 (write only)
 * 8 is STATUS REG 1 (read only) or CONTROL REG 1 (write only)
 * 9 is SCSI SIGNAL REG (read only) or SCSI SIGNAL REG (write only)
 * 10 is SCSI ID/DATA (read/write)
 * 11 is SOURCE/DEST ID (read/write)
 * 12 is MEMORY DATA (read/write)
 * 13 is PORT A (read/write)
 * 14 is PORT B (read/write)
 * 15 is SCSI LATCHED DATA (read only)
 */

/*
 * DMA BYTE COUNT, registers 0,1,&2; R/W
 * register 0 is the low byte of the DMA address
 */
#define DC_REG_LB	0
#define DC_REG_MB	1
#define DC_REG_HB	2

/* EN INT MSK REG 0, register 3, W */
#define IM0_REG		3
#define IM0_SEL_INT	1
#define IM0_RESEL_INT	2
#define IM0_SELOUT_INT	4
#define IM0_CMD_DONE_INT	8
#define IM0_ERROR_INT	0x10
#define IM0_AUTO_ATN	0x20
#define IM0_ARB_SEL_STRT	0x40
#define IM0_NOTUSED	0x80

/* OFFSET CONTROL, register 4, W */
#define OC_REG		4
#define OC_OFFSET_MSK	0xf
#define OC_DEF_OFFSET	7
#define OC_SYNC_XFER_MSK	0x70
#define OC_NOTUSED	0x80
/* these values are for a 20MHz clock and are or'd into the reg. value */
#define SX_5_00MB_SEC	0
#define SX_4_00MB_SEC	0x10
#define SX_3_33MB_SEC	0x20
#define SX_2_86MB_SEC	0x30
#define SX_2_50MB_SEC	0x40
#define SX_2_22MB_SEC	0x50
#define SX_2_00MB_SEC	0x60
#define SX_1_82MB_SEC	0x70

/* FIFO STATUS, register 5, R */
#define FS_REG		5
#define FS_FIFO_CNT_MSK	7
#define FS_FIFO_FULL	8
#define FS_FIFO_MPTY	0x10
#define FS_NO_PEND_ACKS	0x20
#define FS_TEST_BITS	0xc0	/* not used */

/* DMA CNTL, register 5, W */
#define DC_REG		5
#define DC_DMA_EN	1
#define DC_DMA_DIR_OUT	2
#define DC_ODD_XFER_STRT	4	/* only used if 16 bit mode */
#define DC_NOTUSED	0xf0

/* REV CNTL, register 6, R */
#define RC_REG		6
#define RC_REV_LVL_MSK	0x3
#define RC_NOTUSED	0xfc

/* EN INT MSK REG 1, register 6, W */
#define IM1_REG		6
#define IM1_ATN_ON_INT	1	/* target mode */
#define IM1_PH_CHNG_INT	1	/* initiator mode */
#define IM1_SCSI_PE_INT	2
#define IM1_BUS_FREE_INT	4
#define IM1_PH_MISM_INT	8
#define IM1_MEM_PE_INT	0x10
#define IM1_SCSI_RST_INT	0x20
#define IM1_SCSI_REQ_INT	0x40
#define IM1_NOTUSED	0x80

/* STATUS REG 0, register 7, R */
#define SR0_REG		7
#define SR0_DMA_CNT_0	0x1
#define SR0_SCSI_PH_CHNG	0x2	/* initiator mode */
#define SR0_ATN_ASRT	0x2	/* target mode */
#define SR0_SCSI_REQ_ON	0x4
#define SR0_SCSI_PE	0x8
#define SR0_BUS_FREE	0x10
#define SR0_PH_MISM_ERR	0x20
#define SR0_MEM_PE	0x40
#define SR0_SCSI_RST	0x80

/* CONTROL REG 0, register 7, W */
#define CR0_REG		7
#define CR0_SCSI_ID_MSK	0x7
#define CR0_DIFF_MODE	0x8 /* differential mode, else single ended */
#define CR0_PORTA_OUT	0x10
#define CR0_TARGET	0x20 /* run in target mode */
#define CR0_PMEM_W	0x40
#define CR0_PMEM_CYCR	0x80
#define CR0_MASTER	0x07

/* STATUS REG 1, register 8, R */
#define SR1_REG		8
#define SR1_SEL		0x1
#define SR1_RESEL	0x2
#define SR1_SEL_OUT	0x4
#define	SR1_CMD_DONE	0x8
#define SR1_ERROR	0x10	/* if this is set read SR0 for more info */
#define SR1_SCSI_RST_IN	0x20
#define SR1_NOTUSED	0x40
#define SR1_MEM_CYC_CMP	0x80

/* CONTROL REG 1, register 8, W */
#define CR1_REG		8
#define CR1_SW_RESET	0x1
#define CR1_SCSI_RST_O	0x2
#define CR1_CLK_20	0x4	/* 20 MHz clock */
#define CR1_PH_CH_ANY	0x8	/* interrupt on ANY phase change */
#define CR1_PORTB_IO	0x10	/* if 0 used for 16 bit xfers */
#define CR1_NOTUSED	0x20
#define CR1_16BIT_EN	0x40
#define CR1_AUTO_PIO	0x80

/* SCSI SIGNAL REG, register 9, R */
/* read this register to see what signals are on the SCSI bus */
#define SSRR_REG	9
#define SSRR_ACK_IN	0x1
#define SSRR_REQ_IN	0x2
#define SSRR_BSY_IN	0x4
#define SSRR_SEL_IN	0x8
#define SSRR_ATN_IN	0x10
#define SSRR_MSG_IN	0x20
#define SSRR_IO_IN	0x40
#define SSRR_CD_IN	0x80

/* SCSI SIGNAL REG, register 9, W */
/* write this register to set signals on the SCSI bus */
#define SSRW_REG	9
#define SSRW_NOTUSED	0x1
#define SSRW_REQ_OUT	0x2	/* target */
#define SSRW_ACK_OUT	0x2	/* initiator */
#define SSRW_BSY_OUT	0x4
#define SSRW_SEL_OUT	0x8
#define SSRW_ATN_OUT	0x10
#define SSRW_MSG_OUT	0x20
#define SSRW_IO_OUT	0x40
#define SSRW_CD_OUT	0x80

#define PH_MASK		0xE0
#define PH_DO		0
#define PH_DI		0x40
#define PH_CMD		0x80
#define PH_STA		0xC0
#define PH_MO		0xA0
#define PH_MI		0xE0

/*
 * SCSI ID/DATA, register 10, R/W
 * write this register to set the SCSI ID before arbitration, after the
 * ARB/SEL/RESEL are complete, read/write for SCSI bus data. The data from
 * from the SCSI bus are not latched in this register.
 */
#define SIDD_REG	10

/*
 * SOURCE/DEST ID, register 11, R/W. This contains the ID for both the
 * selector and target after a SEL/RESEL.
 */
#define SRC_DEST_REG	11

/* MEMORY DATA, register 12, R/W */
#define MEM_DAT_REG	12

/* PORT A, register 13, R/W */
#define PORTA_REG	13

/* PORT B, register 14, R/W */
#define PORTB_REG	14

/*
 * SCSI LATCHED DATA, register 15, R. Reading this register delivers the
 * data which were latched at the last REQ/ACK cycle. Writing this register
 * in TARGET mode will clear the BSY signal and the SCSI bus enters the
 * bus free phase.
 */
#define SLD_REG		15

/* SCSI BSY RESET, register 15, W */
#define SBR_REG		15	/* target only */
