/* Keyboard driver for PC's and AT's. */

#include "kernel.h"
#include <sgtty.h>
#include <signal.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "proc.h"
#include "tty.h"

/* Standard and AT keyboard.  (PS/2 MCA implies AT throughout.) */
#define KEYBD		0x60	/* I/O port for keyboard data */

/* AT keyboard.  Most of these values are only used for rebooting. */
#define KB_COMMAND	0x64	/* I/O port for commands on AT */
#define KB_GATE_A20	0x02	/* bit in output port to enable A20 line */
#define KB_PULSE_OUTPUT	0xF0	/* base for commands to pulse output port */
#define KB_RESET	0x01	/* bit in output port to reset CPU */
#define KB_STATUS	0x64	/* I/O port for status on AT */

/* PS/2 model 30 keyboard. */
#define PS_KB_STATUS	0x72	/* I/O port for status on ps/2 (???) */
#define PS_KEYBD	0x68	/* I/O port for data on ps/2 */

/* AT and PS/2 model 30 keyboards. */
#define KB_ACK		0xFA	/* keyboard ack response */
#define KB_BUSY		0x02	/* status bit set when KEYBD port ready */
#define LED_CODE	0xED	/* command to keyboard to set LEDs */
#define MAX_KB_ACK_RETRIES 0x1000	/* max #times to wait for kb ack */
#define MAX_KB_BUSY_RETRIES 0x1000	/* max #times to loop while kb busy */

/* All keyboards. */
#define KBIT		0x80	/* bit used to ack characters to keyboard */

/* Scan codes whose action is not completely captured by maps. */
#define DEL_SCAN	  83	/* DEL for use in CTRL-ALT-DEL reboot */
#define ESCAPE_CODE	0xE0	/* beginning of escape sequence */
#define F1		  59	/* function key F1, others follow */
#define F2		  60
#define F3		  61
#define F4		  62
#define F5		  63
#define F8		  66
#define F9		  67
#define F10		  68
#define MINUS_DU	0x35	/* '-' on Dutch extended keybd */
#define NUM_SLASH_DU	0x57	/* numeric keypad slash on Dutch extended kb */
#define SCODE1		  71	/* Home on numeric pad */
#define SCODE2		  81	/* PgDn on numeric pad */
#define TOP_ROW		  14	/* codes below this are shifted if CTRL */

#define NR_SCAN_CODES	0x80	/* Number of scan codes
				 * Actually only 0x69 are used on IBM keyboards,
				 * but 0x80 is the maximum possible number
				 * of scancodes and it makes read/write_kmap
				 * simpler and allows using wierd keyboards (such
				 * as the Amstrad PC1512's) without recompiling
				 * the kernel.
				 */

/* Miscellaneous. */
#define CTRL_S		  31	/* scan code for letter S (for CRTL-S) */
#define CONSOLE		   0	/* line number for console */
#define MEMCHECK_ADR   0x472	/* address to stop memory check after reboot */
#define MEMCHECK_MAG  0x1234	/* magic number to stop memory check */

#define kb_addr(n)	(&kb_lines[CONSOLE])	/* incorrectly ignore n */
#define KB_IBUFSIZE	  32	/* size of keyboard input buffer */

#define ACTION		0x07	/* Action mask for special [] */
#define NORMAL		0x00	/* Actions ... */
#define SHIFT		0x01
#define CONTROL		0x02
#define ALT		0x03
#define CAPSLOCK	0x04
#define SHIFTLOCK	0x05
#define NUMLOCK		0x06
#define K_NUMLOCK	0x40	/* Key responds to num lock */
#define K_CAPSLOCK	0x80	/* Key responds to caps lock */

PRIVATE int alt;		/* alt key state */
PRIVATE int capslock;		/* caps lock key state */
PRIVATE int esc;		/* escape scan code detected? */
PRIVATE int control;		/* control key state */
PRIVATE int caps_off;		/* 1 = normal position, 0 = depressed */
PRIVATE int keyb_type;		/* type of keyboard attached */
PRIVATE int minus_code = MINUS_DU;
				/* numeric minus on extended keyboard */
PRIVATE int numlock;		/* number lock key state */
PRIVATE int num_off;		/* 1 = normal position, 0 = depressed */
PRIVATE int num_slash = NUM_SLASH_DU;
				/* numeric slash on dutch extended keyboard */
PRIVATE int shift;  /* shift key state */
PRIVATE int shiftlock;  /* shift lock key state */

PRIVATE char keymaps [4][NR_SCAN_CODES] = 
{
 {
  0,033,'1','2','3','4','5','6',        	'7','8','9','0','-','=','\b','\t',
  'q','w','e','r','t','y','u','i',      	'o','p','[',']',015,0202,'a','s',
  'd','f','g','h','j','k','l',';',      	047,0140,0200,0134,'z','x','c','v',
  'b','n','m',',','.','/',0201,'*',     	0203,' ',0204,0241,0242,0243,0244,0245,
  0246,0247,0250,0251,0252,0205,0210,0267,  	0270,0271,0211,0264,0265,0266,0214,
  0261,0262,0263,'0',0177,0,0,0,0,		0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0
 },
 {
  0,033,'!','@','#','$','%','^',        	'&','*','(',')','_','+','\b','\t',
  'Q','W','E','R','T','Y','U','I',      	'O','P','{','}',015,0202,'A','S',
  'D','F','G','H','J','K','L',':',      	042,'~',0200,'|','Z','X','C','V',
  'B','N','M','<','>','?',0201,'*',    		0203,' ',0204,0221,0222,0223,0224,0225,
  0226,0227,0230,0231,0232,0204,0213,'7',  	'8','9',0211,'4','5','6',0214,'1',
  '2','3','0','.',0,0,0,0,			0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0
 },
 {
  0
 },
 {
  0,0,0,0,0,0,0,0,				0,0,0,0,0,0,0,0,
  0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,	0x80,0x80,0,0,0,2,0x80,0x80,
  0x80,0x80,0x80,0x80,0x80,0x80,0x80,0,		0,0,1,0,0x80,0x80,0x80,0x80,
  0x80,0x80,0x80,0,0,0,1,0,			3,0,4,0,0,0,0,0,
  0,0,0,0,0,6,0,0x40,				0x40,0x40,0,0x40,0x40,0x40,0,0x40,
  0x40,0x40,0x40,0x40,0,0,0,0,			0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0
 }
};

/* Scan codes to ASCII for unshifted keys */
PRIVATE char * unsh = keymaps [0];

/* Scan codes to ASCII for shifted keys */
PRIVATE char * sh = keymaps [1];

/* Scan codes to ASCII for alt keys */
PRIVATE char * alt_c = keymaps [2];

/* scecial key map. Each entry consists of three fields:
 * action (bits 0 ... 2):
 *  0 Key is not a shift type key.
 *  1 Shift
 *  2 Control
 *  3 Alt
 *  4 Caps-Lock
 *  5 Shift-Lock
 *  6 Num-Lock
 * numlock (bit 6)
 *  0 Key's meaning does change if numlock is active.
 *  1 Key's meaning does not change if numlock is active.
 * capslock (bit 7)
 *  0 Key's meaning does change if capslock is active.
 *  1 Key's meaning does not change if capslock is active.
 *
 * Bits 3 ... 5 are not yet used. If you want additional shift type
 * keys expand the action field if necessary and add lock-sensity fields
 * from high to low bits so they clash with the action field as late
 * as possible.
 *
 * Note that I did not distinguish between left and right shift, alt,
 * and control keys.
 *
 */
PRIVATE char * special = keymaps [3];

PRIVATE char scode_map[] =
  {'H', 'A', 'V', 'S', 'D', 'G', 'C', 'T', 'Y', 'B', 'U'};


/* Keyboard structure, 1 per console. */
struct kb_s {
  int minor;			/* minor number of this line (base 0) */

  char *ibuf;			/* start of input buffer */
  char *ibufend;		/* end of input buffer */
  char *iptr;			/* next free spot in input buffer */

  char ibuf1[KB_IBUFSIZE + 1];	/* 1st input buffer, guard at end */
  char ibuf2[KB_IBUFSIZE + 1];	/* 2nd input buffer (for swapping) */
};

PRIVATE struct kb_s kb_lines[NR_CONS];

FORWARD int kb_ack();
FORWARD int kb_wait();
FORWARD int scan_keyboard();
FORWARD void set_leds();

/*===========================================================================*
 *				keyboard				     *
 *===========================================================================*/
PUBLIC void keyboard()
{
/* A keyboard interrupt has occurred.  Process it. */

  int code, k;
  register struct kb_s *kb;

  /* Fetch the character from the keyboard hardware and acknowledge it. */
  code = scan_keyboard();

  /* Kludge for Extended keyboards.
   * The original code contained a `if (keyb_type == DUTCH_EXT)' here.
   * As the keyboard type is no longer known, I removed this. 
   * Does this make any difference for US keyboards?
   */
  if (esc) {
  	/* Numeric slash gives scan codes 0xE0 0x35. */
  	if (code == minus_code) code = num_slash;
  	esc = FALSE;
  } else {
  	esc = (code == ESCAPE_CODE);
  }	

  /* The IBM keyboard interrupts twice per key, once when depressed, once when
   * released.  Filter out the latter, ignoring all but the shift-type keys.
   * The shift-type keys 29, 42, 54, 56, 58, and 69 must be processed normally.
   */
   
  k = code - 0200;		/* codes > 0200 mean key release */
  if (k > 0) {
  /* A key has been released. */
  if ((special [k] & ACTION) == NORMAL)
  	return;		/* don't call tty_task() */
  } else {
	/* Check to see if character is CTRL-S, to stop output. Setting xoff
	 * to anything other than CTRL-S will not be detected here, but will
	 * be detected later, in the driver.  A general routine to detect any
	 * xoff character here would be complicated since we only have the
	 * scan code here, not the ASCII character.
	 */
	if (!(tty_struct[CONSOLE].tty_mode & RAW) &&
	    control && code == CTRL_S &&
	    tty_struct[CONSOLE].tty_xoff == XOFF_CHAR) {
		tty_struct[CONSOLE].tty_inhibited = STOPPED;
		return;
	}
  }

  /* Call debugger? (as early as practical, not in TTY which may be hung) */
  if (code == F10 && db_exists) {
  	db_enabled = TRUE;
  	db();
  }

  /* Store the character in memory so the task can get at it later. */
  kb = kb_addr(-NR_CONS);
  *kb->iptr = code;
  if (kb->iptr < kb->ibufend) {
  	lock();			/* protect shared variable */
	tty_events += EVENT_THRESHOLD;	/* C doesn't guarantee atomic */
	unlock();
	++kb->iptr;
  }
  /* Else it doesn't fit - discard it. */
}


/*==========================================================================*
 *				kb_read					    *
 *==========================================================================*/
PUBLIC int kb_read(minor, bufindirect, odoneindirect)
int minor;
char **bufindirect;
unsigned char *odoneindirect;
{
/* Swap the keyboard input buffers, giving the old one to TTY. */

  register char *ibuf;
  register struct kb_s *kb;
  int nread;

  kb = kb_addr(minor);
  *odoneindirect = FALSE;
  if (kb->iptr == (ibuf = kb->ibuf)) return 0;
  *bufindirect = ibuf;
  lock();
  nread = kb->iptr - ibuf;
  tty_events -= nread * EVENT_THRESHOLD;
  if (ibuf == kb->ibuf1)
	ibuf = kb->ibuf2;
  else
	ibuf = kb->ibuf1;
  kb->ibufend = ibuf + KB_IBUFSIZE;
  kb->iptr = ibuf;
  unlock();
  kb->ibuf = ibuf;
  return nread;
}


/*===========================================================================*
 *				letter_code				     *
 *===========================================================================*/
PUBLIC int letter_code(scode)
int scode;			/* scan code from key press */
{
/* Convert scan codes from numeric keypad to letters for use in escape seqs. */

  scode &= 0x7F; /* just to be sure */
  if ((special [scode] & K_NUMLOCK) && (shift || !numlock))
	return scode_map[scode - SCODE1];
  return 0;
}


/*===========================================================================*
 *				make_break				     *
 *===========================================================================*/
PUBLIC int make_break(ch)
char ch;			/* scan code of key just struck or released */
{
/* This routine can handle keyboards that interrupt only on key depression,
 * as well as keyboards that interrupt on key depression and key release.
 * For efficiency, the interrupt routine filters out most key releases.
 */

  int c, make, code;

  /* Check for CTRL-ALT-DEL, and if found, reboot the computer. This would
   * be better done in keyboard() in case TTY is hung, except control and
   * alt are set in the high level code.
   */
  if (control && alt && ch == DEL_SCAN) reboot();	/* CTRL-ALT-DEL */

  c = ch & 0177;		/* high-order bit set on key release */
  make = (ch & 0200 ? 0 : 1);	/* 1 when key depressed, 0 when key released */

  if (code = special [c] & ACTION) {
   	/* shift type keys yield actions not characters */
   	switch (code) {
	    case SHIFT:
  		shift = make;
		shiftlock = 0;
		set_leds ();
		break; /* any shift key */
	    case CONTROL:
		control = make; 
		break; /* any control key */
     	    case ALT:
     	    	alt = make; 
     	    	break; /* any alt key */
     	    case CAPSLOCK:
		if (make && caps_off) {
    			/* disable autorepeat for caps lock */
    			capslock = 1 - capslock;
    			set_leds();
   		}
   		caps_off = 1 - make;
   		break; /* caps lock */
	    case SHIFTLOCK: /* for those who want it */
   		shiftlock = 1;
   		set_leds();
   		break; /* shift lock */
	    case NUMLOCK:
   		if (make && num_off) {
    			/* disable autorepeat for num lock */
    			numlock  = 1 - numlock;
    			set_leds();
   		}
   		num_off = 1 - make;
   		break; /* num lock */
  	}
 	return(-1);

  } else {
	/* Ordinary key, i.e. not shift, control, alt, etc. */

	if (alt) {
		code = alt_c[c];
	} else {
		if (special [c] & K_CAPSLOCK) {
			code = ((! shift == ! capslock) ? unsh [c] : sh [c]);
		} else if (special [c] & K_NUMLOCK) {
			code = ((shift || !numlock) ? unsh[c] : sh[c]);
		} else {
			code = (shift || shiftlock || (control && c < TOP_ROW)
				? sh[c] : unsh[c]);
		}
	}
	code &= BYTE;
	if (control) code &= 037;
	if (!make) code = -1; /* key release */
	return(code);
  }
}


/*===========================================================================*
 *				set_leds				     *
 *===========================================================================*/
PRIVATE void set_leds()
{
/* Set the LEDs on the caps lock and num lock keys */

  int leds, data_port, status_port;

  if (!pc_at && !ps) return;	/* PC/XT doesn't have LEDs */
  leds = (numlock << 1) | (capslock << 2);	/* encode LED bits */

  if (ps) {
	data_port = PS_KEYBD;
	status_port = PS_KB_STATUS;
  } else {
	data_port = KEYBD;
	status_port = KB_STATUS;
  }

  kb_wait(status_port);		/* wait for buffer empty  */
  out_byte(data_port, LED_CODE);   /* prepare keyboard to accept LED values */
  kb_ack(data_port);		/* wait for ack response  */

  kb_wait(status_port);		/* wait for buffer empty  */
  out_byte(data_port, leds);	/* give keyboard LED values */
  kb_ack(data_port);		/* wait for ack response  */
}


/*==========================================================================*
 *				kb_wait					    *
 *==========================================================================*/
PRIVATE int kb_wait(status_port)
int status_port;
{
/* Wait until the controller is ready; return zero if this times out. */

  int retries;

  retries = MAX_KB_BUSY_RETRIES + 1;
  while (--retries != 0 && in_byte(status_port) & KB_BUSY)
	;			/* wait until not busy */
  return(retries);		/* nonzero if ready */
}


/*==========================================================================*
 *				kb_ack					    *
 *==========================================================================*/
PRIVATE int kb_ack(data_port)
int data_port;
{
/* Wait until kbd acknowledges last command; return zero if this times out. */

  int retries;

  retries = MAX_KB_ACK_RETRIES + 1;
  while (--retries != 0 && in_byte(data_port) != KB_ACK)
	;			/* wait for ack */
  return(retries);		/* nonzero if ack received */
}

/*===========================================================================*
 *				kb_init					     *
 *===========================================================================*/
PUBLIC void kb_init(minor)
int minor;
{
/* Initialize the keyboard driver. */

  register struct kb_s *kb;

  kb = kb_addr(minor);

  /* Record minor number. */
  kb->minor = minor;

  /* Set up input queue. */
  kb->iptr = kb->ibuf = kb->ibuf1;
  kb->ibufend = kb->ibuf1 + KB_IBUFSIZE;
  kb->iptr = kb->ibuf1;

  /* Set initial values. */
  caps_off = 1;
  num_off = 1;

  set_leds();			/* turn off numlock led */

  scan_keyboard();		/* stop lockup from leftover keystroke */
  enable_irq(KEYBOARD_IRQ);	/* safe now everything initialised! */
}


/*===========================================================================*
 *				func_key				     *
 *===========================================================================*/
PUBLIC int func_key(ch)
char ch;			/* scan code for a function key */
{
/* This procedure traps function keys for debugging and control purposes. */

  if (ch < F1 || ch > F10) return(FALSE);	/* not our job */
  if (ch == F1) p_dmp();	/* print process table */
  if (ch == F2) map_dmp();	/* print memory map */
  if (ch == F3) toggle_scroll();	/* hardware vs. software scrolling */

#if AM_KERNEL
#if !NONET
  if (ch == F4) net_init();	/* Re-initialise the ethernet card */
#endif
  if (ch == F5) amdump();	/* Dump Amoeba statistics. */
#endif /* AM_KERNEL */

  if (ch == F8 && control) sigchar(&tty_struct[CONSOLE], SIGINT);
  if (ch == F9 && control) sigchar(&tty_struct[CONSOLE], SIGKILL);
  return(TRUE);
}


/*==========================================================================*
 *				scan_keyboard				    *
 *==========================================================================*/
PRIVATE int scan_keyboard()
{
/* Fetch the character from the keyboard hardware and acknowledge it. */

  int code;
  int val;

  if (ps) {
	code = in_byte(PS_KEYBD);	/* get the scan code for key struck */
	val = in_byte(0x69);	/* acknowledge it in mysterious ways */
	out_byte(0x69, val ^ 0x10);	/* 0x69 should be equiv to PORT_B */
	out_byte(0x69, val);	/* XOR looks  fishy */
	val = in_byte(0x66);	/* what is 0x66? */
	out_byte(0x66, val & ~0x10);	/* 0x72 for PS_KB_STATUS is fishier */
	out_byte(0x66, val | 0x10);
	out_byte(0x66, val & ~0x10);
  } else {
	code = in_byte(KEYBD);	/* get the scan code for the key struck */
	val = in_byte(PORT_B);	/* strobe the keyboard to ack the char */
	out_byte(PORT_B, val | KBIT);	/* strobe the bit high */
	out_byte(PORT_B, val);	/* now strobe it low */
  }
  return code;
}


/*==========================================================================*
 *				reboot					    *
 *==========================================================================*/
PUBLIC void reboot()
{
/* Reboot the machine. */

  static u16_t magic = MEMCHECK_MAG;

  lock();
  eth_stp();			/* stop ethernet (may be unnecessary) */

  /* Stop BIOS memory test. */
  phys_copy(numap(TTY, (vir_bytes) &magic, sizeof magic),
	    (phys_bytes) MEMCHECK_ADR, (phys_bytes) sizeof magic);
  if (protected_mode) {
	/* Rebooting is nontrivial because the BIOS reboot code is in real
	 * mode and there is no sane way to return to real mode on 286's.
	 */
	if (pc_at) {
		/* Use the AT keyboard controller to reset the processor.
		 * The A20 line is kept enabled in case this code is ever
		 * run from extended memory, and because some machines
		 * appear to drive the fake A20 high instead of low just
		 * after reset, leading to an illegal opode trap.  This bug
		 * is more of a problem if the fake A20 is in use, as it
		 * would be if the keyboard reset were used for real mode.
		 */
		kb_wait();
		out_byte(KB_COMMAND,
			 KB_PULSE_OUTPUT | (0x0F & ~(KB_GATE_A20 | KB_RESET)));
	} else {
		printf("No way to reboot from protected mode on this machine ");
	}
	while (TRUE)
		;		/* no way to recover if the above fails */
  }

  /* In real mode, jumping to the reset address is good enough. */
  reset();
}


/*==========================================================================*
 *				wreboot					    *
 *==========================================================================*/
PUBLIC void wreboot()
{
/* Wait for a keystroke, then reboot the machine.  Don't rely on interrupt
 * to provide the keystroke, since this is usually called after a crash,
 * and possibly before interrupts are initialized.
 */

  register int scancode;

  lock();
  milli_delay(1000);		/* pause for a second to ignore key release */
  scan_keyboard();		/* ack any old input */
  printf("Type any key to reboot\r\n");
  scancode = scan_keyboard();	/* quiescent value (0 on PC, last code on AT)*/
  while(scan_keyboard() == scancode)
	;			/* loop until new keypress or any release */
  reboot();
}


/*==========================================================================*
 *				write_kmap				    *
 *==========================================================================*/
PUBLIC void rw_kmap (m_ptr)
	message * m_ptr;
{
/*
 *	This routine copies a new keyboard mapping from a user process.
 *	The keyboard maps are unshifted, shifted, alt, special. Each
 *	is considered to be 128 bytes long.
 */
 
  phys_bytes here, there, count;
  vir_bytes pos;
  message reply_mess;
  int status;
 
  if ((there = numap (m_ptr->PROC_NR, m_ptr->ADDRESS, m_ptr->COUNT)) == 0) {
 	/* not in user space	*/
  	reply_mess.REP_STATUS = E_BAD_ADDR;
  	goto finis;
  }
  count = m_ptr->COUNT;
  pos = m_ptr->POSITION;
  if (pos >= 512) {
  	/* simulate EOF	*/
  	pos = count = 0;
  } else if (pos + count >= 512) {
  	/* don't allow user to read more than the keymaps	*/
 	count -=  pos + count - 512;
  }
  
  if ((here = umap (proc_addr (SYSTASK), D, (vir_bytes) keymaps + pos, count)) == 0) {
 	/* Oops! Keymap not in task's data space. Really shouldn't happen!
 	 */
 	panic ("Keymap at illegal address\r\n", NO_NUM);
  }
  /* Everything ok. Do it	*/
  switch (m_ptr->m_type) {
  case TTY_READ:
  	phys_copy (here, there, count);
  	break;
  case TTY_WRITE:
  	phys_copy (there, here, count);
  	break;
  default:
  	panic ("Keymap illegal command: %d\r\n", m_ptr->m_type);
  }
  reply_mess.REP_STATUS = count;
 
finis:
  /* Tell FS what happened	*/
  reply_mess.m_type = TASK_REPLY;
  reply_mess.REP_PROC_NR = m_ptr->PROC_NR;
  if ((status = send(m_ptr->m_source, &reply_mess)) != OK)
	printf("\r\nrw_kmap couldn't reply to FS. Status: %d\r\n", status);
}
