/*
 * hsurface.c: main program for generating surface data
 */

#include "heisenberg.h"

#define UNKNOWN			0
#define SET_CHAIN_RESOLUTION	1
#define SET_SLICE_COUNT		2
#define SET_FUDGE		5
#define MESH_SPINAL		6
#define QUAD_SPINAL		7
#define LIST_SPINAL		8
#define SET_STRIP_LIST		9
#define CHAIN_LIST		10
#define RCIRCLE_LIST		11
#define SET_RCIRCLE_RESOLUTION	12

#define NEXTARG --argc; ++argv

int op;
Spinal sp;
Chain ch;
Rcircle rc;
int strip_count = -1;
int strip_list[100];

main(argc, argv)
     int argc;
     char *argv[];
{
  initialize();

  if (parse_args(argc, argv))
    switch (op) {
    case MESH_SPINAL:
      mesh_spinal(&sp, stdout);
      break;
    case QUAD_SPINAL:
      quad_spinal(&sp, stdout, strip_list, strip_count);
      break;
    case LIST_SPINAL:
      list_spinal(&sp, stdout);
      break;
    case CHAIN_LIST:
      printf("LINK\n1\n%1d\n", chain_resolution+1);
      write_chain_list(stdout, &ch, chain_resolution+1);
      break;
    case RCIRCLE_LIST:
      printf("LINK\n1\n%1d\n", rcircle_resolution);
      write_rcircle_list(stdout, &rc);
      break;
    }
  else
    usage();

  exit(0);
}

parse_args(argc, argv)
     int argc;
     char *argv[];
{
  int token,i;
  double d[6];

  if (argc < 3) return(0);

  op = UNKNOWN;
  NEXTARG;
  while (argc) {
    switch (token=fsa_parse(*argv)) {
    case SET_CHAIN_RESOLUTION:
      NEXTARG;
      if (argc < 1) return(0);
      chain_resolution = atoi(*argv);
      break;
    case SET_RCIRCLE_RESOLUTION:
      NEXTARG;
      if (argc < 1) return(0);
      rcircle_resolution = atoi(*argv);
      break;
    case SET_SLICE_COUNT:
      NEXTARG;
      if (argc < 1) return(0);
      slice_count = atoi(*argv);
      break;
    case SET_STRIP_LIST:
      NEXTARG;
      if (strcmp(*argv, "{") != 0) return(0);
      NEXTARG;
      strip_count = 0;
      while (strcmp(*argv, "}") != 0) {
	strip_list[strip_count++] = atoi(*argv);
	NEXTARG;
      }
      break;
    case MESH_SPINAL:
    case QUAD_SPINAL:
    case LIST_SPINAL:
      if (argc < 6) return(0);
      op = token;
      for (i=0; i<6; ++i) {
	NEXTARG;
	d[i] = atof(*argv);
      }
      sp.p1.hor.re	= d[0];
      sp.p1.hor.im	= d[1];
      sp.p1.hor.inf	= (char)0;
      sp.p1.ver		= d[2];
      sp.p2.hor.re	= d[3];
      sp.p2.hor.im	= d[4];
      sp.p2.hor.inf	= (char)0;
      sp.p2.ver		= d[5];
      break;
    case CHAIN_LIST:
      if (argc < 4) return(0);
      op = CHAIN_LIST;
      for (i=0; i<4; ++i) {
	NEXTARG;
	d[i] = atof(*argv);
      }
      ch.cen.hor.re	= d[0];
      ch.cen.hor.im	= d[1];
      ch.cen.hor.inf	= d[2];
      ch.rad		= d[3];
      break;
    case RCIRCLE_LIST:
      if (argc < 5) return(0);
      op = RCIRCLE_LIST;
      for (i=0; i<5; ++i) {
	NEXTARG;
	d[i] = atof(*argv);
      }
      rc.cen.hor.re	= d[0];
      rc.cen.hor.im	= d[1];
      rc.cen.hor.inf	= d[2];
      rc.c_rad.re	= d[3];
      rc.c_rad.im	= d[4];
      rc.c_rad.inf	= (char)0;
      break;
    default:
      break;
    }
    NEXTARG;
  }
  return(1);
}

initialize()
{
  fsa_initialize(UNKNOWN);
  fsa_install("cres",	SET_CHAIN_RESOLUTION);
  fsa_install("rres",	SET_RCIRCLE_RESOLUTION);
  fsa_install("ssc",	SET_SLICE_COUNT);
  fsa_install("ms",	MESH_SPINAL);
  fsa_install("qs",	QUAD_SPINAL);
  fsa_install("ls",	LIST_SPINAL);
  fsa_install("strips",	SET_STRIP_LIST);
  fsa_install("c",  	CHAIN_LIST);
  fsa_install("r",  	RCIRCLE_LIST);
}

fatal(s)
     char s;
{
  fprintf("%s\n", s);
  exit(-1);
}

usage()
{
#define EOUT(string) fprintf(stderr,"%s\n",string)

  EOUT("usage: hsurface [parameter options] {ms | qs} x1 y1 v1 x2 y2 v2 \
> outfile");
  EOUT("");
  EOUT("  parameter options:");
  EOUT("    cres n:");
  EOUT("      set chain resolution (longitudinal increment) to n (an integer \
> 1)");
  EOUT("    ssc n");
  EOUT("      set slice count (latitudinal increment) to n (an integer > 1)");
  EOUT("  ms:");
  EOUT("    generate a mesh for the spinal");
  EOUT("  qs:");
  EOUT("    generate a list of quadrilaterals for the spinal");
  EOUT("  x1 y1 v1:");
  EOUT("    Heisenberg coordinates of first vertex");
  EOUT("  x2 y2 v2:");
  EOUT("    Heisenberg coordinates of second vertex");
  EOUT("");

#undef EOUT
}
