/***************************************************************************
 *                Copyright (C) 1990 by Mark B. Phillips                   *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both      *
 *      that copyright notice and this permission notice appear in         *
 *      supporting documentation, and that the names of Mark B.            *
 *      Phillips, or the University of Maryland not be used in             *
 *      advertising or publicity pertaining to distribution of the         *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the author Mark B. Phillips    *
 *      in any publication which uses part or all of any image produced    *
 *      by this software.                                                  *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/

#include <stdio.h>
#include <bsd/sys/types.h>
/* NOTE: the following had to be changed to <sys/time.h> on the 4D:
 * (it was <bsd/sys/time.h>): */
#include <sys/time.h> 
#include <math.h>
#include "gr.h"
#include "../lgd.h"
#include "dpu.h"
#include "panel.h"
#include GLGHEADER

long GrWindowWidth, GrWindowHeight;

long GrCurrentGID;
long GrGraphicsGID;
long GrPanelGID;

STATIC int GrMainLoopDone;
STATIC GrMenu *GrCurrentMenu=NULL;
STATIC long GrCurrentPup=0;

STATIC int GrBackgroundColor = BLACK;

typedef int (*intfunc)();
STATIC intfunc GrInputFunc = NULL;
STATIC int GrInputFD = -1;

/* aspect ratio: width/height of IRIS graphics window */
double GrAspectRatio = 1.0;	/* (gets reset later to actual value) */


#define ESC	'\033'

static int dx[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static int dy[8] = {0, 0, 0, 0, 0, 0, 0, 0};

static int ViewAction;
static long X0,Y0;

static int TempMessagePresent = 0;
     
/*-----------------------------------------------------------------------
 * Function:     GrInit
 * Description:  Initialize the package
 * Arguments:    name:  name for window -- this is use in indexing the
 *		  user's .deskconfig file.
 *		 fg: flag telling whether program should run in foreground
 *		  or not.  fg=1 means foreground, otherwise runs in
 *		  background.
 * Returns:      nothing
 * Notes:        This must be called before any other GR procedures
 *		 fg determines whether the gl procedure 'foreground'
 *		 is called.
 */
GrInit(name,fg)
     char *name;
     int fg;
{
  static int initialized = 0;
  
  if (!initialized) {
    /* First make sure mex is runing; if not, bail out now! */
    if (!ismex()) {
      printf("\nYou must start mex before running this program!\n\n");
      exit(1);
    }
    if (fg) foreground();
    GrGraphicsGID = winopen(name);
    doublebuffer();
    gconfig();
    GrAdjustToWindowSize();
    frontbuffer(YES);
    GrColor(GrBackgroundColor);
    clear();
    frontbuffer(NO);
    LoadRamps();
    setdepth( LO_SCREEN_COORD, HI_SCREEN_COORD );
    GrDepthCueColor(GR_WHITE);
    initialized = 1;
  }
  GrRedraw();
}

GrDone()
{
  winclose(GrGraphicsGID);
  GrMainLoopDone = YES;
}

GrMainLoop()
{
  GlgEvent event;

  GrPanelGID = GrPanelInit();
  GrPanelDraw();
  GrViewInit();
  
  qdevice(RIGHTMOUSE);
  qdevice(MIDDLEMOUSE);
  qdevice(LEFTMOUSE);
  qdevice(KEYBD);

  GrMainLoopDone = NO;
  SetViewAction(ROTATEVIEW);
  while (!GrMainLoopDone) {

    /* Redraw the graphics window */
    GrRedraw();

    /* Process an event if one has occurred */
    if (qtest()) {
      if (TempMessagePresent) GrPutString("");
      GlgReadEvent(&event);
      ProcessEvent(event);
    }

    if (!GrMainLoopDone) {

      /* Check for and process input */
      if (GrInputFunc != NULL)
	if (GrInputPresent())
	  (*GrInputFunc)();

      /* Update the viewing transformation */
      GrUpdateView();

    }

  }
}

ProcessEvent(event)
     GlgEvent event;
{
/*
printevent("ProcessEvent: ",event);
*/
  switch (event.dev) {

  case REDRAW:
    if (event.val == GrGraphicsGID) {
      GrAdjustToWindowSize();
      GrRedraw();
    }
    else if (event.val == GrPanelGID)
      GrPanelDraw();
    break;
    
  case INPUTCHANGE:
    GrCurrentGID = event.val;
/*
printf("ProcessEvent: GrCurrentGID set to %1d\n", GrCurrentGID);
*/
    break;
    
  default:
    if (GrCurrentGID == GrPanelGID)
      GrPanelProcessEvent(event);
    else if (GrCurrentGID == GrGraphicsGID) {
      
      switch (event.dev) {
	
      case LEFTMOUSE:
	if (event.val) {
/*
printf("ProcessEvent: LEFTMOUSE down\n");	  
*/
	  X0 = getvaluator(MOUSEX);
	  Y0 = getvaluator(MOUSEY);
	}
	break;
	
      case MIDDLEMOUSE:
	if (event.val) {
/*
printf("ProcessEvent: MIDDLEMOUSE down\n");	  
*/
	  X0 = getvaluator(MOUSEX);
	  Y0 = getvaluator(MOUSEY);
	}
	break;
	
      case KEYBD:
	GrProcessKey((int)event.val);
	break;

      default:
	break;
	
      }
    }
    break;			

  }
}

int
  GrProcessKey(key)
int key;
{	   
/*
printf("GrProcessKey: %1d ('%c')\n", key, key);
*/
  switch (key) {
    
  case ESC:
    SetViewAction(RESETVIEW);
    break;

  case 's':
  case 'S':
    SetViewAction(SCALEVIEW);
    break;

  case 'h':
  case 'H':
    SetViewAction(HALTVIEW);
    break;
    
  case 'r':
  case 'R':
    SetViewAction(ROTATEVIEW);
    break;
    
  case 'z':
  case 'Z':
    SetViewAction(ZOOMVIEW);
    break;
    
  case 't':
  case 'T':
    SetViewAction(TRANSLATEVIEW);
    break;

  case 'w':
  case 'W':
    SetViewAction(TWISTVIEW);
    break;
    
  case 'f':
  case 'F':
    SetViewAction(FOVVIEW);
    break;

  case '<':
    {
      char fname[80], buf[90];

      GrPutString("Enter name of state file to read:");
      GrGetString(fname);
      if (strlen(fname) == 0)
	GrPutString("(no file read)");
      else {
	if (GrPanelReadFile(fname)) {
	  sprintf(buf, "read file '%s'", fname);
	  GrPutString(buf);
	}
	else {
	  sprintf(buf, "can't read file '%s'", fname);
	  GrPutString(buf);
	}
      }
      TempMessagePresent = 1;
    }
    break;

  case '>':
    {
      char fname[80], buf[90];

      GrPutString("Enter name of state file to write:");
      GrGetString(fname);
      if (strlen(fname) == 0)
	GrPutString("(no file written)");
      else {
	if (GrPanelWriteFile(fname)) {
	  sprintf(buf, "wrote file '%s'", fname);
	  GrPutString(buf);
	}
	else {
	  sprintf(buf, "can't write file '%s'", fname);
	  GrPutString(buf);
	}
      }
      TempMessagePresent = 1;
    }
    break;

  case '?':
    {
      int x,y,w,h;
      
      winset(GrGraphicsGID);
      getorigin(&x,&y);
      getsize(&w,&h);
      printf("Graphics window is at (%3d,%3d), size %3d X %3d\n",
	     x,y,w,h);
      winset(GrPanelGID);
      getorigin(&x,&y);
      getsize(&w,&h);
      printf("Control panel is at (%3d,%3d), size %3d X %3d\n",
	     x,y,w,h);
      if (GrCurrentGID != 0)
	winset(GrCurrentGID);

      {
	lgd_View3 view;
	extern double LGD_L2dist_vec();

	lgd_inquire_view( &view );
	printvector("eye:   ", view.eye);
	printvector("focus: ", view.focus);
	printvector("up:    ", view.up);
	printf("u1, u2 = %f, %f\n", view.u1, view.u2);
	printf("v1, v2 = %f, %f\n", view.v1, view.v2);
	printf("h1, h2 = %f, %f\n", view.h1, view.h2);
	printf("| eye - focus | = %f\n", LGD_L2dist_vec(view.eye, view.focus));
	printf("dpu_focal_dist = %f\n", dpu_focal_dist);
	printf("\n");
      }

    }
    break;
    
  case 'c':
  case 'C':
    {
      static int on = 1;

      on = !on;
      if (on)
	curson();
      else
	cursoff();
    }
    break;
    
  case 'n':
  case 'N':
    GrToggleNTSC();
    break;
    
  }
}

char *
  viewactionstring(a)
int a;
{
  switch (a) {
  case RESETVIEW: return("RESETVIEW");
  case HALTVIEW: return("HALTVIEW");
  case ROTATEVIEW: return("ROTATEVIEW");
  case ZOOMVIEW: return("ZOOMVIEW");
  case TRANSLATEVIEW: return("TRANSLATEVIEW");
  case SCALEVIEW: return("SCALEVIEW");
  case FOVVIEW: return("FOVVIEW");
  case TWISTVIEW: return("TWISTVIEW");
  default: return("??");
  }
}


GrUpdateView()
{
  lgd_View3 view;

  if (GrCurrentGID == GrGraphicsGID) {
    long newx = getvaluator(MOUSEX);
    long newy = getvaluator(MOUSEY);
    if (getbutton(LEFTMOUSE)) {
      dx[ViewAction] = 3*((double)(newx - X0))/4;
      dy[ViewAction] = 3*((double)(newy - Y0))/4;
    }
    else if (getbutton(MIDDLEMOUSE)) {
      dx[ViewAction] += 3*((double)(newx - X0))/4;
      dy[ViewAction] += 3*((double)(newy - Y0))/4;
    }
    if (GrNTSCMode) {
      setvaluator(MOUSEX, X0=GrNTSCMouseX, 0, XMAXSCREEN);
      setvaluator(MOUSEY, Y0=GrNTSCMouseY, 0, YMAXSCREEN);
    }
    else {
      X0 = newx;
      Y0 = newy;
    }
  }

  switch (ViewAction) {

  case RESETVIEW:
    GrReset();
    /* no break here!  (RESETVIEW implies HALTVIEW) */

  case HALTVIEW:
    dx[ROTATEVIEW] = dy[ROTATEVIEW] = 0;
    dx[TRANSLATEVIEW] = dy[TRANSLATEVIEW] = 0;
    dx[SCALEVIEW] = dy[SCALEVIEW] = 0;
    dx[ZOOMVIEW] = dy[ZOOMVIEW] = 0;
    dx[FOVVIEW] = dy[FOVVIEW] = 0;
    dx[TWISTVIEW] = dy[TWISTVIEW] = 0;
    break;

  default:
    lgd_inquire_view( &view );
    GrRotate(&view, dx[ROTATEVIEW],dy[ROTATEVIEW]);
    GrZoom(&view, dx[ZOOMVIEW],dy[ZOOMVIEW]);
    GrTranslate(&view, dx[TRANSLATEVIEW],dy[TRANSLATEVIEW]);
    GrScale(&view, dx[SCALEVIEW],dy[SCALEVIEW]);
    GrFov(&view, dx[FOVVIEW],dy[FOVVIEW]);
    GrTwist(&view, dx[TWISTVIEW],dy[TWISTVIEW]);
    lgd_set_view( &view );
    break;
  }

}
  
int
  GrPutString(s)
char *s;
{
  GrPanelSetMessage(s);
}

int
  GrGetString(s)
char *s;
{
  GrPanelGetString(s);
}

GrRedraw()
{
  long current_gid;

  current_gid = winget();
  winset(GrGraphicsGID);

  GrColor(GrBackgroundColor);	/* clear the screen */
  zbuffer(TRUE);
  /* We use ZF_GREATER because our screen coords have the positive
     direction pointing OUT of the screen, instead of into it.  Hence
     we want to draw a pixel iff the new z value is greater than the
     current one. */
  zfunction(ZF_GREATER);
  czclear(GrBackgroundColor, LO_SCREEN_COORD);

  linewidth(2);
  GrDrawGraphics();		/* draw the graphics */
  pushmatrix();
  ortho2((Coord)0, (Coord)GrWindowWidth,
	 (Coord)0, (Coord)GrWindowHeight);
  popmatrix();
  swapbuffers();		/* swap buffers */
  zbuffer(FALSE);

  winset(current_gid);

  /* Note: by drawing text after graphics, we make sure the text
     shows up on top of the graphics, rather than under it */
}

GrDrawGraphics()
{
  /* NOTE: This makes use of the fact that dpu_update_display()
     updates the transformation matrix before redrawing.  I don't
     recall why I made dpu_update_display do this.  If it is ever
     changed, this procedure will need to be changed to update
     the transformation matrix before redrawing */
  dpu_update_display();
}

GrAdjustToWindowSize()
{
  int textsep;

  getsize(&GrWindowWidth, &GrWindowHeight);
  GrAspectRatio = (double)GrWindowWidth / (double)GrWindowHeight;
  textsep = getdescender() + getheight() + 5;
  reshapeviewport();
}

GrSetMenu( menu )
GrMenu *menu;
{
  char buf[80];
  int i;

  if (GrCurrentMenu != NULL) freepup(GrCurrentPup);
  sprintf(buf,"%s %%t",menu->title);
  GrCurrentPup = defpup(buf);
  addtopup(GrCurrentPup,"----------");
  GrCurrentMenu = menu;
  for (i=0; i<menu->nopts; ++i) {
    convert_menu_string(buf, menu->list[i].string);
    addtopup(GrCurrentPup, buf);
  }
}
  
GrInquireMenu( menu )
GrMenu **menu;
{
  *menu = GrCurrentMenu;
}

/*-----------------------------------------------------------------------
 * Function:     convert_menu_string
 * Description:  converts a string to a form suitable for mex
 * Arguments IN: *s_in: original string
 *          OUT: *s_out: converted string
 * Returns:      nothing
 * Notes:        This trims off trailing and leading blanks, and converts
 *               strings which consist of ALL blanks to ones consisting
 *               of 8 dashes.
 */
static
  convert_menu_string(s_out,s_in)
char *s_out,*s_in;
{
  int i;

  while (*s_in==' ') ++s_in;	/* Skip over leading blanks */
  if (*s_in=='\0')		/* If that's all there was, */
    strcpy(s_out,"--------");	/*   set answer to dashes.  */
  else {			/* Otherwise, 		    */
    strcpy(s_out,s_in);		/*   copy remainder to answer */
    i = strlen(s_out)-1;	/*   and whack off trailing */
    while (s_out[i]==' ') --i;	/*   blanks                 */
    s_out[i+1] = '\0';
  }
}

GrDoMenu()
{
  int opt;

  opt = dopup(GrCurrentPup) - 2; /* -1 since 1st entry is blank */
  if ((opt>=0) && (opt<=GrCurrentMenu->nopts))
    (*(GrCurrentMenu->list[opt].proc))();
}

GrSetInputFunc(func, fd)
int (*func)();
int fd;
{
  GrInputFunc = func;
  GrInputFD = (func == NULL) ? -1 : fd;
}


/* NOTE: this procedure uses select(2) to poll for the presence of data
 * on a file descriptor.  This is apparently part of the BSD enhancement
 * on the IRIS.  It requires that bsd/sys/types.h and bsd/sys/time.h be
 * included (note that the manual page for select(2) incorrectly says to
 * include sys/types.h (the 'bsd' is missing).  It also requires the
 * -lbsd load flag. */
GrInputPresent()
{
  fd_set readfds;
  static struct timeval zero = {0, 0};

  if (GrInputFD != -1) {
    FD_ZERO(&readfds);
    FD_SET(GrInputFD, &readfds);
    select(1, &readfds, NULL, NULL, &zero);
    if (FD_ISSET(GrInputFD, &readfds)) return(1);
  }
  return(0);
}

/***************************************************************/

printevent(s, event)
char *s;
GlgEvent event;
{
  printf("%sdev = %1d, val = %1d, x = %1f, y = %1f\n",
	 s, event.dev, event.val, event.x, event.y);
}

SetViewAction(action)
int action;
{
  ViewAction = action;
  GrSetViewActionDisplay(action);
}
