#include <stdio.h>
#include <math.h>
#include "art.h"
#include "macro.h"

/*
 * This is an implementation of Ken Perlin's noise function as
 * described in "Hypertexture, Computer Graphics, Vol 23, No.3,
 * July 1989, pp 255-256". Most of the function names are as
 * given in the paper. Also, I think that there may be an error
 * in the paper as the given formular of OMEGA would always
 * produce Zero if the actual point falls on a lattice point.
 * (ie. GAMMA(i, j, k) . (u, v, w) = 0). Anyway, I added in 
 * a psuedo random value for the noise at each lattice point
 * as well as the gradient values (as decribed in his '85 paper
 * and it seems to work (Maybe not so good .... but)
 */

typedef	struct {
	float	x;
	float	y;
	float	z;
	float	r;
} vector4;
	

static	vector4	*G;
static	int	*P;
extern	float randtable[];
#define NUMPTS	512
/*
 * random number pointers
 * Note that this has to be different than the randnum() macro as we
 * need consistent random numbers and init_noise may be called at
 * at any time. Eg say the same scene but with more samples in the
 * lights etc etc - this would cause textures to be different.
 */
float   *rp = randtable, *erp = &randtable[NUMPTS];	/* One more! */
#define randinx()       ((rp == erp) ? *(rp = randtable) : *rp++)

#define	ABS(x)	((x) < 0 ? -(x) : (x))

/*
 * Hashing function
 */
#define	phi(i)	P[ABS(i) % NUMPTS]


/*
 * GAMMA
 *
 *	Hashes into the lattice and returns the Gradient vector
 *	(and the noise value G.r) for the lattice point i, j, k.
 */
static vector4 *
GAMMA(i, j, k)
	int	i, j, k;
{
	register int l;
	
	l = j + phi(k);
	l = phi(i + phi(l));

	return(&G[l]);
}

/*
#define GAMMA(i, j, k)	(&G[phi(i + phi(j + phi(k)))])
*/

/*
 * OMEGA
 *
 *	Evaulates the spline knot at the lattice point i, j, k, 
 *	interpolating between it and the actual point u, v, w.
 */
static float
OMEGA(i, j, k, u, v, w)
	int	i, j, k;
	float	u, v, w;
{
	float	a;
	vector4	*V;

	V = GAMMA(i, j, k);
	a = V->r + (V->x * u + V->y * v + V->z * w);

	if (a == 0.0)
		return(0.0);

	u = ABS(u);
	v = ABS(v);
	w = ABS(w);

	a *= (u < 1.0 ? u * u * (2.0 * u - 3.0) + 1.0 : 0.0);
	a *= (v < 1.0 ? v * v * (2.0 * v - 3.0) + 1.0 : 0.0);
	a *= (w < 1.0 ? w * w * (2.0 * w - 3.0) + 1.0 : 0.0);

	return(a);
}
	

/*
 * noise
 *
 *	Sums up the contibutions from the spline knots (lattice points)
 *	Around the (hashed) value of x, y, z.
 */
float
noise(p)
	vector *p;
{
	register float	x, y, z, xm1, ym1, zm1, sum = 0.0;
	int	fx = floor(p->x);
	int	fy = floor(p->y);
	int	fz = floor(p->z);
	int	fxp1;
	int	fyp1;
	int	fzp1;

	x = p->x - (float)fx;
	y = p->y - (float)fy;
	z = p->z - (float)fz;

	xm1 = x - 1;
	ym1 = y - 1;
	zm1 = z - 1;

	fxp1 = ABS(fx + 1);
	fyp1 = ABS(fy + 1);
	fzp1 = ABS(fz + 1);

	fx = ABS(fx);
	fy = ABS(fy);
	fz = ABS(fz);

	sum += OMEGA(fx, fy, fz, x, y, z);
	sum += OMEGA(fx, fy, fzp1, x, y, zm1);

	sum += OMEGA(fx, fyp1, fz, x, ym1, z);
	sum += OMEGA(fx, fyp1, fzp1, x, ym1, zm1);

	sum += OMEGA(fxp1, fy, fz, xm1, y, z);
	sum += OMEGA(fxp1, fy, fzp1, xm1, y, zm1);

	sum += OMEGA(fxp1, fyp1, fz, xm1, ym1, z);
	sum += OMEGA(fxp1, fyp1, fzp1, xm1, ym1, zm1);

	return(sum);
}

#define OFFSET1	1000.0
#define OFFSET2	2000.0

/*
 * Vnoise
 *
 *	A vector valued noise function.
 * 	(Again, from the '89 paper pp 259)
 */
void
Vnoise(p, v)
	vector	*p, *v;
{
	vector s;

	s.x = p->x - OFFSET1;
	s.y = p->y - OFFSET1;
	s.z = p->z - OFFSET1;

	v->x = noise(&s);

	v->y = noise(p);

	s.x = p->x + OFFSET2;
	s.y = p->y + OFFSET2;
	s.z = p->z + OFFSET2;

	v->z = noise(&s);
}

/*
 * init_noise
 *
 *	Initialises the psuedo random gradient table G(.x, .y. z) and a
 *	psuedo random value at the lattice points (G.r). Also initialises
 *	the index table P.
 */
init_noise()
{
	vector4	v;
	int	i, n = 0, bonk;
	float	a;

	G = (vector4 *)scalloc(sizeof(vector4), NUMPTS);
	P = (int *)scalloc(sizeof(int), NUMPTS);

	i = 0;
	while (n < NUMPTS) {
		v.x = 2.0 * randinx() - 1.0;
		v.y = 2.0 * randinx() - 1.0;
		v.z = 2.0 * randinx() - 1.0;
		v.r = 2.0 * randinx() - 1.0;

		if (a = dprod(v, v) <= 1.0) {
			/*
			 * Normalise (the x, y, z compenents of) v...
			 */
			a = sqrt((double)a);
			if (a > 0.0) {
				v.x /= a;
				v.y /= a;
				v.z /= a;
			}
			G[n].x = v.x;
			G[n].y = v.y;
			G[n].z = v.z;
			G[n].r = v.r;
			n++;
		}
	}


	/*
	 * Set a random permutation of the first NUMPTS integers
	 */
	for (n = 0; n < NUMPTS; n++)
		P[n] = n;

	for (n = 0; n < NUMPTS; n++) {
		i = (int)(randtable[n] * (NUMPTS - 1));
		bonk = P[n];
		P[n] = P[i];
		P[i] = bonk;
	}
}

