/*
 * $Id: signal.c,v 2.3 1991/08/28 19:56:34 steve Exp $
 * $__Header$
 *
 * <signal.h> implementation for the UNICOS 5.1.9 version of the
 * Unidata POSIX programming environment.
 */

#include "udposix.h"
#include <stdio.h>
#include <signal.h>
#include <errno.h>

#undef	SIGMASK
#define	SIGMASK(num)		((sigset_t)1 << (num)-1)

/*
 * Simulate the POSIX sigaction() function as best we can.
 *
 * NOTES:
 *   1. The UNICOS sigset() function used below will only block the signal
 *	that invoked the registered signal handler.  This is less powerful
 *	than the POSIX sigaction() function, which can be used to block other
 *	signals as well.  Stated another way, the 'sa_mask' member of the
 *	argument 'act' is ignored and programs should try to not depend on it.
 *
 *   2.	Similarly, the 'sa_flags' member is unusable and, consequently, 
 *	ignored.
 */
    int
sigaction(sig, act, oact)
    int				sig;
    const struct sigaction	*act;
    struct sigaction		*oact;
{
    void	(*old_handler)();

    if (act != NULL) {
	if ((old_handler = sigset(sig, act->sa_handler)) == SIG_ERR)
	    return -1;
    } else {
	if ((old_handler = sigset(sig, SIG_IGN)) == SIG_ERR)
	    return -1;
	(void) sigset(sig, old_handler);
    }

    if (oact != NULL) {
	oact->sa_handler	= old_handler;
	oact->sa_mask		= 0;
	oact->sa_flags		= 0;
    }

    return 0;
}


/*
 * Add a signal to a signal-mask.
 */
    int
sigaddset(mask, num)
    sigset_t	*mask;
    int		num;
{
    int		retval;

    if (num < 1 || mask == NULL) {
	errno	= EINVAL;
	retval	= -1;
    } else {
	*mask	|= SIGMASK(num);
	retval	= 0;
    }

    return retval;
}


/*
 * Delete a signal from a signal-mask.
 */
    int
sigdelset(mask, num)
    sigset_t	*mask;
    int		num;
{
    int		retval;

    if (num < 1 || mask == NULL) {
	errno	= EINVAL;
	retval	= -1;
    } else {
	*mask	&= ~SIGMASK(num);
	retval	= 0;
    }

    return retval;
}


/*
 * Clear a signal-mask.
 */
    int
sigemptyset(mask)
    sigset_t	*mask;
{
    int		retval;

    if (mask == NULL) {
	errno	= EINVAL;
	retval	= -1;
    } else {
	*mask	= 0;
	retval	= 0;
    }

    return retval;
}


/*
 * Process a signal-mask.
 */
    int
sigprocmask(action, in_mask, out_mask)
    int			action;
    const sigset_t	*in_mask;
    sigset_t		*out_mask;
{
    int		MyErrno	= errno;

    errno	= 0;

    if (out_mask != NULL)
	*out_mask	= sigblock(0L);

    if (in_mask != NULL) {
	switch (action) {
	case SIG_BLOCK:
	    (void)sigblock(*in_mask);
	    break;
	case SIG_UNBLOCK:
	    (void)sigsetmask(sigblock(0L) & ~*in_mask);
	    break;
	case SIG_SETMASK:
	    (void)sigsetmask(*in_mask);
	    break;
	default:
	    errno	= EINVAL;
	}
    }

    if (errno == 0) {
	errno	= MyErrno;
	MyErrno	= 0;
    } else {
	MyErrno	= -1;
    }

    return MyErrno;
}


/*
 * Block the signals of a signal-mask and suspend the process.
 */
    int
sigsuspend(mask)
    const sigset_t	*mask;
{
    return bsdsigp(*mask);
}
