//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : base_vector.h 
// Author      : Frank Lehmann (FL), Markus Maurer (MM) 
//               Stefan Neis (SN), Thomas Papanikolaou (TP)
//               Patrick Theobald (PT)
// Last change : FL/MM, Feb 15 1995, initial version
//               FL/MM, Mar  8 1995
//               FL/MM, May 10 1995, added get_data_address 
//                                   clean-up
//		 FL/MM, May 15 1995, added typedef size_type and
//				     changed type of corresponding variables
//               FL/MM, May 23 1995, removed #define LiDIA_VECTOR_NO_RANGE_CHECK
//		 FL/MM, Jul  6 1995, changed internal size_type of base_vector
//				     to the global LiDIA-type lidia_size_t
//               FL/MM, Jul 24 1995, added 'const' to T* parameters of constructors
//		 FL/MM, Aug 30 1995, changed typedef int lidia_size_t to
//				     "#define ..."
//		 FL/MM, Aug 30 1995, changed "error_handler" to "lidia_error_handler"
//		 FL/MM, Sep 22 1995, removed "#define lidia_size_t ..." and
//				     "#ifdef LiDIA_VECTOR_INTERNAL_DATA_ACCESS ..."
//		 FL/MM, Sep 29 1995, changed "debug_handler" to "debug_handler_l"
//               PT,    Oct 28 1995, new template concept   
//		 MM,	Feb 01 1996, added const for first argument of "set_data(...)"            
//               SN,PT  Mar 01 1996, added HEADBANGER defines

#ifndef LIDIA_BASE_VECTOR_H
#define LIDIA_BASE_VECTOR_H

#define  EXPAND  'E'
#define  FIXED   'F'
#define  DEFAULT_RATIO 2.0

#if defined(HAVE_MAC_DIRS) || defined(__MWERKS__)
#include <LiDIA:lidia.h>
#else
#include <LiDIA/lidia.h>
#endif

#ifdef __GNUG__
class crt;
class bigint_matrix;
class bigint_lattice_gensys;
class bigfloat_lattice_gensys;
template < class T > class base_matrix;
template < class T > class math_matrix;
#endif

template <class T> 
class base_vector
{
#ifdef __GNUG__
  friend class crt;
  friend class bigint_lattice_gensys;
  friend class bigfloat_lattice_gensys;
  friend class base_matrix < T >;
  friend class math_matrix < T >;
  friend class bigint_matrix;
#endif

 protected:
  
  T *data;
  
  lidia_size_t length;
  lidia_size_t allocated;
  
  char mode;
  float dyn_exp_ratio;
  
  char sort_dir;                                 /* sort-direction */
  int (*el_cmp) ( const T & a, const T & b ) ;   /* compare-func. for vec.elements */

 protected:
     
  T * copy_data(T *d, const T *vd, lidia_size_t l) ;
  
  /**
   ** constructors
   **/
  
 public:

  base_vector();
  base_vector(lidia_size_t all, char md);                   /* creates vector of cap. "all" */
  base_vector(lidia_size_t all, lidia_size_t len);
  base_vector(lidia_size_t all, lidia_size_t len, char md); /* creates vector of cap. "all" and sz. len */
  base_vector(const base_vector < T > & v);
  base_vector(const base_vector < T > & v, char md);        /* initializes with vector v */

#ifndef HEADBANGER
  base_vector(const T *v, lidia_size_t len);
  base_vector(const T *v, lidia_size_t len, char md);        /* initializes with first len elts. of T */
#endif
  
  /**
   ** destructor
   **/
  
 public:
 
  virtual ~base_vector ( ) ;
      
  /**
   ** Input / Output
   **/

 public: 
 
  inline friend ostream & operator << (ostream & s, const base_vector < T > & x)
    {x.write(s); return s;}
  inline friend istream & operator >> (istream & s, base_vector < T > & x)
    {x.read(s); return s;}

 protected:

  void write(ostream & out) const;
  void read(istream & in);
  
  /**
   ** structur functions: functions for reading and modifying the variables of the vector 
   **/
 
 public:
 
  inline lidia_size_t capacity() const             
    {return allocated;}                       /* return the capacity of the vector */
  void set_capacity(lidia_size_t all);        /* set the capacity to all */
  
  void kill();                                /* deallocate the elements of the vector */
  
  inline lidia_size_t size() const                 
    {return length;}                          /* return the size ot the vector */
  void set_size(lidia_size_t len);            /* set the size to len */
     
  inline float exp_ratio() const
    {return dyn_exp_ratio;}                   /* return expansion ratio of the vector */
  void set_exp_ratio (float ratio);           /* set the expansion ratio to ratio */
  
  void set_mode(char md);                     /* set the mode of the vector to md */
  
  /**
   ** access functions
   **/
 
 public:
 
  T & operator[] (lidia_size_t i);              /* returns the (i+1)-st element of the vector */
  const T & operator[] (lidia_size_t i) const;  /* returns the (i+1)-st element of the vector */
                                                /* only if i is less than size() */
  T & member(lidia_size_t i) const;             /* returns the (i+1)-st element of a vector */
                                                /* only if i is less than size() */ 

  void set_data(const T * d, lidia_size_t l);         /* set vector = d */
  T *get_data() const;                          /* returns a pointer to a copy of the elements of the vector */

#ifdef __GNUG__
 private:
#else
 public:
#endif

#ifndef HEADBANGER
  inline T *get_data_address() const
    {return data;}                              /* returns a pointer to the internal elts of the vector */
#endif
 
  /**
   ** assignment 
   **/

 public: 

  base_vector < T > & operator = (const base_vector < T > &  v);  

#ifndef HEADBANGER
  friend void assign(base_vector < T > &v, const base_vector < T > &w)
    {v.assign(w);}
  void assign(const base_vector < T > &);
#endif
     
  /**
   ** reverse 
   **/

 public:

  void reverse();                            /* reverse the order of the elements of the vector */
  void reverse(const base_vector < T > & b);
  
  /**
   ** swap functions
   **/

 public:

  inline friend void swap(base_vector < T > &a, base_vector < T > &b)
    {a.swap(b);}                             /* exchange information of two vectors */

 protected:

  void swap(base_vector < T > &);
  
  /**
   ** concat / divide / shift / assign / remove_from
   **/
 
 public:
  
  void concat(const base_vector < T > & b, const base_vector < T > & c);
  
  void shift_left(lidia_size_t pos, lidia_size_t num);
  void shift_right(lidia_size_t pos, lidia_size_t num);
  
  void assign(lidia_size_t at, const base_vector< T > & v, lidia_size_t from, lidia_size_t to);
  void remove_from(lidia_size_t pos, lidia_size_t l = 1);
};

#endif














