/******************************************************************************
 ** LiDIA's Temporary Variable Manager
 ** Original Author: Ralf Dentzer
 ** Rewritten By: Thomas Papanikolaou
 ** Copyright (c) by the LiDIA Group
 ** All rights reserved
 *****************************************************************************/

#include <LiDIA/mm.h>
#include <sys/types.h>

#define MM_SIZEOF_SIZET SIZEOF_INT

/******************************************************************************
 ** special function definition. mm_priv_malloc stores the allocated
 ** size in the first sizeof(size_t) bytes, where free can then find it.
 *****************************************************************************/


static void
mm_store_size(ptr, size)
  void *ptr;
  mm_size_t size;
{
  register char *s = (char *) &size;
  register char *p = (char *) ptr;
#if MM_SIZEOF_SIZET == 2
  *p++ = *s++;
#elif MM_SIZEOF_SIZET == 4
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
#elif MM_SIZEOF_SIZET == 8
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
#endif
  *p = *s;
}

static void
mm_fetch_size(size, ptr)
  mm_size_t *size;
  void *ptr;
{
  register char *s = (char *) size;
  register char *p = (char *) ptr;
#if MM_SIZEOF_SIZET == 2
  *s++ = *p++;
#elif MM_SIZEOF_SIZET == 4
  *s++ = *p++;
  *s++ = *p++;
  *s++ = *p++;
#elif MM_SIZEOF_SIZET == 8
  *s++ = *p++;
  *s++ = *p++;
  *s++ = *p++;
  *s++ = *p++;
  *s++ = *p++;
  *s++ = *p++;
  *s++ = *p++;
#endif
  *s = *p;
}

static void *
mm_priv_malloc(size)
  mm_size_t size;
{
  char *ptr = (char *) malloc(size + MM_SIZEOF_SIZET);
  register char *s = (char *) &size;
  register char *p = (char *) ptr;
#if MM_SIZEOF_SIZET == 2
  *p++ = *s++;
#elif MM_SIZEOF_SIZET == 4
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
#elif MM_SIZEOF_SIZET == 8
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
#endif
  *p = *s;
  return (ptr + MM_SIZEOF_SIZET);
}

static void *
mm_priv_calloc(size)
  mm_size_t size;
{
  char *ptr = (char *) calloc(size + MM_SIZEOF_SIZET, 1);
  register char *s = (char *) &size;
  register char *p = (char *) ptr;
#if MM_SIZEOF_SIZET == 2
  *p++ = *s++;
#elif MM_SIZEOF_SIZET == 4
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
#elif MM_SIZEOF_SIZET == 8
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
  *p++ = *s++;
#endif
  *p = *s;
  return (ptr + MM_SIZEOF_SIZET);
}

mm_size_t
mm_priv_sizeof(ptr)
  void *ptr;
{
  mm_size_t l;
  mm_fetch_size(&l, ((char *) ptr) - MM_SIZEOF_SIZET);
  return l;
}

static void
mm_priv_free(ptr)
  void *ptr;
{
  MM_FREE((char *) ptr - MM_SIZEOF_SIZET);
}

/******************************************************************************
 ** memory element definition
 *****************************************************************************/

typedef union mm_u_mem_elt
{
  union mm_u_mem_elt *next;
  char mem;
}            mm_elt, *mm_ptr_mem_elt;

typedef struct mm_s_mem_list
{
  mm_ptr_mem_elt free;
  mm_size_t size;
}             mm_t_mem_list;

/******************************************************************************
 ** list of free memory elements and their sizes
 *****************************************************************************/

static mm_t_mem_list mm_list[MM_LISTMAX];

/******************************************************************************
 ** is mm_list initialized?
 *****************************************************************************/

static int mm_list_init = 0;

/******************************************************************************
 ** variables for holding statistics
 *****************************************************************************/

#ifdef MM_STATISTICS
static mm_size_t mm_elts_used[MM_LISTMAX];
static mm_size_t mm_elts_allocated[MM_LISTMAX];
static mm_size_t mm_elts_freed[MM_LISTMAX];
#endif

/******************************************************************************
 ** the bit-lengths of the numbers 0...255
 *****************************************************************************/

static unsigned char log2_table[] = {
  0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4,
  5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
  6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
  6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8
};

/******************************************************************************
 ** macro to calculate the bit-length of an ulong a and add it to i
 *****************************************************************************/

#if defined(__alpha)
#define mm_log2(i, a) if (a & 0xffffffff00000000) i += 32, a >>= 32; \
                      if (a & 0xffff0000) i += 16, a >>= 16; \
                      if (a & 0xff00) i += 8, a >>= 8; \
                      i += log2_table[a]
#else
#define mm_log2(i, a) if (a & 0xffff0000) i += 16, a >>= 16; \
                      if (a & 0xff00) i += 8, a >>= 8; \
                      i += log2_table[a]
#endif

/******************************************************************************
 ** initialization routine: sets up lists of elements of 2^n elelements
 *****************************************************************************/

#ifdef MM_STATISTICS

#define MM_INITIALIZE()                                                       \
                                                                              \
if (!mm_list_init)                                                            \
{ mm_size_t i, j = 1 << MM_LISTMIN;                                           \
  mm_list_init = 1;                                                           \
  for (i = MM_LISTMIN; i < MM_LISTMAX; i++, j <<= 1)                          \
  { mm_list[i].size = j;                                                      \
    mm_list[i].free = NULL;                                                   \
    mm_elts_used[i] = 0;                                                      \
    mm_elts_allocated[i] = 0;                                                 \
    mm_elts_freed[i] = 0;                                                     \
  }                                                                           \
}

#else

#define MM_INITIALIZE()                                                       \
                                                                              \
if (!mm_list_init)                                                            \
{ mm_size_t i, j = 1 << MM_LISTMIN;                                           \
  mm_list_init = 1;                                                           \
  for (i = MM_LISTMIN; i < MM_LISTMAX; i++, j <<= 1)                          \
  { mm_list[i].size = j;                                                      \
    mm_list[i].free = NULL;                                                   \
  }                                                                           \
}
#endif

void
mm_initialize()
{
  MM_INITIALIZE();
}

/******************************************************************************
 ** releases the memory collected in the free lists
 *****************************************************************************/

#define MM_COLLECT()                                                          \
{                                                                             \
  register mm_size_t j;                                                       \
  register mm_ptr_mem_elt u;                                                  \
  fprintf(stderr, "\nmm:: garbage collection:: ");                            \
  for (j = MM_LISTMIN; j < MM_LISTMAX; j++)                                   \
    {                                                                         \
      while ((u = mm_list[j].free))                                           \
	{                                                                     \
	  mm_list[j].free = u->next;                                          \
	  MM_FREE((char *) u - MM_SIZEOF_SIZET);                              \
          MM_STATISTIC(mm_elts_freed[j]++);                                   \
	}                                                                     \
    }                                                                         \
  fprintf(stderr, "done");                                                    \
}

void
mm_collect()
{
  MM_COLLECT();
}

/******************************************************************************
 ** sets size = next_pow2_of(size) and allocates size mm_units. This
 ** call does not check if the memory manager is initialized.
 *****************************************************************************/

void *
mm_malloc_without_init(size)
  mm_size_t size;
{
  register mm_size_t i = MM_LISTMIN;
  register mm_ptr_mem_elt u;
  if (size)
  {
    size--;
    size >>= i;
    mm_log2(i, size);
    if (i >= MM_LISTMAX)
      mm_error("mm_malloc", "exceeded MM_LISTMAX");
  }
  size = mm_list[i].size;
  u = mm_list[i].free;
  if (u)
  {
    mm_list[i].free = u->next;
    MM_STATISTIC(mm_elts_used[i]++);
  }
  else
  {
    u = (mm_ptr_mem_elt) mm_priv_malloc(size * mm_unit_size);
    if (!u)
    {
      MM_COLLECT();
      u = (mm_ptr_mem_elt) mm_priv_malloc(size * mm_unit_size);
      if (!u)
	mm_error("mm_malloc", "memory full");
    }
    MM_STATISTIC(mm_elts_used[i]++;
    mm_elts_allocated[i]++);
  }
  return (void *) u;
}

/******************************************************************************
 ** sets size = next_pow2_of(size) and allocates size mm_units. This
 ** call checks if the memory manager is initialized.
 *****************************************************************************/

void *
mm_malloc(size)
  mm_size_t size;
{
  MM_INITIALIZE();
  return mm_malloc_without_init(size);
}

/******************************************************************************
 ** sets size = next_pow2_of(size) and allocates size mm_units. This
 ** call does not check if the memory manager is initialized. The memory is
 ** cleared.
 *****************************************************************************/

void *
mm_calloc_without_init(nmemb, size)
  mm_size_t nmemb;
  mm_size_t size;
{
  register mm_size_t i = MM_LISTMIN;
  register mm_ptr_mem_elt u;
  size *= nmemb;
  if (size)
  {
    size--;
    size >>= i;
    mm_log2(i, size);
    if (i >= MM_LISTMAX)
      mm_error("mm_calloc", "exceeded MM_LISTMAX");
  }
  size = mm_list[i].size;
  u = mm_list[i].free;
  if (u)
  {
    char *p, *e;
    mm_list[i].free = u->next;
    size *= mm_unit_size;
    for (p = (char *) u, e = p + size; p != e;)
      *p++ = (char) 0;
    MM_STATISTIC(mm_elts_used[i]++);
  }
  else
  {
    u = (mm_ptr_mem_elt) mm_priv_calloc(size * mm_unit_size);
    if (!u)
    {
      MM_COLLECT();
      u = (mm_ptr_mem_elt) mm_priv_calloc(size * mm_unit_size);
      if (!u)
	mm_error("mm_calloc", "memory full");
    }
    MM_STATISTIC(mm_elts_used[i]++;
    mm_elts_allocated[i]++);
  }
  return (void *) u;
}

/******************************************************************************
 ** sets size = next_pow2_of(size) and allocates size mm_units. This
 ** call checks if the memory manager is initialized. The memory is cleared.
 *****************************************************************************/

void *
mm_calloc(nmemb, size)
  mm_size_t nmemb;
  mm_size_t size;
{
  MM_INITIALIZE();
  return mm_calloc_without_init(nmemb, size);
}

/******************************************************************************
 ** reallocs the memory u
 *****************************************************************************/

void *
mm_realloc_without_init(u, malloc_size)
  void *u;
  mm_size_t malloc_size;
{
  if (malloc_size == 0)
  {
    mm_free(u);
    return (u = (void *) 0);
  }
  else if (u == (void *) 0)
    return (u = mm_malloc_without_init(malloc_size));
  else
  {
    mm_size_t old_size = mm_priv_sizeof(u);
    if (malloc_size < old_size)
      return u;
    else
    {
      char *cp = (char *) mm_malloc_without_init(malloc_size);
      register char *p = cp, *q = (char *) u, *e = cp + old_size;
      for (; p != e;)
	*p++ = *q++;
      mm_free(u);
      return (void *) (u = cp);
    }
  }
}

/******************************************************************************
 ** reallocs the memory u
 *****************************************************************************/

void *
mm_realloc(u, malloc_size)
  void *u;
  mm_size_t malloc_size;
{
  MM_INITIALIZE();
  return mm_realloc(u, malloc_size);
}

/******************************************************************************
 ** releases the memory u
 *****************************************************************************/

void
mm_free(u)
  void *u;
{
  register mm_size_t size = mm_priv_sizeof(u);
  register mm_size_t i = MM_LISTMIN;
  register mm_ptr_mem_elt v = (mm_ptr_mem_elt) u;
  if (size)
  {
    size--;
    size >>= i;
    mm_log2(i, size);
  }
  v->next = mm_list[i].free;
  mm_list[i].free = v;
  MM_STATISTIC(mm_elts_used[i]--);
}

/******************************************************************************
 ** sets size = next_pow2_of(size) and releases size mm_units in u
 *****************************************************************************/

void
mm_free_with_size(u, size)
  void *u;
  mm_size_t size;
{
  register mm_size_t i = MM_LISTMIN;
  register mm_ptr_mem_elt v = (mm_ptr_mem_elt) u;
  if (size)
  {
    size--;
    size >>= i;
    mm_log2(i, size);
  }
  v->next = mm_list[i].free;
  mm_list[i].free = v;
  MM_STATISTIC(mm_elts_used[i]--);
}

/******************************************************************************
 ** prints a message s::t
 *****************************************************************************/

void
mm_message(s, t)
  char *s;
  char *t;
{
  fprintf(stderr, "mm:: message:: %s:: %s\n", s, t);
}

/******************************************************************************
 ** prints a warning s::t
 *****************************************************************************/

void
mm_warning(s, t)
  char *s;
  char *t;
{
  fprintf(stderr, "mm:: warning:: %s:: %s\n", s, t);
}

/******************************************************************************
 ** prints an error s::t and exits using exit(1)
 *****************************************************************************/

void
mm_error(s, t)
  char *s;
  char *t;
{
  MM_STATISTIC(mm_statistics());
  fprintf(stderr, "mm:: error:: %s:: %s\n", s, t);
  exit(1);
}

/******************************************************************************
 ** prints long memory statistics for each free list
 *****************************************************************************/

#ifdef MM_STATISTICS
void
mm_long_statistics()
{
  register mm_size_t i;
  fprintf(stderr, "\nMM memory management statistics:");
  fprintf(stderr,
	  "\nsize(%4s)  size(bytes)  allocated        used       freed\n",
	  mm_unit_str);
  for (i = MM_LISTMIN; i < MM_LISTMAX; i++)
  {
    fprintf(stderr, "%10ld  %10lu  %10lu  %10lu  %10lu\n",
	    mm_list[i].size, mm_list[i].size * mm_unit_size,
	    mm_elts_allocated[i], mm_elts_used[i],
	    mm_elts_freed[i]);
  }
}
#else
void
mm_long_statistics()
{
}
#endif

/******************************************************************************
 ** prints memory statistics for the memory used
 *****************************************************************************/

#ifdef MM_STATISTICS
void
mm_statistics()
{
  register mm_size_t i, al = 0, us = 0, fr = 0;
  for (i = MM_LISTMIN; i < MM_LISTMAX; i++)
  {
    al += (mm_elts_allocated[i] << i) * mm_unit_size;
    us += (mm_elts_used[i] << i) * mm_unit_size;
    fr += (mm_elts_freed[i] << i) * mm_unit_size;
  }
  fprintf(stderr, "\nbytes allocated=%ld used=%ld freed=%ld", al, us, fr);
}
#else
void
mm_statistics()
{
}
#endif
/*****************************************************************************/
