#define LIDIA_POINTER_ACCESS
#if defined(HAVE_MAC_DIRS) || defined(__MWERKS__)
#include <LiDIA:alg_number.h>
#include <LiDIA:debug.h>
#include <LiDIA:base_vector.h>
#else
#include <LiDIA/alg_number.h>
#include <LiDIA/debug.h>
#include <LiDIA/base_vector.h>
#endif

#ifdef LIDIA_DEBUG
int module::count = 0;
#endif

// Constructors & destructor:
module::module(const alg_number & a, const alg_number & b):
    base(a.degree(), 1), den(a.den), O(a.which_order())
{
    debug_handler_c("module","in module(const a_n, const a_n)",1,
		    count++;
		    cout << "\nNow we have "<<count<<" modules!\n");
    if (b.is_zero())
      base.sto_column_vector(a.coeff, a.degree(),0);
    else if (a.is_zero()){
      base.sto_column_vector(b.coeff, b.degree(),0);
      den.assign(b.den);
    }
    else {
	bigint d = gcd(a.den, b.den);
	bigint e = b.den / d;
	den *= e;
	base.set_no_of_columns(2);
	base.sto_column_vector((a.coeff) * e, a.degree(), 0);
	base.sto_column_vector((b.coeff) * (a.den / d), a.degree(), 1);
    }
    debug_handler_c("module","in module(const a_n, const a_n)",3,
		    cout << " Now the module is "<<(*this));
//    multiply(*this,*this,*O);
    debug_handler_c("module","in module(const a_n, const a_n)",3,
		    cout << " So the ideal is "<<(*this));
}    

module::module(const base_matrix <bigint> & B,
	     const bigint & d,
	     const order * O1):base(B), den(d), O(O1)
{
    debug_handler_c("module",
		    "in module(const b_m &, const bigint &, const order *)",1,
		    count++;
		    cout << "\nNow we have "<<count<<" modules!\n");
    if (B.get_no_of_rows() != O1->degree())
	error_handler("module","module(const bigint_matrix &, ....):: "
		      "Dimension of matrix does not match degree of order");
    normalize();
}

module::module(const order * O1):base(O1->degree(), 1), den(1), O(O1)
{
    debug_handler_c("module","in module(const order *)",1,
		    count++;
		    cout << "\nNow we have "<<count<<" modules!\n");
}

module::module(const module & M):base(M.base), den(M.den), O(M.O)
{
    debug_handler_c("module","in module(const module &)",1,
		    count++;
		    cout << "\nNow we have "<<count<<" modules!\n");
}

module::~module()
{
    debug_handler_c("module","in ~module()",1,
		    count--;
		    cout << "\nNow we have only "<<count<<" modules!\n");
}

// member-functions
bool module::is_zero() const
{
    bool res=true;
    for (lidia_size_t i = 0; res && (i < base.get_no_of_columns()); i++)
	res = base.is_column_zero(i);
    return res;
}

void module::normalize()
{
    if (den.is_negative()) den.negate();
    bigint d = den;
    register lidia_size_t i,j;
    for(i=0; i<degree(); i++)
	for (j=0; j<base.get_no_of_columns(); j++)
	    d = gcd (base.member(i,j), d);
    if (!d.is_one()) {
	den  /= d;
	base /= d;
    }
}

void module::invert()
{
    debug_handler("module","in member - function invert()");
    // Produce a basis of full Z-rank
    lidia_size_t n=base.get_no_of_columns();
    if (n !=degree()){
	multiply(*this,*this,*O);
	if (base.get_no_of_columns()!=degree())
	    lidia_error_handler("module","invert():: Cannot invert (0)");
	n = degree();
    }
    bigint d=den;
    den = 1;		//Multiply *this by den !!

    //Now we have an integral Z -- module of full rank.
    // Compute and save the norm
    bigint N=norm(*this).numerator();	//it's anyway an integer!!
    // Compute: O ---> Hom( M --> O/(N(M)) )
    //		a ---> (    b --> a b      )
    alg_number a,b,y;

    bigint_matrix Map(n*n,n*(n+1));
    bigint * tmp = new bigint[n], * tmp2, rem;

    register lidia_size_t i;

    for (i=0;i<n;tmp[i++] = 0){
 	tmp[i]=1;		//tmp=e_i;
 	// compute the images of the basis of O
 	a= alg_number(tmp,1,O);
 	//a=e_i
 	for (register lidia_size_t j=0; j < n; j++){
                        // compute the images of the basis of M
                        // under the MULT-with-a-homomorphism
 	    b=alg_number(tmp2 = base.column(j), 1, O);
 	    delete[] tmp2;
 	    y=a*b;   // the image under the MULT-with-a-homomorphism
	    tmp2 = y.coeff_vector().get_data();
	    for(register lidia_size_t k=0;k<n;k++){
		remainder(rem, tmp2[k], N);
		Map.sto(j*n+k,i,rem);
	    }
	    delete[] tmp2;
 	}
    }
    delete[] tmp;

    // Complete initialization of map by use of a (n*n)x(n*n) diagonal matrix,
    // with N(M) on the diagonal.
    register lidia_size_t j=n;
    for(i=0;i<n*n;i++,j++)
	Map.sto(i,j,N);

    // Computing the kernel of this matrix and selecting the right rows
    bigint_matrix kern = kernel(Map);
    kern.set_no_of_rows(n);
    assign(module(kern, N, O));
    base.hnf();
    normalize();
    if (d != 1) multiply(*this,*this,d);
}

void module::assign_zero()
{
    base.set_no_of_columns(1);
    for (lidia_size_t i=0;i<degree();i++)
	base.sto(i,0,bigint(0));
    den = 1;    
}

void module::assign(const bigint & a)
{
    debug_handler("module","in member - function assign(const bigint &)");
    base.set_no_of_columns(1);
    base.sto_column_vector((O->get_one()) * a,degree(),0);
    den = 1;
}

void module::assign(const alg_number & a)
{
    debug_handler("module","in member - function assign(a_n &)");
    base.set_no_of_columns(1);
    base.set_no_of_rows(a.degree());
    O = a.O;
    base.sto_column_vector(a.coeff_vector(),degree(),0);
    den = a.den;
}

void module::assign(const module & a)
{
    debug_handler("module","in member - function assign(module &)");
    O = a.O;
    base = a.base;
    den = a.den;
}

// Procedural versions:
void add(module & c, const module & a, const module & b)
{
    debug_handler("module","in function "
		  "add(module &, const module &, const module &)");
    if (a.O != b.O) {
	lidia_error_handler("module","add(...):: Addition of modules from "
		      "different orders is not yet implemented");
    }

    bool inflate = (a.base.get_no_of_columns() < a.degree()|| 
		    b.base.get_no_of_columns() < b.degree()) &&
 	           (a.base.get_no_of_columns() + b.base.get_no_of_columns()
		             >= b.degree());
//    module a=aa;
//     if (a.base.get_no_of_columns() < a.degree())
// 	multiply(a,a,*a.O);
//     debug_handler_l("module","in add(...) -- inflated first argument",0);

//     module b=bb;
//     if (b.base.get_no_of_columns() < b.degree())
// 	multiply(b,b,*b.O);
//     debug_handler_l("module","in add(...) -- inflated second argument",0);

    bigint d = gcd(a.den,b.den);
    bigint e = a.den / d;
    bigint_matrix tmp(a.degree(), a.degree()<<1);

    debug_handler_c("module", "add", 0,
		    cout<< "Adding "<<a<<" and "<<b<<flush);

    bigint_matrix multhelp1, multhelp2;
    multiply(multhelp1, a.base, b.den / d);
    multiply(multhelp2, b.base, e);
    tmp.compose_h(multhelp1, multhelp2);
//  i.e. tmp.compose_h(a.base * (b.den/d), b.base * e);
    debug_handler_c("module", "in add(...) -- tmp is", 0, cout<<tmp);

    tmp.hnf();
    debug_handler_c("module", "in add(...) -- reduced tmp is", 0, cout<<tmp);

    c.O = a.which_order();
    c.base.set_no_of_rows(a.degree());
    c.base.set_no_of_columns(tmp.rank());
    tmp.split_h(tmp, c.base);
    multiply(c.den, e, b.den);
    c.normalize();
    if (inflate) multiply(c, c, *c.O);
}

void intersect(module & c, const module & aa, const module & bb)
{
    debug_handler("module","in function intersect(module, const & module, const & module)");
     if (aa.O != bb.O) {
	lidia_error_handler("module","intersect(...):: Intersection of modules from "
		      "different orders is not yet implemented");
    }

    module a=aa;
    if (a.base.get_no_of_columns()<a.degree())
	multiply(a,a,*a.O);
    debug_handler_l("module","in intersect(...) -- inflated first argument",0);

    module b=bb;
    if (b.base.get_no_of_columns()<b.degree())
	multiply(b,b,*b.O);
    debug_handler_l("module","in intersect(...) -- inflated second argument",0);

    bigint d = gcd(a.den,b.den);
    bigint e = a.den / d;
    c.O = a.which_order();
    c.base.set_no_of_rows(a.degree());
    bigint_matrix tmp2(a.degree(), a.base.get_no_of_columns()
		                 + b.base.get_no_of_columns());
    bigint_matrix tmp1, kern;
    multiply(tmp1, a.base, b.den / d);    
    tmp2.compose_h(tmp1, (-b.base) * e);
    kern.kernel(tmp2);
    bigint * tmp, * tmp3;
    tmp2.set_no_of_columns(a.degree());
    for (lidia_size_t i=0;i<a.degree();i++){
	tmp2.sto_column(tmp=(tmp1 * (tmp3 = kern.column(i))), a.degree(), i);
	delete[] tmp;
	delete[] tmp3;
    }
    tmp1.image(tmp2);
    tmp1.hnf();
    c.base.set_no_of_columns(tmp1.rank());
    kern.set_no_of_rows(a.degree());
    debug_handler_c("module", "in intersect(...) -- starting divide", 0,
		    cout <<"tmp1.columns ="<<tmp1.get_no_of_columns();
		    cout <<"kern.columns ="<<kern.get_no_of_columns();
		    cout <<"c.base.columns ="<<c.base.get_no_of_columns();
		    cout <<"tmp1.rows ="<<tmp1.get_no_of_rows();
		    cout <<"kern.rows ="<<kern.get_no_of_rows();
		    cout <<"c.base.rows ="<<c.base.get_no_of_rows());
    tmp1.split_h(kern,c.base);
    c.den = e * b.den;
    c.normalize();
}

void multiply(module & c, const module & a, const module & b)
{
    debug_handler("module","in function multiply(module, const & module, const & module)"); 
    if (a.O != b.O) {
	lidia_error_handler("module","multiply(...):: Multiplication of modules "
		      "from different orders is not yet implemented");
    }

    c.O = a.which_order();
    c.base.set_no_of_rows(a.degree());

    bool inflate = a.base.get_no_of_columns() < a.degree() &&
	           b.base.get_no_of_columns() < b.degree() &&
 	           a.base.get_no_of_columns() * b.base.get_no_of_columns()
		             >= b.degree();

    bigint_matrix tmp1(a.degree(), a.base.get_no_of_columns()
		                 * b.base.get_no_of_columns());
    alg_number n1,n2;
    bigint * tmp;

    register lidia_size_t i;

    debug_handler_c("module","multiply(...)",0,cout<<a<<" and "<<b);
    for(i=0; i<a.base.get_no_of_columns(); i++){
	alg_number n1(tmp=a.base.column(i), 1, a.which_order());
	delete[] tmp;
	for(register lidia_size_t j=0; j<b.base.get_no_of_columns(); j++){
	    alg_number n2(tmp=b.base.column(j), 1, b.which_order());
	    delete[] tmp;
	    debug_handler_c("module","multiply::multiply elements",1,
			    cout<<i*b.base.get_no_of_columns()+j);
	    tmp1.sto_column_vector((n1*n2).coeff_vector(),
				   b.degree(),i*b.base.get_no_of_columns()+j);
	}

    }
    tmp1.hnf();
    if ((i=tmp1.rank())==0){
	c.base.set_no_of_columns(1);
	inflate = false;
    }
    else{
	c.base.set_no_of_columns(i);
    }
    tmp1.split_h(tmp1, c.base);
    c.den = a.den * b.den;
    c.normalize();
    if (inflate) multiply(c, c, *c.O);
    debug_handler("module","multiply::end of call");
}

void multiply(module &c, const module &a, const bigint & b)
{
    bigint d = gcd(a.den, b);

    c.O = a.which_order();
    c.den = a.den / d;
    multiply(c.base, a.base, (b / d));
}

void multiply(module &c, const bigint &b, const module &a)
{
    bigint d = gcd(a.den, b);

    c.O = a.which_order();
    c.den = a.den / d;
    multiply(c.base, a.base, (b / d));
}

void divide(module &c, const module &a, const bigint & b)
{
    debug_handler("alg_numbers",
		  "in function divide(module &, const module &, const bigint &)");

    c.O = a.which_order();
    c.base = a.base;

    if (&c != &a){
	c.den = b;
	c.normalize();
	c.den *= a.den;
    }
    else{
	bigint tmp=a.den;
	c.den = b;
	c.normalize();
	c.den *= tmp;
    }
}

//void divide(module & c, const module & a, const module & b)
//{
//    if (a.O != b.O) {
//	lidia_error_handler("module","divide(...):: Division of modules from "
//		      "different orders is not yet implemented");
//    }
//    multiply(c,a,inverse(b));
//}

void divide(module & c, const module & aa, const module & bb)
{
    debug_handler("module","in function divide(module &, const module &, "
		  "const module &");
    if (aa.O != bb.O) {
	lidia_error_handler("module","divide(...):: Division of modules from "
		      "different orders is not yet implemented");
    }
    module a = aa, b = bb;
    // Produce a basis of full Z-rank
    lidia_size_t n=b.base.get_no_of_columns();
    if (n != b.degree()){
	multiply(b,b,*(b.O));
	if (b.base.get_no_of_columns()!=b.degree())
	    lidia_error_handler("module","invert():: Cannot invert (0)");
	n = b.degree();
    }
    if (a.base.get_no_of_columns() != a.degree()){
	multiply(a, a, *(a.O));
	if (a.base.get_no_of_columns()!=a.degree()){
	    c.O =  a.O;
	    c.base.set_no_of_rows(a.base.get_no_of_rows());
	    c.assign_zero();
	    return;
	}
    }
    bigint d=b.den;
    b.den = 1;		// Multiply b by den !!

    //Now we have two integral Z -- modules of full rank.
    // Compute and save the norm of B (which is an integer !!)
    bigint N=exp(b).numerator();
    //    bigint Nsicher(N);
    //    power(N,a.den, a.degree());
    //    N = (N * norm(a)).numerator();
    //    multiply(N,N,Nsicher);

    // Compute: A ---> Hom( B --> A/(N(B) A) )
    //		a ---> (    b --> a b      )

    alg_number x,y,z;

    bigint_matrix Map(n*n,n*(n+1));
    bigint * tmp, rem;

    register lidia_size_t i;

    for (i=0;i<a.base.get_no_of_columns();i++){
 	// compute the images of the basis of A
 	x=alg_number(tmp = a.base.column(i), 1, a.O);
	delete[] tmp;
 	for (register lidia_size_t j=0; j < n; j++){
                        // compute the images of the basis of B
                        // under the MULT-with-x-homomorphism
 	    y=alg_number(tmp = b.base.column(j), 1, b.O);
 	    delete[] tmp;
 	    z=x*y;   // the image under the MULT-with-a-homomorphism
	    tmp = z.coeff_vector().get_data();
	    //  Put the image in the matrix of the map
	    for(register lidia_size_t k=0;k<n;k++)
	      Map.sto(j*n+k,i,tmp[k]);
	    delete[] tmp;
 	}
    }

    // Complete initialization of map by use of a (n*n)x(n*n) diagonal matrix,
    // containing N(B)* A ... on the diagonal.
    bigint_matrix NtimesA;
    multiply(NtimesA, a.base, N);

    register lidia_size_t j=n;
    for(i=0;i<n*n;i++,j++)
      for (register lidia_size_t k=0;k<n;k++)
	Map.sto(k+(i/n)*n,j,NtimesA(k,i % n));
   
    // Computing the kernel of this matrix and selecting the right rows
    debug_handler_c("module","divide",2,
		    cout << "Computing kernel of " << Map<<endl);
    bigint_matrix kern = kernel(Map);
    debug_handler_c("module","divide",2,
		    cout << "Verify kernel: " <<Map*kern<<endl);
    kern.set_no_of_rows(n);
    debug_handler_c("module","divide",2,
		  cout << "Kernel is " << kern<<endl);
    c.assign(module(a.base*kern, N, a.O));
    c.base.hnf();
    c.normalize();
    if (!a.den.is_one()) divide(c, c, a.den);
    if (!d.is_one()) multiply(c, c, d);
    debug_handler_c("module","divide",2,
		    cout << "so c is " << c<<endl);
    debug_handler_c("module","divide",2,
		    cout << "Verify: "<< bb*c<<a*N<<endl);
}

void remainder(module &c, const module &a, const bigint &p)
{
    bigint_matrix garbage;
    c.O = a.which_order();
    if (!a.den.is_one())
	error_handler("module", "remainder:: Reducing mod p for a fractional "
		      "ideal ??");
    c.den.assign_one();

    remainder(garbage, a.base, p);
    garbage.hnf();
    register long i=garbage.rank();
    if (i==0){
	c.base.set_no_of_columns(1);
    }
    else{
	c.base.set_no_of_columns(i);
    }
    garbage.split_h(garbage, c.base);
}

void divide(module & C, const module & A, const module & B, const bigint & p)
{
// this is a special case of ideal division :

// A and B in O both contain the prime number p, which generates pO
// in addition, the ideal B/A must lie in O

// Then the ideal C+pO= (B+pO)/(A+pO) is the kernel of the map

//      f:  O/pO ----> (A+pO ----> (O/pO / B+pO))
//           a   |----->  ( x |----> a*x )

// we denote O/pO + B/pO by O1

  //    lidia_size_t i,j,k,l,m,n;
  //    alg_number a,x,y;

//   static ideal_mod_p U;
//   int U_initialized=0;

//   if (!U_initialized) 
// 	{
// 	U_initialized++;
// 	U=the_whole_ring();
// 	}

//   n=dimension;
//   if (A==U)
//      {
//      C=B;
//      return C;
//      }

//   if (B==A) 
// 	{
// 	C=the_whole_ring();
// 	return C;
// 	}

  //    lidia_size_t na=A.base.get_no_of_columns(), nb=B.base.get_no_of_columns();

//     bigmod_matrix O1(n,n,p), M(n-nb,na,p);
//     bigmod_matrix VW(na*(n-nb),n,p);
//     bigmod_matrix KERN_VW(p);
//     bigmod_matrix v(n,2,p);

//     debug_handler("module","in function divide(module &, const module &, "
// 		  "const module &, const bigint &)");

// //   O1=basis_completion(B);  // compute the quotient by basis completion

//     bigint * tmp = new bigint[n];
//     bigint * tmp2;
//     for (i=0;i<n;tmp[i++] = 0){   // compute the images of the basis of O/pO
// 	tmp[i]=1;
// 	a = alg_number(tmp,1,A.O);
// 	// a = e_i
// 	for (j=0;j<na;j++){
//                         // compute the images of the basis of A+pO
//                         // under the MULT-with-a-homomorphism
// 	    x=alg_number(tmp2 = A.base.column(j), 1, A.O);
// 	    delete[] tmp2;
// 	    y=a*x;   // the image under the MULT-with-a-homomorphism

// //           v=solve(O1,y);

// // cout << "solution of O1*x=y:\n" << v << "\n" << flush;
// // store the vector in M and forget the part of B,
// // i.e. the first nb entries

// 	    for (m=nb;m<n;m++){
//  		M.sto(m-nb,j,v.member(k,1));
// 	    }

// 	    // M is the matrix of the MULT-with-a hom.
// 	    // the rows contain the images of this hom
// 	}

// //	cout << "M =\n" << M << "++++++++++++++++\n";

// // now the matrix will be represented as vector 
// // in the matrix VW, which describes the map f

// //       for (l=0;l<na;l++)
// //           for (k=0;k<n-nb;k++)
// //      		VW.sto(k+(l-1)*(n-nb),i,M.member(k,l));
// // cout << "operator/(ideal_mod_p,ideal_mod_p): the matrix VW : \n" << VW;

// //	C.kernel(VW,factor);
// //      }
// //	cout << "operator/ ... ENDS,result is:\n" << C <<flush;
//     }
}

void power(module & c, const module & a, const bigint & b)
{
    debug_handler("module", "in function power(module &, "
		  "const module &, const bigint &)");
    bigint exponent;
    const order * O = a.which_order();
    module multiplier(O);
    
    c.O=O;
    if (b.is_negative())
 	power(c, inverse(a), -b);
    else if (b.is_zero() || a.is_one())
 	c.assign_one();
    else{	
 	exponent.assign(b);
 	multiplier.assign(a);
 	c.assign_one();
  	while (exponent.is_gt_zero()){
 	    if (!exponent.is_even())
 		multiply(c, c, multiplier);
 	    square(multiplier, multiplier);
 	    exponent.divide_by_2();
 	}
    }
}

// Comparision:
// By now, only comparision of modules over the same order is implemented.
// One function to compute the needed information:
void module::compare(const module & a, bool & equal,
		     bool & a_in_this) const 
{
    debug_handler("module","in member - function compare(...)");
    if (O != (a.O))	//we should use *O != *(a.O)
	error_handler("module","compare::You tried to compare modules over "
		      "different orders!");
    if (base.get_no_of_columns() != base.get_no_of_rows()){
	if (a.base.get_no_of_columns() != a.base.get_no_of_rows())
	    ((*this)*(*O)).compare(a*(*O), equal, a_in_this);
	else
	    ((*this)*(*O)).compare(a, equal, a_in_this);
	return;
    }
    else if (a.base.get_no_of_columns() != a.base.get_no_of_rows()){
	compare(a*(*O), equal, a_in_this);
	return;
    }

    // Check for equality - also useful, in case you're not checking 
    // for equality since either fast or sufficient
    debug_handler_c("module","compare",1,
		    cout<<"comparing"<<(*this)<<" and "<<a);
    if (den != a.den) equal = false;
    else if (base != a.base) equal = false;
    else {
	equal = true;
	return;
    }
    
    //Check for inclusion - useless, if you check for (in-)equality!!!
    bigint rem;
    remainder(rem, den, a.den);
    if (!rem.is_zero()){
	a_in_this = false;
	return;
    }

    bigint_matrix T;
    T.reginvimage(base, a.base);
    a_in_this=true;
    for (lidia_size_t i=0; a_in_this && (i < degree()); i++)
	if (T.member(degree(),i) != bigint(1)) a_in_this=false;
}

bool operator ==(const module & a, const module & b)
{
    bool equal, b_in_a;
    debug_handler_l("module","operator ==",2);
    a.compare(b, equal, b_in_a);
    return equal;
}

bool operator !=(const module & a, const module & b)
{ 
    bool equal, b_in_a;
    a.compare(b, equal, b_in_a);
    return !equal;
}

bool operator <=(const module & a, const module & b) // Same as divisibility!!!!
{
    bool equal, a_in_b;
    b.compare(a, equal, a_in_b);
    return (equal || a_in_b);
}

bool operator <(const module & a, const module & b)
{ 
    bool equal, a_in_b;
    b.compare(a, equal, a_in_b);
    return (!equal && a_in_b);
}

bool operator >=(const module & a, const module & b)
{
    bool equal, b_in_a;
    a.compare(b, equal, b_in_a);
    return (equal || b_in_a);
}

bool operator >(const module & a, const module & b)
{
    bool equal, b_in_a;
    a.compare(b, equal, b_in_a);
    return (!equal && b_in_a);
}

// Some number-theoretic functions:
bigrational norm(const module & a)	// Norm
{
    if (a.base.get_no_of_columns()== a.degree()){
	bigint normdenominator;
	power(normdenominator, a.den, bigint(a.degree()));
	return bigrational(a.base.det(),normdenominator);
    }
    else {
	// Since everything else is interpreted as O-module base!
	return norm(a*(*a.O));
    }
}

bigrational exp(const module & a)
{
    
    lidia_size_t n = a.base.get_no_of_columns();
    if (n!=a.degree()) return exp(a*(*a.O));
    bigint_matrix res(n+1,n), Id(n,n);
    
    Id.diag(1,0);
    res.reginvimage(a.base,Id);
    
    bigint result=res(n,0);
    for (lidia_size_t i=1;i<n;i++)
	result = lcm(result,  res(n,i));
    return bigrational(result,a.den);
}

order module::ring_of_multipliers(const bigint &p, bigint & factor) const
{
    // We are here in a member function of `Ip'
    // Consider the kernel C of the map:
    // O/pO -> End(Ip/pIp)
    //   a  -> (b -> ab)
    // then the result O' fullfills: O' =1/p * C
    // quite similar to `pseudo-radical', but more complicated, since
    // now we have matrices for each (b -> ab).

    debug_handler("module","in member - function ring_of_multipliers(const bigint &, bigint &)");
    register lidia_size_t i,j,k;
    register lidia_size_t n=degree();
    alg_number a,b,y;

    bigint_matrix init(n,n*n);
    bigint_matrix B(base);
    bigmod_matrix C;
    bigmod_matrix VW(n * n, n, p);
    bigint_matrix v;
    bigint * tmp = new bigint[n];
    bigint * tmp2;
    bigint rem;
    
    if (base.get_no_of_columns()!= n)
	B.assign((*this * (*O)).base);

    for (i=0;i<n;tmp[i++] = 0){
	tmp[i]=1;		//tmp=e_i;
	// compute the images of the basis of O/pO
	a= alg_number(tmp,1,O);
	//a=e_i
	for (j=0; j < n; j++){
                       // compute the images of the basis of A+pO
                       // under the MULT-with-a-homomorphism
	    b=alg_number(tmp2 = B.column(j), 1, O);
	    delete[] tmp2;
	    y=a*b;   // the image under the MULT-with-a-homomorphism
	    init.sto_column_vector(y.coeff_vector(),n,i*n+j);
	}
    }
    delete[] tmp;

// the image must be written relative to base;
  debug_handler_c("module", "in member - function ring_of_multipliers(...)",
		  1, cout <<"solve "<<B<<init<<endl<<flush );
  v.reginvimage(B, init);
  debug_handler_c("module", "in member - function ring_of_multipliers(...)",
		  1, cout <<"solution is "<<v<<endl<<flush);
  // move the result to the Matrix VW
  for (i=0;i<n;i++){
    for (j=0; j < n; j++){
      for (k=0;k<n;k++){
	VW.sto(k+j*n,i,v.member(k,i*n+j));
      }
    }
  }
  debug_handler_c("module", "in member - function ring_of_multipliers(...)",
		  1, cout << "Compute kernel of"<<VW);
  C.kernel(VW,factor);
  if (!factor.is_one()){
    debug_handler_c("module", "in member - function ring_of_"
		    "multipliers(...)", 1,
		    cout << "While computing kernel mod "<< p;
		    cout <<" found factor "<< factor << endl);
    return (*O);
  }
  debug_handler_c("module", "in member - function ring_of_multipliers(...)",
		  1, cout << "Kernel is " << C << flush);
 
  // interpret C as a module M and lift it into the order!!
  init.set_no_of_columns(k=C.get_no_of_columns());
  base_vector <bigint> new_tmp;
  for (j = 0; j < n; j++){
    C.row_vector(new_tmp,j);
    init.sto_row_vector(new_tmp,k,j);
  }
  module M(init,1,O);
  M+= p * (*O);	// hopefully interpreted as module-mult.
  debug_handler_c("module","in member - function ring_of_multipliers(...)",
		  4,cout <<"Module is"<<M<<flush);

//  Instead of creating the multiplication table, we create the base:
  if (O->K != NULL){
    M.den *= p;
    if (O->base_computed()){
      debug_handler_c("module",
		      "in member - function ring_of_multipliers(...)",
		      4,cout <<"Multiply module by "<<O->base<<flush);
      multiply(M.base, O->base, M.base);
      M.den *= O->den;
    }
    M.normalize();
    debug_handler_c("module","ring_of_multipliers(...)",4,
		    cout << "Transformation is  1/" << M.den << " * ";
		    cout << M.base << endl << flush);

    order result(M.base, M.den, O->K);
    debug_handler_c("module","ring_of_multipliers(...)",4,
		    cout << "So the new order has the following table";
		    cout << result << endl << flush;
		    cout << "and the Discriminant is" <<disc(result);
		    cout << endl << flush);
    return result;
  }
  else{
    // If we can't create the base (because there is no field)
    // create the MT instead.
    base_vector <bigint> tmp;
    init.set_no_of_columns((n*(n+1))/2);
    for (i = 0; i < n; i++){
      M.base.column_vector(tmp,i);
      a=alg_number(tmp,1,O);
      for (j = 0; j <=i; j++){
	M.base.column_vector(tmp,j);
	b=alg_number(tmp,1,O);
	init.sto_column_vector((a*b).coeff_vector(), n, (i*(i+1))/2+j);
      }
    }
    debug_handler_c("module",
		    "in member - function ring_of_multipliers(...)",3,
		    cout <<"solve "<<M.base<<init<<endl<<flush );

    init.reginvimage(M.base, init);
    init.set_no_of_rows(n);

    debug_handler_c("module",
		    "in member - function ring_of_multipliers(...)",3,
		    cout << "p * MT is" << trans(init) << flush);

    for (i = 0; i < init.get_no_of_rows(); i++)
      for (j = 0; j < init.get_no_of_columns(); j++){
	bigint q,r;
	div_rem(q,r,init.member(i,j),p);
	if (!r.is_zero())
	  error_handler("module", "ring_of multipliers::internal error::"
			"division by p unsuccesful");
	init.sto(i,j,q);
      }
    debug_handler_c("module",
		    "in member - function ring_of_multipliers(...)",3,
		    cout << "MT is" << trans(init) << flush);
    return order(trans(init));
  }
}

// Other functions:
void invert(module &c, const module & a)
{
    c = a;
    c.invert();
}

module inverse(const module & a)
{
    module c = a;
    c.invert();
    return c;
}

void square(module & a, const module & b)
{
    debug_handler("modules",
		  "in function square(a_n &, const a_n &)");
    multiply(a,b,b);
}

void swap(module & a, module & b)
{
    swap (a.den, b.den);
    swap(a.base,b.base);
    const order * O = a.O;
    a.O = b.O;
    b.O = O;
}

// random numbers
void module::randomize(const bigint & b)
{
  debug_handler("module", "in member-function "
		"randomize(const bigint &)");
  den.assign_one();
  base.set_no_of_columns(2);
  base.sto_column_vector(alg_number(::randomize(b),O).coeff_vector(),
			 degree(),0);
  for (register lidia_size_t j=0;j<degree();j++)
    base.sto(j,1,::randomize(b));
  multiply(*this, *this, *O);
  den = ::randomize(b);
  normalize();
}

// In-/Output:
ostream& operator <<(ostream & s, const module & a)
{
  s << a.base;
  if (!(a.is_zero() || a.den.is_one()))
    s <<" / "<< a.den;
  return s;
}

istream& operator >>(istream & s, module & a)
{
  if (current_order == NULL){
    if (current_number_field == NULL)
      lidia_error_handler ("module","operator >>::No number_field or order!");
    else{
      order * dummy_order = new order(current_number_field);
      // this may lead to memory-problems, but since you typically
      // shouldn't need this feature very often, it's not worth 
      // the trouble doing this cleaner...
      a.O = current_order = dummy_order;
    }
  }
  else
    a.O = current_order;
  s >> a.base;
  char c;
  do{
    s.get(c);
  } while (isspace(c) && c != '\n'); 
  if (c == '/'){
    s >> a.den;
    a.normalize();
  }
  else{
    a.den = 1;
    if (c != '\n' && c != '\r')
      s.putback(c);
  }
  return s;
}
