## Copyright (C) 2009-2015   Lukas F. Reichlin
##
## This file is part of LTI Syncope.
##
## LTI Syncope is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## LTI Syncope is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with LTI Syncope.  If not, see <http://www.gnu.org/licenses/>.

## -*- texinfo -*-
## @deftypefn {Function File} {[@var{sys}, @var{x0}, @var{info}] =} moen4 (@var{dat}, @dots{})
## @deftypefnx {Function File} {[@var{sys}, @var{x0}, @var{info}] =} moen4 (@var{dat}, @var{n}, @dots{})
## @deftypefnx {Function File} {[@var{sys}, @var{x0}, @var{info}] =} moen4 (@var{dat}, @var{opt}, @dots{})
## @deftypefnx {Function File} {[@var{sys}, @var{x0}, @var{info}] =} moen4 (@var{dat}, @var{n}, @var{opt}, @dots{})
## Estimate state-space model using combined subspace method:
## @acronym{MOESP} algorithm for finding the matrices A and C,
## and @acronym{N4SID} algorithm for finding the matrices B and D.
## If no output arguments are given, the singular values are
## plotted on the screen in order to estimate the system order.
##
## @strong{Inputs}
## @table @var
## @item dat
## iddata set containing the measurements, i.e. time-domain signals.
## @item n
## The desired order of the resulting state-space system @var{sys}.
## If not specified, @var{n} is chosen automatically according
## to the singular values and tolerances.
## @item @dots{}
## Optional pairs of keys and values.  @code{'key1', value1, 'key2', value2}.
## @item opt
## Optional struct with keys as field names.
## Struct @var{opt} can be created directly or
## by function @command{options}.  @code{opt.key1 = value1, opt.key2 = value2}.
## @end table
##
##
## @strong{Outputs}
## @table @var
## @item sys
## Discrete-time state-space model.
## @item x0
## Initial state vector.  If @var{dat} is a multi-experiment dataset,
## @var{x0} becomes a cell vector containing an initial state vector
## for each experiment.
## @item info
## Struct containing additional information.
## @table @var
## @item info.K
## Kalman gain matrix.
## @item info.Q
## State covariance matrix.
## @item info.Ry
## Output covariance matrix.
## @item info.S
## State-output cross-covariance matrix.
## @item info.L
## Noise variance matrix factor. LL'=Ry.
## @end table
## @end table
##
##
##
## @strong{Option Keys and Values}
## @table @var
## @item 'n'
## The desired order of the resulting state-space system @var{sys}.
## @var{s} > @var{n} > 0.
##
## @item 's'
## The number of block rows @var{s} in the input and output
## block Hankel matrices to be processed.  @var{s} > 0.
## In the MOESP theory, @var{s} should be larger than @var{n},
## the estimated dimension of state vector.
##
## @item 'alg', 'algorithm'
## Specifies the algorithm for computing the triangular
## factor R, as follows:
## @table @var
## @item 'C'
## Cholesky algorithm applied to the correlation
## matrix of the input-output data.  Default method.
## @item 'F'
## Fast QR algorithm.
## @item 'Q'
## QR algorithm applied to the concatenated block
## Hankel matrices.
## @end table
##
## @item 'tol'
## Absolute tolerance used for determining an estimate of
## the system order.  If  @var{tol} >= 0,  the estimate is
## indicated by the index of the last singular value greater
## than or equal to @var{tol}.  (Singular values less than @var{tol}
## are considered as zero.)  When  @var{tol} = 0,  an internally
## computed default value,  @var{tol} = @var{s}*@var{eps}*SV(1),  is used,
## where  SV(1)  is the maximal singular value, and @var{eps} is
## the relative machine precision.
## When @var{tol} < 0,  the estimate is indicated by the
## index of the singular value that has the largest
## logarithmic gap to its successor.  Default value is 0.
##
## @item 'rcond'
## The tolerance to be used for estimating the rank of
## matrices. If the user sets @var{rcond} > 0,  the given value
## of @var{rcond} is used as a lower bound for the reciprocal
## condition number;  an m-by-n matrix whose estimated
## condition number is less than  1/@var{rcond}  is considered to
## be of full rank.  If the user sets @var{rcond} <= 0,  then an
## implicitly computed, default tolerance, defined by
## @var{rcond} = m*n*@var{eps},  is used instead, where @var{eps} is the
## relative machine precision.  Default value is 0.
##
## @item 'confirm'
## Specifies whether or not the user's confirmation of the
## system order estimate is desired, as follows:
## @table @var
## @item true
## User's confirmation.
## @item false
## No confirmation.  Default value.
## @end table
##
## @item 'noiseinput'
## The desired type of noise input channels.
## @table @var
## @item 'n'
## No error inputs.  Default value.
## @iftex
## @tex
## $$ x_{k+1} = A x_k + B u_k $$
## $$ y_k = C x_k + D u_k $$
## @end tex
## @end iftex
## @ifnottex
## @example
## x[k+1] = A x[k] + B u[k]
## y[k]   = C x[k] + D u[k]
## @end example
## @end ifnottex
##
## @item 'e'
## Return @var{sys} as a (p-by-m+p) state-space model with
## both measured input channels u and noise channels e
## with covariance matrix @var{Ry}.
## @iftex
## @tex
## $$ x_{k+1} = A x_k + B u_k + K e_k $$
## $$ y_k = C x_k + D u_k + e_k $$
## @end tex
## @end iftex
## @ifnottex
## @example
## x[k+1] = A x[k] + B u[k] + K e[k]
## y[k]   = C x[k] + D u[k] +   e[k]
## @end example
## @end ifnottex
##
## @item 'v'
## Return @var{sys} as a (p-by-m+p) state-space model with
## both measured input channels u and white noise channels v
## with identity covariance matrix.
## @iftex
## @tex
## $$ x_{k+1} = A x_k + B u_k + K L v_k $$
## $$ y_k = C x_k + D u_k + L v_k $$
## $$ e = L v, \\ L L^T = R_y $$
## @end tex
## @end iftex
## @ifnottex
## @example
## x[k+1] = A x[k] + B u[k] + K L v[k]
## y[k]   = C x[k] + D u[k] +   L v[k]
## e = L v,  L L' = Ry
## @end example
## @end ifnottex
##
## @item 'k'
## Return @var{sys} as a Kalman predictor for simulation.
## @iftex
## @tex
## $$ \\widehat{x}_{k+1} = A \\widehat{x}_k + B u_k + K (y_k - \\widehat{y}_k) $$
## $$ \\widehat{y}_k = C \\widehat{x}_k + D u_k $$
## @end tex
## @end iftex
## @ifnottex
## @example
## ^          ^                        ^
## x[k+1] = A x[k] + B u[k] + K(y[k] - y[k])
## ^          ^
## y[k]   = C x[k] + D u[k]
## @end example
## @end ifnottex
##
## @iftex
## @tex
## $$ \\widehat{x}_{k+1} = (A-KC) \\widehat{x}_k + (B-KD) u_k + K y_k $$
## $$ \\widehat{y}_k = C \\widehat{x}_k + D u_k + 0 y_k $$
## @end tex
## @end iftex
## @ifnottex
## @example
## ^               ^
## x[k+1] = (A-KC) x[k] + (B-KD) u[k] + K y[k]
## ^          ^
## y[k]   = C x[k] + D u[k] + 0 y[k]
## @end example
## @end ifnottex
## @end table
## @end table
##
##
## @strong{Algorithm}@*
## Uses SLICOT IB01AD, IB01BD and IB01CD by courtesy of
## @uref{http://www.slicot.org, NICONET e.V.}
##
## @end deftypefn

## Author: Lukas Reichlin <lukas.reichlin@gmail.com>
## Created: May 2012
## Version: 0.1

function [sys, x0, info] = moen4 (varargin)

  if (nargin == 0)
    print_usage ();
  endif

  if (nargout == 0)
    __slicot_identification__ ("moen4", nargout, varargin{:});
  else
    [sys, x0, info] = __slicot_identification__ ("moen4", nargout, varargin{:});
  endif

endfunction


%!shared SYS, X0, INFO, Ae, Be, Ce, De, Ke, Qe, Rye, Se, X0e
%!
%! Y = [ 4.7661   5.5451   5.8503   5.3766   4.8833   5.4865   3.5378   5.3155   6.0530   4.3729
%!       4.7637   5.1886   5.9236   5.6818   4.8858   5.1495   3.5549   5.5329   6.0799   4.7417
%!       4.8394   4.8833   5.9212   5.8235   4.8931   4.8442   3.4938   5.4450   6.1287   5.0884
%!       5.0030   4.6000   5.9773   5.9529   4.7148   4.5414   3.4474   5.3961   6.0799   5.1861
%!       5.0176   4.2704   5.7405   6.0628   4.4511   4.2679   3.4401   5.2740   6.1678   5.0372
%!       5.0567   4.0384   5.3888   6.0897   4.2337   4.0604   3.4083   5.0274   6.1947   4.7856
%!       5.1544   3.8381   5.0005   6.0750   4.0433   3.9602   3.4108   4.7441   6.2362   4.5634
%!       5.3619   3.7112   4.8491   6.0262   3.8650   3.7893   3.4523   4.6684   6.0530   4.5341
%!       5.4254   3.5915   4.9444   5.9944   3.7576   3.6428   3.6818   4.6513   5.6525   4.7050
%!       5.5695   3.5353   5.1739   6.0775   3.6696   3.5256   4.0604   4.5146   5.2740   4.7417
%!       5.6818   3.4865   5.3693   5.8577   3.5939   3.4987   4.4413   4.2679   4.8589   4.6489
%!       5.7429   3.4767   5.4474   5.7014   3.5475   3.4547   4.8540   4.2606   4.5341   4.4315
%!       5.8039   3.4254   5.6037   5.7307   3.5060   3.4083   5.1544   4.2630   4.4560   4.2386
%!       5.9187   3.3815   5.7307   5.7844   3.4547   3.3790   5.4254   4.1898   4.6196   4.0652
%!       5.8210   3.3693   5.8503   5.8235   3.3986   3.3766   5.5964   4.2777   4.8662   3.9431
%!       5.4474   3.3644   5.9798   5.8943   3.3619   3.3619   5.5866   4.6000   5.1177   3.8113
%!       5.0616   3.3473   5.9920   5.7624   3.3400   3.3595   5.3546   4.9322   5.1666   3.6916
%!       4.6293   3.3815   6.0848   5.4157   3.3742   3.3693   5.0274   5.2838   5.0567   3.6525
%!       4.2679   3.4206   5.9407   4.9615   3.5207   3.3986   4.8638   5.5280   5.0030   3.8259
%!       4.0115   3.4132   5.8039   4.5952   3.7136   3.5793   4.7612   5.7405   5.0982   4.2240
%!       3.8503   3.4523   5.7917   4.3314   3.7576   3.9480   4.5707   5.8748   5.3253   4.4242
%!       3.7112   3.6355   5.6037   4.2972   3.7795   4.4120   4.3681   5.9554   5.5671   4.4291
%!       3.5695   4.0384   5.2643   4.5829   3.6965   4.5854   4.3974   5.9920   5.4670   4.3192
%!       3.5182   4.3754   4.9468   4.8613   3.7771   4.5146   4.5732   5.8455   5.2521   4.1385
%!       3.6525   4.7270   4.6196   5.1739   3.8870   4.3436   4.8418   5.5280   4.9468   3.9651
%!       3.8186   5.0567   4.5146   5.1666   3.9041   4.1556   5.2032   5.0616   4.8809   3.8870
%!       3.8626   5.2985   4.4340   4.9199   3.8503   3.9847   5.4523   4.7344   4.9810   3.8015
%!       4.0115   5.5329   4.2850   4.6074   3.9651   4.0433   5.6525   4.5341   5.2252   3.7014
%!       4.3534   5.4670   4.1214   4.3705   4.2826   4.3070   5.8552   4.5341   5.4596   3.6403
%!       4.7050   5.1959   3.9456   4.1825   4.5219   4.4218   5.9065   4.6977   5.7234   3.7673
%!       5.0836   4.8858   3.9847   4.0384   4.7148   4.3534   5.9529   4.7441   5.7917   4.1507
%!       5.3449   4.7637   4.2191   4.1458   4.9712   4.2240   5.8284   4.6196   5.9065   4.6489
%!       5.2740   4.8760   4.5463   4.4315   5.2203   4.0530   5.7917   4.6440   5.9920   4.9908
%!       5.1275   5.0420   4.8735   4.5561   5.5329   3.9407   5.7991   4.8320   5.8357   5.0884
%!       4.7612   5.2838   5.1544   4.4804   5.6525   3.8381   5.8137   5.1324   5.5280   5.0225
%!       4.4511   5.4914   5.3888   4.3754   5.7820   3.7307   5.8772   5.4108   5.1422   4.7832
%!       4.2215   5.5964   5.6135   4.3705   5.9554   3.6525   5.9554   5.6257   4.7759   4.6855
%!       4.0457   5.6721   5.8357   4.5585   6.0359   3.6110   5.7820   5.6037   4.4902   4.6660
%!       3.8748   5.7722   5.8845   4.8589   6.1190   3.5646   5.5182   5.3155   4.2362   4.7075
%!       3.7307   5.8308   5.9554   4.8955   6.1336   3.4963   5.1275   4.9615   4.0237   4.9126
%!       3.6623   5.9334   5.7624   4.7417   6.1532   3.4621   4.7637   4.6196   3.8870   5.1959
%!       3.5768   5.8992   5.4596   4.7441   6.1922   3.4547   4.4926   4.3583   3.7527   5.4157
%!       3.5427   5.9358   5.0616   4.8760   6.1434   3.4254   4.2337   4.1556   3.6818   5.6232
%!       3.4792   5.8943   4.7075   5.1055   6.1678   3.3790   4.0115   4.0335   3.8064   5.7405
%!       3.4547   5.9187   4.4584   5.2398   5.9920   3.4328   3.8552   3.8870   4.1458   5.8992
%!       3.3595   5.9944   4.2679   5.5182   5.6525   3.6232   3.6916   3.7722   4.6000   5.9285
%!       3.2985   5.9578   4.0530   5.6525   5.4596   3.9749   3.6355   3.6403   5.0030   6.0506
%!       3.2252   6.0311   3.9431   5.7234   5.4376   4.3803   3.8186   3.5329   5.3033   6.1532
%!       3.2008   6.0628   3.8259   5.8552   5.3400   4.7148   4.1556   3.4352   5.5524   5.9651
%!       3.2252   6.0408   3.9676   5.9627   5.0982   5.0738   4.5903   3.4279   5.6159   5.5866
%!       3.2276   6.0970   4.2801   5.9847   4.7856   5.3693   4.9883   3.4230   5.5231   5.3815
%!       3.2740   6.1239   4.4804   5.9847   4.4926   5.6037   5.0762   3.3986   5.6110   5.3717
%!       3.4572   6.1629   4.4926   6.0555   4.2362   5.7453   4.9077   3.6037   5.7136   5.4865
%!       3.8674   6.0408   4.3900   6.0628   4.0677   5.6525   4.6489   4.0237   5.8455   5.5671
%!       4.3217   5.8455   4.1971   6.0555   3.9334   5.4010   4.3778   4.4511   5.8992   5.8210
%!       4.4926   5.7722   4.1116   6.0701   3.8235   5.0152   4.2166   4.7930   5.9944   5.9138
%!       4.4315   5.7991   3.9822   5.7844   3.7307   4.7099   4.2875   4.9029   6.0921   5.9944
%!       4.2435   5.9236   3.8674   5.4401   3.6110   4.4169   4.5903   4.7808   6.0921   6.0115
%!       4.0506   5.9285   3.7673   5.0567   3.5646   4.2362   4.8467   4.5903   6.1434   5.9993
%!       3.8577   6.0018   3.8723   4.9419   3.5500   4.2362   5.1397   4.3363   6.1532   6.0188
%!       3.7307   6.0018   4.2362   5.0103   3.5573   4.2484   5.3888   4.1458   6.2337   5.8210
%!       3.7917   6.0604   4.6635   5.1348   3.5134   4.2215   5.6892   4.2166   6.1873   5.7282
%!       3.9212   5.8821   4.9712   5.3131   3.5158   4.2972   5.8845   4.4340   6.0140   5.7405
%!       3.9554   5.5109   5.0665   5.4792   3.6941   4.5903   6.0433   4.7148   5.8357   5.7649
%!       3.8479   5.3229   4.9029   5.6232   4.0726   4.8931   6.1703   5.0982   5.7746   5.8821
%!       3.7258   5.3717   4.6757   5.5622   4.4804   5.1348   6.2118   5.3595   5.6867   5.9260
%!       3.6110   5.4547   4.3925   5.3302   4.7050   5.4279   6.2508   5.5695   5.5378   5.7502
%!       3.7160   5.4376   4.0994   5.0103   4.6123   5.3790   6.2093   5.7722   5.3278   5.4157
%!       4.0921   5.1593   4.1141   4.6660   4.3851   5.3644   6.0140   5.9212   5.0543   4.9956
%!       4.4804   4.9029   4.3265   4.4145   4.2020   5.4523   5.7014   6.0555   4.7002   4.8613
%!       4.8149   4.5878   4.6440   4.2020   4.0262   5.5671   5.4694   5.9627   4.3949   4.9029
%!       5.0543   4.5024   4.9712   4.0482   3.9041   5.6721   5.4792   5.6428   4.1800   5.1031
%!       5.3033   4.5952   5.1593   4.0799   3.7746   5.7698   5.5573   5.4352   4.0433   5.3644
%!       5.4865   4.8247   5.3888   4.1898   3.6916   5.8308   5.7282   5.3888   3.8772   5.5964
%!       5.6721   5.0640   5.5768   4.1312   3.8455   5.9236   5.8821   5.5378   3.7527   5.7527
%!       5.7795   5.2716   5.6525   4.0042   4.2020   5.9651   5.9847   5.6818   3.7282   5.8455
%!       5.7991   5.4670   5.8039   3.9163   4.5854   6.0579   5.9016   5.7014   3.8699   5.9285
%!       5.6648   5.6159   5.9138   3.9602   4.9029   6.0506   5.5817   5.6159   4.2069   6.0066
%!       5.2911   5.5280   5.8870   4.1996   5.2569   6.0726   5.3717   5.6672   4.3558   5.8406
%!       4.8809   5.2545   5.7991   4.6245   5.5109   6.1116   5.4181   5.7405   4.4267   5.5182
%!       4.5585   4.8833   5.7307   4.8833   5.6403   6.0701   5.5109   5.8039   4.4535   5.1739
%!       4.1849   4.5170   5.7624   5.1373   5.8430   5.8967   5.6672   5.8821   4.5219   4.7392
%!       3.8894   4.1971   5.8137   5.3790   5.9749   5.7551   5.7917   5.9505   4.3925   4.4584
%!       3.7087   4.0018   5.8210   5.6232   5.9358   5.7185   5.6989   6.0726   4.1556   4.4267
%!       3.6232   3.8064   5.9285   5.7624   5.8210   5.8210   5.4840   6.1483   3.9651   4.6025
%!       3.5695   3.9041   6.0140   5.8333   5.5280   6.0018   5.1544   6.1165   3.8772   4.8223
%!       3.7185   3.9236   5.7649   5.6867   5.1715   6.0018   4.9810   6.1776   3.9700   5.1837
%!       4.0335   3.8699   5.4132   5.3668   4.8101   5.9016   5.0616   6.2020   4.2582   5.4303
%!       4.4120   3.8064   5.0982   5.2252   4.4535   5.5573   5.1959   6.2069   4.4218   5.6525
%!       4.6293   3.7209   4.6782   5.2398   4.3803   5.1739   5.3595   5.9920   4.3363   5.8210
%!       4.5585   3.8186   4.3729   5.3546   4.5659   4.8003   5.6159   5.5646   4.2997   5.7063
%!       4.3949   4.1409   4.3925   5.5085   4.8052   4.4315   5.7624   5.1788   4.3925   5.3693
%!       4.1800   4.5292   4.5903   5.5964   5.1251   4.1947   5.8577   4.9981   4.6757   5.0274
%!       4.1971   4.8052   4.9199   5.7527   5.3546   4.0066   5.9480   5.0518   4.7612   4.7050
%!       4.4315   5.0860   5.0176   5.8748   5.5891   3.8503   5.8357   5.2325   4.6587   4.4145
%!       4.7148   5.3400   4.8589   5.9065   5.7649   3.7478   5.7063   5.4840   4.4902   4.1458
%!       4.9615   5.5329   4.6757   5.8943   5.9236   3.6428   5.4987   5.6867   4.3070   3.9651
%!       5.3009   5.5768   4.6196   5.7429   5.9407   3.5915   5.1886   5.8992   4.1263   4.0335
%!       5.5671   5.6672   4.8345   5.4474   5.8577   3.5695   5.1177   5.8699   3.9724   4.3729
%!       5.6818   5.7917   5.0909   5.0250   5.6941   3.5280   5.1910   5.9773   4.0775   4.6831 ](:);
%!
%!
%!
%! U = [ 6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   3.4100   6.4100   6.4100   3.4100
%!       3.4100   3.4100   3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   3.4100
%!       6.4100   3.4100   3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   3.4100
%!       6.4100   3.4100   3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   3.4100
%!       6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   3.4100   6.4100   3.4100   6.4100
%!       6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   6.4100   3.4100   3.4100   6.4100
%!       6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   6.4100   3.4100   3.4100   3.4100
%!       6.4100   3.4100   6.4100   3.4100   3.4100   3.4100   6.4100   3.4100   3.4100   3.4100
%!       6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   6.4100   6.4100   3.4100   3.4100
%!       6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   6.4100   3.4100   6.4100   3.4100
%!       6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   6.4100   3.4100   6.4100   3.4100
%!       3.4100   3.4100   6.4100   6.4100   3.4100   3.4100   6.4100   6.4100   6.4100   3.4100
%!       3.4100   3.4100   6.4100   6.4100   3.4100   3.4100   3.4100   6.4100   6.4100   3.4100
%!       3.4100   3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   3.4100   3.4100
%!       3.4100   3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   3.4100   3.4100
%!       3.4100   3.4100   3.4100   3.4100   6.4100   3.4100   6.4100   6.4100   6.4100   6.4100
%!       3.4100   3.4100   6.4100   3.4100   3.4100   6.4100   3.4100   6.4100   6.4100   6.4100
%!       3.4100   3.4100   6.4100   3.4100   3.4100   6.4100   3.4100   6.4100   6.4100   3.4100
%!       3.4100   6.4100   3.4100   6.4100   3.4100   6.4100   3.4100   6.4100   6.4100   3.4100
%!       3.4100   6.4100   3.4100   6.4100   3.4100   3.4100   6.4100   6.4100   3.4100   3.4100
%!       3.4100   6.4100   3.4100   6.4100   6.4100   3.4100   6.4100   3.4100   3.4100   3.4100
%!       6.4100   6.4100   3.4100   6.4100   3.4100   3.4100   6.4100   3.4100   3.4100   3.4100
%!       3.4100   6.4100   6.4100   3.4100   3.4100   3.4100   6.4100   3.4100   6.4100   3.4100
%!       3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   3.4100   6.4100   3.4100
%!       6.4100   6.4100   3.4100   3.4100   6.4100   6.4100   6.4100   3.4100   6.4100   3.4100
%!       6.4100   3.4100   3.4100   3.4100   6.4100   6.4100   6.4100   6.4100   6.4100   3.4100
%!       6.4100   3.4100   3.4100   3.4100   3.4100   3.4100   6.4100   6.4100   6.4100   6.4100
%!       6.4100   3.4100   6.4100   3.4100   6.4100   3.4100   6.4100   3.4100   6.4100   6.4100
%!       6.4100   6.4100   6.4100   6.4100   6.4100   3.4100   3.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   6.4100   6.4100
%!       3.4100   6.4100   6.4100   3.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100
%!       3.4100   6.4100   6.4100   3.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100
%!       3.4100   6.4100   6.4100   3.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100
%!       3.4100   6.4100   6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   3.4100   6.4100
%!       3.4100   6.4100   6.4100   6.4100   6.4100   3.4100   3.4100   3.4100   3.4100   3.4100
%!       3.4100   6.4100   6.4100   6.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100
%!       3.4100   6.4100   6.4100   3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100
%!       3.4100   6.4100   3.4100   3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   3.4100   6.4100   3.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   6.4100   6.4100   3.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   6.4100   6.4100   6.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   3.4100   6.4100   6.4100   3.4100   6.4100   3.4100
%!       3.4100   6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   3.4100
%!       3.4100   6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   3.4100   6.4100   3.4100   3.4100   6.4100   6.4100
%!       6.4100   6.4100   3.4100   6.4100   3.4100   6.4100   3.4100   6.4100   6.4100   6.4100
%!       6.4100   3.4100   3.4100   6.4100   3.4100   3.4100   3.4100   6.4100   6.4100   6.4100
%!       6.4100   6.4100   3.4100   6.4100   3.4100   3.4100   3.4100   6.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   3.4100   3.4100   3.4100   6.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   6.4100   6.4100   3.4100   3.4100   6.4100   3.4100   6.4100   3.4100
%!       6.4100   6.4100   6.4100   6.4100   3.4100   3.4100   6.4100   6.4100   6.4100   6.4100
%!       3.4100   3.4100   6.4100   6.4100   3.4100   6.4100   6.4100   6.4100   3.4100   6.4100
%!       3.4100   3.4100   3.4100   6.4100   6.4100   6.4100   6.4100   6.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   6.4100   6.4100   6.4100   6.4100   6.4100   6.4100
%!       3.4100   6.4100   3.4100   3.4100   6.4100   6.4100   6.4100   6.4100   3.4100   6.4100
%!       3.4100   6.4100   3.4100   3.4100   3.4100   6.4100   6.4100   6.4100   6.4100   3.4100
%!       6.4100   3.4100   3.4100   3.4100   3.4100   3.4100   6.4100   6.4100   3.4100   3.4100
%!       6.4100   3.4100   6.4100   3.4100   3.4100   6.4100   3.4100   6.4100   3.4100   3.4100
%!       6.4100   3.4100   6.4100   3.4100   3.4100   6.4100   3.4100   6.4100   3.4100   6.4100
%!       6.4100   3.4100   6.4100   3.4100   3.4100   6.4100   6.4100   3.4100   3.4100   6.4100
%!       6.4100   6.4100   6.4100   3.4100   3.4100   6.4100   6.4100   3.4100   3.4100   6.4100
%!       6.4100   6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   6.4100   3.4100   6.4100
%!       6.4100   6.4100   6.4100   3.4100   3.4100   6.4100   6.4100   6.4100   3.4100   6.4100
%!       6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   6.4100   6.4100   3.4100   6.4100
%!       6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   6.4100   6.4100   3.4100   6.4100
%!       6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   6.4100   6.4100
%!       3.4100   6.4100   6.4100   6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   6.4100
%!       3.4100   3.4100   6.4100   6.4100   6.4100   6.4100   6.4100   6.4100   3.4100   3.4100
%!       3.4100   3.4100   3.4100   6.4100   6.4100   6.4100   6.4100   6.4100   3.4100   3.4100
%!       3.4100   3.4100   6.4100   6.4100   6.4100   6.4100   6.4100   6.4100   6.4100   3.4100
%!       3.4100   3.4100   6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100
%!       3.4100   3.4100   6.4100   6.4100   6.4100   6.4100   6.4100   6.4100   3.4100   3.4100
%!       3.4100   3.4100   6.4100   6.4100   6.4100   6.4100   3.4100   6.4100   3.4100   6.4100
%!       3.4100   3.4100   6.4100   6.4100   3.4100   6.4100   3.4100   6.4100   3.4100   6.4100
%!       3.4100   6.4100   6.4100   6.4100   3.4100   6.4100   3.4100   6.4100   3.4100   6.4100
%!       6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   6.4100   6.4100   6.4100   6.4100
%!       6.4100   3.4100   3.4100   3.4100   3.4100   3.4100   6.4100   6.4100   6.4100   6.4100
%!       6.4100   3.4100   3.4100   6.4100   3.4100   3.4100   6.4100   6.4100   3.4100   6.4100
%!       3.4100   3.4100   3.4100   6.4100   6.4100   3.4100   6.4100   3.4100   3.4100   6.4100
%!       3.4100   6.4100   3.4100   6.4100   6.4100   3.4100   6.4100   3.4100   3.4100   3.4100
%!       3.4100   6.4100   6.4100   6.4100   6.4100   3.4100   6.4100   3.4100   6.4100   3.4100
%!       3.4100   6.4100   6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   6.4100   3.4100
%!       6.4100   6.4100   6.4100   6.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100
%!       6.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   6.4100   3.4100   3.4100
%!       6.4100   6.4100   3.4100   6.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100
%!       6.4100   6.4100   3.4100   6.4100   6.4100   3.4100   3.4100   6.4100   3.4100   3.4100
%!       6.4100   6.4100   6.4100   3.4100   6.4100   3.4100   3.4100   6.4100   3.4100   6.4100
%!       6.4100   6.4100   6.4100   3.4100   3.4100   3.4100   6.4100   6.4100   3.4100   6.4100
%!       6.4100   6.4100   6.4100   3.4100   6.4100   3.4100   6.4100   6.4100   6.4100   6.4100
%!       3.4100   6.4100   6.4100   6.4100   3.4100   3.4100   6.4100   6.4100   6.4100   6.4100
%!       3.4100   6.4100   6.4100   6.4100   3.4100   3.4100   6.4100   6.4100   6.4100   6.4100
%!       3.4100   6.4100   6.4100   3.4100   3.4100   3.4100   3.4100   6.4100   6.4100   6.4100 ](:);
%!
%!
%! DAT = iddata (Y, U);
%!
%! [SYS, X0, INFO] = moen4 (DAT, "s", 15, "rcond", 0.0, "tol", -1.0, "confirm", false);
%!
%! Ae = [  0.8924   0.3887   0.1285   0.1716
%!        -0.0837   0.6186  -0.6273  -0.4582
%!         0.0052   0.1307   0.6685  -0.6755
%!         0.0055   0.0734  -0.2148   0.4788 ];
%!
%! Ce = [ -0.4442   0.6663   0.3961   0.4102 ];
%!
%! Be = [ -0.2142
%!        -0.1968
%!         0.0525
%!         0.0361 ];
%! 
%! De = [ -0.0041 ];
%!
%! Ke = [ -1.9513
%!        -0.1867
%!         0.6348
%!        -0.3486 ];
%!
%! Qe = [  0.0052   0.0005  -0.0017   0.0009
%!         0.0005   0.0000  -0.0002   0.0001
%!        -0.0017  -0.0002   0.0006  -0.0003
%!         0.0009   0.0001  -0.0003   0.0002 ];
%!
%! Rye = [ 0.0012 ];
%!
%! Se = [ -0.0025
%!        -0.0002
%!         0.0008
%!        -0.0005 ];
%!
%! X0e = [ -11.496422
%!          -0.718576
%!          -0.014211
%!           0.500073 ];    # X0e is not from SLICOT
%!
%! ## The SLICOT test for IB01CD uses COMUSE=C, not COMUSE=U.
%! ## This means that they don't use the matrices B and D
%! ## computed by IB01BD.  They use only A and C from IB01BD,
%! ## while B and D are from SLICOT routine IB01CD.
%! ## Therefore they get slightly different matrices B and D
%! ## and finally a different initial state vector X0.
%!
%!assert (SYS.A, Ae, 1e-4);
%!assert (SYS.B, Be, 1e-4);
%!assert (SYS.C, Ce, 1e-4);
%!assert (SYS.D, De, 1e-4);
%!assert (INFO.K, Ke, 1e-4);
%!assert (INFO.Q, Qe, 1e-4);
%!assert (INFO.Ry, Rye, 1e-4);
%!assert (INFO.S, Se, 1e-4);
%!assert (X0, X0e, 1e-4);


## [96-003] Data of a 120 MW power plant (Pont-sur-Sambre, France)
%!shared SYS, Ae, Be, Ce, De
%! U = [  -811        -592         421        -680        -681
%!        -812        -619         477        -685        -651
%!        -817        -565         538        -678        -677
%!        -695        -725         536        -674        -702
%!        -697        -571         531        -676        -685
%!        -697        -618         533        -681        -721
%!        -702        -579         549        -677        -699
%!        -703        -487         575        -677        -694
%!        -705        -449         561        -679        -678
%!        -705        -431         563        -680        -692
%!        -707        -502         561        -679        -686
%!        -707        -583         530        -676        -751
%!        -710        -458         540        -677        -700
%!        -713        -469         543        -679        -731
%!        -715        -506         549        -684        -635
%!        -713        -590         532        -681        -715
%!        -714        -582         528        -676        -696
%!        -713        -575         538        -679        -690
%!        -716        -382         557        -682        -701
%!        -716        -476         558        -679        -690
%!        -718        -425         565        -678        -686
%!        -719        -409         562        -679        -615
%!        -719        -508         523        -677        -737
%!        -721        -569         523        -679        -722
%!        -723        -434         542        -681        -735
%!        -723        -395         544        -676        -704
%!        -723        -428         542        -677        -729
%!        -722        -402         537        -677        -706
%!        -725        -380         534        -681        -696
%!        -726        -324         549        -676        -701
%!        -726        -211         578        -675        -614
%!        -727        -113         569        -677        -738
%!        -727        -208         554        -676        -737
%!        -727        -320         523        -684        -697
%!        -727         944         605        -680        -587
%!        -729         396         654        -681        -708
%!        -729         754         637        -682        -685
%!        -727         419         522        -677        -715
%!        -729         378         494        -668        -703
%!        -728         363         493        -669        -682
%!        -729         390         496        -665        -713
%!        -729         106         483        -664        -711
%!        -729          32         495        -661        -718
%!        -729         517         585        -661        -641
%!        -729         455         625        -659        -703
%!        -730         521         649        -687        -669
%!        -730         540         627        -689        -705
%!        -731         514         605        -694        -682
%!        -585         525         558        -685        -611
%!        -586         534         520        -680        -668
%!        -586         539         531        -681        -679
%!        -585         519         507        -682        -663
%!        -588         513         505        -667        -668
%!        -587         509         539        -680        -616
%!        -587         512         535        -668        -628
%!        -588         514         557        -667        -648
%!        -588         553         563        -676        -613
%!        -589         519         559        -684        -638
%!        -589         521         563        -682        -652
%!        -588         518         547        -678        -597
%!        -589         552         549        -688        -630
%!        -589         520         535        -685        -623
%!        -589         547         542        -678        -619
%!        -589         549         531        -684        -524
%!        -588         544         522       -1540        -580
%!        -588         564         555       -1538        -584
%!        -588         684         545       -1541        -564
%!        -590         558         546       -1541        -609
%!        -589         552         537       -1550        -601
%!        -591         532         526       -1548        -580
%!        -590         544         524       -1542        -565
%!        -591         559         535       -1538        -604
%!        -592         555         542       -1548        -629
%!        -591         577         532       -1549        -587
%!        -593         581         530       -1543        -585
%!        -592         562         540       -1548        -583
%!        -591         568         546       -1536        -587
%!        -593         550         557       -1533        -569
%!        -592         550         537       -1518        -527
%!        -593         568         551       -1533        -582
%!        -590         528         540       -1529        -492
%!        -590         542         532       -1525        -585
%!        -590         556         535       -1522        -606
%!        -591         637         535       -1516        -571
%!        -591         608         539       -1512        -582
%!        -591         545         527       -1510        -577
%!        -591         603         534       -1507        -548
%!        -592         567         521       -1507        -539
%!        -594         560         530       -1503        -583
%!        -422         549         534       -1487        -542
%!           5         619         550       -1488        -609
%!           5         572         541       -1487        -545
%!         185         564         529       -1488        -528
%!         185         571         531       -1497        -512
%!         187         590         545       -1489        -472
%!         186         658         544       -1485        -501
%!         185         639         553       -1486        -512
%!         187         604         532       -1486        -490
%!         186         584         535       -1486        -505
%!         187         572         541       -1488        -479
%!         186         570         531       -1486        -477
%!         187         579         539       -1488        -482
%!         187         694         537       -1487        -512
%!         187         727         546       -1494        -474
%!         186         838         545       -1493        -470
%!         186         703         527       -1492        -510
%!         185         634         502       -1490        -486
%!         185         659         502       -1490        -446
%!         185         569         483       -1488        -472
%!         184         552         494       -1487        -468
%!         183         541         492       -1487        -411
%!         184         544         519       -1486        -471
%!         184         577         538       -1487        -420
%!         183         536         546       -1490        -455
%!         183         586         567       -1487        -421
%!         183         554         566       -1485        -449
%!         183         564         558       -1486        -455
%!         182         558         546       -1486        -426
%!         183         659         543       -1486        -398
%!         182         545         531       -1487        -425
%!         182         542         513       -1487        -297
%!         181         549         502       -1486        -407
%!         180         558         963       -1487        -391
%!         182         546        1105       -1487        -388
%!         179         540        1166       -1487        -387
%!         181         519        1221       -1487        -390
%!         179         510        1254       -1486        -382
%!         176         528        1289       -1487        -379
%!         178         510        1310       -1486        -389
%!         178         504        1313       -1484        -357
%!         178         507        1328       -1484        -383
%!         178         519        1333       -1489        -426
%!         183         509        1317       -1483        -377
%!         177         585        1331       -1483        -352
%!         176         765        1334       -1484        -375
%!         196         689        1328       -1484        -381
%!         178         570        1306       -1486        -398
%!         176         997        1338       -1483        -360
%!         176         839        1318       -1487        -366
%!         176         879        1324         -90        -394
%!         175         763        1302         -92        -436
%!         175         739        1307         -92        -430
%!         176         791        1336         -86        -446
%!         175         774        1301         -91        -405
%!         174         782        1304         -91        -414
%!         189         722        1335         -92        -360
%!         173         871        1341         -91        -417
%!         173         825        1337         -86        -383
%!         193         836        1332         -86        -449
%!         174         832        1325         -87        -432
%!         175         834        1326         -88        -383
%!         176         899        1332         -86        -341
%!         195         827        1296         -84        -371
%!         174         897        1311         -86        -366
%!         195         848        1320         -86        -416
%!         192         777        1286         -83         126
%!         171         788        1309         -84         153
%!         171         810        1307         -84         173
%!         174         758        1297         -85         182
%!         188         910        1329       -1036         236
%!         189         944        1344       -1669         245
%!         196         859        1330       -1668         249
%!         190         797        1320       -1669         231
%!         191         784        1341       -1665         248
%!         184         737        1335       -1656         240
%!         182         733        1342       -1655         245
%!         182         696        1350       -1653         253
%!         185         614        1341       -1658         236
%!         188         708        1370       -1654         238
%!         191         729        1366       -1655         237
%!         183         714        1380       -1651         262
%!         186         695        1378       -1651         257
%!         189         758        1387       -1649         276
%!         164         572        1364       -1649         257
%!         163         587        1343       -1650         246
%!         161         683        1353       -1658         261
%!         160         572        1359       -1654         261
%!         160         607        1355       -1651         267
%!         158         580        1349       -1655         273
%!         161         631        1362       -1652         272
%!         160         706        1382       -1657         298
%!         161         601        1356       -1654         277
%!         159         570        1354       -1653         274
%!         158         547        1347       -1657         261
%!         158         582        1332       -1657         257
%!         157         570        1340       -1657         267
%!         154         556        1343       -1657         268
%!         157         537        1345       -1657        -425
%!         158         555        1331       -1653        -581
%!         158         551        1315       -1654        -643
%!         159         590        1322       -1656        -687
%!         160         566        1315       -1657        -737
%!         160         553        1315       -1653        -767
%!         161         644        1327       -1396        -731
%!         159         640        1335        -577        -639
%!         161         726        1334        -577        -730
%!         175         729        1310        -573        -711
%!         175         854        1330        -576        -690
%!         161         729        1313        -573        -636
%!         176         706        1314        -583        -745 ];
%!
%!
%! Y = [   117         129         -47
%!         113         141         -42
%!          83         150         -37
%!         144         174          -3
%!         156         196          19
%!         174         192           6
%!         171         193           3
%!         169         224          14
%!         155         211          14
%!         137         175           4
%!         130         165          13
%!         145         172          21
%!         128         173          30
%!         119         194          30
%!         127         188          11
%!         147         176           5
%!         162         183          19
%!         173         178          14
%!         131         161           4
%!         112         139         -11
%!          91         133         -10
%!          75         127          -7
%!          80         115         -10
%!          87         132           6
%!          51         151           6
%!          28         167           3
%!          22         167          -3
%!           9         171           4
%!           7         158           6
%!         -11         155           5
%!         -87         149          -8
%!        -153         146         -12
%!        -184         116         -20
%!        -167         118           8
%!        -563          78          -4
%!        -694          -3        -155
%!       -1000         -39        -168
%!       -1101         -28        -135
%!       -1080          55           6
%!       -1075         162          89
%!       -1035         254         117
%!        -894         329         148
%!        -747         409         218
%!        -774         416         231
%!        -781         361         189
%!        -825         288         171
%!        -897         220         166
%!        -960         175         169
%!        -994         157         196
%!       -1009         202         237
%!       -1003         267         258
%!        -981         326         267
%!        -947         361         276
%!        -921         369         280
%!        -894         369         292
%!        -876         359         302
%!        -869         356         318
%!        -857         334         316
%!        -857         336         320
%!        -868         331         323
%!        -873         316         324
%!        -870         301         326
%!        -867         294         323
%!        -878         293         322
%!        -877         300         309
%!        -891         317         233
%!        -919         308         160
%!        -919         296         111
%!        -898         304          94
%!        -884         336         101
%!        -876         344          92
%!        -877         347          77
%!        -859         354          79
%!        -852         368          84
%!        -852         362          79
%!        -846         337          67
%!        -846         322          71
%!        -836         325          82
%!        -831         330          81
%!        -848         332          84
%!        -834         334          89
%!        -830         314          87
%!        -838         322          93
%!        -859         313          73
%!        -864         317          59
%!        -852         318          48
%!        -850         313          59
%!        -858         327          65
%!        -849         332          82
%!        -822         332          77
%!        -721         377         121
%!        -517         492         193
%!        -306         596         226
%!        -117         683         244
%!          22         716         212
%!          87         684         157
%!         151         642         129
%!         198         599         105
%!         233         549          93
%!         244         512          83
%!         255         487          70
%!         247         453          49
%!         225         437          32
%!         175         410           4
%!         105         386         -12
%!          86         379         -35
%!         101         382         -16
%!         111         412           3
%!         158         451          31
%!         217         486          37
%!         259         504          29
%!         301         503          24
%!         317         495           7
%!         354         472          -3
%!         361         441          -8
%!         368         413         -32
%!         362         381         -34
%!         354         370         -42
%!         310         360         -36
%!         318         369         -36
%!         313         372         -36
%!         298         377         -28
%!         311         341         -34
%!         339         292         -33
%!         362         223         -52
%!         380         161         -73
%!         397         113         -90
%!         402          64        -117
%!         398          30        -136
%!         396          12        -148
%!         390         -22        -189
%!         377         -47        -211
%!         366         -54        -227
%!         331         -52        -240
%!         227         -74        -289
%!         168        -117        -355
%!         149        -104        -357
%!          10        -101        -344
%!         -77        -115        -374
%!        -116         -88        -330
%!        -106         -53        -223
%!         -82          -4         -98
%!         -99           5         -40
%!         -89          22           3
%!         -79          34          26
%!         -74          34          42
%!        -101          28          61
%!        -102          28          73
%!        -124          22          74
%!        -125          19          90
%!        -132           9         100
%!        -167           8         102
%!        -161           7          90
%!        -185          19         105
%!        -207          29         110
%!        -295          24         159
%!        -361          17         293
%!        -411          12         436
%!        -401          48         540
%!        -441          86         601
%!        -475         118         553
%!        -448         146         472
%!        -401         172         428
%!        -366         197         400
%!        -304         231         391
%!        -252         253         387
%!        -183         261         399
%!        -123         255         405
%!        -107         247         397
%!         -76         228         392
%!         -71         205         389
%!         -66         192         381
%!         -92         171         371
%!         -40         178         356
%!         -15         185         353
%!         -27         170         359
%!          15         173         351
%!          23         175         356
%!          44         178         338
%!          47         171         323
%!           4         153         294
%!          23         141         295
%!          39         141         306
%!          52         139         313
%!          53         142         313
%!          69         142         312
%!          74         136         306
%!         196         164         277
%!         347         208         133
%!         461         230         -58
%!         526         204        -251
%!         581         161        -385
%!         588         119        -458
%!         549          63        -528
%!         497           5        -550
%!         420         -20        -498
%!         327         -49        -464
%!         198         -78        -438
%!         154         -80        -409
%!         130         -60        -377 ];
%!
%! DAT = iddata (Y, U, 1228.8);
%!
%! [SYS, X0] = moen4 (DAT, "s", 10, "n", 8, "rcond", 0.0, "tol", -1.0);
%!
%! Ae = [  0.9811   0.0574   0.3270   0.0003   0.0358   0.0403  -0.1366   0.0276
%!         0.1043   0.7634  -0.1308   0.2252   0.0203  -0.0117  -0.2328  -0.2516
%!        -0.0612   0.1437   0.8378  -0.2400   0.0367  -0.3205  -0.0367  -0.0978
%!        -0.0213   0.0149   0.0706   0.8902   0.2415  -0.1329   0.3080  -0.0143
%!        -0.0076   0.0680   0.0557  -0.0469   0.7084   0.2852   0.2565   0.1508
%!        -0.0016   0.0603   0.0071  -0.0448  -0.0870   0.8608   0.1495  -0.1148
%!         0.0046   0.0120   0.0040  -0.0272   0.0117  -0.0654   0.8348  -0.4557
%!        -0.0058   0.0133  -0.0112  -0.0171  -0.0353  -0.0892   0.3328   0.7650 ];
%!
%! Ce = [ -0.1635   0.6294   0.1157  -0.2129   0.0812  -0.0238   0.0745   0.2027
%!         0.1775   0.1736  -0.2839  -0.0750  -0.4986  -0.1254   0.3740   0.1598
%!         0.2012  -0.0045  -0.4447  -0.3289   0.4767  -0.3377   0.0200  -0.0978 ];
%!
%! Be = [ -0.0436   0.1911  -0.1345   0.0409   0.2828
%!         0.5541  -0.4223   0.0001  -0.0151  -0.1778
%!        -0.3644   0.2868   0.0754  -0.0457  -0.1691
%!        -0.0372   0.1270  -0.0772  -0.0714  -0.0255
%!        -0.1251  -0.0021   0.1094   0.0576   0.2121
%!        -0.1372   0.1139   0.0030   0.0141   0.0493
%!        -0.0187   0.0712  -0.0042   0.0075  -0.0221
%!        -0.0834   0.0509   0.0322  -0.0089   0.0009 ];
%!
%! De = [  0.1581  -0.3111  -0.0350   0.0179  -0.1403
%!        -0.0037  -0.0461  -0.1177   0.0092  -0.0242
%!         0.0476  -0.0237  -0.0159   0.0174   0.0464 ];
%!
%!assert (abs (SYS.A), abs (Ae), 1e-4);
%!assert (abs (SYS.B), abs (Be), 1e-4);
%!assert (abs (SYS.C), abs (Ce), 1e-4);
%!assert (abs (SYS.D), abs (De), 1e-4);


## [96-007] Data of a CD-player arm
%!shared SYS, Ae, Be, Ce, De
%! UY = [  0.0531   -0.0313    0.0495    0.0342
%!         0.0526   -0.0328    0.0525    0.0342
%!         0.0504   -0.0482    0.0485    0.0567
%!         0.0423   -0.1048    0.0285    0.1542
%!         0.0269   -0.1515    0.0555    0.1192
%!         0.0072   -0.1404    0.0635    0.2067
%!        -0.0091   -0.0936    0.1885   -0.0558
%!        -0.0272   -0.0770    0.1145    0.2292
%!        -0.0602   -0.0720    0.0985    0.2767
%!        -0.0962   -0.1012    0.2485   -0.0733
%!        -0.1266   -0.2732    0.2705    0.3017
%!        -0.1548   -0.4999    0.3675    0.2567
%!        -0.1831   -0.6836    0.3665    0.3492
%!        -0.2225   -0.8477    0.3145    0.7492
%!        -0.2638   -0.9533    0.4615    0.4067
%!        -0.2865   -0.9926    0.5025    0.4917
%!        -0.2877   -0.9519    0.5355    0.5942
%!        -0.2652   -0.8100    0.5835    0.2917
%!        -0.2229   -0.6546    0.4355    0.4417
%!        -0.1690   -0.4802    0.3175    0.2667
%!        -0.1022   -0.3147    0.2265    0.1067
%!        -0.0262   -0.2181    0.1655    0.1717
%!         0.0508   -0.1214    0.1295    0.0167
%!         0.1181    0.0027   -0.0375    0.0642
%!         0.1639    0.1290   -0.2015    0.0767
%!         0.1915    0.2491   -0.2445   -0.0933
%!         0.2141    0.3501   -0.2085   -0.1833
%!         0.2383    0.4384   -0.2055   -0.2083
%!         0.2591    0.5325   -0.2915   -0.1533
%!         0.2720    0.6505   -0.3725   -0.2108
%!         0.2855    0.7959   -0.3975   -0.4083
%!         0.3036    0.9837   -0.4585   -0.4258
%!         0.3180    1.2078   -0.5175   -0.4108
%!         0.3278    1.4001   -0.4415   -0.6958
%!         0.3333    1.5244   -0.4445   -0.7183
%!         0.3201    1.6123   -0.6065   -0.4658
%!         0.2920    1.6983   -0.5645   -0.8133
%!         0.2720    1.7388   -0.3875   -1.0058
%!         0.2556    1.6854   -0.3775   -0.6208
%!         0.2279    1.5986   -0.4115   -0.6783
%!         0.1954    1.4945   -0.4215   -0.7808
%!         0.1667    1.3238   -0.3475   -0.6483
%!         0.1384    1.0917   -0.2985   -0.5033
%!         0.1039    0.8539   -0.2995   -0.3083
%!         0.0697    0.6475   -0.1155   -0.4483
%!         0.0391    0.4233   -0.0315   -0.2033
%!         0.0027    0.2235   -0.0775    0.0317
%!        -0.0330    0.0966    0.0155   -0.2458
%!        -0.0606   -0.0817    0.1125    0.0542
%!        -0.0827   -0.3000    0.2685    0.1242
%!        -0.0939   -0.4675    0.2865    0.0642
%!        -0.0950   -0.5866    0.1685    0.3717
%!        -0.0840   -0.6702    0.2735    0.1017
%!        -0.0611   -0.7676    0.2505    0.2867
%!        -0.0432   -0.7891    0.1245    0.5342
%!        -0.0289   -0.6736    0.1945    0.0842
%!        -0.0126   -0.5516    0.1885    0.2392
%!         0.0003   -0.4055    0.1795    0.1967
%!         0.0159   -0.2097    0.1605   -0.1208
%!         0.0348   -0.0408    0.0925    0.0217
%!         0.0506    0.1208    0.1395   -0.1408
%!         0.0627    0.3078    0.0795   -0.2183
%!         0.0723    0.4773    0.0175   -0.3158
%!         0.0792    0.5095    0.0215   -0.3533
%!         0.0752    0.4314    0.0085   -0.1533
%!         0.0590    0.4053    0.0425   -0.2683
%!         0.0378    0.4174    0.0245   -0.2658
%!         0.0103    0.4410    0.0295   -0.1783
%!        -0.0258    0.4940    0.0905   -0.2633
%!        -0.0652    0.4837    0.1385   -0.2758
%!        -0.1035    0.3611    0.1805   -0.2258
%!        -0.1429    0.1376    0.2035   -0.0383
%!        -0.1847   -0.2251    0.3095    0.0167
%!        -0.2340   -0.7017    0.3245    0.3692
%!        -0.3036   -1.1050    0.3355    0.7517
%!        -0.3827   -1.3696    0.5995    0.4917
%!        -0.4557   -1.6423    0.7655    0.7017
%!        -0.5309   -1.9082    0.7975    1.1367
%!        -0.6018   -2.0464    0.9805    0.8317
%!        -0.6458   -2.0844    1.1445    0.7692
%!        -0.6613   -2.0949    1.2125    0.9217
%!        -0.6589   -2.0794    1.1595    0.9317
%!        -0.6399   -1.9906    1.1525    0.7942
%!        -0.6009   -1.8632    1.1895    0.6867
%!        -0.5490   -1.7523    1.0765    0.7617
%!        -0.4912   -1.6356    0.9615    0.6342
%!        -0.4272   -1.5132    0.8845    0.5817
%!        -0.3567   -1.3945    0.8295    0.5517
%!        -0.2766   -1.3215    0.7085    0.4392
%!        -0.1868   -1.3317    0.5175    0.4792
%!        -0.0971   -1.3609    0.3765    0.5317
%!        -0.0192   -1.3473    0.2125    0.6367
%!         0.0466   -1.2796    0.0715    0.5192
%!         0.1089   -1.2096   -0.0385    0.4117
%!         0.1644   -1.1582   -0.1425    0.5742
%!         0.2056   -1.0522   -0.2015    0.4542
%!         0.2331   -0.9216   -0.3305    0.3817
%!         0.2478   -0.8330   -0.4485    0.4142
%!         0.2518   -0.7613   -0.4545    0.3017
%!         0.2470   -0.7081   -0.4315    0.2892
%!         0.2276   -0.6960   -0.4605    0.3442
%!         0.1865   -0.7076   -0.5155    0.3692
%!         0.1253   -0.7396   -0.4485    0.3242
%!         0.0472   -0.8146   -0.3375    0.4292
%!        -0.0473   -0.8961   -0.2235    0.4967
%!        -0.1521   -0.9436   -0.0515    0.4692
%!        -0.2577   -0.9516    0.1265    0.5667
%!        -0.3484   -0.8765    0.3575    0.3617
%!        -0.4103   -0.7632    0.4915    0.2542
%!        -0.4523   -0.6816    0.4825    0.4667
%!        -0.4835   -0.5694    0.5725    0.2892
%!        -0.4953   -0.4093    0.6665    0.1417
%!        -0.4786   -0.2418    0.7055    0.0917
%!        -0.4345   -0.0829    0.6615    0.0192
%!        -0.3719    0.0565    0.5455    0.0617
%!        -0.2949    0.2073    0.4655   -0.0808
%!        -0.2016    0.3566    0.3255   -0.1433
%!        -0.0946    0.3950    0.1765   -0.1383
%!         0.0180    0.2803    0.0325   -0.1083
%!         0.1239    0.1148   -0.1245    0.0392
%!         0.2137   -0.0357   -0.2785    0.0642
%!         0.2851   -0.1776   -0.4545    0.1267
%!         0.3386   -0.2838   -0.5135    0.1392
%!         0.3724   -0.3703   -0.5395    0.1992
%!         0.3804   -0.4262   -0.5995    0.2692
%!         0.3606   -0.3882   -0.6205    0.1717
%!         0.3143   -0.2657   -0.6005    0.1667
%!         0.2456   -0.0453   -0.4925    0.0092
%!         0.1633    0.2726   -0.3835   -0.1683
%!         0.0757    0.6140   -0.2705   -0.3183
%!        -0.0088    0.8954   -0.1175   -0.5333
%!        -0.0827    1.0590   -0.0395   -0.5358
%!        -0.1377    1.1857    0.0545   -0.7408
%!        -0.1707    1.3715    0.0725   -0.7933
%!        -0.1911    1.5865    0.0695   -0.7758
%!        -0.2027    1.7486    0.1445   -0.9933
%!        -0.2042    1.8322    0.1265   -0.8733
%!        -0.1901    1.9362    0.2065   -1.0933
%!        -0.1550    2.0756    0.2185   -1.2008
%!        -0.1150    2.2175    0.0835   -0.9258
%!        -0.0782    2.4161    0.1225   -1.3158
%!        -0.0382    2.5968    0.0995   -1.3508
%!         0.0009    2.6974    0.0685   -1.2508
%!         0.0392    2.7343    0.0955   -1.4683
%!         0.0755    2.6634    0.0165   -1.2133
%!         0.1076    2.5477    0.0225   -1.2458
%!         0.1431    2.4412   -0.0305   -1.2558
%!         0.1844    2.3062   -0.1165   -1.0908
%!         0.2260    2.1947   -0.1755   -1.1083
%!         0.2587    2.1622   -0.2835   -0.9508
%!         0.2798    2.1829   -0.3055   -1.0508
%!         0.2967    2.2071   -0.3375   -1.1083
%!         0.3114    2.2195   -0.3515   -1.0408
%!         0.3235    2.2042   -0.3295   -1.1133
%!         0.3315    2.0932   -0.3775   -0.9583
%!         0.3365    1.8795   -0.3815   -0.9233
%!         0.3446    1.6517   -0.4025   -0.8583
%!         0.3561    1.4464   -0.4285   -0.7183
%!         0.3619    1.2298   -0.4995   -0.5708
%!         0.3524    1.0610   -0.6035   -0.3683
%!         0.3355    0.9913   -0.5375   -0.5383
%!         0.3266    0.9109   -0.4835   -0.5358
%!         0.3218    0.7731   -0.5415   -0.2683
%!         0.3110    0.6149   -0.5715   -0.2408
%!         0.2969    0.4368   -0.5525   -0.1883
%!         0.2860    0.2525   -0.4825   -0.1483
%!         0.2740    0.0546   -0.5115    0.0767
%!         0.2521   -0.1300   -0.5075    0.2217
%!         0.2261   -0.2571   -0.3805    0.1317
%!         0.2065   -0.3238   -0.3155    0.1867
%!         0.1931   -0.2987   -0.3305    0.1942
%!         0.1808   -0.2329   -0.3565    0.2092
%!         0.1728   -0.2060   -0.2655    0.1117
%!         0.1778   -0.1696   -0.1935    0.0042
%!         0.1933   -0.0814   -0.2805    0.1292
%!         0.2136    0.0401   -0.3195    0.0017
%!         0.2433    0.1480   -0.3015   -0.1308
%!         0.2830    0.1864   -0.3395   -0.1183
%!         0.3231    0.1801   -0.5095   -0.0508
%!         0.3548    0.1985   -0.6255   -0.0283
%!         0.3828    0.2212   -0.5505   -0.2358
%!         0.4071    0.2130   -0.6185   -0.0833
%!         0.4161    0.2078   -0.7505   -0.0233
%!         0.4070    0.1729   -0.7735   -0.1208
%!         0.3820    0.0580   -0.7255    0.0892
%!         0.3472   -0.1245   -0.6195    0.0492
%!         0.3091   -0.3883   -0.6545    0.2617
%!         0.2664   -0.6903   -0.6355    0.5492
%!         0.2262   -1.0013   -0.4655    0.4792
%!         0.1930   -1.3467   -0.4925    0.8692
%!         0.1636   -1.6417   -0.5175    0.9717
%!         0.1427   -1.8525   -0.4415    0.9067
%!         0.1266   -2.0179   -0.4015    1.2292
%!         0.1118   -2.1203   -0.3485    1.0842
%!         0.1018   -2.1909   -0.3845    1.0817
%!         0.0912   -2.2222   -0.3635    1.2492
%!         0.0774   -2.1564   -0.2115    0.9892
%!         0.0614   -2.0300   -0.1995    1.0467
%!         0.0438   -1.8759   -0.1595    0.9517
%!         0.0385   -1.7096    0.0155    0.5367
%!         0.0457   -1.5773    0.0455    0.6742
%!         0.0461   -1.4479   -0.0415    0.7242
%!         0.0371   -1.2867   -0.0325    0.3842
%!         0.0259   -1.1849    0.0445    0.4142
%!         0.0133   -1.1751    0.1485    0.4092
%!        -0.0023   -1.1735    0.1255    0.3992
%!        -0.0234   -1.1757    0.0705    0.5492
%!        -0.0427   -1.2288    0.1825    0.3642
%!        -0.0546   -1.3379    0.1545    0.5367
%!        -0.0693   -1.4457    0.0385    0.8092
%!        -0.0862   -1.4768    0.0965    0.6067
%!        -0.0960   -1.4632    0.1635    0.6542
%!        -0.1013   -1.4416    0.1445    0.7117
%!        -0.1098   -1.3817    0.0555    0.6417
%!        -0.1307   -1.2943    0.0305    0.7067
%!        -0.1687   -1.1853    0.1585    0.6017
%!        -0.2203   -1.0350    0.2505    0.5317
%!        -0.2805   -0.8197    0.3175    0.4542
%!        -0.3427   -0.5362    0.4545    0.3067
%!        -0.3950   -0.2783    0.6245    0.1217
%!        -0.4257   -0.1265    0.6745   -0.0083
%!        -0.4347   -0.0352    0.6115    0.0642
%!        -0.4273    0.0849    0.6455   -0.0608
%!        -0.4080    0.2669    0.6375   -0.0983
%!        -0.3808    0.4593    0.5595   -0.1183
%!        -0.3378    0.6035    0.5465   -0.4158
%!        -0.2776    0.6956    0.4895   -0.3233
%!        -0.2139    0.7767    0.4145   -0.2483
%!        -0.1485    0.8759    0.3105   -0.5033
%!        -0.0798    0.9496    0.1465   -0.4033
%!        -0.0138    0.9867    0.1065   -0.4133
%!         0.0481    1.0353    0.0825   -0.5458
%!         0.1014    1.0584   -0.0285   -0.3958
%!         0.1417    1.0195   -0.1245   -0.3508
%!         0.1810    0.9643   -0.1175   -0.5058
%!         0.2304    0.8987   -0.1325   -0.4633
%!         0.2784    0.8223   -0.3155   -0.1833
%!         0.3158    0.7741   -0.4165   -0.3058
%!         0.3477    0.7241   -0.4475   -0.3608
%!         0.3700    0.6195   -0.5515   -0.0458
%!         0.3809    0.5139   -0.5905   -0.1633
%!         0.3896    0.4472   -0.5885   -0.1983
%!         0.3935    0.3697   -0.5985    0.0417
%!         0.3879    0.3296   -0.6335   -0.0308
%!         0.3781    0.3844   -0.6795   -0.1183
%!         0.3661    0.4741   -0.6555   -0.0908
%!         0.3493    0.6039   -0.5805   -0.2358
%!         0.3240    0.7004   -0.5635   -0.2533
%!         0.2858    0.6189   -0.5785   -0.0933
%!         0.2432    0.4427   -0.4435   -0.2383
%!         0.2074    0.2354   -0.3355   -0.1458
%!         0.1672    0.0128   -0.4075    0.2967
%!         0.1183   -0.1060   -0.3275    0.0767
%!         0.0720   -0.1569   -0.2185    0.0792
%!         0.0281   -0.1837   -0.1755    0.2942
%!        -0.0110   -0.1465   -0.0965    0.0092
%!        -0.0430   -0.1124   -0.1035    0.1892
%!        -0.0694   -0.0907    0.0345    0.1267
%!        -0.0822   -0.0402    0.1395   -0.0883
%!        -0.0840    0.0366    0.0545    0.1267
%!        -0.0813    0.1044    0.1125   -0.0758
%!        -0.0744    0.0936    0.1615   -0.0783
%!        -0.0736    0.0579    0.1215    0.1442
%!        -0.0809    0.0546    0.1445   -0.0458
%!        -0.0869    0.0402    0.1665   -0.0033
%!        -0.0882    0.0025    0.2585    0.0167
%!        -0.0891   -0.0893    0.2335    0.0617
%!        -0.0989   -0.2016    0.1095    0.2567
%!        -0.1157   -0.2898    0.1755    0.1517
%!        -0.1279   -0.4203    0.2505    0.2192
%!        -0.1321   -0.5523    0.2425    0.2892
%!        -0.1353   -0.6380    0.1515    0.3892
%!        -0.1451   -0.7266    0.0955    0.5492
%!        -0.1517   -0.8032    0.1775    0.3517
%!        -0.1402   -0.8444    0.1445    0.4217
%!        -0.1103   -0.8473    0.0825    0.4467
%!        -0.0677   -0.8711    0.0725    0.3342
%!        -0.0286   -0.9469   -0.0805    0.6842
%!         0.0023   -0.9579   -0.1245    0.4242
%!         0.0381   -0.9479   -0.1135    0.2117
%!         0.0664   -0.9993   -0.2065    0.7217
%!         0.0742   -1.0048   -0.1365    0.4642
%!         0.0735   -1.0174   -0.0945    0.2817
%!         0.0619   -1.1243   -0.1855    0.6692
%!         0.0339   -1.2704   -0.1235    0.5067
%!        -0.0041   -1.4761   -0.0475    0.6292
%!        -0.0512   -1.6788    0.0215    0.8992
%!        -0.0973   -1.7716    0.1645    0.6467
%!        -0.1341   -1.8061    0.1835    0.7942
%!        -0.1695   -1.7935    0.1665    0.9717
%!        -0.1988   -1.7171    0.2675    0.6517
%!        -0.2107   -1.6368    0.3285    0.6417
%!        -0.2119   -1.5489    0.3375    0.7267
%!        -0.2123   -1.4181    0.3615    0.5617
%!        -0.2172   -1.2714    0.3455    0.5342
%!        -0.2319   -1.0671    0.3335    0.4817
%!        -0.2558   -0.7123    0.3665    0.2467
%!        -0.2838   -0.2669    0.4385    0.0517
%!        -0.3123    0.1531    0.5195   -0.1483
%!        -0.3390    0.5226    0.5215   -0.3008
%!        -0.3576    0.8555    0.4985   -0.4808
%!        -0.3583    1.1684    0.5345   -0.7233
%!        -0.3436    1.4304    0.5085   -0.6883
%!        -0.3233    1.6427    0.4625   -0.7483
%!        -0.2932    1.8196    0.4685   -1.0258
%!        -0.2534    1.8815    0.3835   -0.8933
%!        -0.2164    1.8872    0.2795   -0.7883
%!        -0.1816    1.9679    0.2695   -0.9983
%!        -0.1423    2.0907    0.2485   -1.0408
%!        -0.1081    2.2150    0.1715   -0.9933
%!        -0.0923    2.2823    0.0785   -0.9883
%!        -0.0926    2.2505    0.0865   -1.0533
%!        -0.0974    2.2151    0.1815   -1.1133
%!        -0.1033    2.1746    0.2105   -1.0133
%!        -0.1130    2.0369    0.1895   -0.9683
%!        -0.1261    1.8613    0.1905   -0.8858
%!        -0.1377    1.7249    0.2185   -0.8358
%!        -0.1403    1.5793    0.2285   -0.8208
%!        -0.1310    1.4183    0.2345   -0.7683
%!        -0.1180    1.2871    0.2165   -0.6133
%!        -0.1102    1.1936    0.1955   -0.5383
%!        -0.1015    1.1864    0.2075   -0.7283
%!        -0.0886    1.2592    0.1275   -0.5833
%!        -0.0749    1.3410    0.1485   -0.5658
%!        -0.0470    1.4143    0.2895   -0.9458
%!        -0.0031    1.4604    0.1465   -0.6733
%!         0.0402    1.5447   -0.0435   -0.6083
%!         0.0876    1.6959   -0.0645   -1.0383
%!         0.1412    1.7869   -0.1525   -0.8083
%!         0.1883    1.8245   -0.2255   -0.8208
%!         0.2252    1.8679   -0.3275   -0.9933
%!         0.2467    1.9045   -0.4395   -0.8208
%!         0.2469    1.9533   -0.4565   -0.9058
%!         0.2301    1.9731   -0.5135   -0.9308
%!         0.2034    1.9264   -0.4865   -0.9233
%!         0.1679    1.8261   -0.4125   -0.8458
%!         0.1191    1.7087   -0.3975   -0.7083
%!         0.0578    1.6395   -0.3765   -0.7758
%!        -0.0120    1.5785   -0.3385   -0.6708
%!        -0.0860    1.4653   -0.1925   -0.6008
%!        -0.1515    1.3092    0.0135   -0.6958
%!        -0.2008    1.0958    0.0985   -0.5583
%!        -0.2418    0.8423    0.0795   -0.3258
%!        -0.2793    0.5898    0.1115   -0.2908
%!        -0.3053    0.3063    0.2445   -0.3208
%!        -0.3232   -0.0402    0.2745    0.0467
%!        -0.3460   -0.3642    0.2835    0.3317
%!        -0.3602   -0.6486    0.4845    0.0767
%!        -0.3510   -1.0138    0.5625    0.3067
%!        -0.3365   -1.3782    0.4035    0.8092
%!        -0.3285   -1.6311    0.3775    0.7192
%!        -0.3177   -1.8760    0.4785    0.7742
%!        -0.3025   -2.0846    0.5275    1.0067
%!        -0.2861   -2.1696    0.4745    1.0042
%!        -0.2632   -2.1988    0.4385    0.9742
%!        -0.2280   -2.2098    0.4705    0.9567
%!        -0.1835   -2.2121    0.3835    1.0492
%!        -0.1343   -2.2346    0.2365    1.0167
%!        -0.0799   -2.2940    0.2005    0.9267
%!        -0.0296   -2.4018    0.1485    1.1567
%!         0.0054   -2.5377    0.0265    1.2892
%!         0.0379   -2.6953   -0.0205    1.1017
%!         0.0850   -2.8762   -0.0155    1.1667
%!         0.1365   -3.0209   -0.1045    1.4792
%!         0.1758   -3.0638   -0.2865    1.4992
%!         0.2069   -3.0199   -0.3975    1.3217
%!         0.2399   -2.9711   -0.3785    1.2517
%!         0.2687   -2.9543   -0.4125    1.3817
%!         0.2813   -2.9447   -0.4615    1.3142
%!         0.2737   -2.9591   -0.4575    1.2217
%!         0.2385   -3.0112   -0.4585    1.4592
%!         0.1778   -3.0317   -0.3535    1.3667
%!         0.1115   -2.9655   -0.1805    1.1367
%!         0.0469   -2.8414   -0.0535    1.2642
%!        -0.0254   -2.7009    0.0305    1.2717
%!        -0.1049   -2.5296    0.0665    1.1567
%!        -0.1749   -2.2903    0.2005    0.8842
%!        -0.2261   -2.0294    0.3485    0.7692
%!        -0.2709   -1.8409    0.3965    0.9217
%!        -0.3160   -1.7059    0.4925    0.6592
%!        -0.3553   -1.5587    0.5645    0.5542
%!        -0.3887   -1.4311    0.6065    0.6467
%!        -0.4132   -1.3691    0.6915    0.4717
%!        -0.4202   -1.3607    0.7465    0.4992
%!        -0.4079   -1.3743    0.8165    0.5242
%!        -0.3790   -1.3781    0.7885    0.4892
%!        -0.3393   -1.3442    0.6355    0.5792
%!        -0.2925   -1.2459    0.5555    0.4817
%!        -0.2378   -1.1148    0.5695    0.3467
%!        -0.1856   -0.9931    0.4825    0.4667
%!        -0.1486   -0.8185    0.3805    0.3767
%!        -0.1216   -0.5874    0.3885    0.0717
%!        -0.1015   -0.3821    0.3235    0.1667
%!        -0.0898   -0.1778    0.2855    0.0667
%!        -0.0712    0.0346    0.3355   -0.2858
%!        -0.0379    0.2085    0.3175   -0.2483
%!        -0.0027    0.3772    0.2065   -0.2333
%!         0.0236    0.5655    0.0175   -0.3458
%!         0.0438    0.7329   -0.0445   -0.4733
%!         0.0635    0.8253    0.0225   -0.5433
%!         0.0801    0.8649    0.0045   -0.4633
%!         0.0903    0.9308   -0.0115   -0.6133
%!         0.0901    1.0208   -0.0875   -0.5258
%!         0.0744    1.1279   -0.1185   -0.4933
%!         0.0558    1.2299   -0.0115   -0.8233
%!         0.0408    1.2587   -0.0005   -0.6233
%!         0.0191    1.2574    0.0385   -0.5433
%!        -0.0063    1.2484    0.0785   -0.7158
%!        -0.0255    1.1921    0.0205   -0.5933
%!        -0.0336    1.1108    0.0425   -0.6808
%!        -0.0357    0.9984    0.0135   -0.4883
%!        -0.0415    0.8908    0.0155   -0.3583
%!        -0.0416    0.8448    0.0965   -0.6158
%!        -0.0291    0.8026    0.0265   -0.4233
%!        -0.0135    0.7560   -0.0525   -0.2958
%!         0.0064    0.7406   -0.0195   -0.4958
%!         0.0347    0.7367   -0.0165   -0.4208
%!         0.0633    0.7939   -0.0945   -0.3233
%!         0.0912    0.9579   -0.1195   -0.5658
%!         0.1225    1.1168   -0.1215   -0.6283
%!         0.1480    1.1866   -0.2055   -0.4508
%!         0.1662    1.2186   -0.1875   -0.7083
%!         0.1851    1.2276   -0.2065   -0.7058
%!         0.1996    1.2388   -0.2605   -0.4633
%!         0.2131    1.3063   -0.2195   -0.7858
%!         0.2323    1.3793   -0.3485   -0.6333
%!         0.2558    1.4242   -0.3975   -0.6083
%!         0.2937    1.4094   -0.3555   -0.9133
%!         0.3392    1.2810   -0.5475   -0.4583
%!         0.3757    1.1218   -0.6745   -0.4483
%!         0.4103    0.9746   -0.7225   -0.6508
%!         0.4441    0.8090   -0.8085   -0.2808
%!         0.4710    0.6612   -0.8195   -0.3258
%!         0.4931    0.5291   -0.8975   -0.3208
%!         0.5052    0.4500   -0.9735   -0.1608
%!         0.4969    0.4825   -1.0065   -0.2058
%!         0.4640    0.5571   -1.0395   -0.1783
%!         0.4134    0.6007   -0.9225   -0.3358
%!         0.3525    0.5695   -0.7705   -0.2733
%!         0.2808    0.4708   -0.6625   -0.1683
%!         0.2038    0.3850   -0.5825   -0.2383
%!         0.1330    0.3329   -0.4905   -0.1808
%!         0.0800    0.2964   -0.3315   -0.2783
%!         0.0499    0.2154   -0.2585   -0.2158
%!         0.0356    0.0501   -0.2945    0.0017
%!         0.0345   -0.1225   -0.2425   -0.1633
%!         0.0424   -0.2963   -0.2555    0.0417
%!         0.0454   -0.4329   -0.3185    0.2467
%!         0.0459   -0.4041   -0.2455   -0.0408
%!         0.0501   -0.2796   -0.1785    0.0867
%!         0.0562   -0.1394   -0.1245    0.0392
%!         0.0708    0.0026   -0.1095   -0.2233
%!         0.0921    0.1000   -0.2235    0.0242
%!         0.1173    0.2755   -0.1985   -0.2208
%!         0.1574    0.5562   -0.1505   -0.5558
%!         0.2034    0.7663   -0.3005   -0.3233
%!         0.2345    0.8696   -0.3995   -0.4508
%!         0.2466    0.8681   -0.4305   -0.5183
%!         0.2426    0.8135   -0.4365   -0.4033
%!         0.2294    0.8375   -0.3955   -0.5383
%!         0.2117    0.9195   -0.3865   -0.4633
%!         0.1889    1.0417   -0.3415   -0.5258
%!         0.1615    1.2338   -0.3415   -0.6758
%!         0.1260    1.4168   -0.4065   -0.6508
%!         0.0810    1.5369   -0.3505   -0.7708
%!         0.0342    1.5810   -0.1615   -0.8558
%!        -0.0091    1.5314   -0.0225   -0.8083
%!        -0.0503    1.4241   -0.0285   -0.7283
%!        -0.0876    1.2739    0.0045   -0.7208
%!        -0.1141    1.0825    0.0965   -0.7008
%!        -0.1308    0.9130    0.1345   -0.5608
%!        -0.1446    0.7945    0.1385   -0.5108
%!        -0.1608    0.7183    0.1425   -0.4833
%!        -0.1846    0.7185    0.1855   -0.4508
%!        -0.2169    0.7908    0.2355   -0.5133
%!        -0.2502    0.8577    0.2985   -0.5833
%!        -0.2747    0.8571    0.4505   -0.6633
%!        -0.2911    0.7965    0.5145   -0.5358
%!        -0.3055    0.7070    0.5125   -0.5008
%!        -0.3153    0.5460    0.5205   -0.5583
%!        -0.3242    0.2987    0.4875   -0.2383
%!        -0.3392    0.0706    0.5795   -0.1583
%!        -0.3528   -0.0710    0.6875   -0.2433
%!        -0.3649   -0.1759    0.6485    0.0217
%!        -0.3827   -0.3188    0.6335    0.1142
%!        -0.3987   -0.5033    0.7005    0.0142
%!        -0.4061   -0.6932    0.7185    0.2192
%!        -0.4107   -0.8736    0.7015    0.4317
%!        -0.4089   -1.0089    0.7395    0.3192
%!        -0.3942   -1.1045    0.7075    0.4167
%!        -0.3738   -1.2074    0.5835    0.6192
%!        -0.3490   -1.3199    0.5075    0.5067
%!        -0.3125   -1.4543    0.4785    0.5492
%!        -0.2685   -1.5904    0.4425    0.6667
%!        -0.2324   -1.6779    0.2805    0.7667
%!        -0.2177   -1.7222    0.1215    0.8817
%!        -0.2177   -1.7124    0.2035    0.6517
%!        -0.2191   -1.6440    0.2945    0.6017
%!        -0.2262   -1.5442    0.2725    0.7342
%!        -0.2446   -1.4283    0.2725    0.5967
%!        -0.2675   -1.3620    0.3145    0.5392
%!        -0.2852   -1.3847    0.3815    0.5242
%!        -0.2915   -1.4221    0.3885    0.5442
%!        -0.2859   -1.4241    0.3955    0.6142
%!        -0.2662   -1.4117    0.4345    0.5817
%!        -0.2261   -1.4121    0.3795    0.5092
%!        -0.1681   -1.4674    0.1875    0.5967
%!        -0.1067   -1.5856    0.0025    0.7967
%!        -0.0484   -1.7123    0.0105    0.7017
%!         0.0055   -1.8149   -0.0275    0.7892
%!         0.0441   -1.8852   -0.1755    1.0267
%!         0.0662   -1.9856   -0.1775    0.8417
%!         0.0769   -2.2193   -0.1235    1.0092
%!         0.0690   -2.4788   -0.1095    1.2917
%!         0.0433   -2.5547   -0.1075    1.1892
%!         0.0121   -2.4502   -0.0515    1.1292
%!        -0.0163   -2.3174    0.0695    1.0167
%!        -0.0431   -2.2535    0.0355    1.0567
%!        -0.0690   -2.2237   -0.0085    1.0267
%!        -0.0830   -2.1826    0.0905    0.8217
%!        -0.0802   -2.1610    0.1245    0.9717
%!        -0.0644   -2.1314    0.0895    0.8942
%!        -0.0389   -2.0083   -0.0235    0.7767
%!        -0.0144   -1.7807   -0.0925    0.8117
%!         0.0039   -1.5045   -0.0555    0.5167
%!         0.0181   -1.2185   -0.0975    0.3967
%!         0.0257   -0.9135   -0.1055    0.2917
%!         0.0253   -0.5901   -0.0405    0.0192
%!         0.0119   -0.2804   -0.0405   -0.0208
%!        -0.0223    0.0284   -0.0525   -0.1183
%!        -0.0722    0.3668    0.0375   -0.3658
%!        -0.1248    0.6931    0.2185   -0.5433
%!        -0.1762    0.9826    0.3255   -0.5983
%!        -0.2279    1.2387    0.3505   -0.7408
%!        -0.2801    1.4081    0.3595   -0.7783
%!        -0.3313    1.4824    0.4385   -0.7633
%!        -0.3714    1.4586    0.5705   -0.8483
%!        -0.3940    1.3294    0.5915   -0.6908
%!        -0.4058    1.2356    0.5915   -0.5733
%!        -0.4124    1.2560    0.5965   -0.6108
%!        -0.4123    1.3012    0.5405   -0.5908
%!        -0.3988    1.2608    0.5535   -0.6983
%!        -0.3752    1.0714    0.5255   -0.4733
%!        -0.3563    0.8491    0.5215   -0.2658
%!        -0.3428    0.6776    0.5805   -0.4083
%!        -0.3337    0.4690    0.4925   -0.1158
%!        -0.3337    0.2571    0.5005    0.0117
%!        -0.3316    0.0998    0.6345   -0.1208
%!        -0.3140   -0.0275    0.7025   -0.0458
%!        -0.2848   -0.1338    0.6455    0.0317
%!        -0.2575   -0.2528    0.4625    0.1967
%!        -0.2342   -0.3685    0.4365    0.0917
%!        -0.2083   -0.4770    0.4665    0.0767
%!        -0.1875   -0.6038    0.3695    0.3942
%!        -0.1767   -0.6891    0.3645    0.2767
%!        -0.1647   -0.7274    0.4225    0.1342
%!        -0.1496   -0.7568    0.3685    0.2642
%!        -0.1427   -0.8146    0.2315    0.3767
%!        -0.1438   -0.9273    0.2285    0.3517
%!        -0.1384   -1.0284    0.4105    0.2117
%!        -0.1249   -1.1491    0.4185    0.4717
%!        -0.1147   -1.3249    0.2665    0.6917
%!        -0.0989   -1.4528    0.2795    0.4867
%!        -0.0619   -1.5434    0.3525    0.5667
%!        -0.0081   -1.6087    0.2615    0.7242
%!         0.0507   -1.6067    0.0855    0.6692
%!         0.1051   -1.5640   -0.0405    0.7317
%!         0.1482   -1.4225   -0.0795    0.6492
%!         0.1804   -1.1516   -0.1505    0.4267
%!         0.2027   -0.8082   -0.2635    0.3492
%!         0.2135   -0.4165   -0.2375    0.1192
%!         0.2148   -0.0383   -0.1745   -0.1108
%!         0.2076    0.3144   -0.3045   -0.1433
%!         0.2004    0.6244   -0.3415   -0.4108
%!         0.2110    0.7909   -0.2105   -0.6583
%!         0.2361    0.8213   -0.2295   -0.3833
%!         0.2602    0.7620   -0.3495   -0.3533
%!         0.2841    0.6144   -0.4395   -0.4783
%!         0.3056    0.4209   -0.4835   -0.1683
%!         0.3176    0.2084   -0.4535   -0.0783
%!         0.3250   -0.0407   -0.4695   -0.0783
%!         0.3287   -0.3328   -0.5005    0.1792
%!         0.3187   -0.6266   -0.5085    0.4117
%!         0.2961   -0.8894   -0.5195    0.4617
%!         0.2740   -1.0959   -0.4995    0.4492
%!         0.2542   -1.1684   -0.4795    0.6467
%!         0.2321   -1.0800   -0.3895    0.5542
%!         0.2110   -0.8808   -0.3495    0.3067
%!         0.1900   -0.6415   -0.3965    0.3167
%!         0.1682   -0.4753   -0.3005    0.0467
%!         0.1458   -0.4414   -0.2055    0.0917
%!         0.1163   -0.4386   -0.1425    0.2617
%!         0.0891   -0.3733    0.0045   -0.0933
%!         0.0760   -0.3217    0.0415   -0.0408
%!         0.0669   -0.3010   -0.0135    0.1467
%!         0.0560   -0.2167   -0.0305   -0.0708
%!         0.0476   -0.1060   -0.0405   -0.0458
%!         0.0458   -0.0230    0.0035   -0.1208
%!         0.0540    0.0155   -0.0085   -0.1733
%!         0.0668   -0.0147   -0.1005    0.0117
%!         0.0792   -0.0398   -0.1425   -0.0508
%!         0.0933   -0.0347   -0.1735   -0.0508
%!         0.1089   -0.0374   -0.2045   -0.0283
%!         0.1195   -0.0380   -0.2625    0.0292
%!         0.1195   -0.0179   -0.2725    0.0267
%!         0.1110    0.0673   -0.2335   -0.1483
%!         0.0916    0.2159   -0.2865   -0.0383
%!         0.0596    0.3960   -0.2405   -0.1733
%!         0.0267    0.6076   -0.1025   -0.4333
%!        -0.0017    0.8519   -0.0395   -0.3883
%!        -0.0248    1.1211    0.0015   -0.6533
%!        -0.0414    1.3389   -0.0415   -0.6833
%!        -0.0505    1.4805   -0.0145   -0.7208
%!        -0.0418    1.5473    0.0785   -1.0008
%!        -0.0182    1.5086   -0.0465   -0.7158
%!         0.0047    1.4786   -0.1455   -0.6858
%!         0.0286    1.4618   -0.1135   -0.8933
%!         0.0544    1.3645   -0.1525   -0.6733
%!         0.0735    1.2553   -0.1945   -0.6583
%!         0.0804    1.1186   -0.2285   -0.5433
%!         0.0768    0.9336   -0.1345   -0.4483
%!         0.0700    0.7648   -0.0355   -0.4333
%!         0.0642    0.5964   -0.0615   -0.2583
%!         0.0608    0.4208   -0.0495   -0.2258
%!         0.0602    0.2307   -0.0155   -0.0408
%!         0.0659    0.0514    0.0435   -0.0433
%!         0.0820   -0.0945    0.0105   -0.0583
%!         0.0956   -0.2256   -0.1235    0.2667
%!         0.0948   -0.2807   -0.1055    0.1717
%!         0.0868   -0.2945   -0.0515    0.0367
%!         0.0754   -0.3494   -0.0825    0.1942
%!         0.0574   -0.4033   -0.0815    0.1867
%!         0.0357   -0.4587   -0.0115    0.1717
%!         0.0156   -0.5346    0.0505    0.1867
%!        -0.0023   -0.6237    0.0135    0.3017
%!        -0.0165   -0.7228    0.0525    0.2717
%!        -0.0230   -0.7928    0.1605    0.2417
%!        -0.0284   -0.8219    0.1615    0.4642
%!        -0.0327   -0.8036    0.1625    0.2917
%!        -0.0210   -0.7466    0.1915    0.1567
%!         0.0111   -0.7242    0.2005    0.3042
%!         0.0592   -0.7573    0.1465    0.2667
%!         0.1231   -0.7844   -0.0095    0.2467
%!         0.1999   -0.7794   -0.1375    0.2692
%!         0.2804   -0.7513   -0.2825    0.3117
%!         0.3560   -0.6774   -0.4315    0.2467
%!         0.4198   -0.5854   -0.5625    0.2242
%!         0.4650   -0.5277   -0.6685    0.3242
%!         0.4975   -0.4888   -0.6675    0.1642
%!         0.5329   -0.4856   -0.6705    0.0967
%!         0.5714   -0.5286   -0.7535    0.2292
%!         0.5974   -0.5528   -0.8835    0.3717
%!         0.6036   -0.5237   -0.9285    0.3542
%!         0.6018   -0.4513   -0.8545    0.1417
%!         0.5995   -0.3414   -0.8395    0.2067
%!         0.5916   -0.2010   -0.8205    0.1942
%!         0.5754   -0.0626   -0.8065    0.0892
%!         0.5542    0.0470   -0.8155    0.0967
%!         0.5375    0.0963   -0.7775   -0.0808
%!         0.5306    0.0637   -0.8045    0.0367
%!         0.5296    0.0065   -0.7795    0.0492
%!         0.5319   -0.0821   -0.7785   -0.0208
%!         0.5239   -0.2330   -0.9285    0.2767
%!         0.4911   -0.3534   -0.9875    0.2492
%!         0.4368   -0.4041   -0.9135    0.2767
%!         0.3774   -0.4396   -0.6965    0.2317
%!         0.3299   -0.5189   -0.5085    0.1617
%!         0.2946   -0.5876   -0.5285    0.4617
%!         0.2732   -0.5118   -0.4495    0.1917
%!         0.2735   -0.3676   -0.4335    0.1092
%!         0.2897   -0.2488   -0.4875    0.2317
%!         0.3188   -0.1107   -0.4445   -0.0758
%!         0.3580    0.0310   -0.4655    0.0367
%!         0.3999    0.2099   -0.4825   -0.0933
%!         0.4416    0.4537   -0.6305   -0.2708
%!         0.4758    0.6744   -0.8075   -0.1533
%!         0.5033    0.8167   -0.7235   -0.5058
%!         0.5249    0.8812   -0.7385   -0.3783
%!         0.5302    0.9226   -0.8115   -0.2858
%!         0.5213    0.9572   -0.8195   -0.5783
%!         0.4967    0.9528   -0.8725   -0.2483
%!         0.4535    0.9563   -0.7685   -0.3808
%!         0.4022    0.9923   -0.6685   -0.5258
%!         0.3435    1.0462   -0.6205   -0.2433
%!         0.2816    1.1181   -0.4365   -0.5508
%!         0.2276    1.1811   -0.3685   -0.5858
%!         0.1752    1.2622   -0.3765   -0.4508
%!         0.1188    1.4145   -0.3025   -0.6808
%!         0.0624    1.5738   -0.1975   -0.7083
%!         0.0118    1.6516   -0.0645   -0.8358
%!        -0.0333    1.6513   -0.0625   -0.8383
%!        -0.0826    1.5769   -0.1065   -0.6158
%!        -0.1318    1.4702    0.0125   -0.7833
%!        -0.1670    1.4121    0.1115   -0.7908
%!        -0.1917    1.3594    0.0985   -0.5433
%!        -0.2125    1.2994    0.1295   -0.6108
%!        -0.2249    1.2767    0.2415   -0.6833
%!        -0.2319    1.2354    0.2795   -0.5283
%!        -0.2430    1.1914    0.2665   -0.4958
%!        -0.2598    1.1599    0.3615   -0.5608
%!        -0.2818    1.0552    0.4915   -0.4008
%!        -0.3076    0.8983    0.5765   -0.2833
%!        -0.3210    0.7477    0.6535   -0.4083
%!        -0.3101    0.6146    0.6765   -0.3383
%!        -0.2900    0.5163    0.5625    0.0067
%!        -0.2723    0.4557    0.4575   -0.0683
%!        -0.2465    0.3869    0.4035   -0.2233
%!        -0.2099    0.2499    0.3495    0.0092
%!        -0.1739    0.1037    0.3055    0.0267
%!        -0.1512    0.0119    0.1695    0.1517
%!        -0.1484   -0.0519    0.0805    0.2842
%!        -0.1510   -0.0699    0.2045   -0.0533
%!        -0.1495   -0.0575    0.2485    0.0517
%!        -0.1572   -0.0494    0.2205    0.2867
%!        -0.1715   -0.0612    0.3355   -0.0658
%!        -0.1810   -0.1391    0.3715    0.0442
%!        -0.1916   -0.2317    0.3255    0.2367
%!        -0.2045   -0.3066    0.3375    0.0842
%!        -0.2152   -0.4015    0.3625    0.2167
%!        -0.2228   -0.5104    0.4315    0.2292
%!        -0.2264   -0.6669    0.3695    0.2817
%!        -0.2322   -0.8304    0.2035    0.5217
%!        -0.2424   -0.9202    0.2115    0.4992
%!        -0.2438   -0.9549    0.3045    0.3742
%!        -0.2288   -0.9232    0.2735    0.3792
%!        -0.2076   -0.8649    0.1505    0.5142
%!        -0.1881   -0.8226    0.1345    0.4142
%!        -0.1670   -0.7409    0.1615    0.2617
%!        -0.1451   -0.6425    0.1015    0.3617
%!        -0.1230   -0.5638    0.1275    0.1917
%!        -0.0984   -0.5363    0.2195    0.1492
%!        -0.0803   -0.5953    0.1945    0.3917
%!        -0.0729   -0.6555    0.1085    0.3267
%!        -0.0632   -0.6865    0.1145    0.1717
%!        -0.0489   -0.7249    0.1485    0.3442
%!        -0.0439   -0.7028    0.0995    0.4717
%!        -0.0451   -0.6042    0.0625    0.2117
%!        -0.0406   -0.5269    0.0245    0.1367
%!        -0.0383   -0.4894   -0.0435    0.3392
%!        -0.0485   -0.4438   -0.0455    0.2067
%!        -0.0679   -0.4331   -0.0395    0.2042
%!        -0.0895   -0.4768    0.0445    0.2217
%!        -0.1077   -0.4958    0.1265    0.0817
%!        -0.1329   -0.4951    0.0275    0.3717
%!        -0.1737   -0.5221    0.0445    0.3217
%!        -0.2132   -0.5739    0.2165    0.0742
%!        -0.2403   -0.6589    0.2895    0.3267
%!        -0.2575   -0.7592    0.3165    0.3442
%!        -0.2639   -0.8295    0.2885    0.3417
%!        -0.2594   -0.8865    0.2775    0.3992
%!        -0.2440   -0.9962    0.2605    0.3367
%!        -0.2267   -1.1301    0.0985    0.6167
%!        -0.2183   -1.1910    0.0655    0.6417
%!        -0.2123   -1.1804    0.1735    0.4192
%!        -0.2063   -1.1308    0.1145    0.5367
%!        -0.2103   -1.0522   -0.0135    0.6292
%!        -0.2177   -0.9723    0.0595    0.3867
%!        -0.2085   -0.8974    0.2295    0.2317
%!        -0.1816   -0.8520    0.1905    0.4017
%!        -0.1471   -0.8541    0.0715    0.3917
%!        -0.1048   -0.8622    0.0615    0.2592
%!        -0.0576   -0.8463    0.0615    0.3767
%!        -0.0136   -0.8192   -0.0185    0.3392
%!         0.0235   -0.7788   -0.1415    0.3292
%!         0.0551   -0.7021   -0.1095    0.2867
%!         0.0856   -0.6467   -0.0675    0.1667
%!         0.1120   -0.6449   -0.2705    0.3217
%!         0.1297   -0.6291   -0.4115    0.2392
%!         0.1433   -0.6098   -0.3755    0.1167
%!         0.1493   -0.5649   -0.4015    0.2967
%!         0.1413   -0.4200   -0.4795    0.1742
%!         0.1251   -0.2349   -0.5045   -0.0333
%!         0.1035   -0.0723   -0.4195   -0.0283
%!         0.0714    0.0871   -0.3485   -0.0533
%!         0.0300    0.2866   -0.3345   -0.1883
%!        -0.0115    0.5233   -0.2235   -0.3508
%!        -0.0474    0.7916   -0.0725   -0.4258
%!        -0.0765    1.1562   -0.0105   -0.6133
%!        -0.0981    1.5229   -0.0365   -0.7958
%!        -0.1162    1.7436   -0.0585   -0.7983
%!        -0.1316    1.8822    0.0025   -0.9008
%!        -0.1349    1.9916    0.0605   -1.0358
%!        -0.1202    2.0391    0.0065   -1.0233
%!        -0.0932    2.0123   -0.0625   -0.9483
%!        -0.0609    1.9210   -0.0805   -0.9583
%!        -0.0293    1.8059   -0.1785   -0.8458
%!        -0.0052    1.7253   -0.2625   -0.7733
%!         0.0091    1.7317   -0.2195   -0.8883
%!         0.0063    1.7910   -0.1925   -0.7333
%!        -0.0175    1.8525   -0.1535   -0.7433
%!        -0.0460    1.9114   -0.0405   -1.0208
%!        -0.0656    1.9435    0.0365   -0.9383
%!        -0.0793    1.9967    0.1075   -0.9433
%!        -0.0890    2.0655    0.1275   -1.0708
%!        -0.0971    2.0547    0.0985   -0.9808
%!        -0.1056    1.9936    0.1425   -1.0508
%!        -0.1133    1.9289    0.1445   -1.0233
%!        -0.1202    1.8841    0.1445   -1.0283
%!        -0.1278    1.8653    0.1605   -1.1133
%!        -0.1456    1.8273    0.0905   -0.9433
%!        -0.1796    1.7730    0.0805   -0.9833
%!        -0.2191    1.6584    0.1575   -1.0358
%!        -0.2544    1.4743    0.2565   -0.9083
%!        -0.2854    1.3089    0.3565   -0.9083
%!        -0.3208    1.1494    0.3295   -0.6758
%!        -0.3642    0.9927    0.3655   -0.6683
%!        -0.4052    0.8263    0.5005   -0.7658
%!        -0.4443    0.5780    0.5355   -0.4133
%!        -0.4891    0.3103    0.6355   -0.3558
%!        -0.5373    0.1107    0.7275   -0.2808
%!        -0.5849    0.0334    0.7645   -0.1008
%!        -0.6212    0.1017    0.8415   -0.3308
%!        -0.6375    0.2141    0.8055   -0.2608
%!        -0.6349    0.3187    0.8195   -0.2683
%!        -0.6100    0.4173    0.8785   -0.5558
%!        -0.5747    0.4404    0.6735   -0.2058
%!        -0.5453    0.3679    0.5525   -0.1733
%!        -0.5116    0.2496    0.6265   -0.4183
%!        -0.4674    0.1314    0.6485   -0.1408
%!        -0.4214    0.0316    0.6365   -0.1358
%!        -0.3811   -0.0349    0.5035   -0.0783
%!        -0.3498   -0.0504    0.4445    0.0092
%!        -0.3165   -0.0772    0.5055   -0.2258
%!        -0.2736   -0.1911    0.4325   -0.0983
%!        -0.2333   -0.3845    0.3495    0.0917
%!        -0.2116   -0.6080    0.2855    0.1917
%!        -0.2140   -0.7873    0.2095    0.3642
%!        -0.2278   -0.9055    0.1795    0.2642
%!        -0.2368   -1.0029    0.1665    0.3317
%!        -0.2350   -1.0601    0.2255    0.3617
%!        -0.2232   -1.0793    0.2145    0.2592
%!        -0.2166   -1.0744   -0.0305    0.4942
%!        -0.2288   -0.9663   -0.1795    0.4292
%!        -0.2482   -0.7630   -0.0595    0.1117
%!        -0.2598   -0.5967    0.1215    0.0967
%!        -0.2618   -0.4925    0.2025    0.0492
%!        -0.2590   -0.4010    0.1605    0.0367
%!        -0.2550   -0.3328    0.1185    0.0867
%!        -0.2431   -0.2946    0.1605   -0.0808
%!        -0.2166   -0.2787    0.1645   -0.0383
%!        -0.1817   -0.3122    0.1755    0.1067
%!        -0.1408   -0.4076    0.2485   -0.0058
%!        -0.0955   -0.5121    0.1335    0.1192
%!        -0.0580   -0.5406   -0.0965    0.2292
%!        -0.0371   -0.4996   -0.1945    0.1292
%!        -0.0357   -0.4631   -0.1495    0.2142
%!        -0.0497   -0.4436   -0.0065    0.0717
%!        -0.0737   -0.4519   -0.0165    0.0992
%!        -0.1119   -0.4687   -0.0865    0.2892
%!        -0.1561   -0.4543    0.0375   -0.0208
%!        -0.1950   -0.4876    0.1145    0.0817
%!        -0.2343   -0.5827    0.1375    0.2967
%!        -0.2695   -0.6924    0.2895    0.0017
%!        -0.2983   -0.8815    0.3295    0.3067
%!        -0.3325   -1.1117    0.3055    0.5492
%!        -0.3656   -1.2956    0.3705    0.2967
%!        -0.3906   -1.4858    0.4285    0.5417
%!        -0.4198   -1.6773    0.4905    0.8242
%!        -0.4544   -1.8112    0.5555    0.6917
%!        -0.4817   -1.9092    0.5635    0.7317
%!        -0.4984   -1.9624    0.5865    0.8517
%!        -0.5052   -1.9578    0.6725    0.7392
%!        -0.5043   -1.9750    0.6675    0.8467
%!        -0.5005   -2.0244    0.6645    0.8942
%!        -0.4895   -2.0492    0.7325    0.7292
%!        -0.4713   -2.0865    0.6905    0.9517
%!        -0.4485   -2.1329    0.7045    0.9417
%!        -0.4108   -2.1318    0.7525    0.7467
%!        -0.3562   -2.1064    0.6665    0.9792
%!        -0.2897   -2.0510    0.5845    0.8717
%!        -0.2097   -1.9414    0.4475    0.7142
%!        -0.1240   -1.8255    0.2975    0.8092
%!        -0.0458   -1.7261    0.1825    0.7217
%!         0.0176   -1.5932    0.0225    0.7017
%!         0.0662   -1.4228   -0.0535    0.5917
%!         0.1049   -1.3030   -0.0845    0.4792
%!         0.1335   -1.2418   -0.1745    0.5192
%!         0.1483   -1.1914   -0.2385    0.4542
%!         0.1466   -1.1992   -0.2885    0.5092
%!         0.1282   -1.2520   -0.2755    0.5542
%!         0.0992   -1.2430   -0.2205    0.5267
%!         0.0699   -1.1510   -0.1345    0.4517
%!         0.0478   -1.0541   -0.0455    0.3867
%!         0.0322   -0.9852   -0.0195    0.4267
%!         0.0207   -0.8964    0.0005    0.2817
%!         0.0094   -0.8166   -0.0155    0.3042
%!        -0.0108   -0.7687    0.0055    0.3492
%!        -0.0421   -0.6628    0.0855    0.1892
%!        -0.0816   -0.5166    0.1335    0.2067
%!        -0.1240   -0.4531    0.1985    0.1517
%!        -0.1587   -0.4955    0.2175    0.1317
%!        -0.1765   -0.5726    0.2375    0.2242
%!        -0.1697   -0.6386    0.2575    0.1567
%!        -0.1396   -0.6945    0.1815    0.2542
%!        -0.0986   -0.7609    0.0965    0.3417
%!        -0.0577   -0.8911    0.0185    0.4417
%!        -0.0179   -1.0590    0.0145    0.4917
%!         0.0260   -1.1811    0.0785    0.4217
%!         0.0683   -1.2444    0.0305    0.6217
%!         0.0949   -1.1920   -0.0365    0.6392
%!         0.1073   -1.0508   -0.0215    0.4067
%!         0.1155   -0.9975   -0.0025    0.4092
%!         0.1214   -1.0199    0.0115    0.4092
%!         0.1192   -0.9884    0.0035    0.4167
%!         0.1011   -0.8955   -0.0175    0.4642
%!         0.0706   -0.7734   -0.0055    0.3317
%!         0.0435   -0.6574    0.0385    0.1417
%!         0.0254   -0.5681    0.0495    0.1792
%!         0.0072   -0.4466    0.0325    0.2567
%!        -0.0080   -0.2535    0.0875   -0.0533
%!        -0.0115   -0.0873    0.0575   -0.1208
%!        -0.0112   -0.0130   -0.0395    0.0817
%!        -0.0089    0.0369    0.0145   -0.1758
%!         0.0030    0.0533    0.0625   -0.1708
%!         0.0194   -0.0025    0.0185    0.0467
%!         0.0374   -0.0338   -0.0445   -0.0783
%!         0.0607    0.0134   -0.0985   -0.0633
%!         0.0869    0.0666   -0.1175   -0.0358
%!         0.1116    0.1083   -0.1945   -0.0683
%!         0.1329    0.1776   -0.2645   -0.0883
%!         0.1515    0.2269   -0.2495   -0.1808
%!         0.1643    0.2195   -0.3015   -0.0533
%!         0.1677    0.2351   -0.3745   -0.0333
%!         0.1713    0.2860   -0.3445   -0.2508
%!         0.1813    0.2769   -0.3195   -0.1283
%!         0.1898    0.1852   -0.3195   -0.0083
%!         0.1937    0.0800   -0.3415   -0.0683
%!         0.1960    0.0125   -0.3745    0.0767
%!         0.2058   -0.0679   -0.2675   -0.0358
%!         0.2313   -0.2067   -0.2205   -0.0433
%!         0.2604   -0.2975   -0.3315    0.2442
%!         0.2806   -0.2723   -0.3645    0.0692
%!         0.2919   -0.2029   -0.3955    0.0117
%!         0.2899   -0.0986   -0.4325    0.0517
%!         0.2731    0.0407   -0.3855   -0.1783
%!         0.2420    0.1252   -0.3585   -0.1008
%!         0.1990    0.1201   -0.2645   -0.1433
%!         0.1551    0.0612   -0.2165   -0.2383
%!         0.1148   -0.0002   -0.2735   -0.0133
%!         0.0804   -0.0214   -0.2015   -0.1033
%!         0.0626   -0.0484   -0.0865   -0.2383
%!         0.0590   -0.1404   -0.1335    0.0367
%!         0.0604   -0.2218   -0.2005    0.0567
%!         0.0731   -0.2085   -0.1645   -0.1383
%!         0.0997   -0.1695   -0.1575   -0.0058
%!         0.1295   -0.1504   -0.2145    0.0242
%!         0.1556   -0.1147   -0.2605   -0.0783
%!         0.1749   -0.1237   -0.2505    0.0192
%!         0.1857   -0.2165   -0.2005   -0.0008
%!         0.1859   -0.2956   -0.2755    0.1167
%!         0.1726   -0.2906   -0.3025    0.1892
%!         0.1556   -0.2457   -0.1625   -0.0583
%!         0.1417   -0.2264   -0.1545    0.0942
%!         0.1269   -0.2038   -0.1905    0.0842
%!         0.1129   -0.1753   -0.1785   -0.0658
%!         0.0992   -0.1663   -0.1525    0.1442
%!         0.0875   -0.1169   -0.0275   -0.1183
%!         0.0790   -0.0582   -0.0545   -0.0283
%!         0.0663   -0.0174   -0.0605    0.0767
%!         0.0562    0.0665    0.0935   -0.3133
%!         0.0531    0.1512    0.0685   -0.1408
%!         0.0461    0.2151    0.0115   -0.1208
%!         0.0374    0.2488    0.0475   -0.3083
%!         0.0370    0.2115    0.1245   -0.2358
%!         0.0492    0.1377    0.1595   -0.3158
%!         0.0650    0.0180   -0.0255   -0.0283
%!         0.0750   -0.1500   -0.0805    0.0317
%!         0.0891   -0.3396    0.0055   -0.0733
%!         0.1133   -0.5416   -0.0395    0.1742
%!         0.1401   -0.7139   -0.1485    0.2417
%!         0.1603   -0.9009   -0.2485    0.4742
%!         0.1780   -1.1065   -0.1815    0.4742
%!         0.2099   -1.2319   -0.1665    0.3667
%!         0.2553   -1.2788   -0.3375    0.6817
%!         0.3076   -1.2666   -0.3695    0.5117
%!         0.3685   -1.2250   -0.3785    0.4217
%!         0.4305   -1.1970   -0.5295    0.5917
%!         0.4856   -1.1524   -0.7095    0.4792
%!         0.5373   -1.1079   -0.7375    0.4042
%!         0.5897   -1.0977   -0.6755    0.3842
%!         0.6362   -1.0851   -0.8095    0.4792
%!         0.6699   -1.0496   -0.9775    0.4067
%!         0.6894   -0.9976   -1.0365    0.3367
%!         0.6901   -0.9060   -1.0275    0.4142
%!         0.6719   -0.7677   -0.9995    0.2142
%!         0.6371   -0.6139   -1.0315    0.2267
%!         0.5849   -0.4463   -0.9335    0.1717
%!         0.5219   -0.2783   -0.7995    0.0167
%!         0.4573   -0.1421   -0.7785    0.0567
%!         0.4020   -0.0451   -0.6615   -0.1833
%!         0.3559    0.0389   -0.5805   -0.0758
%!         0.3050    0.2057   -0.5415    0.0042
%!         0.2521    0.4271   -0.4775   -0.3708
%!         0.2065    0.5643   -0.4755   -0.3033
%!         0.1686    0.6331   -0.3475   -0.3783
%!         0.1358    0.6722   -0.2895   -0.3708
%!         0.1037    0.6797   -0.3195   -0.2408
%!         0.0778    0.6585   -0.2575   -0.4533
%!         0.0624    0.5950   -0.2515   -0.2558
%!         0.0517    0.5407   -0.2285   -0.1908
%!         0.0473    0.5332   -0.1815   -0.3383
%!         0.0465    0.6105   -0.1795   -0.2558
%!         0.0382    0.7590   -0.1585   -0.3383
%!         0.0154    0.8366   -0.1745   -0.3208
%!        -0.0179    0.8604   -0.0875   -0.3733
%!        -0.0470    0.8872    0.1005   -0.5033
%!        -0.0626    0.8752    0.1875   -0.3783
%!        -0.0602    0.8813    0.2065   -0.5008
%!        -0.0402    0.8863    0.1245   -0.4583
%!        -0.0124    0.8391    0.0235   -0.3433
%!         0.0192    0.7919   -0.0165   -0.4783
%!         0.0554    0.7361   -0.0555   -0.3733
%!         0.0929    0.6751   -0.0475   -0.3683
%!         0.1304    0.6184   -0.0965   -0.3933
%!         0.1639    0.5700   -0.2405   -0.2308
%!         0.1967    0.6093   -0.2875   -0.4483
%!         0.2338    0.7093   -0.3125   -0.4933
%!         0.2645    0.8005   -0.3845   -0.3158
%!         0.2869    0.8915   -0.3885   -0.5983
%!         0.3091    0.9385   -0.4365   -0.6158
%!         0.3251    0.9464   -0.5195   -0.4483
%!         0.3311    0.9586   -0.5255   -0.6083
%!         0.3313    0.9269   -0.5245   -0.5558
%!         0.3224    0.8837   -0.5195   -0.4508
%!         0.3000    0.9051   -0.5285   -0.4733
%!         0.2679    0.9557   -0.5235   -0.4883
%!         0.2360    1.0002   -0.4485   -0.6033
%!         0.2076    0.9830   -0.4105   -0.5083
%!         0.1760    0.8692   -0.4285   -0.3633
%!         0.1395    0.7508   -0.4055   -0.3758
%!         0.1052    0.6499   -0.3275   -0.3558
%!         0.0774    0.5699   -0.2655   -0.3033
%!         0.0517    0.6138   -0.2495   -0.3008
%!         0.0213    0.7287   -0.1975   -0.3383
%!        -0.0184    0.7977   -0.1255   -0.3633
%!        -0.0668    0.8589   -0.0445   -0.4283
%!        -0.1192    0.9156    0.0605   -0.5133
%!        -0.1747    0.9026    0.1815   -0.4483
%!        -0.2323    0.8126    0.3325   -0.4383
%!        -0.2839    0.6700    0.4405   -0.4483
%!        -0.3233    0.5204    0.4685   -0.3258
%!        -0.3485    0.4290    0.5085   -0.3033
%!        -0.3550    0.4219    0.5735   -0.3533
%!        -0.3426    0.4701    0.5585   -0.3108
%!        -0.3172    0.5523    0.4785   -0.3108
%!        -0.2791    0.6432    0.4565   -0.4733
%!        -0.2315    0.6805    0.4115   -0.4533
%!        -0.1894    0.6917    0.2995   -0.3708
%!        -0.1591    0.7669    0.2685   -0.5033
%!        -0.1352    0.8755    0.2685   -0.5383
%!        -0.1136    0.9348    0.2695   -0.5183
%!        -0.0901    0.9419    0.2325   -0.6033
%!        -0.0643    0.9225    0.1315   -0.5033
%!        -0.0407    0.8522    0.0945   -0.4833
%!        -0.0225    0.7295    0.0265   -0.4108
%!        -0.0137    0.6081   -0.0405   -0.3008
%!        -0.0151    0.4741   -0.0395   -0.2883
%!        -0.0274    0.3231   -0.0395   -0.1408
%!        -0.0520    0.2134   -0.0215   -0.0958
%!        -0.0864    0.1254   -0.0295   -0.0233
%!        -0.1263    0.0256    0.0055    0.0442
%!        -0.1653   -0.0778    0.0825    0.0617
%!        -0.1988   -0.1712    0.1285    0.1667
%!        -0.2238   -0.2647    0.1945    0.1292
%!        -0.2412   -0.3741    0.1865    0.2617
%!        -0.2549   -0.4006    0.1935    0.3267
%!        -0.2573   -0.3240    0.2765    0.0892
%!        -0.2433   -0.2505    0.2645    0.1417
%!        -0.2231   -0.1534    0.2415    0.1142
%!        -0.2069   -0.0410    0.2085    0.0342
%!        -0.2000    0.0238    0.1645    0.0642
%!        -0.2001    0.0544    0.1985   -0.0808
%!        -0.2037   -0.0099    0.2065    0.0317
%!        -0.2104   -0.1788    0.2635    0.1042
%!        -0.2160   -0.3564    0.2935    0.1217
%!        -0.2186   -0.5024    0.2335    0.2917
%!        -0.2186   -0.5871    0.1745    0.3217
%!        -0.2108   -0.5916    0.1735    0.2567
%!        -0.1901   -0.5845    0.2265    0.1692
%!        -0.1679   -0.6241    0.1165    0.4067
%!        -0.1573   -0.6377    0.0325    0.3817
%!        -0.1530   -0.6189    0.0765    0.1342
%!        -0.1569   -0.6339    0.0265    0.4667
%!        -0.1734   -0.6051    0.0855    0.3067
%!        -0.1951   -0.5319    0.1385    0.1567
%!        -0.2237   -0.4474    0.1275    0.3742
%!        -0.2571   -0.3082    0.2075    0.0317
%!        -0.2879   -0.2211    0.1775    0.0992
%!        -0.3188   -0.1793    0.1985    0.1917
%!        -0.3429   -0.0881    0.3475   -0.1158
%!        -0.3541   -0.0297    0.3825   -0.0333
%!        -0.3618    0.0134    0.3265    0.0117
%!        -0.3716    0.0785    0.2805   -0.0033
%!        -0.3719    0.0910    0.3965   -0.1883
%!        -0.3546    0.0535    0.4395   -0.1708
%!        -0.3345    0.0320    0.3055    0.1492
%!        -0.3160    0.0821    0.3255   -0.1533
%!        -0.2892    0.1773    0.3235   -0.2208
%!        -0.2587    0.2922    0.2255   -0.0133
%!        -0.2214    0.4446    0.2095   -0.3608
%!        -0.1671    0.5549    0.2005   -0.4208
%!        -0.1050    0.5845    0.1275   -0.2683
%!        -0.0486    0.5966   -0.0525   -0.3083
%!         0.0032    0.5948   -0.1795   -0.4133
%!         0.0516    0.5806   -0.2155   -0.3683
%!         0.0863    0.5808   -0.2465   -0.2633
%!         0.1019    0.5570   -0.2995   -0.3258
%!         0.1007    0.4895   -0.3555   -0.2833
%!         0.0823    0.4475   -0.3445   -0.2083
%!         0.0486    0.4517   -0.3085   -0.1983
%!         0.0106    0.4461   -0.2495   -0.2433
%!        -0.0204    0.3968   -0.1525   -0.2433
%!        -0.0427    0.3374   -0.1145   -0.1308
%!        -0.0570    0.3425   -0.1035   -0.2283
%!        -0.0662    0.4094   -0.1785   -0.1583
%!        -0.0749    0.4980   -0.1755   -0.1458
%!        -0.0767    0.5981   -0.0495   -0.3908
%!        -0.0689    0.6888   -0.0465   -0.2683
%!        -0.0577    0.7881   -0.0605   -0.3833
%!        -0.0469    0.8619   -0.1055   -0.4733
%!        -0.0422    0.8321   -0.1485   -0.3083
%!        -0.0387    0.7466   -0.0695   -0.5408
%!        -0.0309    0.6634   -0.0885   -0.3933
%!        -0.0268    0.5794   -0.0795   -0.2908
%!        -0.0307    0.5002   -0.0445   -0.4458
%!        -0.0508    0.4760   -0.1505   -0.1608
%!        -0.0894    0.5679   -0.0875   -0.3133
%!        -0.1283    0.7078    0.1175   -0.6508
%!        -0.1663    0.7919    0.1845   -0.3833
%!        -0.2133    0.8265    0.2345   -0.4483
%!        -0.2599    0.8374    0.3125   -0.6758
%!        -0.2969    0.8162    0.3435   -0.4808
%!        -0.3210    0.7420    0.4165   -0.5058
%!        -0.3215    0.5730    0.4795   -0.5808
%!        -0.3013    0.2979    0.4085   -0.1883
%!        -0.2680    0.0143    0.3635   -0.1408
%!        -0.2172   -0.1931    0.2695   -0.1933
%!        -0.1568   -0.3482    0.0795    0.1542
%!        -0.0990   -0.5055    0.0565    0.0917
%!        -0.0475   -0.7178   -0.0035    0.2642
%!        -0.0025   -0.9341   -0.0295    0.3692
%!         0.0428   -1.0834   -0.0325    0.2067
%!         0.0814   -1.2097   -0.2115    0.6567
%!         0.1090   -1.3175   -0.1885    0.4517
%!         0.1336   -1.4127   -0.1795    0.3767
%!         0.1445   -1.5065   -0.3165    0.8592
%!         0.1424   -1.5894   -0.2155    0.4442
%!         0.1379   -1.7094   -0.2005    0.5842
%!         0.1194   -1.7628   -0.2475    0.9192
%!         0.0871   -1.6474   -0.1865    0.5092
%!         0.0541   -1.4894   -0.1495    0.4967
%!         0.0205   -1.3807   -0.0495    0.4817
%!        -0.0189   -1.3325   -0.0365    0.4492
%!        -0.0630   -1.3452   -0.0175    0.4417
%!        -0.1055   -1.4067    0.0835    0.4317
%!        -0.1473   -1.4711    0.1405    0.6042
%!        -0.1886   -1.4663    0.2055    0.4892
%!        -0.2252   -1.4405    0.2435    0.5292
%!        -0.2545   -1.4530    0.3345    0.5367
%!        -0.2726   -1.4790    0.3915    0.4667
%!        -0.2804   -1.5291    0.3805    0.6292
%!        -0.2800   -1.6064    0.4145    0.5967
%!        -0.2711   -1.6577    0.4125    0.6642
%!        -0.2543   -1.6503    0.4445    0.6142
%!        -0.2332   -1.6413    0.3945    0.6267
%!        -0.2186   -1.6598    0.2995    0.8167
%!        -0.2105   -1.6905    0.3655    0.6167
%!        -0.2016   -1.7773    0.3605    0.6767
%!        -0.1990   -1.8767    0.3035    0.8442
%!        -0.2109   -1.9359    0.2905    0.8542
%!        -0.2309   -1.9978    0.3625    0.8267
%!        -0.2464   -2.0725    0.4795    0.7117
%!        -0.2598   -2.1566    0.3915    1.0417
%!        -0.2729   -2.2060    0.4415    0.9092
%!        -0.2729   -2.2146    0.5805    0.6867
%!        -0.2651   -2.1907    0.4845    1.0592
%!        -0.2563   -2.0544    0.4705    0.7217
%!        -0.2420   -1.8793    0.4675    0.5817
%!        -0.2313   -1.7438    0.4625    0.7617
%!        -0.2284   -1.5795    0.5095    0.4092
%!        -0.2306   -1.4005    0.4165    0.4717
%!        -0.2380   -1.2226    0.4445    0.3817
%!        -0.2426   -1.0615    0.5195    0.1817
%!        -0.2387   -0.9443    0.4795    0.2317
%!        -0.2262   -0.8374    0.4195    0.1167
%!        -0.2090   -0.7585    0.3505    0.1867
%!        -0.1902   -0.6965    0.3525    0.1167
%!        -0.1702   -0.6160    0.2955    0.0917
%!        -0.1530   -0.5535    0.2285    0.1892
%!        -0.1379   -0.5331    0.2845    0.0267
%!        -0.1232   -0.5745    0.2955    0.1692
%!        -0.1141   -0.6693    0.2535    0.3117
%!        -0.1051   -0.7692    0.2395    0.2017
%!        -0.0888   -0.8714    0.2465    0.3167
%!        -0.0706   -0.9784    0.2345    0.4467
%!        -0.0576   -1.0470    0.1525    0.4467
%!        -0.0533   -1.0386    0.0805    0.4942
%!        -0.0564   -0.9563    0.0985    0.4417
%!        -0.0595   -0.8278    0.1525    0.2792
%!        -0.0590   -0.6936    0.1335    0.2642
%!        -0.0571   -0.6112    0.1255    0.2042
%!        -0.0560   -0.6221    0.1465    0.1867
%!        -0.0570   -0.6806    0.1365    0.2192
%!        -0.0627   -0.7056    0.1195    0.1917
%!        -0.0791   -0.6703    0.1115    0.3067
%!        -0.1047   -0.5488    0.2345    0.1217
%!        -0.1276   -0.3868    0.3215   -0.0508
%!        -0.1452   -0.2989    0.2885    0.0592
%!        -0.1579   -0.3142    0.3435   -0.1133
%!        -0.1698   -0.3930    0.3155    0.1142
%!        -0.1872   -0.4033    0.3395    0.1592
%!        -0.1990   -0.3194    0.4595   -0.2333
%!        -0.2078   -0.2826    0.3695    0.1642
%!        -0.2264   -0.2406    0.3875    0.1117
%!        -0.2433   -0.1958    0.4715   -0.2008
%!        -0.2512   -0.2855    0.4245    0.1467
%!        -0.2515   -0.4154    0.4715    0.0942
%!        -0.2396   -0.5006    0.4585    0.1692
%!        -0.2129   -0.5207    0.4565    0.2067
%!        -0.1705   -0.4377    0.3985    0.0617
%!        -0.1208   -0.3532    0.2195    0.2492
%!        -0.0686   -0.3256    0.2095    0.0667
%!        -0.0112   -0.3184    0.1815    0.0317
%!         0.0433   -0.3352    0.0535    0.2317
%!         0.0918   -0.4017   -0.0255    0.0992
%!         0.1426   -0.5515   -0.0725    0.1567
%!         0.1971   -0.7994   -0.1095    0.3167
%!         0.2489   -1.0643   -0.2455    0.4467
%!         0.2920   -1.1858   -0.3815    0.4692
%!         0.3209   -1.1632   -0.4625    0.5017
%!         0.3329   -1.1275   -0.5055    0.5192
%!         0.3346   -1.1285   -0.4975    0.3542
%!         0.3295   -1.1701   -0.5175    0.5017
%!         0.3165   -1.1697   -0.4415    0.4092
%!         0.2959   -1.0758   -0.4215    0.3492
%!         0.2624   -0.9392   -0.4725    0.4917
%!         0.2233   -0.7484   -0.3445    0.0942
%!         0.1902   -0.5707   -0.2465    0.1017
%!         0.1574   -0.4626   -0.2065    0.2217
%!         0.1245   -0.3397   -0.1795   -0.0258
%!         0.0987   -0.1692   -0.2115    0.0592
%!         0.0892    0.0685   -0.1245   -0.2233
%!         0.1028    0.3453   -0.1275   -0.3658
%!         0.1273    0.5826   -0.2405   -0.2433
%!         0.1497    0.7764   -0.2575   -0.4733
%!         0.1646    0.9231   -0.3065   -0.4433
%!         0.1718    1.0326   -0.3265   -0.4683
%!         0.1809    1.1361   -0.2795   -0.6558
%!         0.1953    1.2657   -0.2505   -0.6283
%!         0.2050    1.4464   -0.2775   -0.6108
%!         0.2068    1.6188   -0.3545   -0.7183
%!         0.2126    1.7093   -0.3495   -0.9183
%!         0.2281    1.7044   -0.3355   -0.8058
%!         0.2457    1.6703   -0.3325   -0.7608
%!         0.2611    1.6585   -0.3575   -0.9008
%!         0.2661    1.6143   -0.4705   -0.6558
%!         0.2578    1.5015   -0.4585   -0.7583
%!         0.2451    1.2956   -0.4175   -0.8033
%!         0.2207    1.0253   -0.4485   -0.3858
%!         0.1768    0.8074   -0.3655   -0.4483
%!         0.1220    0.6616   -0.2795   -0.4508
%!         0.0620    0.5891   -0.2285   -0.3108
%!         0.0002    0.5812   -0.1485   -0.4083
%!        -0.0594    0.5578   -0.0655   -0.3083
%!        -0.1114    0.5396    0.0895   -0.3433
%!        -0.1434    0.5741    0.2125   -0.4708
%!        -0.1458    0.6506    0.2315   -0.4908
%!        -0.1214    0.7486    0.1975   -0.5258
%!        -0.0806    0.8194    0.1325   -0.4908
%!        -0.0301    0.8304    0.0945   -0.5433
%!         0.0299    0.7454    0.0665   -0.5858
%!         0.0900    0.5891   -0.0315   -0.3883
%!         0.1351    0.4885   -0.1265   -0.2783
%!         0.1632    0.4505   -0.1695   -0.3683
%!         0.1762    0.3982   -0.2285   -0.2433
%!         0.1710    0.3510   -0.2465   -0.1358
%!         0.1548    0.3268   -0.1755   -0.2458
%!         0.1398    0.2582   -0.1175   -0.2033
%!         0.1272    0.1084   -0.1365   -0.0433
%!         0.1135   -0.0557   -0.1765    0.0192
%!         0.1006   -0.2035   -0.1505    0.0517
%!         0.0938   -0.3889   -0.0855    0.1217
%!         0.0944   -0.6404   -0.0965    0.2717
%!         0.1026   -0.9610   -0.1245    0.3942
%!         0.1201   -1.2709   -0.1525    0.5192
%!         0.1463   -1.4491   -0.2075    0.6042
%!         0.1781   -1.5392   -0.2885    0.5967
%!         0.2058   -1.5973   -0.4185    0.7367
%!         0.2213   -1.5830   -0.4225    0.7092
%!         0.2301   -1.5379   -0.3435    0.5192
%!         0.2335   -1.5207   -0.3875    0.6617
%!         0.2284   -1.5386   -0.3855    0.6292
%!         0.2245   -1.5724   -0.3055    0.4667
%!         0.2271   -1.5665   -0.2665    0.5442
%!         0.2266   -1.5256   -0.3165    0.5842
%!         0.2127   -1.4837   -0.3835    0.5992
%!         0.1882   -1.4411   -0.2855    0.4917
%!         0.1628   -1.3895   -0.1655    0.4392
%!         0.1359   -1.2722   -0.1825    0.5567
%!         0.1069   -1.0629   -0.1565    0.3892
%!         0.0833   -0.7969   -0.0815    0.2242
%!         0.0679   -0.5100   -0.0695    0.1892
%!         0.0615   -0.2389   -0.1075   -0.0033
%!         0.0664    0.0303   -0.1485   -0.0908
%!         0.0822    0.3079   -0.1295   -0.1783
%!         0.1140    0.5658   -0.1465   -0.4283
%!         0.1627    0.7941   -0.3005   -0.5208
%!         0.2134    0.9634   -0.4675   -0.4783
%!         0.2515    1.0687   -0.5295   -0.5583
%!         0.2738    1.1300   -0.5445   -0.6233
%!         0.2803    1.1322   -0.5615   -0.6283
%!         0.2668    1.0979   -0.5675   -0.5358
%!         0.2311    1.0801   -0.4795   -0.5458
%!         0.1799    1.0888   -0.3715   -0.6558
%!         0.1121    1.1266   -0.4035   -0.4358
%!         0.0311    1.2269   -0.3055   -0.5658
%!        -0.0402    1.3304   -0.0415   -0.9458
%!        -0.0970    1.3157    0.0015   -0.6333
%!        -0.1536    1.2545   -0.0375   -0.5383
%!        -0.2054    1.2245    0.0305   -0.7933
%!        -0.2445    1.1415    0.1235   -0.5958
%!        -0.2725    1.0068    0.2105   -0.5658
%!        -0.2878    0.8388    0.2305   -0.5408
%!        -0.2910    0.6395    0.2735   -0.3933
%!        -0.2874    0.5035    0.3115   -0.3133
%!        -0.2779    0.4273    0.2875   -0.2908
%!        -0.2564    0.3242    0.2965   -0.3508
%!        -0.2262    0.1788    0.3125   -0.1683
%!        -0.1995    0.0612    0.3315   -0.0883
%!        -0.1818    0.0445    0.2725   -0.1558
%!        -0.1738    0.0600    0.1725   -0.0008
%!        -0.1659    0.0355    0.2655   -0.2008
%!        -0.1461   -0.0219    0.3315   -0.2508
%!        -0.1285   -0.0978    0.1775    0.1717
%!        -0.1227   -0.0947    0.1175   -0.0583
%!        -0.1180    0.0203    0.1475   -0.2958
%!        -0.1209    0.1808    0.0885   -0.0583
%!        -0.1447    0.3916    0.0655   -0.2633
%!        -0.1867    0.6153    0.1075   -0.3983
%!        -0.2364    0.7858    0.2675   -0.4633
%!        -0.2797    0.8695    0.3975   -0.6508
%!        -0.3113    0.8415    0.3245   -0.4733
%!        -0.3328    0.7698    0.3295   -0.4683
%!        -0.3393    0.6590    0.4225   -0.5358
%!        -0.3336    0.4096    0.3615   -0.2583
%!        -0.3268    0.0623    0.2565   -0.0833
%!        -0.3236   -0.2521    0.2545    0.0217
%!        -0.3274   -0.4415    0.3015    0.1842
%!        -0.3401   -0.4715    0.3165    0.1442
%!        -0.3613   -0.3861    0.2875    0.1192
%!        -0.3928   -0.2126    0.3465    0.1017
%!        -0.4270    0.0263    0.5095   -0.1383
%!        -0.4483    0.2006    0.5895   -0.2933
%!        -0.4524    0.2266    0.5585   -0.1858
%!        -0.4430    0.1870    0.5985   -0.2108
%!        -0.4182    0.1108    0.6425   -0.2158
%!        -0.3771   -0.0376    0.5545   -0.0483
%!        -0.3187   -0.1650    0.4475   -0.0533
%!        -0.2411   -0.2002    0.3935   -0.0583
%!        -0.1509   -0.2036    0.2985    0.0642
%!        -0.0553   -0.2188    0.1135    0.0367
%!         0.0448   -0.2304   -0.0875   -0.0033
%!         0.1459   -0.2073   -0.1945    0.0142
%!         0.2399   -0.1473   -0.2935   -0.0083
%!         0.3167   -0.0702   -0.4665   -0.0133
%!         0.3672   -0.0015   -0.6225   -0.0183
%!         0.3880    0.0378   -0.6905   -0.0008
%!         0.3879    0.0570   -0.6595   -0.1233
%!         0.3785    0.0166   -0.6325   -0.1958
%!         0.3514   -0.0870   -0.7245    0.1167
%!         0.2945   -0.1448   -0.7365    0.1692
%!         0.2217   -0.1525   -0.6035   -0.0558
%!         0.1512   -0.1462   -0.5155    0.0192
%!         0.0848   -0.0799   -0.4575    0.1092
%!         0.0256    0.0141   -0.3195   -0.0783
%!        -0.0216    0.0462   -0.2265   -0.0958
%!        -0.0663    0.0041   -0.2675    0.1267
%!        -0.1129   -0.0417   -0.2395    0.0467
%!        -0.1489   -0.0732   -0.0465   -0.1133
%!        -0.1735   -0.1401    0.0665    0.1492
%!        -0.1959   -0.2071    0.0965    0.1167
%!        -0.2177   -0.2551    0.0575    0.0867
%!        -0.2382   -0.3215    0.0755    0.3017
%!        -0.2444   -0.3974    0.2365    0.0692
%!        -0.2240   -0.5052    0.2665    0.1617
%!        -0.1884   -0.5983    0.1795    0.4567
%!        -0.1441   -0.5736    0.1695    0.2567
%!        -0.0834   -0.5133    0.1105    0.2092
%!        -0.0064   -0.5083   -0.0085    0.2642
%!         0.0780   -0.4662   -0.1285    0.2517
%!         0.1629   -0.3218   -0.1705    0.1792
%!         0.2469   -0.1316   -0.2225   -0.0458
%!         0.3188    0.0698   -0.4895    0.0292
%!         0.3657    0.2711   -0.6915   -0.0558
%!         0.3947    0.3881   -0.6545   -0.3083
%!         0.4166    0.3752   -0.6155   -0.2233
%!         0.4322    0.3157   -0.6895   -0.1933
%!         0.4374    0.2851   -0.8475   -0.1083
%!         0.4346    0.2856   -0.8755   -0.1283
%!         0.4354    0.2982   -0.8335   -0.3133
%!         0.4374    0.3127   -0.9455   -0.0808
%!         0.4289    0.3315   -0.9695   -0.0958
%!         0.4117    0.3484   -0.8695   -0.2483
%!         0.3875    0.3695   -0.8635   -0.1233
%!         0.3534    0.4046   -0.8965   -0.2008
%!         0.3109    0.4103   -0.8595   -0.2033
%!         0.2621    0.3735   -0.6865   -0.1533
%!         0.2091    0.3471   -0.5615   -0.1958
%!         0.1487    0.3500   -0.6085   -0.0883
%!         0.0755    0.4095   -0.5965   -0.0558
%!        -0.0007    0.5566   -0.3735   -0.2433
%!        -0.0624    0.7375   -0.1545   -0.3708
%!        -0.1055    0.9106   -0.1155   -0.3558
%!        -0.1340    1.0728   -0.0775   -0.4608
%!        -0.1476    1.2037   -0.0005   -0.5433
%!        -0.1445    1.3093    0.0225   -0.5708
%!        -0.1205    1.4124    0.0125   -0.6983
%!        -0.0755    1.5512   -0.0005   -0.7383
%!        -0.0197    1.7541   -0.0045   -0.7633
%!         0.0390    1.9343   -0.0825   -0.9433
%!         0.0935    1.9883   -0.2615   -0.9758
%!         0.1276    1.9447   -0.3705   -0.8358
%!         0.1302    1.8773   -0.3225   -0.8033
%!         0.1103    1.7874   -0.2095   -0.9333
%!         0.0776    1.6447   -0.2025   -0.8183
%!         0.0259    1.5027   -0.2675   -0.5433
%!        -0.0390    1.4181   -0.1615   -0.7108
%!        -0.0957    1.2941   -0.0355   -0.7858
%!        -0.1442    1.0659   -0.0665   -0.4158
%!        -0.1955    0.8406   -0.0395   -0.3733
%!        -0.2470    0.6814    0.0665   -0.4008
%!        -0.2951    0.5478    0.1495   -0.2533
%!        -0.3391    0.4030    0.1875   -0.2308
%!        -0.3757    0.2509    0.2445   -0.1508
%!        -0.3997    0.0973    0.4195   -0.1908
%!        -0.4133   -0.1007    0.4435    0.0092
%!        -0.4258   -0.3112    0.3315    0.2217
%!        -0.4290   -0.4816    0.4055    0.0417
%!        -0.4116   -0.6439    0.5125    0.1667
%!        -0.3855   -0.7518    0.4715    0.3892
%!        -0.3658   -0.8248    0.3845    0.3667
%!        -0.3557   -0.9450    0.3525    0.4867
%!        -0.3478   -1.0281    0.4715    0.4367
%!        -0.3283   -1.0926    0.5655    0.3717
%!        -0.2970   -1.2113    0.4865    0.5767
%!        -0.2629   -1.3095    0.4455    0.5692
%!        -0.2277   -1.4162    0.4285    0.5542
%!        -0.1940   -1.5722    0.2955    0.7242
%!        -0.1658   -1.7209    0.1985    0.7442
%!        -0.1448   -1.8574    0.2275    0.7892
%!        -0.1375   -1.9598    0.2405    0.9342
%!        -0.1496   -1.9831    0.1605    0.9617
%!        -0.1754   -1.9571    0.0975    0.8592
%!        -0.2079   -1.9523    0.1425    0.9042
%!        -0.2432   -1.9481    0.2485    0.9167
%!        -0.2732   -1.8977    0.2955    0.8142
%!        -0.2909   -1.8860    0.2895    0.8067
%!        -0.2993   -1.9321    0.2625    0.8917
%!        -0.3030   -1.9254    0.2225    0.9267
%!        -0.2944   -1.8378    0.2495    0.7292
%!        -0.2680   -1.7114    0.2665    0.6842
%!        -0.2381   -1.6035    0.2105    0.8692
%!        -0.2142   -1.5457    0.1815    0.7167
%!        -0.1883   -1.5002    0.1355    0.6142
%!        -0.1554   -1.4121    0.0995    0.6692
%!        -0.1165   -1.2429    0.1295    0.4617
%!        -0.0822   -1.0065    0.0435    0.5217
%!        -0.0685   -0.7615   -0.0505    0.5117
%!        -0.0718   -0.5596   -0.0245    0.1767
%!        -0.0823   -0.4178   -0.0155    0.2067
%!        -0.0983   -0.3206    0.0595    0.2192
%!        -0.1120   -0.2728    0.1795    0.0367
%!        -0.1158   -0.3035    0.2005    0.0992
%!        -0.1137   -0.4179    0.1345    0.2017
%!        -0.1111   -0.5628    0.0385    0.2992
%!        -0.1055   -0.6991    0.0785    0.2917
%!        -0.0926   -0.8462    0.1725    0.3192
%!        -0.0805   -0.9604    0.1215    0.4942
%!        -0.0829   -0.9911    0.0025    0.5617
%!        -0.0989   -0.9748    0.0055    0.4517
%!        -0.1198   -0.9313    0.0895    0.3517
%!        -0.1470   -0.8188    0.0865    0.4717
%!        -0.1825   -0.5767    0.1175    0.3517
%!        -0.2078   -0.2504    0.2775   -0.1133
%!        -0.2118    0.0583    0.2575   -0.1083
%!        -0.2050    0.3599    0.1075   -0.1258
%!        -0.1866    0.6426    0.1025   -0.5283
%!        -0.1520    0.8930    0.1295   -0.5483
%!        -0.1091    1.1596    0.1155   -0.6283
%!        -0.0649    1.3843   -0.0185   -0.8033
%!        -0.0246    1.5324   -0.1305   -0.8133
%!         0.0046    1.6659   -0.1205   -0.8933
%!         0.0178    1.7974   -0.1815   -0.8083
%!         0.0241    1.8634   -0.1665   -0.9808
%!         0.0406    1.7408   -0.1085   -1.0333
%!         0.0601    1.4745   -0.2045   -0.5733
%!         0.0706    1.2181   -0.2795   -0.5683
%!         0.0761    0.9181   -0.3655   -0.4383
%!         0.0813    0.5557   -0.3455   -0.1333
%!         0.0948    0.1924   -0.1705   -0.2608
%!         0.1104   -0.2105   -0.2785    0.2767
%!         0.1106   -0.5572   -0.3925    0.5542
%!         0.1075   -0.7736   -0.2685    0.2067
%!         0.1162   -0.9587   -0.2095    0.4942
%!         0.1310   -1.0968   -0.1835    0.5617
%!         0.1419   -1.1625   -0.2115    0.5117
%!         0.1387   -1.1825   -0.2185    0.6392
%!         0.1236   -1.1781   -0.1115    0.3592
%!         0.1019   -1.2740   -0.1395    0.5267
%!         0.0726   -1.4467   -0.0835    0.6167
%!         0.0408   -1.5448    0.1065    0.4117
%!         0.0016   -1.5764    0.0855    0.7292
%!        -0.0515   -1.5247    0.0165    0.6792
%!        -0.0978   -1.3839    0.1515    0.2642
%!        -0.1198   -1.2895    0.3075    0.3542
%!        -0.1232   -1.2714    0.3285    0.4542
%!        -0.1137   -1.2652    0.2525    0.3392
%!        -0.0937   -1.2585    0.1695    0.4317
%!        -0.0676   -1.2220    0.1545    0.4642
%!        -0.0336   -1.1269    0.1285    0.2617
%!         0.0051   -0.9943    0.0255    0.3567
%!         0.0358   -0.8408   -0.0185    0.3917
%!         0.0583   -0.6968   -0.0065    0.1442
%!         0.0780   -0.6212   -0.1115    0.1767
%!         0.0897   -0.6090   -0.2035    0.2367
%!         0.0864   -0.6100   -0.1845    0.2517
%!         0.0693   -0.5884   -0.0925    0.3142
%!         0.0521   -0.5452    0.0195    0.0817
%!         0.0422   -0.5450   -0.0485    0.2167
%!         0.0304   -0.5974   -0.0775    0.3592
%!         0.0179   -0.6380    0.0365    0.1317
%!         0.0090   -0.6589    0.0385    0.3142
%!         0.0029   -0.6802    0.0675    0.3042
%!         0.0083   -0.7241    0.1465    0.1517
%!         0.0292   -0.8090    0.1455    0.3517
%!         0.0665   -0.9103    0.1315    0.2267
%!         0.1175   -0.9751   -0.0355    0.3342
%!         0.1700   -0.9540   -0.1515    0.4492
%!         0.2239   -0.8772   -0.1285    0.1217
%!         0.2782   -0.7929   -0.2845    0.2592
%!         0.3202   -0.6856   -0.4145    0.2367
%!         0.3485   -0.5863   -0.4115   -0.0058
%!         0.3620   -0.5588   -0.4445    0.1667
%!         0.3588   -0.6270   -0.4405    0.1317
%!         0.3445   -0.7807   -0.4595    0.1717
%!         0.3223   -0.9466   -0.4315    0.3642
%!         0.2962   -1.0383   -0.3285    0.3042
%!         0.2693   -1.0274   -0.3485    0.3917
%!         0.2392   -0.9591   -0.3755    0.3742
%!         0.2038   -0.9085   -0.3265    0.3042
%!         0.1593   -0.8824   -0.2615    0.4042
%!         0.1061   -0.8394   -0.1675    0.3567
%!         0.0545   -0.7459   -0.0505    0.2442
%!         0.0122   -0.6037    0.0655    0.1592
%!        -0.0254   -0.4611    0.0855    0.2192
%!        -0.0626   -0.3247    0.0695    0.1617
%!        -0.0875   -0.1777    0.1505   -0.1158
%!        -0.0905   -0.0481    0.2245   -0.0983
%!        -0.0766    0.0802    0.2345   -0.1633
%!        -0.0549    0.2306    0.1405   -0.2558
%!        -0.0366    0.3284    0.0435   -0.1458
%!        -0.0207    0.3329    0.1075   -0.3483
%!         0.0026    0.3131    0.1475   -0.3983
%!         0.0262    0.3154    0.1045   -0.1858
%!         0.0400    0.3280    0.0795   -0.2658
%!         0.0476    0.3332    0.0685   -0.3658
%!         0.0513    0.3067    0.0155   -0.3033
%!         0.0436    0.2422   -0.0525   -0.2133
%!         0.0191    0.1745   -0.0125   -0.1783
%!        -0.0154    0.1173    0.1125   -0.2333
%!        -0.0520    0.0566    0.1405   -0.1883
%!        -0.0912   -0.0241    0.1125   -0.1083
%!        -0.1337   -0.1957    0.1615   -0.0583
%!        -0.1775   -0.4680    0.2625    0.0967
%!        -0.2182   -0.7258    0.3555    0.1792
%!        -0.2523   -0.9660    0.3795    0.3267
%!        -0.2796   -1.2414    0.4355    0.4692
%!        -0.3002   -1.4781    0.5115    0.5442
%!        -0.3161   -1.6356    0.4915    0.7167
%!        -0.3243   -1.7215    0.5205    0.6517
%!        -0.3173   -1.7280    0.6125    0.5417
%!        -0.3033   -1.6605    0.5655    0.7367
%!        -0.2915   -1.5221    0.5125    0.5692
%!        -0.2740   -1.4133    0.5035    0.3317
%!        -0.2508   -1.3893    0.4615    0.5317
%!        -0.2274   -1.3097    0.5105    0.3317
%!        -0.2072   -1.1267    0.4385    0.2692
%!        -0.2012   -0.8891    0.3175    0.3692
%!        -0.2073   -0.6029    0.3925   -0.0383
%!        -0.2127   -0.3371    0.4275   -0.1183
%!        -0.2136   -0.1458    0.4635   -0.1808
%!        -0.2108   -0.0281    0.4615   -0.3008
%!        -0.2106    0.0331    0.3555   -0.0583
%!        -0.2045    0.0892    0.4145   -0.3483
%!        -0.1740    0.1431    0.4105   -0.4808
%!        -0.1301    0.2082    0.2505   -0.1883
%!        -0.0931    0.3034    0.1765   -0.3183
%!        -0.0690    0.3629    0.0665   -0.2933
%!        -0.0537    0.4034    0.0305   -0.3458
%!        -0.0359    0.4826    0.0655   -0.5758
%!        -0.0191    0.5747    0.0015   -0.3583
%!        -0.0119    0.6836    0.0125   -0.4958
%!        -0.0100    0.8082   -0.0155   -0.6258
%!        -0.0110    0.9267   -0.0725   -0.5783
%!        -0.0131    1.0266   -0.0315   -0.7833
%!        -0.0191    1.0669   -0.0235   -0.6333
%!        -0.0329    1.0852    0.0475   -0.6808
%!        -0.0476    1.1486    0.1005   -0.8408
%!        -0.0613    1.2331    0.0555   -0.6658
%!        -0.0695    1.2863    0.1305   -0.8658
%!        -0.0629    1.2536    0.1935   -0.9183
%!        -0.0502    1.1622    0.1295   -0.6433
%!        -0.0420    1.0762    0.1075   -0.7283
%!        -0.0378    0.9463    0.0865   -0.6458
%!        -0.0383    0.7694    0.0815   -0.4908
%!        -0.0424    0.6196    0.1055   -0.5508
%!        -0.0527    0.4838    0.0825   -0.3458
%!        -0.0750    0.3620    0.1135   -0.2508
%!        -0.1029    0.3257    0.1775   -0.3258
%!        -0.1237    0.3916    0.2245   -0.4083
%!        -0.1362    0.4953    0.2215   -0.4708
%!        -0.1525    0.5921    0.2075   -0.3783
%!        -0.1774    0.6786    0.2845   -0.5608
%!        -0.2054    0.7145    0.3265   -0.5958
%!        -0.2364    0.6962    0.3055   -0.4108
%!        -0.2619    0.6799    0.4275   -0.6658
%!        -0.2722    0.6582    0.4825   -0.6183
%!        -0.2772    0.6341    0.3975   -0.3433
%!        -0.2753    0.6139    0.3925   -0.6433
%!        -0.2569    0.5279    0.3515   -0.5758
%!        -0.2352    0.3880    0.2905   -0.2933
%!        -0.2256    0.2776    0.2425   -0.2983
%!        -0.2253    0.2197    0.2435   -0.2883
%!        -0.2240    0.1661    0.3435   -0.3258
%!        -0.2194    0.0552    0.3135   -0.1083
%!        -0.2107   -0.0586    0.2945   -0.0883
%!        -0.1862   -0.1745    0.3505   -0.1983
%!        -0.1449   -0.3729    0.2845    0.1492
%!        -0.0923   -0.5517    0.2365    0.0692
%!        -0.0331   -0.6555    0.0805    0.1867
%!         0.0188   -0.7057   -0.0615    0.4342
%!         0.0632   -0.6388   -0.0115    0.0117
%!         0.1011   -0.5252   -0.1195    0.1917
%!         0.1224   -0.4038   -0.1795    0.1617
%!         0.1294   -0.2700   -0.1185   -0.2408
%!         0.1151   -0.2191   -0.2065    0.1567
%!         0.0729   -0.1642   -0.1515   -0.0483
%!         0.0192   -0.0712   -0.0105   -0.3533
%!        -0.0439   -0.0512    0.0485    0.0167
%!        -0.1182   -0.0612    0.2055   -0.1708
%!        -0.1887   -0.0999    0.2975   -0.2733
%!        -0.2494   -0.1813    0.3405   -0.0633
%!        -0.2982   -0.2755    0.4775   -0.1683
%!        -0.3334   -0.4156    0.5205   -0.0333
%!        -0.3602   -0.5483    0.5275    0.1142
%!        -0.3767   -0.6765    0.6015    0.0492
%!        -0.3788   -0.8510    0.6165    0.1667
%!        -0.3733   -0.9980    0.5815    0.3117
%!        -0.3666   -1.1136    0.5495    0.3617
%!        -0.3586   -1.2004    0.5285    0.3817
%!        -0.3497   -1.1950    0.5185    0.4042
%!        -0.3411   -1.1416    0.4925    0.3942
%!        -0.3308   -1.1319    0.4865    0.3667
%!        -0.3151   -1.1897    0.5185    0.4042
%!        -0.2909   -1.2479    0.5295    0.4217
%!        -0.2559   -1.2221    0.4895    0.3667
%!        -0.2137   -1.1087    0.4145    0.3742
%!        -0.1715   -0.9637    0.3655    0.3142
%!        -0.1311   -0.8325    0.3315    0.2117
%!        -0.0919   -0.6882    0.2665    0.1967
%!        -0.0507   -0.5098    0.2545    0.0167
%!        -0.0053   -0.3457    0.2025   -0.0408
%!         0.0384   -0.2101    0.0915   -0.0308
%!         0.0784   -0.1111    0.0025   -0.1683
%!         0.1181   -0.0751   -0.0865   -0.1308
%!         0.1605   -0.0730   -0.1175   -0.2033
%!         0.2059   -0.1001   -0.2135   -0.2008
%!         0.2462   -0.2077   -0.3615   -0.0183
%!         0.2772   -0.4270   -0.4225   -0.0558
%!         0.3007   -0.7541   -0.4925    0.1717
%!         0.3146   -1.1303   -0.5105    0.3567
%!         0.3186   -1.4819   -0.5215    0.4617
%!         0.3091   -1.7927   -0.5565    0.7917
%!         0.2890   -2.0763   -0.4785    0.7642
%!         0.2692   -2.3198   -0.4585    0.8617
%!         0.2490   -2.4531   -0.4775    1.1192
%!         0.2292   -2.4601   -0.4055    0.9467
%!         0.2131   -2.4495   -0.3885    0.9992
%!         0.1952   -2.4696   -0.3775    1.0642
%!         0.1710   -2.4586   -0.3765    1.0317
%!         0.1456   -2.4272   -0.2785    0.9392
%!         0.1230   -2.4762   -0.1635    0.9317
%!         0.0921   -2.5929   -0.1965    1.2292
%!         0.0478   -2.6509   -0.1055    1.0467
%!        -0.0025   -2.6497    0.0055    1.0742
%!        -0.0578   -2.6410    0.0905    1.2692
%!        -0.1096   -2.5957    0.2545    0.9617
%!        -0.1513   -2.5087    0.2895    1.1817
%!        -0.1795   -2.3898    0.4275    1.0492
%!        -0.1794   -2.2767    0.5375    0.6942
%!        -0.1570   -2.1483    0.3505    1.0492
%!        -0.1248   -1.8914    0.3095    0.7342
%!        -0.0796   -1.5767    0.2835    0.4517
%!        -0.0317   -1.3301    0.1355    0.6567
%!         0.0092   -1.1046    0.0485    0.3492
%!         0.0457   -0.8055   -0.0545    0.2217
%!         0.0755   -0.4035   -0.0825    0.1117
%!         0.0943    0.0059   -0.1495   -0.0883
%!         0.1011    0.3715   -0.2115   -0.2533
%!         0.0943    0.7130   -0.1945   -0.4308
%!         0.0694    1.0155   -0.1785   -0.3908
%!         0.0356    1.3196   -0.0615   -0.7733
%!         0.0046    1.5734   -0.0275   -0.8458
%!        -0.0314    1.7202   -0.0395   -0.6133
%!        -0.0681    1.8521    0.1035   -1.0408
%!        -0.0942    1.9659    0.1055   -0.9358
%!        -0.1141    2.0471    0.0675   -0.8083
%!        -0.1209    2.0915    0.1535   -1.1308
%!        -0.1063    2.0434    0.1825   -1.0608
%!        -0.0888    1.9430    0.0735   -0.8483
%!        -0.0915    1.8424   -0.1075   -0.6683
%!        -0.1083    1.8053   -0.0255   -0.9108
%!        -0.1202    1.8339    0.1335   -0.9908
%!        -0.1316    1.8332    0.1095   -0.6858
%!        -0.1505    1.8459    0.1255   -0.9133
%!        -0.1761    1.8897    0.1255   -0.8433
%!        -0.2072    1.8891    0.1965   -0.7508
%!        -0.2260    1.8554    0.3535   -1.1083
%!        -0.2255    1.8173    0.3455   -0.8533
%!        -0.2197    1.7742    0.3785   -0.7258
%!        -0.2068    1.6928    0.4225   -0.9683
%!        -0.1823    1.5929    0.2655   -0.7583
%!        -0.1528    1.5525    0.1635   -0.6833
%!        -0.1153    1.5768    0.1865   -0.8233
%!        -0.0639    1.6492    0.1845   -0.8233
%!        -0.0036    1.7389    0.0385   -0.7983
%!         0.0568    1.8252   -0.1795   -0.8408
%!         0.1135    1.9303   -0.3065   -0.9008
%!         0.1663    2.0578   -0.3395   -0.9508
%!         0.2171    2.2149   -0.3865   -1.1108
%!         0.2609    2.3688   -0.5255   -1.0583
%!         0.2903    2.4811   -0.5965   -1.0783
%!         0.3116    2.5456   -0.5985   -1.2933
%!         0.3302    2.5368   -0.6915   -1.2033
%!         0.3386    2.4999   -0.7635   -1.1133
%!         0.3349    2.4705   -0.7265   -1.1508
%!         0.3272    2.3935   -0.6665   -1.1583
%!         0.3200    2.2373   -0.7235   -1.0883
%!         0.3069    2.0341   -0.8575   -0.8508
%!         0.2852    1.8725   -0.7935   -0.9233
%!         0.2581    1.7540   -0.7195   -0.8208
%!         0.2203    1.6180   -0.7705   -0.5283
%!         0.1816    1.4957   -0.6725   -0.8258
%!         0.1549    1.3512   -0.5665   -0.7058
%!         0.1282    1.1388   -0.5585   -0.2658
%!         0.1008    0.9536   -0.4685   -0.5208
%!         0.0874    0.8015   -0.3885   -0.4808
%!         0.0840    0.6058   -0.3185   -0.1583
%!         0.0854    0.4115   -0.2425   -0.2633
%!         0.0948    0.2596   -0.2405   -0.1983
%!         0.1089    0.1118   -0.2375   -0.0158
%!         0.1282   -0.0454   -0.1845   -0.0983
%!         0.1559   -0.2334   -0.2045   -0.0283
%!         0.1806   -0.4327   -0.3215    0.2567
%!         0.1940   -0.5798   -0.3475    0.2092
%!         0.2039   -0.6991   -0.3195    0.1717
%!         0.2138   -0.7922   -0.3635    0.3517
%!         0.2221   -0.8503   -0.4145    0.3267
%!         0.2299   -0.9164   -0.4665    0.3792
%!         0.2400   -0.9682   -0.4465    0.3692
%!         0.2545   -1.0512   -0.4625    0.3767
%!         0.2670   -1.2018   -0.5555    0.6217
%!         0.2737   -1.3174   -0.5085    0.5317
%!         0.2764   -1.4090   -0.4815    0.6092
%!         0.2693   -1.5248   -0.5265    0.8217
%!         0.2531   -1.6471   -0.4865    0.6892
%!         0.2321   -1.8015   -0.4525    0.8867
%!         0.2056   -1.9496   -0.3925    1.0067
%!         0.1804   -2.0230   -0.3695    0.8967
%!         0.1599   -1.9852   -0.4155    1.0367
%!         0.1436   -1.8214   -0.3455    0.8392
%!         0.1329   -1.6423   -0.3275    0.7317
%!         0.1199   -1.5459   -0.3995    0.8667
%!         0.1026   -1.4838   -0.3475    0.6542
%!         0.0885   -1.4278   -0.2525    0.5967
%!         0.0761   -1.3787   -0.2145    0.6842
%!         0.0645   -1.2595   -0.1895    0.5417
%!         0.0588   -1.0485   -0.1635    0.3892
%!         0.0552   -0.8090   -0.1595    0.4317
%!         0.0479   -0.5480   -0.1485    0.2867
%!         0.0420   -0.3196   -0.1045    0.0467
%!         0.0378   -0.2052   -0.0815    0.0967
%!         0.0258   -0.1789   -0.0605    0.0942
%!         0.0006   -0.2476   -0.0715    0.1042
%!        -0.0359   -0.4128   -0.0735    0.2417
%!        -0.0730   -0.6137    0.0265    0.2317
%!        -0.0993   -0.8314    0.0895    0.3617
%!        -0.1153   -0.9895    0.0605    0.5417
%!        -0.1249   -1.0287   -0.0115    0.6092
%!        -0.1250   -1.0177   -0.0275    0.5817
%!        -0.1093   -0.9871    0.0055    0.4892
%!        -0.0832   -0.9130   -0.0565    0.6292
%!        -0.0526   -0.8006   -0.0585    0.5317
%!        -0.0077   -0.7273    0.0065    0.2742
%!         0.0553   -0.7710   -0.0665    0.4267
%!         0.1270   -0.9222   -0.1935    0.4617
%!         0.1985   -1.1668   -0.3345    0.6017
%!         0.2688   -1.4234   -0.3835    0.6767
%!         0.3422   -1.5956   -0.4405    0.5992
%!         0.4068   -1.7452   -0.6875    0.9367
%!         0.4443   -1.9072   -0.8405    0.9867
%!         0.4553   -2.0635   -0.8345    0.9092
%!         0.4476   -2.2464   -0.8135    1.0567
%!         0.4252   -2.4329   -0.7825    1.0967
%!         0.3876   -2.5541   -0.7555    1.2767
%!         0.3383   -2.5291   -0.6055    1.1367
%!         0.2853   -2.3477   -0.5295    1.0567
%!         0.2304   -2.0738   -0.5415    1.0692
%!         0.1821   -1.7182   -0.3605    0.5542
%!         0.1429   -1.3468   -0.2425    0.5842
%!         0.1032   -1.0234   -0.2025    0.4742
%!         0.0679   -0.7153   -0.1825    0.0467
%!         0.0414   -0.4370   -0.1845    0.1467
%!         0.0237   -0.2054   -0.0365   -0.1708
%!         0.0103   -0.0543   -0.0845   -0.1233
%!        -0.0138   -0.0096   -0.1945    0.1042
%!        -0.0442    0.0084   -0.0665   -0.3308
%!        -0.0775    0.0274   -0.0545   -0.0058
%!        -0.1239    0.0529   -0.0365    0.0892
%!        -0.1740    0.1041    0.0725   -0.2208
%!        -0.2155    0.1382    0.1855    0.0092
%!        -0.2412    0.1550    0.3675   -0.2133
%!        -0.2493    0.1734    0.3055   -0.1183
%!        -0.2515    0.2194    0.2315    0.0167
%!        -0.2507    0.3222    0.3455   -0.2983
%!        -0.2498    0.4421    0.3345   -0.1158
%!        -0.2534    0.5567    0.3655   -0.2708
%!        -0.2559    0.6360    0.3995   -0.3958
%!        -0.2582    0.6567    0.4175   -0.2383
%!        -0.2604    0.6446    0.4565   -0.4358
%!        -0.2618    0.5815    0.3395   -0.2633
%!        -0.2665    0.5052    0.3425   -0.2233
%!        -0.2708    0.5100    0.4235   -0.3608
%!        -0.2728    0.6079    0.3875   -0.2883
%!        -0.2742    0.7203    0.3615   -0.4683
%!        -0.2795    0.7793    0.3085   -0.3733
%!        -0.2924    0.8673    0.3805   -0.4158
%!        -0.3020    0.9995    0.5025   -0.6983
%!        -0.3037    1.0591    0.4755   -0.5208
%!        -0.3040    1.0871    0.5245   -0.5933
%!        -0.3049    1.1507    0.5525   -0.6233
%!        -0.3058    1.2644    0.5195   -0.6358
%!        -0.3004    1.4334    0.4955   -0.8358
%!        -0.2875    1.5499    0.4055   -0.6808
%!        -0.2652    1.5844    0.4605   -0.8508
%!        -0.2284    1.5971    0.3825   -0.8783
%!        -0.1892    1.6190    0.1465   -0.5958
%!        -0.1537    1.6781    0.1075   -0.9083
%!        -0.1213    1.6967    0.0525   -0.7883
%!        -0.1047    1.6351    0.0045   -0.5783
%!        -0.1042    1.5375    0.0525   -0.8158
%!        -0.1170    1.3787    0.0375   -0.5208
%!        -0.1449    1.2041    0.1245   -0.4508
%!        -0.1783    1.0765    0.1865   -0.4958
%!        -0.2074    0.9696    0.2285   -0.3758
%!        -0.2269    0.8117    0.3565   -0.4383
%!        -0.2434    0.5263    0.3285   -0.0883
%!        -0.2634    0.1883    0.3275    0.0292
%!        -0.2766   -0.1724    0.4035   -0.0358
%!        -0.2855   -0.6067    0.3515    0.5092
%!        -0.2971   -0.9636    0.3975    0.5592
%!        -0.3002   -1.1742    0.4425    0.4867
%!        -0.2883   -1.3005    0.3975    0.6967
%!        -0.2630   -1.3348    0.3585    0.5892
%!        -0.2244   -1.3593    0.2655    0.6317
%!        -0.1732   -1.4072    0.2425    0.5767
%!        -0.1119   -1.4491    0.2175    0.5367
%!        -0.0519   -1.5377    0.1075    0.7042
%!        -0.0045   -1.6482    0.0515    0.6342
%!         0.0222   -1.7353   -0.0215    0.8067
%!         0.0255   -1.7833   -0.0095    0.8492
%!         0.0200   -1.7957    0.0855    0.6017
%!         0.0138   -1.8427    0.0605    0.8717
%!         0.0079   -1.9256    0.0775    0.8342
%!         0.0100   -2.0111    0.0645    0.7467
%!         0.0160   -2.0729   -0.0805    1.0792
%!         0.0231   -2.0740   -0.0975    0.8967
%!         0.0388   -2.0445   -0.1065    0.8617
%!         0.0597   -1.9906   -0.1435    1.0067
%!         0.0807   -1.9170   -0.1825    0.8017
%!         0.0969   -1.8479   -0.3205    0.9292
%!         0.1029   -1.7495   -0.3275    0.8542
%!         0.1043   -1.6604   -0.2355    0.6392
%!         0.1001   -1.6211   -0.2395    0.8142
%!         0.0841   -1.5806   -0.2095    0.7817
%!         0.0630   -1.5319   -0.1415    0.6667
%!         0.0489   -1.5022   -0.0715    0.6367
%!         0.0469   -1.4700   -0.0835    0.6692
%!         0.0566   -1.3234   -0.1335    0.6092
%!         0.0811   -1.0251   -0.0935    0.2617
%!         0.1115   -0.7103   -0.2095    0.3017
%!         0.1288   -0.4227   -0.3555    0.2692
%!         0.1352   -0.1704   -0.2835   -0.2308
%!         0.1320   -0.0594   -0.2505   -0.0183
%!         0.1065   -0.0067   -0.2275    0.0967
%!         0.0680    0.1685   -0.1365   -0.3083
%!         0.0335    0.4148   -0.0715   -0.3258
%!         0.0049    0.5997   -0.0175   -0.3758
%!        -0.0195    0.7006   -0.0445   -0.4033
%!        -0.0351    0.7637   -0.0225   -0.4233
%!        -0.0299    0.8425    0.0905   -0.6383
%!        -0.0055    0.9276    0.0615   -0.5283
%!         0.0178    1.0462   -0.0975   -0.3783
%!         0.0368    1.2729   -0.1145   -0.6858
%!         0.0674    1.4969   -0.0265   -0.9283
%!         0.1083    1.6066   -0.1025   -0.7658
%!         0.1448    1.7228   -0.2695   -0.7558
%!         0.1778    1.9169   -0.3095   -1.0808
%!         0.2111    2.1262   -0.3605   -1.0633
%!         0.2372    2.3387   -0.4795   -1.0733
%!         0.2550    2.5757   -0.5415   -1.3358
%!         0.2666    2.7617   -0.4935   -1.4108
%!         0.2636    2.7910   -0.4535   -1.3158
%!         0.2380    2.7246   -0.5335   -1.2708
%!         0.1975    2.6043   -0.4965   -1.3383
%!         0.1524    2.3566   -0.3465   -1.1458
%!         0.1037    2.0830   -0.2385   -0.9383
%!         0.0570    1.8917   -0.1875   -1.0133
%!         0.0150    1.7363   -0.2315   -0.7908
%!        -0.0242    1.6378   -0.1455   -0.8158
%!        -0.0589    1.5875   -0.0665   -0.9083
%!        -0.0979    1.5152   -0.1345   -0.6183
%!        -0.1459    1.4203   -0.0325   -0.7233
%!        -0.1901    1.3044    0.1315   -0.7433
%!        -0.2209    1.1849    0.2195   -0.6183
%!        -0.2339    1.0649    0.2275   -0.6983
%!        -0.2355    0.9365    0.1425   -0.4533
%!        -0.2337    0.8784    0.1855   -0.4658
%!        -0.2214    0.8990    0.2485   -0.6558
%!        -0.1964    0.9349    0.1745   -0.5258
%!        -0.1659    0.9811    0.1785   -0.6333
%!        -0.1372    0.9916    0.1725   -0.5783
%!        -0.1158    0.9395    0.1425   -0.5083
%!        -0.0970    0.8657    0.1345   -0.6183
%!        -0.0792    0.8086    0.0965   -0.4358
%!        -0.0657    0.8370    0.1385   -0.4958
%!        -0.0552    0.9112    0.1155   -0.5783
%!        -0.0518    0.9266   -0.0175   -0.4133
%!        -0.0560    0.9192   -0.0425   -0.5208
%!        -0.0578    0.8932    0.0005   -0.5783
%!        -0.0564    0.8007   -0.0235   -0.4083
%!        -0.0591    0.6940   -0.0925   -0.3358
%!        -0.0655    0.6191   -0.1015   -0.3833
%!        -0.0735    0.5441   -0.0995   -0.2783
%!        -0.0841    0.4197   -0.1085   -0.1608
%!        -0.0892    0.2639   -0.0675   -0.2483
%!        -0.0801    0.1240   -0.0085   -0.2133
%!        -0.0661   -0.0027   -0.0185   -0.0408
%!        -0.0647   -0.0929   -0.1255    0.0367
%!        -0.0828   -0.1350   -0.1705    0.0342
%!        -0.1111   -0.1750   -0.0195   -0.0983
%!        -0.1424   -0.2431    0.1145   -0.0458
%!        -0.1823   -0.3343    0.1235    0.1642
%!        -0.2282   -0.4034    0.1845    0.0017
%!        -0.2684   -0.4463    0.2355    0.0017
%!        -0.3042   -0.4858    0.2235    0.1692
%!        -0.3374   -0.5038    0.2585    0.0742
%!        -0.3619   -0.5260    0.3695    0.0692
%!        -0.3751   -0.5692    0.4925    0.0867
%!        -0.3804   -0.6271    0.4655    0.1342
%!        -0.3838   -0.6946    0.3635    0.2492
%!        -0.3849   -0.7012    0.4405    0.1317
%!        -0.3802   -0.6456    0.5245    0.1767
%!        -0.3685   -0.5710    0.5145    0.1717
%!        -0.3432   -0.4960    0.4965   -0.0458
%!        -0.3091   -0.5027    0.3875    0.1642
%!        -0.2816   -0.5650    0.2855    0.2767
%!        -0.2575   -0.6307    0.2745    0.0492
%!        -0.2333   -0.7767    0.2115    0.3092
%!        -0.2177   -0.9320    0.2115    0.4467
%!        -0.2103   -0.9808    0.2295    0.3017
%!        -0.2077   -0.9820    0.1235    0.4517
%!        -0.2091   -0.9672    0.0925    0.4017
%!        -0.2067   -0.9320    0.1505    0.2667
%!        -0.1964   -0.9354    0.1875    0.3117
%!        -0.1862   -1.0067    0.1385    0.4092
%!        -0.1835   -1.1249    0.0845    0.4417
%!        -0.1878   -1.2659    0.1145    0.4717
%!        -0.1971   -1.4192    0.1755    0.5892
%!        -0.2082   -1.5634    0.2345    0.5917
%!        -0.2205   -1.6030    0.2205    0.7092
%!        -0.2334   -1.4712    0.2515    0.5992
%!        -0.2378   -1.2802    0.2915    0.3067
%!        -0.2371   -1.1340    0.1805    0.4742
%!        -0.2418   -1.0239    0.1895    0.3717
%!        -0.2467   -0.9961    0.2855    0.1917
%!        -0.2506   -1.0394    0.2625    0.4892
%!        -0.2481   -1.0708    0.3205    0.2867
%!        -0.2255   -1.1285    0.3115    0.2942
%!        -0.1851   -1.2190    0.2515    0.5392
%!        -0.1284   -1.2780    0.2205    0.3617
%!        -0.0568   -1.3066    0.0485    0.4992
%!         0.0190   -1.3093   -0.0745    0.5842
%!         0.0980   -1.3169   -0.0975    0.3817
%!         0.1790   -1.3629   -0.1905    0.4917
%!         0.2470   -1.3928   -0.3335    0.5942
%!         0.2906   -1.3799   -0.4495    0.5667
%!         0.3119   -1.3872   -0.4345    0.4267
%!         0.3070   -1.4366   -0.4805    0.6267
%!         0.2694   -1.4497   -0.5195    0.7092
%!         0.2157   -1.4180   -0.3445    0.3267
%!         0.1572   -1.4131   -0.2795    0.5917
%!         0.0900   -1.4020   -0.2475    0.6467
%!         0.0252   -1.3726   -0.1455    0.2942
%!        -0.0329   -1.3701   -0.1075    0.5992
%!        -0.0863   -1.3388    0.0435    0.4792
%!        -0.1235   -1.2708    0.1565    0.2142
%!        -0.1478   -1.2320    0.1245    0.4817
%!        -0.1728   -1.2034    0.1555    0.4392
%!        -0.1952   -1.1628    0.2365    0.2667
%!        -0.2103   -1.1065    0.2765    0.3317
%!        -0.2217   -0.9840    0.3265    0.2767
%!        -0.2321   -0.8129    0.3975    0.1317
%!        -0.2489   -0.6673    0.3905    0.1867
%!        -0.2823   -0.5045    0.3435    0.2217
%!        -0.3208   -0.2971    0.4725   -0.1183
%!        -0.3473   -0.1283    0.6145   -0.1808
%!        -0.3655    0.0017    0.6595   -0.0633
%!        -0.3824    0.1148    0.6585   -0.2158
%!        -0.3984    0.1791    0.6025   -0.1708
%!        -0.4094    0.2068    0.6665   -0.2083
%!        -0.4057    0.1834    0.7515   -0.2908
%!        -0.3881    0.1013    0.7285   -0.1283
%!        -0.3652    0.0076    0.6865   -0.0358
%!        -0.3341   -0.0852    0.6455   -0.0683
%!        -0.2841   -0.1695    0.5685   -0.0783
%!        -0.2157   -0.2379    0.4245    0.0317
%!        -0.1411   -0.3204    0.2975    0.1167
%!        -0.0709   -0.4622    0.1965    0.1842
%!        -0.0090   -0.6683    0.0635    0.2967
%!         0.0471   -0.8651   -0.0275    0.2617
%!         0.0923   -0.9752   -0.1435    0.4992
%!         0.1202   -0.9762   -0.1835    0.5592
%!         0.1442   -0.9324   -0.0975    0.1967
%!         0.1675   -0.9128   -0.2015    0.4142
%!         0.1732   -0.8419   -0.3305    0.5317
%!         0.1625   -0.6695   -0.2815    0.1442
%!         0.1441   -0.4938   -0.2375    0.1892 ];
%!
%! DAT = iddata (UY(:, 3:4), UY(:, 1:2));
%!
%! [SYS, X0] = moen4 (DAT, "s", 15, "n", 8, "rcond", 0.0, "tol", -1.0);
%!
%! Ae = [  0.9893   0.0081  -0.0844   0.0299   0.1262   0.0815  -0.0379   0.1779
%!         0.0076   0.9694   0.1352   0.1793   0.0965  -0.1041  -0.0373  -0.0045
%!         0.0018  -0.0115   0.9413  -0.3450   0.1674   0.0302  -0.1098  -0.0087
%!         0.0036  -0.0197   0.1766   0.6550  -0.1178   0.6168   0.0028   0.0085
%!        -0.0010  -0.0032  -0.0507   0.0624   0.9293  -0.0034   0.0229  -0.4511
%!         0.0032  -0.0012  -0.0397   0.1255   0.1578   0.1833   0.7586   0.3125
%!         0.0029  -0.0048   0.0422   0.2343   0.0321  -0.6549  -0.0374   0.2232
%!        -0.0036   0.0028   0.0080  -0.2437   0.1074   0.1599  -0.1012   0.2106 ];
%!
%! Ce = [ -0.2226   0.2072  -0.3129  -0.3937   0.1722   0.3232  -0.2113   0.2928
%!        -0.1680  -0.2205   0.0698   0.4049   0.3210  -0.1143  -0.2451   0.4844 ];
%!
%! Be = [  0.0857   0.0026
%!        -0.6433  -0.0314
%!         0.2445   0.0044
%!         1.8942   0.0488
%!         0.0549   0.0151
%!        -2.2093  -0.0622
%!        -2.5072  -0.0925
%!         0.8189   0.0280 ];
%!
%! De = [ -0.4997   0.0451
%!        -1.0011  -0.5567 ];
%!
%!assert (SYS.A, Ae, 1e-4);
%!assert (SYS.B, Be, 1e-4);
%!assert (SYS.C, Ce, 1e-4);
%!assert (SYS.D, De, 1e-4);
