/* -*-C-*-
********************************************************************************
*
* File:         xtiff.c
* RCS:          $Header: xtiff.c,v 1.4 90/01/27 08:22:35 salem Rel $
* Description:  Display a TIFF image in an X11 window
* Author:       Chip Chapin, HP/CLL, chip@hpcllcc.hp.com
* Created:      Tue Jul  4 18:45:47 1989
* Modified:     Sat Jan  7 08:22:35 1990 (Jim Salem) salem@think.com
* Language:     C
* Package:      TIFF utilities
* Status:       Experimental (Do Not Distribute)
*
********************************************************************************
* $Log:	xtiff.c,v $
 *
 * Revision 1.4  90/01/27  08:22:35 salem (Jim Salem)
 *  Fixed Colormap handling.  It now conforms to the 5.0 spec by using the
 *  16bit color values rather than just the lower 8 bits.  The maximum
 *  srip value can now be 15 due to this change.  The stripping code now
 *  is much slower (this could be fixed . . . in fact, alot could be fixed.)
 *  Also, fixed calls to rgbi so that it is always passed 16 bit values.
 *
 * Revision 1.3  89/09/26  10:51:45  10:51:45  chip (Chip Chapin)
 * Release 1.5 -- Fix silly bug: duplicated declarations from merging files
 * caused 6-plane display failure (I'm surprised it worked at all).
 * 
 * Revision 1.2  89/09/20  14:30:58  14:30:58  chip (Chip Chapin)
 * Release 1.4
 * 
 * Revision 1.1  89/09/20  12:52:26  12:52:26  chip (Chip Chapin)
 * Initial revision
 * 
*
* 890708 cc	Add -p (private colormap) option
* 890708 cc	Initial HP release, version 1.1
* 890710 cc	Print image size & tiff dir with # colors.
* 890718 cc	Use DefaultDepth instead of hard-wired 8-plane.
*		(thanks to Chris Christensen, John Silva, Walter Underwood)
* 890719 cc	Added wunder's color-by-color mod to stripping algorithm.
* 890719 cc	Make MAXCOLORS dynamic, instead of 256.
* 890720 cc	Fix order of TIFFClose/RemapColors.
* 890726 cc	Fix bug with expanded images in non-8-plane environment.
* 890726 cc	Release 1.2
* 890812 cc	Fix bug with BW tif's where width%8 != 0.
*		Also fix bug in photometric range mapping.
* 890812 cc	Release 1.3
* 890814 cc	Make errors in TIFFReadScanLine fatal.  Seems to corrupt heap.
*/

/*
* Derived from sgigt.c, as distributed with tifflib (4/89), by Sam Leffler
*		(sam@ucbvax.berkeley.edu), and
*         from xgif.c by John Bradley (University of Pennsylvania,
*		bradley@cis.upenn.edu), and
*	  from xgifload.c by John Bradley and Patrick Noughton.
*
* The file sgigt.c bore no notice of any kind, but appears to originate with
* Leffler.  Some of the other files in his TIFF library distribution
* bear the following notice:
*
	# TIFF Library Tools
	#
	# Copyright (c) 1988 by Sam Leffler.
	# All rights reserved.
	#
	# This file is provided for unrestricted use provided that this
	# legend is included on all tape media and as a part of the
	# software program in whole or part.  Users may copy, modify or
	# distribute this file at will.
	#
*
* The file xgif.c bore no copyright notice, but acknowledged Bradley as author.
* The file xgifload.c bore the following notice:
*
 * xgifload.c  -  based strongly on...
 *
 * gif2ras.c - Converts from a Compuserve GIF (tm) image to a Sun Raster image.
 *
 * Copyright (c) 1988, 1989 by Patrick J. Naughton
 *
 * Author: Patrick J. Naughton
 * naughton@wind.sun.com
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 * event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
* <end of notice from xgifload.c>
*
* (whew!) This derivative program is subject to the same notices made by the
* authors of its precursors.
*
********************************************************************************
*/

#ifndef lint
static char rcs_id[] = "@(#)$Header: xtiff.c,v 1.3 89/09/26 10:51:45 chip Rel $";
#endif

#include <stdio.h>
/* #include <malloc.h> */
#include <memory.h>
long	strtol();

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "tiffio.h"

#ifdef SYSV
# define rindex strrchr
# define bcopy(a, b, c)  (void) memcpy(b, a, c)
#endif
char *rindex();

typedef unsigned char  byte;
typedef unsigned short RGBvalue;

#define CENTERX(f,x,str) ((x)-XTextWidth(f,str,strlen(str))/2)
#define CENTERY(f,y) ((y)-((f->ascent+f->descent)/2)+f->ascent)
#define MAXEXPAND 16
/* #define MAXCOLORS 256 */

/* X stuff */
Display       *theDisp;
int           theScreen, dispcells;
Colormap      theCmap;
Window        rootW, mainW;
GC            theGC;
unsigned long fcol,bcol;
Font          mfont;
XFontStruct   *mfinfo;
Visual        *theVisual;
XImage        *theImage, *expImage;
int  	      imageDepth;	/* Depth (planes) of image in X window */
int	      imageQuantum;	/* 8, 16, 32.  Used as "bitmap_pad" */
  

int            iWIDE,iHIGH,eWIDE,eHIGH,expand,strip,nostrip;
char          *cmd;		/* Invocation name of prog, e.g. "xtiff" */
byte	      *Image;		/* raster image: assumes max depth 8 */

/* revised colormap/palette stuff */
u_short	*redcmap, *greencmap, *bluecmap; /* colormap for palette images */
u_short *Red, *Green, *Blue;	/* Derived color map.  Type should be same
				 as "redcmap", etc. above. */
u_char	*Used;			/* True if a color has been used (only useful
				   with palette images) */
int	HaveColorMap;		/* True if raw cmaps above are usable */
int	PrivateColormap;	/* True if -p option given */
unsigned int  	numcols;	/* Number of colors used, or in colormap */
unsigned int	MAXCOLORS;	/* # of colors possible on screen */

/* These all come from the original sgigt stuff */
u_short	width, height;			/* image width & height */
u_short	bitspersample;
u_short	samplesperpixel;
u_short	photometric;
u_short	orientation;
RGBvalue **BWmap;


main(argc, argv)
int argc;
char *argv[];
{
    int i;
    char *cp;
    TIFF *tif;
    char *display, *geom, *fname;
    int  tiffdir;
    XEvent event;
    
    display = geom = fname = NULL;
    redcmap = greencmap = bluecmap = NULL;
    Red     = Green     = Blue     = NULL;
    BWmap = NULL;
    cmd = argv[0];
    expImage = NULL;
    expand = 1; strip = 0; nostrip = 0; numcols = 0;
    tiffdir = 0;			/* Default, first image */
    HaveColorMap = 0;			/* Default, no colormap from file */
    PrivateColormap = 0;		/* Default, use standard X map */

    /* Options processing from xgif */
    for (i = 1; i < argc; i++) {
        char *strind;
	
        if (!strncmp(argv[i],"-g",2)) {		/* geometry */
            i++;
            geom = argv[i];
            continue;
	}
	
        if (argv[i][0] == '=') {		/* old-style geometry */
            geom = argv[i];
            continue;
	}
	
        if (!strncmp(argv[i],"-d",2)) {		/* display */
            i++;
            display = argv[i];
            continue;
	}
	
        strind = index(argv[i], ':');		/* old-style display */
        if(strind != NULL) {
            display = argv[i];
            continue;
	}
	
        if (!strcmp(argv[i],"-e")) {		/* expand */
            i++;
            expand=atoi(argv[i]);
            continue;
	}

	if (!strcmp(argv[i],"-p")) {		/* private colormap */
	    PrivateColormap = 1;
	    continue;
	}
	
        if (!strcmp(argv[i],"-s")) {		/* strip */
            i++;
            strip=atoi(argv[i]);
            continue;
	}
	
        if (!strcmp(argv[i],"-ns")) {		/* nostrip */
            nostrip++;
            continue;
	}
	
        if (argv[i][0] != '-') {		/* the file name */
	    if (fname == NULL)
		fname = argv[i];
            continue;
	}

	if (argv[i][0] == '-') {		/* may be TIFF dir */
#ifdef HAVE_STRTOL
	    tiffdir=(int)strtol(argv[i]+1, &cp, 10);
	    if (cp == argv[i]+1) Syntax(cmd);
#else
	    tiffdir = atoi(argv[i+1]);
#endif
	    continue;
	}
	
        Syntax(cmd);
    }
    
    if (fname==NULL) fname="-";
    if (expand<1 || expand>MAXEXPAND) Syntax(cmd);
    if (strip<0 || strip>15) Syntax(cmd);
    /*****************************************************/

    
    tif = TIFFOpen(fname, "r");
    if (tif == NULL)
	exit(-1);
    if (tiffdir && !TIFFSetDirectory(tif, tiffdir)) {
	fprintf(stderr, "%s: Error in %s seeking to directory %d.\n",
		cmd, fname, tiffdir);
	exit(-1);
    }
    if (!TIFFGetField(tif, TIFFTAG_BITSPERSAMPLE, &bitspersample))
	bitspersample = 1;
    switch (bitspersample) {
      case 1: case 2: case 4:
      case 8: case 16:
	break;
      default:
	fprintf(stderr, "%s: Sorry, can't handle %d-bit pictures\n",
		cmd, bitspersample);
	exit(-1);
    }
    if (!TIFFGetField(tif, TIFFTAG_SAMPLESPERPIXEL, &samplesperpixel))
	samplesperpixel = 1;
    switch (samplesperpixel) {
      case 1: case 3: case 4:
	break;
      default:
	fprintf(stderr, "%s: Sorry, can't handle %d-channel images\n",
		cmd, samplesperpixel);
	exit(-1);
    }
    TIFFGetField(tif, TIFFTAG_IMAGEWIDTH, &width);
    TIFFGetField(tif, TIFFTAG_IMAGELENGTH, &height);
    
    cp = rindex(fname, '/');
    if (cp == NULL)
	cp = fname;
    else
	cp++;
    fname=cp;

    /*** All kinds of stuff below is almost directly from xgif */
    /* Open up the display */
    if ( (theDisp=XOpenDisplay(display)) == NULL) {
        fprintf(stderr, "%s: Can't open display\n",argv[0]);
        exit(1);
    }
    
    theScreen = DefaultScreen(theDisp);
    rootW     = RootWindow(theDisp,theScreen);
    theGC     = DefaultGC(theDisp,theScreen);
    fcol      = WhitePixel(theDisp,theScreen);
    bcol      = BlackPixel(theDisp,theScreen);

    /* The whole business of selecting a visual needs to be revamped.
       For example, the default visual may not offer as much depth as the
       hardware is capable of supporting.  We should really use XGetVisualInfo
       and look for the best visual we can use.  Meantime...
     */
    theVisual = DefaultVisual(theDisp,theScreen);
    dispcells = DisplayCells(theDisp, theScreen); /* colormap size */
    MAXCOLORS = dispcells;		/* cheap hack for now */
    imageDepth =DefaultDepth(theDisp,theScreen);  /* Bits per pixel */
    imageQuantum = 8;	/* Can't deal with anything else right now */

    /* Note -- this doesn't really do what we want.  It does *not* guarantee
       a color display with a writable colormap. --cc
     */
    if (dispcells<=2) 
        FatalError("This program requires a color display, pref. 8 bits.");

    /* What a token... This is really bogus */
    if (imageDepth > 8)
	printf("Oh wow, you've got %d planes!  I've never tried that before,\nso I'm probably going to croak...\n",
	       imageDepth);

    /* Get dynamic visual.  Method & code described by Dave Lemke */
    /* We can adapt this to choosing a visual when we get around to it.
       See also Nye vol 1, section 7.7
     */
    /* XVisualInfo *visinfo, *vlist, *v;	/* move decls when done! */
    /* visinfo = (XVisualInfo*) 0;
    /* vlist =XGetVisualInfo(theDisp, VisualNoMask, &vinfo_template, &num_vis);
    /* find the first dynamic visual and break out of the loop */
    /* for (v=vlist; v < vlist + num_vis; v++)
	/* dynamic classed visuals are odd-numbered -- see X.h */
	/* if (v->class & 0x1) {
	    /* visinfo = v;
	    /* break;
	/* }
    /* if (visinfo == (XVisualInfo*)0)
	/* FatalError( "This program requires a dynamic visual" );
    /* theVisual = visinfo->visual;
       */

    /* Make the colormap with the proper visual */
    if (PrivateColormap)
	theCmap   = XCreateColormap(theDisp, rootW, theVisual, AllocAll);
    else
	theCmap   = DefaultColormap(theDisp, theScreen);
    
    /* Allocate the X Image */
    Image = (byte *) malloc(width*height*sizeof(byte));
    if (!Image) FatalError("not enough memory for XImage");
    
    theImage = XCreateImage(theDisp,theVisual,imageDepth,ZPixmap,0,Image,
			    width,height,imageQuantum,0);
    if (!theImage) FatalError("unable to create XImage");
    
    /* Possibly expand the image */
    iWIDE = theImage->width;  iHIGH = theImage->height;
    eWIDE = iWIDE * expand;  eHIGH = iHIGH * expand;
    
    /* Clip the window to fit on the screen */
    if (eWIDE > DisplayWidth(theDisp,theScreen)) 
        eWIDE = DisplayWidth(theDisp,theScreen);
    if (eHIGH > DisplayHeight(theDisp,theScreen)) 
        eHIGH = DisplayHeight(theDisp,theScreen);
        
    /**************** Create/Open X Resources ***************/
    if ((mfinfo = XLoadQueryFont(theDisp,"variable"))==NULL)
	FatalError("couldn't open 'variable' font\n");
    mfont=mfinfo->fid;
    XSetFont(theDisp,theGC,mfont);
    XSetForeground(theDisp,theGC,fcol);
    XSetBackground(theDisp,theGC,bcol);
    
    Used = (u_char*)calloc(MAXCOLORS, sizeof(u_char));
    if (!Used) FatalError("Not enough memory");

    if (gt(tif, width, height, Image)) {
	RemapColors(fname, tiffdir);
	TIFFClose(tif);  /* AFTER RemapColors, please.  Otherwise this can
			    deallocate our palette. */
	
	CreateMainWindow(geom,argc,argv);
	Resize(eWIDE, eHIGH);
	XSelectInput(theDisp, mainW, ExposureMask | KeyPressMask 
		     | StructureNotifyMask);
	XMapWindow(theDisp,mainW);
	
	/**************** Main loop *****************/
	while (1) {
	    XNextEvent(theDisp, &event);
	    HandleEvent(&event);
        }
    }

    if (PrivateColormap) XFreeColormap(theDisp, theCmap);
    return 0;
}


/******* copped from xgif **********/
HandleEvent(event)
XEvent *event;
/****************/
{
    switch (event->type) {
      case Expose: {
	  XExposeEvent *exp_event = (XExposeEvent *) event;
	  
	  if (exp_event->window==mainW) 
	      DrawWindow(exp_event->x,exp_event->y,
			 exp_event->width, exp_event->height);
      }
	break;
	
      case KeyPress: {
	  XKeyEvent *key_event = (XKeyEvent *) event;
	  char buf[128];
	  KeySym ks;
	  XComposeStatus status;
	  
	  XLookupString(key_event,buf,128,&ks,&status);
	  if (buf[0]=='q' || buf[0]=='Q') Quit();
      }
	break;
	
      case ConfigureNotify: {
	  XConfigureEvent *conf_event = (XConfigureEvent *) event;
	  
	  if (conf_event->window == mainW && 
	      (conf_event->width != eWIDE || conf_event->height != eHIGH))
	      Resize(conf_event->width, conf_event->height);
      }
	break;
	
	
      case CirculateNotify:
      case MapNotify:
      case DestroyNotify:
      case GravityNotify:
      case ReparentNotify:
      case UnmapNotify:       break;
	
      default:		/* ignore unexpected events */
	break;
    }  /* end of switch */
}


#define	howmany(x, y)	(((x)+((y)-1))/(y))

/* All this "gt" stuff used to declare "raster" as "u_long *raster" */
static
    gt(tif, w, h, raster)
TIFF *tif;
int w, h;
byte *raster;
{
    u_short minsamplevalue, maxsamplevalue, planarconfig;
    RGBvalue *Map;
    int e;
    
    if (!TIFFGetField(tif, TIFFTAG_PHOTOMETRIC, &photometric)) {
	switch (samplesperpixel) {
	  case 1:
	    photometric = PHOTOMETRIC_MINISBLACK;
	    break;
	  case 3: case 4:
	    photometric = PHOTOMETRIC_RGB;
	    break;
	  default:
	    fprintf(stderr, "Missing needed \"%s\" tag.\n",
		    "PhotometricInterpretation");
	    return (0);
	}
	printf("No \"PhotometricInterpretation\" tag, assuming %s.\n",
	       photometric == PHOTOMETRIC_RGB ? "RGB" : "min-is-black");
    }
    if (!TIFFGetField(tif, TIFFTAG_MINSAMPLEVALUE, &minsamplevalue))
	minsamplevalue = 0;
    if (!TIFFGetField(tif, TIFFTAG_MAXSAMPLEVALUE, &maxsamplevalue))
	maxsamplevalue = (1<<bitspersample)-1;
    Map = NULL;
    switch (photometric) {
      case PHOTOMETRIC_RGB:
	if (minsamplevalue == 0 && maxsamplevalue == 65535)
	    /* Does not require any mapping */
	    break;
	/* fall thru... */
      case PHOTOMETRIC_MINISBLACK:
      case PHOTOMETRIC_MINISWHITE: {
	  register int x, range;

	  /* map each sample value onto 0..65535 */
	  /* 890812 cc -- fix added to get correct values */
	  range = maxsamplevalue - minsamplevalue;
	  Map = (RGBvalue *)malloc((range+1) * sizeof (RGBvalue));
	  if (Map == NULL) {
	      fprintf(stderr, "%s.\n",
		      "No space for photometric conversion table");
	      return (0);
	  }
	  if (photometric == PHOTOMETRIC_MINISWHITE) {
	      for (x = 0; x <= range; x++)
		  Map[x] = ((range - x) * 65535) / range;
	  } else {
	      for (x = 0; x <= range; x++)
		  Map[x] = (x * 65535) / range;
	  }
	  if (bitspersample != 8 && photometric != PHOTOMETRIC_RGB) {
	      if (!makebwmap(Map))
		  return (0);
	      /* no longer need Map, free it */
	      free((char *)Map);
	      Map = NULL;
	  }
	  break;
      }
      case PHOTOMETRIC_PALETTE:
	if (!TIFFGetField(tif, TIFFTAG_COLORMAP,
			  &redcmap, &greencmap, &bluecmap)) {
	    fprintf(stderr, "Missing required \"Colormap\" tag.\n");
	    return (0);
	}
	/* 890706 cc -- We can probably just use their colormap, and
	   save a lot of time. */
	if (bitspersample == 8 && samplesperpixel == 1 && MAXCOLORS >= 256) {
	    /* We're home free.  Somebody's already done all the work */
	    HaveColorMap = 1;
	    numcols = 256;
	}
	break;
    }
    
    if (HaveColorMap == 0) {
	/* Allocate space for derived colormap */
	Red   = (u_short*)malloc(MAXCOLORS*sizeof(u_short));
	Green = (u_short*)malloc(MAXCOLORS*sizeof(u_short));
	Blue  = (u_short*)malloc(MAXCOLORS*sizeof(u_short));
	if (!(Red && Green && Blue))
	    FatalError("Not enough memory");
    }

    TIFFGetField(tif, TIFFTAG_PLANARCONFIG, &planarconfig);
    if (planarconfig == PLANARCONFIG_SEPARATE)
	e = gtseparate(tif, raster, Map, h, w);
    else
	e = gtcontig(tif, raster, Map, h, w);
    if (Map)
	free((char *)Map);
    if (BWmap)
	free((char *)BWmap);
    return (e);
}

setorientation(tif, h)
TIFF *tif;
int h;
{
    int y;
    
    if (!TIFFGetField(tif, TIFFTAG_ORIENTATION, &orientation))
	orientation = ORIENTATION_TOPLEFT;
    switch (orientation) {
      case ORIENTATION_BOTRIGHT:
      case ORIENTATION_RIGHTBOT:	/* XXX */
      case ORIENTATION_LEFTBOT:	/* XXX */
	printf("Warning, using bottom-left orientation.\n");
	orientation = ORIENTATION_BOTLEFT;
	/* fall thru... */
      case ORIENTATION_BOTLEFT:
	/* y = 0; */
	y = h-1;	/* 890706, cc. I reversed this. */
	break;
      case ORIENTATION_TOPRIGHT:
      case ORIENTATION_RIGHTTOP:	/* XXX */
      case ORIENTATION_LEFTTOP:	/* XXX */
	printf("Warning, using top-left orientation.\n");
	orientation = ORIENTATION_TOPLEFT;
	/* fall thru... */
      case ORIENTATION_TOPLEFT:
	/* y = h-1; */
	y = 0;		/* 890706 see comment above */
	break;
    }
    return (y);
}

gtcontig(tif, raster, Map, h, w)
TIFF *tif;
byte *raster;
register RGBvalue *Map;
int h, w;
{
    register u_char *pp;
    register byte *cp;
    register int x;
    int row, y, e;
    u_char *buf;

    buf = (u_char *)malloc((unsigned)TIFFScanlineSize(tif));
    if (buf == 0) {
	FatalError("No space for scanline buffer");
    }
    y = setorientation(tif, h);
    e = 1;		/* default return value == no error */
    for (row = 0; row < h; row++) {
	if (TIFFReadScanline(tif, buf, row, 0) < 0) {
	    e = 0;	       /* return value == error */
	    break;
	}
	pp = buf;
	cp = raster + y*w;
	switch (photometric) {
	  case PHOTOMETRIC_RGB:
	    switch (bitspersample) {
	      case 8:
	      for (x = w; x-- > 0;) {
		*cp++ = rgbi(Map[pp[0]], Map[pp[1]], Map[pp[2]]);
		pp += samplesperpixel;
	      }
	      break;

    	      case 16: {
	        register u_short *wp;
	      
		if (Map) {
		      wp = (u_short *)pp;
		      for (x = w; x-- > 0;) {
			  wp[0] = Map[wp[0]];
			  wp[1] = Map[wp[1]];
			  wp[2] = Map[wp[2]];
			  wp += samplesperpixel;
		      }
		  }
		  wp = (u_short *)pp;
		  for (x = w; x-- > 0;) {
		      *cp++ = rgbi(wp[0], wp[1], wp[2]);
		      wp += samplesperpixel;
		  }
		  break;
	      }
	    }
	    break;
	  case PHOTOMETRIC_PALETTE:
	    if (HaveColorMap) {
		bcopy((char*)pp, (char*)cp, w);	/* copy scanline */
		for (x = w; x-- > 0;) Used[*pp++] = 1;
	    } else {
		/* If palette was bigger than MAXCOLORS, go here */
		for (x = w; x-- > 0;) {
		    RGBvalue c = *pp++;
		    *cp++ = rgbi(redcmap[c],
				 greencmap[c], bluecmap[c]);
		}
	    }
	    break;
	  case PHOTOMETRIC_MINISWHITE:
	  case PHOTOMETRIC_MINISBLACK:
	    if (bitspersample == 8) {
		register RGBvalue c;
		
		for (x = w; x-- > 0;) {
		    c = Map[*pp++];
		    *cp++ = rgbi(c, c, c);
		}
	    } else
		gtbw(bitspersample, w, cp, pp);
	    break;
	}
	/* lrectwrite(0, y, w-1, y+1, raster+y*w); */
	/* y += (orientation == ORIENTATION_TOPLEFT ? -1 : 1); */
	y += (orientation == ORIENTATION_TOPLEFT ? 1 : -1);	/* 890706 cc */
    }
    free((char*)buf);
    return (e);
}

gtseparate(tif, raster, Map, h, w)
TIFF *tif;
byte *raster;
register RGBvalue *Map;
int h, w;
{
    register byte *cp;
    register int x;
    u_char *red;
    int scanline, row, y, e;
    
    scanline = TIFFScanlineSize(tif);
    switch (samplesperpixel) {
      case 1:
	red = (u_char *)malloc((unsigned)scanline);
	if (!red) FatalError("Not enough memory");
	break;
      case 3: case 4:
	red = (u_char *)malloc((unsigned)3*scanline);
	if (!red) FatalError("Not enough memory");
	break;
    }
    y = setorientation(tif, h);
    e = 1;		/* default return value == no error */
    for (row = 0; row < h; row++) {
	cp = raster + y*w;
	if (TIFFReadScanline(tif, red, row, 0) < 0) {
	    e = 0;	/* return error */
  	    break;
  	}
	switch (photometric) {
	  case PHOTOMETRIC_RGB: {
	      register u_char *r, *g, *b;
	      
	      r = red;
	      if (TIFFReadScanline(tif, g = r + scanline, row, 1) < 0) {
		  e = 0;	/* return error */
  		  break;
  	      }
	      if (TIFFReadScanline(tif, b = g + scanline, row, 2) < 0) {
		  e = 0;	/* return error */
  		  break;
  	      }
	      switch (bitspersample) {
		case 8:
		  for (x = 0; x < w; x++)
		      *cp++ = rgbi(Map[*r++], Map[*g++], Map[*b++]);
		  break;
		case 16:
		  for (x = 0; x < w; x++)
#define	wp(x)	((u_short *)(x))
		      *cp++ = rgbi(wp(r)[x],wp(g)[x],wp(b)[x]);
		  break;
#undef	wp
	      }
	      break;
	  }
	  case PHOTOMETRIC_PALETTE: {
	      register u_char *pp = red;
	      if (HaveColorMap) {
		  bcopy((char*)pp, (char*)cp, w);	/* copy scanline */
		  for (x = w; x-- > 0;) Used[*pp++] = 1;
	      } else {
		  /* If palette was bigger than MAXCOLORS, go here */
		  for (x = w; x-- > 0;) {
		      RGBvalue c = *pp++;
		      *cp++ = rgbi(redcmap[c],
				   greencmap[c], bluecmap[c]);
		  }
	      }
	      break;
	  }
	  case PHOTOMETRIC_MINISWHITE:
	  case PHOTOMETRIC_MINISBLACK:
	    if (bitspersample == 8) {
		register u_short *pp = (u_short *)red;
		register RGBvalue c;
		
		for (x = w; x-- > 0;) {
		    c = Map[*pp++];
		    *cp++ = rgbi(c, c, c);
		}
	    } else
		gtbw(bitspersample, w, cp, red);
	    break;
	}
	/* lrectwrite(0, y, w-1, y+1, raster+y*w); */
	/* XPutImage(theDisp,mainW,theGC,expImage,
		  0, y, 0, y, w, 1); */
	/* y += (orientation == ORIENTATION_TOPLEFT ? -1 : 1); */
	y += (orientation == ORIENTATION_TOPLEFT ? 1 : -1);
    }
    if (red)
	free((char*)red);
    return (e);
}

/*
 * Greyscale images with less than 8 bits/sample are handled
 * with a table to avoid lots of shits and masks.  The table
 * is setup so that gtbw (below) can retrieve 8/bitspersample
 * pixel values simply by indexing into the table with one
 * number.
 */
makebwmap(Map)
RGBvalue *Map;
{
    register int i;
    int nsamples = 8 / bitspersample;
    register RGBvalue *p;
    
    BWmap = (RGBvalue **)malloc(
				256*sizeof (RGBvalue *)+(256*nsamples*sizeof(RGBvalue)));
    if (BWmap == NULL) {
	fprintf(stderr, "No space for B&W mapping table.\n");
	return (0);
    }
    p = (RGBvalue *)(BWmap + 256);
    for (i = 0; i < 256; i++) {
	BWmap[i] = p;
	switch (bitspersample) {
	  case 1:
	    *p++ = Map[i>>7];
	    *p++ = Map[(i>>6)&1];
	    *p++ = Map[(i>>5)&1];
	    *p++ = Map[(i>>4)&1];
	    *p++ = Map[(i>>3)&1];
	    *p++ = Map[(i>>2)&1];
	    *p++ = Map[(i>>1)&1];
	    *p++ = Map[i&1];
	    break;
	  case 2:
	    *p++ = Map[i>>6];
	    *p++ = Map[(i>>4)&3];
	    *p++ = Map[(i>>2)&3];
	    *p++ = Map[i&3];
	    break;
	  case 4:
	    *p++ = Map[i>>4];
	    *p++ = Map[i&0xf];
	    break;
	}
    }
    return (1);
}

#define	REPEAT8(op)	REPEAT4(op); REPEAT4(op)
#define	REPEAT4(op)	REPEAT2(op); REPEAT2(op)
#define	REPEAT2(op)	op; op
    
    gtbw(bitspersample, w, cp, pp)
int bitspersample, w;
register byte *cp;
register u_char *pp;
{
    register RGBvalue c, *bw;
    register int x;

    /* 890812 cc -- fix to correctly handle w%8 != 0 */
    switch (bitspersample) {
      case 1:
	for (x = w; x >= 8; x -= 8) {
	    bw = BWmap[*pp++];
	    REPEAT8(c = *bw++; *cp++ = rgbi(c, c, c));
	}
	break;
      case 2:
	for (x = w; x >= 4; x -= 4) {
	    bw = BWmap[*pp++];
	    REPEAT4(c = *bw++; *cp++ = rgbi(c, c, c));
	}
	break;
      case 4:
	for (x = w; x >= 2; x -= 2) {
	    bw = BWmap[*pp++];
	    REPEAT2(c = *bw++; *cp++ = rgbi(c, c, c));
	}
	break;
    }
    /* do remaining oddball pixels (incomplete byte) */
    if (x > 0) {	
	bw = BWmap[*pp];
	for (; x; x--) {
	    c = *bw++;
	    *cp++ = rgbi(c, c, c);
	}
    }
}

/************ Copped from xgif ***********************/

/***********************************/
Syntax(cmd)
char *cmd;
{
    printf("Usage: %s [[-geometry] geom] [[-display] display]\n",cmd);
    printf("       [-p] [-e 1..%d] [-s 0-7] [-ns] filename [-<dirnum>]\n",
	   MAXEXPAND);
    exit(1);
}


/***********************************/
FatalError (identifier)
char *identifier;
{
    fprintf(stderr, "%s: %s\n",cmd, identifier);
    exit(-1);
}


/***********************************/
Quit()
{
    exit(0);
}


/***********************************/
DrawWindow(x,y,w,h)
{
    XPutImage(theDisp,mainW,theGC,expImage,x,y,x,y,w,h);
}


/***********************************/
Resize(w,h)
int w,h;
{
    int  ix,iy,ex,ey;
    byte *ximag,*ilptr,*ipptr,*elptr,*epptr;
    static char *rstr = "Resizing Image.  Please wait...";
    
    /* warning:  this code'll only run machines where int=32-bits */
    
    if (w==iWIDE && h==iHIGH) {		/* very special case */
        if (expImage != theImage) {
            if (expImage) XDestroyImage(expImage);
            expImage = theImage;
            eWIDE = iWIDE;  eHIGH = iHIGH;
	}
    }
    
    else {				/* have to do some work */
        /* if it's a big image, this'll take a while.  mention it */
        if (w*h>(500*500)) {
            XDrawImageString(theDisp,mainW,theGC,CENTERX(mfinfo,w/2,rstr),
			     CENTERY(mfinfo,h/2),rstr, strlen(rstr));
            XFlush(theDisp);
	}
	
	/* first, kill the old expImage, if one exists */
	if (expImage && expImage != theImage) {
            free((char*)(expImage->data));  expImage->data = NULL;
            XDestroyImage(expImage);
	}
	
        /* create expImage of the appropriate size */
        
        eWIDE = w;  eHIGH = h;
        ximag = (byte *) malloc(w*h*sizeof(byte));
        expImage = XCreateImage(theDisp,theVisual,imageDepth,ZPixmap,0,ximag,
				eWIDE,eHIGH,imageQuantum,eWIDE);
	
        if (!ximag || !expImage) {
            fprintf(stderr,"ERROR: unable to create a %dx%d image\n",w,h);
            exit(0);
	}
	
        elptr = epptr = (byte *) expImage->data;
	
        for (ey=0;  ey<eHIGH;  ey++, elptr+=eWIDE) {
            iy = (iHIGH * ey) / eHIGH;
            epptr = elptr;
            ilptr = (byte *) theImage->data + (iy * iWIDE);
            for (ex=0;  ex<eWIDE;  ex++,epptr++) {
                ix = (iWIDE * ex) / eWIDE;
                ipptr = ilptr + ix;
                *epptr = *ipptr;
	    }
	}
    }
}


CreateMainWindow(geom,argc,argv)
char *geom,**argv;
int   argc;
{
    XSetWindowAttributes xswa;
    unsigned int xswamask;
    XSizeHints hints;
    int i,x,y,w,h;
    
    x=y=w=h=1;
    i=XParseGeometry(geom,&x,&y,&w,&h);
    if (i&WidthValue)  eWIDE = w;
    if (i&HeightValue) eHIGH = h;
    
    if (i&XValue || i&YValue) hints.flags = USPosition;  
    else hints.flags = PPosition;
    
    hints.flags |= USSize;
    
    if (i&XValue && i&XNegative) 
        x = XDisplayWidth(theDisp,theScreen)-eWIDE-abs(x);
    if (i&YValue && i&YNegative) 
        y = XDisplayHeight(theDisp,theScreen)-eHIGH-abs(y);
    
    hints.x=x;             hints.y=y;
    hints.width  = eWIDE;  hints.height = eHIGH;
    hints.max_width = DisplayWidth(theDisp,theScreen);
    hints.max_height = DisplayHeight(theDisp,theScreen);
    hints.flags |= PMaxSize;
    
    xswa.background_pixel = bcol;
    xswa.border_pixel     = fcol;
    xswamask = CWBackPixel | CWBorderPixel;
    
    mainW = XCreateWindow(theDisp,rootW,x,y,eWIDE,eHIGH,2,0,CopyFromParent,
                          CopyFromParent, xswamask, &xswa);
    
    XSetStandardProperties(theDisp,mainW,cmd,cmd,None,
			   argv,argc,&hints);

    if (PrivateColormap)
	XSetWindowColormap(theDisp, mainW, theCmap);
    
    if (!mainW) FatalError("Can't open main window");
    
}


/*************************/
RemapColors(fname, tiffdir) /* ok, so I didn't like the old name.  Nyah. */
char *fname;
int tiffdir;
{
    /* we've got the picture loaded, we know what colors are needed. get 'em */
    
    register int   i,j;
    int		   numused;
    static short   lmasks[16]    = {0xffff, 0xfffe, 0xfffc, 0xfff8,
				    0xfff0, 0xffe0, 0xffc0, 0xff80,
				    0xff00, 0xfe00, 0xfc00, 0xf800,
				    0xf000, 0xe000, 0xc000, 0x8000};
    static short   roundvals[16] = {0x0000, 0x0000, 0x0001, 0x0002,
				    0x0004, 0x0008, 0x0010, 0x0020,
				    0x0040, 0x0080, 0x0100, 0x0200,
				    0x0400, 0x0800, 0x1000, 0x2000};
    short          rlmask, glmask, blmask;
    short          rroundval,groundval,broundval;
    byte           *ptr;
    u_char	   *Uptr;	/* Traipse through Used array */
    u_long	   *cols;	/* Array of pixel values.  Must be u_long */
    XColor  	   *defs;	/* Array of color definitions */

    cols = (u_long*)calloc(MAXCOLORS, sizeof(u_long));
    defs = (XColor*)calloc(MAXCOLORS, sizeof(XColor));
    if (!(cols && defs)) FatalError("Not enough memory");

    if (HaveColorMap) {
	/* Use the colormap provided with the image. */
	Red = redcmap; Green = greencmap; Blue = bluecmap;
	/* It may have unused entries, however.  I wonder how many? */
	for (numused=0,Uptr=Used+MAXCOLORS; Uptr != Used;)
	    if (*--Uptr) numused++;
    } else {
	/* All colors are used */
	memset((char*)Used, 1, MAXCOLORS*sizeof(u_char));
	numused = numcols;
    }

    printf( "  There are %d colors (%d used) in %dx%d image %s/%d.\n",
	   numcols, numused, width, height, fname, tiffdir);

    /* Allocate the X colors for this picture */
    if (PrivateColormap) {
	for (i=0; i<numcols; i++) {
	    defs[i].red   = Red[i];
	    defs[i].green = Green[i];
	    defs[i].blue  = Blue[i];
	    defs[i].flags = DoRed | DoGreen | DoBlue;
	    defs[i].pixel = i;
	}
	XStoreColors(theDisp, theCmap, defs, numcols);
    } else {
	
	if (nostrip)  {   /* nostrip was set.  try REAL hard to do it */
	    for (i=j=0; i<numcols; i++) {
		if (Used[i]) {
		    defs[i].red   = Red[i];
		    defs[i].green = Green[i];
		    defs[i].blue  = Blue[i];
		    defs[i].flags = DoRed | DoGreen | DoBlue;
		    if (!XAllocColor(theDisp,theCmap,&defs[i])) { 
			j++;  defs[i].pixel = 0xffff;
		    }
		    cols[i] = defs[i].pixel;
		}
	    }
	    
	    if (j) {		/* failed to pull it off */
		XColor *ctab;
		int    dc;

		ctab = (XColor*)calloc(MAXCOLORS,sizeof(XColor));
		if (!ctab) FatalError("Not enough memory");
		dc = (dispcells<MAXCOLORS) ? dispcells : MAXCOLORS;
		
		fprintf(stderr,"failed to allocate %d out of %d colors.  Trying extra hard.\n",j,numcols);
		
		/* read in the color table */
		for (i=0; i<dc; i++) ctab[i].pixel = i;
		XQueryColors(theDisp, theCmap, ctab, dc);
		
		/* run through the used colors.  Any used color that has a pixel
		   value of 0xffff wasn't allocated.  for such colors, run through
		   the entire X colormap and pick the closest color */
		
		for (i=0; i<numcols; i++)
		    if (Used[i] && cols[i]==0xffff) {
			/* an unallocated pixel */
			int d, mdist, close;
			unsigned long r,g,b;
			
			mdist = 100000;   close = -1;
			r =  Red[i];
			g =  Green[i];
			b =  Blue[i];
			for (j=0; j<dc; j++) {
			    d = abs(r - (ctab[j].red)) +
				abs(g - (ctab[j].green)) +
			        abs(b - (ctab[j].blue));
			    if (d<mdist) { mdist=d; close=j; }
			}
			if (close<0) FatalError("simply can't do it.  Sorry.");
			bcopy(&defs[close],&defs[i],sizeof(XColor));
			cols[i] = ctab[close].pixel;
		    }
	    }	/* end 'failed to pull it off' */
	}
	
	else {          /* strip wasn't set, do the best auto-strip */
	    j = 0;
	    while (strip<16) {
		int color;		/* 0:Blue, 1:Green, 2:Red */
		int laststrip;

		/* Walter Underwood's revised color stripping scheme ... */
		/* We strip one color at a time, in an attempt to do the least
		   amount of damage.  It is not a great color quantization
		   algorithm, but it gives decent results on a 6-plane display.
		 */
		for (color=0 ; color<=2 ; color++ ) {
		    laststrip = (strip==0) ? strip : strip-1;
		    switch(color) {
		      case 0:		/* strip Blue */
			rlmask = lmasks[laststrip];
			rroundval = roundvals[laststrip];
			glmask = lmasks[laststrip];
			groundval = roundvals[laststrip];
			blmask = lmasks[strip];
			broundval = roundvals[strip];
			break;
		      case 1:		/* strip Green and Blue */
			rlmask = lmasks[laststrip];
			rroundval = roundvals[laststrip];
			glmask = lmasks[strip];
			groundval = roundvals[strip];
			blmask = lmasks[strip];
			broundval = roundvals[strip];
			break;
		      case 2:		/* strip Red, Green, and Blue */
			rlmask = lmasks[strip];
			rroundval = roundvals[strip];
			glmask = lmasks[strip];
			groundval = roundvals[strip];
			blmask = lmasks[strip];
			broundval = roundvals[strip];
			break;
		    }
		    for (i=0; i<numcols; i++)
			if (Used[i]) {
			    defs[i].red   = ((Red[i]  &rlmask)+rroundval);
			    defs[i].green = ((Green[i]&glmask)+groundval);
			    defs[i].blue  = ((Blue[i] &blmask)+broundval);
			    defs[i].flags = DoRed | DoGreen | DoBlue;
			    if (!XAllocColor(theDisp,theCmap,&defs[i])) break;
			    cols[i] = defs[i].pixel;
			}
		    
		    if (i<numcols) {		/* failed */
			if (color==2) {
			    /* tried all colors, strip another bit */
			    strip++;
			    j++;
			}
			for (i--; i>=0; i--)
			    if (Used[i])
				XFreeColors(theDisp,theCmap,cols+i,1,0L);
		    }
		    else goto done;

		} /* for */
	    } /* while */
	    
	  done:
	    if (j && strip<16)
		fprintf(stderr,"%s:  %s stripped %d bits\n",cmd,fname,strip);
	    
	    if (strip==16) {
		fprintf(stderr,"UTTERLY failed to allocate the desired colors.\n");
		for (i=0; i<numcols; i++) cols[i]=i;
	    }
	} /* not nostrip */
    
	/* Now remap all the colors in Image to the X colormap entries */
	ptr = Image;
	for (i=0; i<height; i++)
	    for (j=0; j<width; j++,ptr++) 
		*ptr = (byte) cols[*ptr];
    } /* not PrivateColormap */

} /* RemapColors */


/* rgbi -- The original "sgigt", from which xtiff is derived,
   calls the rgbi library routine from SGI.  This version works out OK, but
   is not a
   good substitute for doing a real color analysis in cases where there
   are more than MAXCOLORS colors.  -- cc
   
   Takes R/G/B values (0..65535) and finds a matching colormap entry, or
   allocates a new one up to MAXCOLORS.  If MAXCOLORS is exceeded, it
   finds the nearest existing color and uses that.
   
   Note that these will all be remapped to X colormap entries in the end.
 */

int rgbi(r, g, b)
RGBvalue r, g, b;
{
    register int i;
    static int didWarning=0;
    int mdist, closest, d;
    
    for (i=0; i<numcols; i++)
	/* Look for RGB in our colormap */
	if (r == Red[i] && g == Green[i] && b == Blue[i]) return i;
    
    /* Didn't find a match */
    if (i < MAXCOLORS) {
	/* Inserting a new color */
	numcols++;
	Red[i] = r; Green[i] = g; Blue[i] = b;
	return i;
    } /* new color */
    
    /* No more room for new colors.  Find closest match */
    if (!didWarning) {
	printf("  Warning: MAXCOLORS (%d) exceeded.  The colors will probably not look good.\
              \n           Try using tiffmedian.\n",
		MAXCOLORS);
	didWarning=1;
    }
    /* Look for closest match */
    mdist = 1000000; closest = -1;
    for (i=0; i<MAXCOLORS; i++) {
	d = abs(r - Red[i]) +
	    abs(g - Green[i]) +
	    abs(b - Blue[i]);
	if (d<mdist) { mdist=d; closest=i; }
    }
    return closest;

} /* rgbi */

