 /*
  * Khoros: $Id: update.c,v 1.2 1991/07/15 06:02:57 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: update.c,v 1.2 1991/07/15 06:02:57 khoros Exp $";
#endif

 /*
  * $Log: update.c,v $
 * Revision 1.2  1991/07/15  06:02:57  khoros
 * HellPatch1
 *
  */
 
/*
 *----------------------------------------------------------------------
 *
 *            Copyright 1990 University of New Mexico
 *  
 *  Permission to use, copy, modify, distribute, and sell this
 *  software and its documentation for any purpose is hereby
 *  granted without fee, provided that the above copyright
 *  notice appear in all copies and that both that copyright
 *  notice and this permission notice appear in supporting docu-
 *  mentation, and that the name of UNM not be used in advertis-
 *  ing or publicity pertaining to distribution of the software
 *  without specific, written prior permission.  UNM makes no
 *  representations about the suitability of this software for
 *  any purpose.  It is provided "as is" without express or
 *  implied warranty.
 *  
 *  UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 *  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 *  NESS, IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 *  RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 *  ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 *  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 *  OF THIS SOFTWARE.
 *  
 *----------------------------------------------------------------------
 */

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */
#include "xvdisplay.h"

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>            Update Display Utility Routines            <<<<
   >>>>                                                       <<<<
   >>>>		    xvd_update_image()			      <<<<
   >>>>		    xvd_update_shape()			      <<<<
   >>>>		    xvd_update_clip()			      <<<<
   >>>>		    xvd_update_overlay()		      <<<<
   >>>>		    xvd_update_all()			      <<<<
   >>>>		    xvd_update_xcolors()		      <<<<
   >>>>		    xvd_update_region()			      <<<<
   >>>>		    xvd_update_pixel()			      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */



/************************************************************
*
*  MODULE NAME: xvd_update_image
*
*      PURPOSE: Updates the display image associated with the
*		xvdisplay structure.
*
*        INPUT: xvdisplay - the xvdisplay structure to be updated.
*		image     - the image to be updated.  If NULL then
*			    update the current xvdisplay image.
*
*       OUTPUT: returns True if we are successful, otherwise False.
*
*    CALLED BY: the application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


int xvd_update_image(xvdisplay, image)

DisplayStructure *xvdisplay;
struct xvimage   *image;
{
	int	  i;
	Dimension width, height;
	Arg       args[MaxArgs];
	struct    xvimage *temp, *disp_temp;
	unsigned  long background, foreground;


	temp  = xvdisplay->image;
	disp_temp = xvdisplay->disp_image;
	if (image != NULL)
	{
	   if (image->imagedata == NULL)
	   {
	      xvf_error_wait("Error!  No image data found.  The image is \
probably a colormap rather than an image, since no image data was found. \
Please re-run the program with an image that has column and row size of at \
least 1 ('vfileinfo' can be used to obtain information about the image).",
			"xvd_update_image", NULL);
	      xvdisplay->image = temp;
	      xvdisplay->disp_image = disp_temp;
	      return(False);
	   }
	   xvdisplay->image = image;
	}

	/*
	 *  Check the image to make sure that it is still a proper image
	 */
	if (!xvd_check_image(xvdisplay))
	{
	   xvdisplay->image = temp;
	   xvdisplay->disp_image = disp_temp;
	   return(False);
	}

	/*
	 *  Build a histogram so that we know which colors to allocate
 	 */
	if (!xvd_initialize_colormap(xvdisplay))
	{
	   xvdisplay->image = temp;
	   xvdisplay->disp_image = disp_temp;
	   return(False);
	}

	/*
	 *  Load the xcolors according to the updated image
	 */
	if (!xvd_load_xcolors(xvdisplay))
	{
	   xvdisplay->image = temp;
	   xvdisplay->disp_image = disp_temp;
	   return(False);
	}
	xvd_update_colormap(xvdisplay);

	if (!_xvd_update_ximage(xvdisplay, xvdisplay->disp_image,
		&xvdisplay->ximage, &xvdisplay->pixmap))
	{
	   xvdisplay->image = temp;
	   xvdisplay->disp_image = disp_temp;
	   return(False);
	}

	/*
	 *  Update the display image for the raster widget.
	 */
	if (xvdisplay->raster != NULL)
	{
	   i = 0;
	   foreground = xvd_white;
	   background = xvd_black;
	   if (xvdisplay->image != NULL)
	   {
	      if (xvdisplay->image->data_storage_type  == VFF_TYP_BIT)
	      {
	         background = xvdisplay->xcolors[0].pixel;
	         foreground = xvdisplay->xcolors[1].pixel;
	      }

              width = xvdisplay->image->row_size;
              height = xvdisplay->image->col_size;
	      if (width  > 800) width  = 800;
	      if (height > 750) height = 750;

	      XtSetArg(args[i], XtNwidth,  width);		i++;
	      XtSetArg(args[i], XtNheight, height);		i++;
	   }
	   XtSetArg(args[i], XtNimage, xvdisplay->ximage);	i++;
	   XtSetArg(args[i], XtNpixmap, xvdisplay->pixmap);	i++;
	   XtSetArg(args[i], XtNbackground, background);	i++;
	   XtSetArg(args[i], XtNforeground, foreground);	i++;
	   XtSetValues(xvdisplay->raster, args, i);
	}

	if (image != NULL)
	{
	   if (temp != disp_temp)
	      freeimage(disp_temp);

	   freeimage(temp);
	}
	return(True);

}  /* end xvd_update_image */



/************************************************************
*
*  MODULE NAME: xvd_update_shape
*
*      PURPOSE: Updates the shape image associated with the
*		xvdisplay structure.
*
*        INPUT: xvdisplay - the xvdisplay structure to be updated.
*		shape     - the shape image to be updated.  If NULL then
*			    update the current xvdisplay shape.
*
*       OUTPUT: returns True if we are successful, otherwise False.
*
*    CALLED BY: the application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


int xvd_update_shape(xvdisplay, shape)

DisplayStructure *xvdisplay;
struct xvimage   *shape;
{
	int	i;
	Arg     args[MaxArgs];
	struct  xvimage *temp, *shape_image, *xvd_resize_shape();


	temp = xvdisplay->shape;
	if (shape != NULL)
	{
	   if (shape->imagedata == NULL)
	   {
	      xvf_error_wait("Error!  No 'shape' image data found.  The shape \
image is probably a colormap rather than an image, since no image data was \
found.  Please re-run the program with an image that has column and row size \
of at least 1 ('vfileinfo' can be used to obtain information about the image).",
			"xvd_update_shape", NULL);
	      xvdisplay->shape = temp;
	      return(False);
	   }
	   xvdisplay->shape = shape;
	}

	if (xvdisplay->shape != NULL)
	{
	   if (!xvd_check_shape(xvdisplay))
	   {
	      xvdisplay->shape = temp;
	      return(False);
	   }
	}

	/*
	 *  Resize shape if shape_parent is true.  This tells us that
	 *  the shape image should be used to clip out or not clip out
	 *  the parent.  If the ximage widget is placed inside of the
	 *  user interface then we wouldn't want the shaping of the image
	 *  window to make parent windows disappear.
	 */
	if (!xvdisplay->shape_parent && xvdisplay->shape != NULL)
	   shape_image = xvd_resize_shape(xvdisplay, xvdisplay->shape);
	else
	   shape_image = xvdisplay->shape;

	if (!_xvd_update_ximage(xvdisplay, shape_image, &xvdisplay->xshape,
				&xvdisplay->shape_mask))
	{
	   xvdisplay->shape = temp;
	   return(False);
	}

	if (shape_image != xvdisplay->shape)
	   freeimage(shape_image);

	/*
	 *  Update the overlay image for the raster widget.
	 */
	if (xvdisplay->raster != NULL)
	{
	   i = 0;
	   XtSetArg(args[i], XtNshape, xvdisplay->shape_mask);		i++;
	   XtSetArg(args[i], XtNshapeParent, xvdisplay->shape_parent);	i++;
	   XtSetValues(xvdisplay->raster, args, i);
	}

	if (shape != NULL)
	   freeimage(temp);
	return(True);

}  /* end xvd_update_shape */



/************************************************************
*
*  MODULE NAME: xvd_update_clip
*
*      PURPOSE: Updates the clip image associated with the
*		xvdisplay structure.
*
*        INPUT: xvdisplay - the xvdisplay structure to be updated.
*		clip      - the clip image to be updated.  If NULL then
*			    update the current xvdisplay clip image.
*
*    CALLED BY: the application program
*
*    CALLED BY: leditimage()
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


int xvd_update_clip(xvdisplay, clip)

DisplayStructure *xvdisplay;
struct xvimage   *clip;
{
	int	i;
	Arg     args[MaxArgs];
	struct  xvimage *temp;


	temp = xvdisplay->clip;
	if (clip != NULL)
	{
	   if (clip->imagedata == NULL)
	   {
	      xvf_error_wait("Error!  No 'clip' image data found.  The clip \
image is probably a colormap rather than an image, since no image data was \
found.  Please re-run the program with an image that has column and row size \
of at least 1 ('vfileinfo' can be used to obtain information about the image).",
			"xvd_update_clip", NULL);
	      xvdisplay->clip = temp;
	      return(False);
	   }
	   xvdisplay->clip = clip;
	}

	if (xvdisplay->clip != NULL)
	{
	   if (!xvd_check_clip(xvdisplay))
	   {
	      xvdisplay->clip = temp;
	      return(False);
	   }
	}

	if (!_xvd_update_ximage(xvdisplay, xvdisplay->clip, &xvdisplay->xclip,
			  &xvdisplay->clip_mask))
	{
	   xvdisplay->clip = temp;
	   return(False);
	}

	/*
	 *  Update the overlay image for the raster widget.
	 */
	if (xvdisplay->raster != NULL)
	{
	   i = 0;
	   XtSetArg(args[i], XtNclip, xvdisplay->clip_mask);		i++;
	   XtSetValues(xvdisplay->raster, args, i);
	}

	if (clip != NULL)
	   freeimage(temp);
	return(True);

}  /* end xvd_update_clip */



/************************************************************
*
*  MODULE NAME: xvd_update_overlay
*
*      PURPOSE: Updates the overlay image associated with the
*		xvdisplay structure.
*
*        INPUT: xvdisplay - the xvdisplay structure to be updated.
*		overlay   - the overlay image to be updated.  If NULL then
*			    update the current xvdisplay overlay image.
*
*       OUTPUT: returns True if we are successful, otherwise False.
*
*    CALLED BY: the application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


int xvd_update_overlay(xvdisplay, overlay)

DisplayStructure *xvdisplay;
struct xvimage   *overlay;
{
	int	i;
	Arg     args[MaxArgs];
	struct  xvimage *temp, *ovmask;
	DisplayStructure *ovdisplay;


	temp = xvdisplay->overlay;
	ovmask = xvdisplay->ovmask;
	if (overlay != NULL)
	{
	   if (overlay->imagedata == NULL)
	   {
	      xvf_error_wait("Error!  No 'overlay' image data found.  The \
overlay image is probably a colormap rather than an image, since no image data \
was found.  Please re-run the program with an image that has column and row \
size of at least 1 ('vfileinfo' can be used to obtain information about the \
image).", "xvd_update_overlay", NULL);
	      xvdisplay->overlay = temp;
	      xvdisplay->ovmask  = ovmask;
	      return(False);
	   }
	   xvdisplay->overlay = overlay;
	}

	if (xvdisplay->overlay != NULL)
	{
	   if (!xvd_check_overlay(xvdisplay))
	   {
	      xvdisplay->overlay = temp;
	      xvdisplay->ovmask  = ovmask;
	      return(False);
	   }
	   ovdisplay = xvd_init_image(xvdisplay->display, xvdisplay->overlay,
		   NULL, NULL, NULL, False, True, False, xvdisplay->colormap);
	   if (ovdisplay == NULL)
	   {
	      xvdisplay->overlay = temp;
	      xvdisplay->ovmask  = ovmask;
	      return(False);
	   }
	   xvdisplay->xoverlay = ovdisplay->ximage;
	   xvdisplay->overlay_pixmap = ovdisplay->pixmap;
	   if (!_xvd_update_ximage(ovdisplay, xvdisplay->ovmask, NULL,
			&xvdisplay->overlay_mask))
	   {
	      xvdisplay->overlay = temp;
	      xvdisplay->ovmask  = ovmask;
	      free(ovdisplay);
	      return(False);
	   }
	   free(ovdisplay);
	}
	else
	{
	   xvdisplay->ovmask = NULL;

	   if (!_xvd_update_ximage(xvdisplay, xvdisplay->overlay,
			&xvdisplay->xoverlay, &xvdisplay->overlay_pixmap))
	   {
	      xvdisplay->overlay = temp;
	      xvdisplay->ovmask  = ovmask;
	      return(False);
	   }

	   if (!_xvd_update_ximage(xvdisplay, xvdisplay->ovmask, NULL,
			&xvdisplay->overlay_mask))
	   {
	      xvdisplay->overlay = temp;
	      xvdisplay->ovmask  = ovmask;
	      return(False);
	   }
	}

	/*
	 *  Update the overlay image for the raster widget.
	 */
	if (xvdisplay->raster != NULL)
	{
	   i = 0;
	   XtSetArg(args[i], XtNoverlay, xvdisplay->overlay_pixmap);	i++;
	   XtSetArg(args[i], XtNoverlayMask, xvdisplay->overlay_mask);	i++;
	   XtSetValues(xvdisplay->raster, args, i);
	}

	if (overlay != NULL)
	{
	   if (temp != ovmask) freeimage(ovmask);
	   freeimage(temp);
	}
	return(True);

}  /* end xvd_update_overlay */



/************************************************************
*
*  MODULE NAME: xvd_update_all
*
*      PURPOSE: Updates the all the associated images with the
*		xvdisplay structure at once.
*
*        INPUT: xvdisplay - the xvdisplay structure to be updated.
*		image     - the display image to be updated.
*		shape     - the shape image to be updated.
*		clip      - the clip image to be updated.
*		overlay   - the overlay image to be updated.
*
*       OUTPUT: returns True if we are successful, otherwise False.
*
*    CALLED BY: the application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


int xvd_update_all(xvdisplay, image, shape, clip, overlay)

DisplayStructure *xvdisplay;
struct xvimage   *image, *shape, *clip, *overlay;
{
	struct  xvimage *tmp_image, *tmp_disp, *tmp_shape, *tmp_clip,
			*tmp_overlay;


	tmp_image   = xvdisplay->image;
	tmp_disp    = xvdisplay->disp_image;
	tmp_shape   = xvdisplay->shape;
	tmp_clip    = xvdisplay->clip;
	tmp_overlay = xvdisplay->overlay;
	if (image   != NULL) xvdisplay->image   = image;
	if (shape   != NULL) xvdisplay->shape   = shape;
	if (clip    != NULL) xvdisplay->clip    = clip;
	if (overlay != NULL) xvdisplay->overlay = overlay;

	if (!xvd_update_image(xvdisplay, NULL))
	{
	   xvdisplay->image = tmp_image;
	   xvdisplay->disp_image = tmp_disp;
	   return(False);
	}
	else if (image != NULL)
	   freeimage(image);

	if (!xvd_update_shape(xvdisplay, NULL))
	{
	   xvdisplay->shape = tmp_shape;
	   return(False);
	}
	else if (shape != NULL)
	   freeimage(shape);

	if (!xvd_update_clip(xvdisplay, NULL))
	{
	   xvdisplay->clip  = tmp_clip;
	   return(False);
	}
	else if (clip != NULL)
	   freeimage(clip);

	if (!xvd_update_overlay(xvdisplay, NULL))
	{
	   xvdisplay->overlay = tmp_overlay;
	   return(False);
	}
	else if (overlay != NULL)
	   freeimage(overlay);

	return(True);

}  /* end xvd_update_all */



/************************************************************
*
*  MODULE NAME: xvd_update_xcolors
*
*      PURPOSE: Updates the xcolors in the xvdisplay structure
*
*        INPUT: xvdisplay - the xvdisplay structure to be updated.
*
*       OUTPUT: returns True if we are successful, otherwise False.
*
*    CALLED BY: the application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


int xvd_update_xcolors(xvdisplay)

DisplayStructure *xvdisplay;
{
	int	  i, j, ncolors;
	XColor	  temp[MAX_PIXELS], *xcolors = xvdisplay->xcolors;

	/*
	 *  Store the changed color cells into the current colormap.
	 */
	for (i = xvdisplay->pixelmin, j = 0; i <= xvdisplay->pixelmax; i++)
	{
	    if (xvdisplay->active[i])
	       temp[j++] = xcolors[i];
	}

	/*
	 *  Store the X colors into the colormap
	 */
	XStoreColors(display, xvdisplay->colormap, temp, j);
}



/************************************************************
*
*  MODULE NAME: xvd_update_region
*
*      PURPOSE: Updates a region of the xvdisplay image, rather
*		than the entire image. 
*
*        INPUT: xvdisplay - the xvdisplay structure to be updated.
*		x & y	  - the x & y position in the image to be
*			    updated.
*		w & h	  - the width & height of that region
*
*       OUTPUT: returns True if we are successful, otherwise False.
*
*    CALLED BY: the application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


int xvd_update_region(xvdisplay, x, y, w, h)

DisplayStructure *xvdisplay;
int		 x, y, w, h;
{
	int	i, j, k, index;
	Display *display = xvdisplay->display;
	struct  xvimage *image = xvdisplay->image;

        Arg	    args[2];
	Dimension   xoffset, yoffset;


	if (image == NULL)
	   return(False);
	else if (x < 0 || x >= image->row_size || y < 0 || y >= image->col_size)
	   return(False);

	for (j = y; j < y+h; j++)
	{
	   k = PIXEL(x, j, image->col_size, image->row_size);
	   for (i = x; i < x+w; i++, k++)
	   {
	      index = (int) xvdisplay->disp_image->imagedata[k];
	      if (xvdisplay->histogram[index] == 0)
	      {
	         xvdisplay->histogram[index]++;
	         if (xvd_allocate_color(xvdisplay, index) == False)
	         {
	            if (!xvd_update_image(xvdisplay, NULL))
		       return(False);
	         }
	      }
	      else
	         xvdisplay->histogram[index]++;

	     if (xvdisplay->ximage != NULL)
	        XPutPixel(xvdisplay->ximage, i, j, (unsigned long) index);
	   }
	}

	if (xvdisplay->pixmap != None && xvdisplay->ximage != NULL)
	{
	   XPutImage(display, xvdisplay->pixmap, xvd_gc, xvdisplay->ximage,
			x, y, x, y, w, h);
	}

	if (xvdisplay->raster != NULL)
	{
	   i = 0;
	   XtSetArg(args[i], XtNxoffset, &xoffset);  i++;
	   XtSetArg(args[i], XtNyoffset, &yoffset);  i++;
	   XtGetValues(xvdisplay->raster, args, i);

	   x -= xoffset;
	   y -= yoffset;
	   XClearArea(display, XtWindow(xvdisplay->raster), x, y, w, h, True);
	}
}



/************************************************************
*
*  MODULE NAME: xvd_update_pixel
*
*      PURPOSE: Updates the all the associated images with the
*		xvdisplay structure at once.
*
*        INPUT: xvdisplay - the xvdisplay structure to be updated.
*		x & y	  - the x & y position in the image to be
*			    updated.
*
*       OUTPUT: returns True if we are successful, otherwise False.
*
*    CALLED BY: the application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


int xvd_update_pixel(xvdisplay, x, y)

DisplayStructure *xvdisplay;
int		 x, y;
{
	int	i, index;
	Display *display = xvdisplay->display;
	struct  xvimage *image = xvdisplay->image;

	Arg	  args[2];
	Dimension xoffset, yoffset;


	if (image == NULL)
	   return(False);
	else if (x < 0 || x >= image->row_size || y < 0 || y >= image->col_size)
	   return(False);

	i = PIXEL(x, y, image->col_size, image->row_size);
	index = (int) xvdisplay->disp_image->imagedata[i];
	if (xvdisplay->histogram[index] == 0)
	{
	   xvdisplay->histogram[index]++;
	   if (xvd_allocate_color(xvdisplay, index) == False)
	   {
	      if (!xvd_update_image(xvdisplay, NULL))
		 return(False);
	   }
	}
	else
	   xvdisplay->histogram[index]++;

	if (xvdisplay->ximage != NULL)
	   XPutPixel(xvdisplay->ximage, x, y, (unsigned long) index);

	XSetForeground(display, xvd_gc, (unsigned long) index);
	if (xvdisplay->pixmap != None)
	   XDrawPoint(display, xvdisplay->pixmap, xvd_gc, x, y);

	if (xvdisplay->raster != NULL)
	{
	   i = 0;
	   XtSetArg(args[i], XtNxoffset, &xoffset);  i++;
	   XtSetArg(args[i], XtNyoffset, &yoffset);  i++;
	   XtGetValues(xvdisplay->raster, args, i);

	   x -= xoffset;
	   y -= yoffset;
	   XDrawPoint(display, XtWindow(xvdisplay->raster), xvd_gc, x, y);
	}
}
