 /*
  * Khoros: $Id: ldshift.c,v 1.2 1992/03/20 23:29:08 dkhoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: ldshift.c,v 1.2 1992/03/20 23:29:08 dkhoros Exp $";
#endif

 /*
  * $Log: ldshift.c,v $
 * Revision 1.2  1992/03/20  23:29:08  dkhoros
 * VirtualPatch5
 *
  */

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: ldshift.c
 >>>>
 >>>>      Program Name: dshift
 >>>>
 >>>> Date Last Updated: Sat Feb 15 16:36:25 1992 
 >>>>
 >>>>          Routines: ldshift - the library call for dshift
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* -library_includes_end */


/****************************************************************
*
* Routine Name: ldshift - library call for dshift
*
* Purpose:
*    
*    Shifts or rotates 1-D signal(s)
*    
*    

* Input:
*    
*    *image         (struct xvimage) -  pointer  to  the  input/output
*                   xvimage structure
*    
*    *s_img         (struct xvimage) -  pointer  to  the  shift  value
*                   input image
*    
*    s_img_flg      (int) - flag set to 1 if  shift  value  image  was
*                   specified.
*    
*    s_const        (int) - constant shift value
*    
*    wrap           (int) - flag set to 1 if wrap (rotate)
*    
*    r_pad          (float) - real padding value if wrap != TRUE
*    
*    i_pad          (float) - imaginary pad value if wrap != TRUE
*    
*    process_dir    (int) - data processing direction.  0 = vector,  1
*                   = band.
*    
*    

* Output:
*    
*    *image         (struct xvimage) -  pointer  to  the  input/output
*                   xvimage structure
*    
*    

*
* Written By: Donna Koechner
*    
*    Thu Jun  7 22:31:11 MDT 1990 Donna  Koechner  -  updated  to
*    ghostwriter
*    Audit Done  05-OCT-90  Donna Koechner
*    
*    

****************************************************************/


/* -library_def */
int
ldshift(image, s_img, s_img_flg, s_const, wrap, r_pad, i_pad, process_dir)

struct xvimage  *image,                 /* input/output image */
                *s_img;                 /* shift value image */
int             s_img_flg,              /* set if shift file specified */
                s_const,                /* constant shift value */
                wrap,                   /* wrap flag (wrap is w=TRUE) */
                process_dir;            /* 0=vect proc, 1= band proc */
float           r_pad,                  /* real constant padding value */
                i_pad;                  /* imaginary constant padding value */

/* -library_def_end */

/* -library_code */
{

  int           arith,                  /* scale factor: real=1, cmplx=2 */
                num_vects,              /* number of vectors */
                dimension,              /* dimension of vectors */
                shift_val,              /* local shifting variable */
                i, j,                   /* loop variables */
                t, m,                   /* varibles for toggle of pad array */
                shift_start,            /* starting shift shift-array index */
                shift_stop,             /* stopping shift shift-array index */
                pad_start,              /* starting pad shift-array index */
                pad_stop,               /* stopping pad shift-array index */
                wrap_index,             /* constant index offset for wrapping */
                *s_array;
                

  float         **data,                 /* input data array */
                **shift,                /* shifted data array */
                pad[2];                 /* pad value array *arith */

  char          **dload_vector();


  if( ! (propertype("ldshift",image,VFF_TYP_FLOAT,FALSE)) && 
      ! (propertype("ldshift",image,VFF_TYP_COMPLEX,FALSE)) ) { 
      fprintf(stderr,"ldshift: data storage type must be float or complex.\n");
      return(0);
  }

 /*
  * Set up pad array and arithmetic type
  */


  if (image->data_storage_type == VFF_TYP_COMPLEX)
  {
    arith = 2;
    pad[0] = r_pad;
    pad[1] = i_pad;
  }
  else
  {
    arith = 1;
    pad[0] = r_pad;
    pad[1] = r_pad;
  }

 /*
  * Call dload_vector for input image
  */
  if ( ( data = (float **)dload_vector(image, &num_vects, &dimension, 
                process_dir)) == NULL)
  {
    (void)fprintf(stderr,"ldshift: dload_vector failed\n");
    return(0);
  }

  if (s_img_flg)
  {
   /*
    * make sure shift value image is proper type.
    */
    if(!(propertype("ldshift", s_img, VFF_TYP_4_BYTE, FALSE)) )
    {
      (void)fprintf(stderr,"ldshift: shift value file must be of type int.\n");
      return(0);
    }

   /*
    * verify that shifting value file has same number of points as there
    * are vectors in the input file
    */
    if(num_vects != (s_img->row_size*s_img->col_size*s_img->num_data_bands) )
    {
      (void)fprintf(stderr,"ldshift: shift value image is wrong size (%d)\n",
                    (s_img->row_size*s_img->col_size*s_img->num_data_bands) );
      (void)fprintf(stderr,"number of vectors in input file is %d\n",num_vects);
      return(0);
    }

   /*
    * Assign the image data to the shift array.
    */
    s_array = (int *)s_img->imagedata;
  }

 /*
  * Allocate array for shifted/rotated data.
  */

  if ( (shift = (float **) (malloc ((unsigned int) num_vects *
        sizeof(float))) ) == NULL)
  {
    (void) fprintf(stderr,"ldshift: Cannot allocate for shifted array.\n");
    return(0);
  }
  for (i=0; i<num_vects; i++)
  {
    if ((shift[i] = (float *) (malloc ((unsigned int) dimension * arith *
         sizeof(float) ))) == NULL)
    {
      (void)fprintf(stderr,"ldshift: Cannot alloc for shifted array.\n");
      return(0);
    }
  }

 /*
  * If shift image flag is not set, assign shift value and check bounds.
  */
  if ( ! s_img_flg )
  {
    shift_val = s_const;
    if( (abs(shift_val)) > dimension)
    {
      (void)fprintf(stderr,"ldshift: shift_val out of range.");
      (void)fprintf(stderr,"  shift_val = %d, dimension = %d.\n",
                    shift_val, dimension);
      return(0);
    }
  }

 
  for ( i=0; i<num_vects; i++)
  {
    if (s_img_flg)
    {
      shift_val = s_array[i]; 
      if( (abs(shift_val)) > dimension)
      {
        (void)fprintf(stderr,"ldshift: shift_val for vector %d out of range.\n",
                      i);
        (void)fprintf(stderr,"shift_val = %d, dimension = %d.\n",
                      shift_val, dimension);
        return(0);
      }
    }

    if (shift_val > 0)
    {
      shift_start = shift_val * arith;
      shift_stop  = dimension * arith;
      pad_start   = 0;
      pad_stop    = shift_val * arith;
      wrap_index  = (shift_val - dimension) * arith;
    }

    else  /* (shift_val < 0) */
    {
      shift_start = 0;
      shift_stop  = (dimension + shift_val) * arith;
      pad_start   = (dimension + shift_val) * arith;
      pad_stop    = dimension * arith;
      wrap_index  = (shift_val + dimension) * arith;
    }

    if ( (shift_val == 0) || ( (abs(shift_val) == dimension) && wrap ) )
    {
      bcopy(data[i], shift[i], dimension*arith*sizeof(float));
    }

    else
    {
      for ( j=shift_start; j<shift_stop; j++)
      {
        shift[i][j] = data[i][j-(shift_val*arith)];
      }

      if ( wrap )
      {
        for ( j=pad_start; j<pad_stop; j++ )
        {
          shift[i][j] = data[i][j - wrap_index];
        }
      }

      else 
      {
        m=0;
        t=1;
        for( j=pad_start; j<pad_stop; j++)
        {
          shift[i][j] = pad[m];
          m += t;
          t *= -1;
        }
      }

    }  /* end of single vector shift */

  }  /* end of loop through vectors */


 /*
  * Unload the shifted vector array back into the image
  */
  if (!(dunload_vector((char **)shift, image, image->data_storage_type, 
                       num_vects, dimension, process_dir)) )
  {
    (void)fprintf(stderr,"ldshift: dunload_vector failed!\n");
    return(0);
  }

  for (i=0; i<num_vects; i++)
  {
    free(data[i]);
    free(shift[i]);
  }
  free(data);
  free(shift);

  return(1);

}  /* end of ldshift.c */
/* -library_code_end */
