/*LINTLIBRARY*/

/*  @(#)gl.c 1.3 92/04/16
 *
 *  Popi device driver for GL 
 *  Written by Glenn Geers, University of Sydney.
 *
 * NOTE: This file was edited with tabstops set to 4 spaces.
 * In vi...  :se ts=4 sw=4
 *
 * GL also supports ibm and epson printers and HP laserjets through the 
 * GLMODE environment variable. GLMODE=256 -> epson/ibm printers.
 *
 * ADVANTAGES OF GL
 *		GL runs under the following operating systems without mods
 * 1. Microport UNIX/286
 * 2. Xenix/286
 * 3. DOS
 *		The following devices are supported:
 * 1. Hercules
 * 2. CGA - low res (320 * 200) - not supported with popi
 * 3. CGA - high res (640 x 200)
 * 4. EGA - 640 x 350 and CGA emulation
 * 5. VGA - emulating EGA or CGA only; fixes will be forthcoming for hi-res.
 * 6. Epson printers
 * 7. HP laserjet
 *
 * BSD(3) library functions are also supported.
 * 
 *
 * HOW TO PORT GL TO 386 XENIX
 * 		Replace g_init.c in the GL suite with the one provided with this 
 * distribution for 386 support of EGA, CGA, Hercules and printers.
 * Define XENIX_386 in config.h and then make.
 * It may also be possible that LJ_PRINTER is not defined in gl.h.
 * It's value may be found in modes.h where it *is* defined.
 *
 *
 * BUGS:
 *		GL does not support high res VGA modes. Hercules graphics is not 
 *  supported on early versions of Xenix without a custom kernel driver.
 *  386 support is not available in the standard GL library softawre.
 *
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

/*  These are the exportable routines used by the popi program.
 *
 *  disp_init(argc, argv)	- called from main at the start.
 *  disp_finish()		- called from main prior to exit.
 *  disp_imgstart(w,h,c,l)	- called prior to drawing an image.
 *  disp_imgend()		- called after drawing an image.
 *  disp_putline(l,y,w,c)	- to draw an image scanline.
 *  disp_getchar()		- to get the next character typed.
 *  disp_prompt()		- display popi prompt and clear input buffer.
 *  disp_error(errtype,pos)	- display error message.
 *  disp_percentdone(n)		- display percentage value of conversion.
 *  disp_resize(w,h)            - resize popi image window (width, height).
 *  disp_colormap(n,r,g,b)      - load new colormap
 */

#include "popi.h"
#include <stdio.h>
#include <gl.h>     /* redefines AND and OR hence the #undefs at the end */

/* Display specifics */
#define HERC_X	720
#define HERC_Y	348

#define EGA_X	640
#define EGA_Y	350

#define CGA_X	640
#define CGA_Y	200

#define EPSON_X 720
#define EPSON_Y 960

#define HPLJ_X 1380
#define HPLJ_Y 1120


/* A couple of useful defines */
#define TRUE 1
#define FALSE 0

/* Array containing threshold values for dithering. */
pixel_t thresh[BITSPERPIXEL][BITSPERPIXEL] =
{
    {	0, 128,	 32, 160,   8, 136,  40, 168, },
    { 192,  64,	224,  96, 200,	72, 232, 104, },
    {  48, 176,	 16, 144,  56, 184,  24, 152, },
    { 240, 112,	208,  80, 248, 120, 216,  88, },
    {  12, 140,	 44, 172,   4, 132,  36, 164, },
    { 204,  76,	236, 108, 196,	68, 228, 100, },
    {  60, 188,	 28, 156,  52, 180,  20, 148, },
    { 252, 124,	220,  92, 244, 116, 212,  84, },
};

static int printer;
static int mode;

/*ARGSUSED*/
void
disp_init(argc,argv)		/* called from main at the start. */
int argc;
char *argv[];
{
	int xsize, ysize;
	char *getenv();
	
	printer = FALSE;

	if ((mode = atoi(getenv("GLMODE"))) == 0) {
		FPRINTF(stderr, "You must set the GLMODE environment variable - ");
		FPRINTF(stderr, "Exiting\n");
		exit(1);
	};

	if (mode > MAXVIDEO)
		printer = TRUE;

	switch (mode) {
		case HERC_P0_MODE:
				xsize = HERC_X;
				ysize = HERC_Y;
				break;
		case EGA_COLOR_MODE:
				xsize = EGA_X;
				ysize = EGA_Y;
				break;
		case CGA_HI_RES_MODE:
				xsize = CGA_X;
				ysize = CGA_Y;
				break;
		case IBM_PRINTER:
				xsize = EPSON_X;
				ysize = EPSON_Y;
				break;
		case LJ_PRINTER:
				xsize = HPLJ_X;
				ysize = HPLJ_Y;
				break;
		default:
				FPRINTF(stderr, "Unknown mode - check GLMODE");
				FPRINTF(stderr, " - Exiting\n");
				exit(1);
	}

	if (Xsize > xsize)
		Xsize = xsize;

	if (Ysize > ysize)
		Ysize = ysize;

	return;
}

void
disp_finish()			/* called from main prior to exit. */
{
	/* Do nothing in this case */

	return;
}


/* Called prior to drawing an image. */

void
disp_imgstart(width, height, ncolors, len)
int width, height, ncolors, len ;
{
	g_init(mode); 	
	g_pix_mode(OR);     
	g_pix_color(WHITE);

	if (!printer)	/* don't print a blank page ! */
		g_clear();

	return;
}

void
disp_imgend()			/* called after drawing an image. */
{
	if (!printer)   /* don't wait if we are going to print */
    	getchar();	/* display the image until a carriage return */

	g_finish(); 

	return;
}

void
disp_putline(line, y, width, ncolors)     /* draw image scanline y. */
pixel_t *line;
int y, width, ncolors;
{
    register int x;

	    for (x = 0; x < width; ++x)
		{
			pixel_t		val;

			val = *line++ < thresh[y % BITSPERPIXEL][x % BITSPERPIXEL] ?
						(pixel_t)0 : (pixel_t)1;

			switch (printer) {
				case FALSE:    /* most common, video display */
						if (val == (pixel_t)1)
							p_wr_pix(x, y);  /* light image -> lit pixel */
						break;
				case TRUE:
						if (val == (pixel_t)0)
							p_wr_pix(x, y);  /* dark image -> lit pixel */
						break;
				default:
						fprintf(stderr, "Fatal display error - Exiting\n");
						exit(1);
			}
		}
		return;
}

int
disp_getchar()			/* get next user typed character. */
{
    return(getchar());
}

int
disp_prompt()			/* display popi prompt. */
{
    char	*prompt = "-> ";

    PRINTF(prompt);
    return sizeof prompt - 1;
}

void
disp_error(errtype, pos)	/* display error message. */
int	errtype;
int pos;
{
    extern int	errno;
    extern char	*sys_errlist[];

    if (errtype	& ERR_PARSE)
    {
		int	i;

		for (i=1; i < pos; ++i)
	    	PUTC('-', stderr);

		PUTC('^', stderr);
		PUTC('\n', stderr);
    }

    FPRINTF(stderr, "%s\n", ErrBuf);
    /* we assume errno hasn't been reset by the	preceding output */
    if (errtype	& ERR_SYS)
		FPRINTF(stderr,	"\t(%s)\n", sys_errlist[errno]);

	return;
}

void
disp_percentdone(percent)
int	percent;
{
    static int	lastpercent = 100;

    if (!Verbose)
		return;

    if (percent	== 100)
    {
		printf("\r    \n");
		return;
    }
    if (percent	!= lastpercent)
    {
		printf("\r%2d%%	", percent);
		fflush(stdout);
		lastpercent = percent;
    }
	return;
}


void
disp_resize(width, height)    /* Resize popi image window. */
int width, height ;
{
}


void
disp_colormap(n, red, green, blue)      /* Load new colormap. */
int n ;
unsigned char *red, *green, *blue ;
{
}


/* Best to be sure */
#undef AND
#undef OR
