/*+-----------------------------------------------------------------------+
 *| The following module contains routines which access the advanced      |
 *| graphics modes of a Hercules Graphics Station Card.                   |
 *|                                                                       |
 *| Author: Michael S. A. Robb         Version: 1.1        Date: 16/06/93 |
 *+-----------------------------------------------------------------------+
 */

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <dos.h>
#include <mem.h>

#include "hardware.h"
#include "colours.h"
#include "tmsmodes.h"

/*+-----------------------------------------------------------------------+
 *| The following variables are used to store the current dimensions of   |
 *| the screen.                                                           |
 *+-----------------------------------------------------------------------+
 */

WORD     screen_width;     /* Stores screen pixel width    */
WORD     screen_height;    /* Stores screen pixel height.  */
LONG     screen_pitch;     /* Stores screen line pitch.    */
LONG     screen_vramwidth; /* Stores VRAM width in pixels. */

/*+-----------------------------------------------------------------------+
 *| The following data defines the 512 x 256 x 32 bit graphics mode.      |
 *+-----------------------------------------------------------------------+
 */

TMS34010_MODE mode512x256x32bit =
  {
  "512 x 256 x 32 bits",
  8,   12,   76,   80,             /* Horizontal timings.   */
  1,   35,  547,  548,             /* Vertical   timings.   */

  MODE_UNINTERLACED | MODE_2KBYTE, /* Display    controls.  */
  0xFFFD,
  0x0000,

  CLOCK_20000MHZ,                  /* External   controls.  */
  PIXEL_SIZE32,
  VSCAN_UNINTERLACED,
  DEFAULT_24BIT,

  512, 256, 32,                    /* Screen dimensions.    */
  0x4000,                          /* Display pitch - bits. */
  };

/*+-----------------------------------------------------------------------+
 *| The following data defines the 512 x 480 x 32 bit graphics mode.      |
 *+-----------------------------------------------------------------------+
 */

TMS34010_MODE mode512x480x32bit =
  {
  "512 x 480 x 32 bits",
  8,   12,   76,   80,             /* Horizontal timings.   */
  1,   35,  515,  525,             /* Vertical   timings.   */

  MODE_UNINTERLACED | MODE_2KBYTE, /* Display    controls.  */
  0xFFFC,
  0x0000,

  CLOCK_20000MHZ,                  /* External   controls.  */
  PIXEL_SIZE32,
  VSCAN_UNINTERLACED,
  DEFAULT_24BIT,

  512, 480, 32,                    /* Screen dimensions.    */
  0x4000,                          /* Display pitch - bits. */
  };

/*+-----------------------------------------------------------------------+
 *| The following data defines the 512 x 512 x 32 bit graphics mode.      |
 *+-----------------------------------------------------------------------+
 */

TMS34010_MODE mode512x512x32bit =
  {
  "512 x 512 x 32 bits",
   8,  12,   76,  80,              /* Horizontal timings.   */
   1,  16,  528, 544,              /* Vertical   timings.   */

  MODE_UNINTERLACED | MODE_2KBYTE, /* Display    controls.  */
  0xFFFC,
  0x0000,

  CLOCK_20000MHZ,                  /* External   controls.  */
  PIXEL_SIZE32,
  VSCAN_UNINTERLACED,
  DEFAULT_24BIT,

  512, 512, 32,                    /* Screen dimensions.    */
  0x4000,                          /* Display pitch - bits. */
  };

/*+-----------------------------------------------------------------------+
 *| The following routine is used to set the value of a TMS34010 host     |
 *| interface register.                                                   |
 *+-----------------------------------------------------------------------+
 */

void tms34010_sethostregister( address, data )
  ADDRESS   address;
  WORD      data;
  {
  *( (WORD far *) address ) = data;
  }

/*+-----------------------------------------------------------------------+
 *| The following routine is used to get the value of a TMS34010 host     |
 *| interface register.                                                   |
 *+-----------------------------------------------------------------------+
 */

WORD tms34010_gethostregister( address )
  ADDRESS address;
  {
  return( *( (WORD far *) address) );
  }

/*+-----------------------------------------------------------------------+
 *| The following routine is used to set the address accessed via the     |
 *| TMS34010 host registers.                                              |
 *+-----------------------------------------------------------------------+
 */

void tms34010_setaddress( address )
  ADDRESS address;
  {
  tms34010_sethostregister( CPU_ADDRLO,   (WORD) ( address &  0xFFFFL ) );
  tms34010_sethostregister( CPU_ADDRHI,   (WORD) ( address >> 16L     ) );
  }

/*+-----------------------------------------------------------------------+
 *| The following routine is used to set the value of TMS34010 registers. |
 *+-----------------------------------------------------------------------+
 */

void tms34010_setregister( address, data )
  ADDRESS address;
  WORD    data;
  {
  tms34010_sethostregister( CPU_ADDRLO,   (WORD) ( address &  0xFFFFL ) );
  tms34010_sethostregister( CPU_ADDRHI,   (WORD) ( address >> 16L     ) );
  tms34010_sethostregister( CPU_SLOWDATA, data );
  }

/*+-----------------------------------------------------------------------+
 *| The following routine is used to get the value of TMS34010 registers. |
 *+-----------------------------------------------------------------------+
 */

WORD tms34010_getregister( address )
  ADDRESS address;
  {
  tms34010_sethostregister( CPU_ADDRLO,   (WORD) ( address &  0xFFFFL ) );
  tms34010_sethostregister( CPU_ADDRHI,   (WORD) ( address >> 16L     ) );

  return( tms34010_gethostregister( CPU_SLOWDATA ) );
  }

/*+-----------------------------------------------------------------------+
 *| The following routine is used to set up a TMS34010 graphics mode.     |
 *+-----------------------------------------------------------------------+
 */

void tms34010_mode( mode )
  TMS34010_MODE *mode;
  {
  WORD n, *data = &mode -> tms_hesync;

  tms34010_sethostregister(   CPU_CONTROL,    HOST_AUTOINCR );
  tms34010_setaddress(        IO_HESYNC                   );

  for ( n = 10; n--; data++ )
    tms34010_sethostregister( CPU_SLOWDATA, *data );

  tms34010_sethostregister( CPU_CONTROL,    HOST_NOINCR   );
  tms34010_setregister(     IO_DPYTAP,      mode -> tms_dpytap );
  tms34010_setregister(     CONFIG1_WRITE,  mode -> tms_clockbase | 0x0008 );
  tms34010_setregister(     CONFIG2_WRITE,  mode -> tms_pixelsize |
                                            mode -> tms_videotiming );

  tms34010_setregister(     RAMDAC_COMMAND, mode -> tms_ramdac      );
  tms34010_setregister(     CONFIG1_WRITE,  mode -> tms_clockbase & 0x00F7 );
  tms34010_setaddress(      NULL );

  screen_width     = mode -> tms_xmax;
  screen_height    = mode -> tms_ymax;
  screen_pitch     = mode -> tms_dpitch;

  screen_vramwidth = mode -> tms_dpitch / mode -> tms_psize;
  }

/*+-----------------------------------------------------------------------+
 *| The following routine is used to restore the VGA graphics mode.       |
 *+-----------------------------------------------------------------------+
 */

void tms34010_setvga( void )
  {
  tms34010_sethostregister( CPU_CONTROL,   HOST_AUTOINCR );

  tms34010_setregister(     CONFIG1_WRITE, 0x000A        );
  tms34010_setregister(     CONFIG2_WRITE, 0x000C        );

  outportb(                 0x03C6,        0x004B );

  tms34010_setregister(     CONFIG1_WRITE, 0x0002        );
  tms34010_sethostregister( CPU_CONTROL,   HOST_NOINCR   );

  textmode( 0x00 );
  textmode( 0x03 );

  clrscr();
  }

/*+----------------------------------------------------------------------+
 *| This subroutine is used to fill a block of memory with a specific    |
 *| colour.                                                              |
 *+----------------------------------------------------------------------+
 */

void tms34010_fillblockaddr32( base, xlo, ylo, width, height, col )
  ADDRESS  base;
  WORD     xlo, ylo;
  WORD     width, height;
  long     col;
  {
  WORD y, size;
  WORD col_lo = col & 0xFFFFL;
  WORD col_hi = col >> 16;

  ADDRESS address = (( (ADDRESS) xlo + (ADDRESS) ylo * 512 ) << 5) + base;

  tms34010_sethostregister( CPU_CONTROL, HOST_AUTOINCR  );

  for ( y = height; y--; address += 0x4000L )
    {
    tms34010_setaddress( address );

    for ( size = width; size--; )
      {
      tms34010_sethostregister( CPU_SLOWDATA, col_hi );
      tms34010_sethostregister( CPU_SLOWDATA, col_lo );
      }
    }

  tms34010_sethostregister( CPU_CONTROL, HOST_NOINCR );
  }



