/*
 * Copyright (c) 1989 by Apollo Computer, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The authors
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 * event will the authors be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * Copyright (c) 1989 Carnegie-Mellon University
 * Copyright (c) 1988 Carnegie-Mellon University
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 *
 *
 * Bug reports/fixes, comments and enhancements should be sent to:
 *       nawaf@osf.org
 *       Nawaf Bitar
 *       Open Software Foundation
 *       11 Cambridge Center
 *       Cambridge, MA 02142
 *
 * Contributors:  Nawaf Bitar
 *		  Tom Lemaire
 *		  Vidas Neverauskas
 *		  Ed Sharpe
 *
 */

#ifndef	_MACHINE_MMU_
#define	_MACHINE_MMU_
   
#define	M68030

/*
 * Parameters which determine the 'geometry' of the M68K page tables in memory.
 * We assume three levels of page tables.
 * Note that the geometry of the page tables used by LOCORE is fixed.
 */
#ifdef	LOCORE

#define L1_BITS  2  /* do not change */
#define L2_BITS  9  /* do not change */
#define L3_BITS  8  /* do not change */
#define L4_BITS  0  /* do not change */
#define PG_BITS  (13) /* do not change */

#else	LOCORE

#define	L1_BITS  4
#define	L2_BITS  8
#define L3_BITS  8
#define L4_BITS  0
#define	PG_BITS  12

#endif	LOCORE

/*
 * Value to load into M68K Translation Control register.
 */
#define IS (32-PG_BITS-L1_BITS-L2_BITS-L3_BITS-L4_BITS)
#define TC                                              \
	(0x80000000) +		/* Enable bit */	\
	(PG_BITS<<20) +		/* page size */		\
	(IS << 16) +		/* initial shift */	\
	(L1_BITS << 12) +	/* TIA */		\
	(L2_BITS << 8) +	/* TIB */		\
	(L3_BITS << 4) +	/* TIC */		\
	L4_BITS			/* TID */

/*
 * Shifts and masks for converting between vaddr's and pfn's.
 */
#define PMAP_PGBYTES	(1<<PG_BITS)		/* bytes per M68K page     */
#define	PMAP_PGOFSET	(PMAP_PGBYTES-1)	/* byte offset into page   */
#define PMAP_PGSHIFT	PG_BITS			/* number of bits to shift */
#define	PMAP_PGMASK	(~PMAP_PGOFSET)		/* page mask               */

#ifdef	LOCORE

#define	GET_L1_INDEX(reg,scr)                     \
	movel	#32-L1_BITS,scr;                  \
	lsrl	scr,reg;                          \
	andl	#((1<<L1_BITS)-1),reg

#define	GET_L2_INDEX(reg,scr)                     \
	movel	#32-L1_BITS-L2_BITS,scr;          \
	lsrl	scr,reg;                          \
	andl	#((1<<L2_BITS)-1),reg

#define	GET_L3_INDEX(reg,scr)                     \
	movel	#32-L1_BITS-L2_BITS-L3_BITS,scr;  \
	lsrl	scr,reg;                          \
	andl	#((1<<L3_BITS)-1),reg

#define GET_PAGE_COUNT(reg,scr)	                  \
	movel	#PG_BITS,scr;                     \
	lsrl	scr,reg

#else	LOCORE

/*
 * Convert bytes to pages and convert pages to bytes.
 * No rounding is used.
 */
#define	PMAP_BTOP(x)		(((unsigned)(x)) >> PMAP_PGSHIFT)
#define	PMAP_PTOB(x)		(((unsigned)(x)) << PMAP_PGSHIFT)

/*
 * Round off or truncate to the nearest page.  These will work for
 * either addresses or counts.  (i.e. 1 byte rounds to 1 page bytes).
 */
#define PMAP_TRUNC_PAGE(x)	(((unsigned)(x)) & PMAP_PGMASK)
#define PMAP_ROUND_PAGE(x)	PMAP_TRUNC_PAGE((x)+ PMAP_PGOFSET)


/*
 * M68K Long Format Table Descriptor.
 */
typedef struct l1_entry {
	unsigned long	
			lu:1,		/* lower/upper limit	*/	
			limit:15,	/* table search limit	*/
#ifdef	M68851
			ral:3,		/* read-access level	*/
			wal:3,		/* write-access level	*/
			share:1,	/* shared-global bit	*/
#else
			:7,
#endif	M68851
			sup:1,		/* supervisor access bit*/
			:4,		/* unused, always zero	*/
			pg_used:1,	/* reference bit	*/
			prot:1,		/* protection- rd/wr	*/
			dtype:2;	/* descriptor type	*/

	unsigned long	table_addr:28,	/* base address of descriptor table */
			:4;		/* not used */
} pt_ldesc_t;
                               
/* template for bit-manipulation */
typedef union pt_ldesc_template {
	pt_ldesc_t	pt_ldesc;
	struct double_bits {
		unsigned long lo;
		unsigned long hi;
	} bits;
} pt_ldesc_template_t;

#define	PT_LDESC_NULL	((pt_ldesc_t *) 0)



/*
 * M68K Short Format Table Descriptor.
 */
typedef struct pt_sdesc {     
	unsigned long	
			table_addr:28,	/* descriptor table base addr	*/
			pg_used:1,	/* page has been referenced	*/
			prot:1,		/* protection - read/write     	*/
			dtype:2;	/* descriptor type		*/
} pt_sdesc_t;

typedef union pt_sdesc_template {
	pt_sdesc_t	pt_sdesc;
	unsigned long	bits;
} pt_sdesc_template_t;

#define	PT_SDESC_NULL	((pt_sdesc_t *) 0)
  

/*
 * M68K Short Format Page Descriptor.
 */
typedef struct pt_entry {
	unsigned long	
#if	PMAP_PGBYTES==8192
			pfn:19,		/* page frame number                  */
			:4,
#else
#if	PMAP_PGBYTES==4096
			pfn:20,		/* page frame number                  */
			:3,
#else
			:23,		/*                                    */
#endif	PMAP_PGBYTES==4096
#endif	PMAP_PGBYTES==8192
			wired:1,	/* page-wired bit; (stolen from pfn)  */
#ifdef	M68851
			gate:1,		/* gate-ing for module descriptors    */
#else
			:1,
#endif	M68851
			no_cache:1,	/* cache inhibit bit                  */
#ifdef	M68851
			lock:1,		/* exempt page from ATC replacement   */
#else
			:1,
#endif	M68851
			modified:1,	/* page has been modified             */
			pg_used:1,	/* page has been (used) referenced    */
			prot:1,		/* protection - read/write            */
			dtype:2;	/* descriptor type                    */
} pt_entry_t;

/* pte template for bit-manipulation. */
typedef union pte_template {
	pt_entry_t	pte;
	unsigned long	bits;
} pte_template_t;

#define	PT_ENTRY_NULL	((pt_entry_t *) 0)



/*
 * Level 1 page table entries are M68K long-format table descriptors,
 * level two page table entries are M68K short-format table descriptors,
 * level three are M68k short-format page descriptors.
 */
typedef	pt_ldesc_t	l1_entry_t;  
#define	L1_ENTRY_NULL	PT_LDESC_NULL

typedef	pt_sdesc_t	l2_entry_t;
#define	L2_ENTRY_NULL	PT_SDESC_NULL


/*
 * protection codes (pt_entry_t.prot) .
 */
#define M68K_UR		1		/* user readable */
#define M68K_UW		0		/* user writeable */
#define M68K_KR		1		/* kernel readable */
#define M68K_KW		0		/* kernel writeable */

/*
 * descriptor types (pt_entry_t.dtype, l2_entry_t.dtype, l1_entry_t.dtype)
 */
#define DT_INVALID      0
#define DT_PAGE         1
#define	DT_SHORT	2
#define	DT_LONG         3
                                                            
/*
 * Number of entries in a single page table.
 */
#define	POWER2(n)	(1<<(n))
#define	L1_ENTRIES	(POWER2(L1_BITS))
#define	L2_ENTRIES	(POWER2(L2_BITS))
#define	L3_ENTRIES	(POWER2(L3_BITS))

/*
 * Size in bytes of a single page table.
 */
#define	L1_SIZE		(L1_ENTRIES * sizeof (struct l1_entry))
#define	L2_SIZE		(L2_ENTRIES * sizeof (struct pt_sdesc))
#define	L3_SIZE		(L3_ENTRIES * sizeof (struct pt_entry))

/*
 * Shifts and masks to map from virt addresses to level indices.
 */
#define	L1_SHIFT	(PG_BITS + L3_BITS + L2_BITS)
#define	L2_SHIFT	(PG_BITS + L3_BITS)
#define	L3_SHIFT	PG_BITS

#define	L1_MASK	((POWER2(L1_BITS)-1) << (32-L1_BITS))
#define	L2_MASK	((POWER2(L2_BITS)-1) << (32-L1_BITS-L2_BITS))
#define	L3_MASK	((POWER2(L3_BITS)-1) << (32-L1_BITS-L2_BITS-L3_BITS))

#define	L1_NEXT(va) ((va+(1<<(32-L1_BITS)))         & L1_MASK)
#define	L2_NEXT(va) ((va+(1<<(32-L1_BITS-L2_BITS))) & (L1_MASK|L2_MASK))

#define	L1IDX(va)	((va & L1_MASK) >> L1_SHIFT)
#define	L2IDX(va)	((va & L2_MASK) >> L2_SHIFT)
#define	L3IDX(va)	((va & L3_MASK) >> L3_SHIFT)

#define	L1ENT(map,va)	((l1_entry_t *)(map->lvl1_vaddr + L1IDX(va)))

/*
 * The following stuff is to handle level2 and level3 'table groups'.
 * When the underlying storage for page tables at either level is not
 * zone-based, but instead is allocated in whole VM pages via kmem_alloc,
 * we set up 'groups' of page tables, where a group is defined as the
 * number of page tables which fit into a single VM page.  If the
 * underlying storage is allocated from zones, the group size is 1.
 */

#if	L2_ZONE
#define L2_GROUP_SIZE 1
#define L2_FREE(tbl)	zfree(lvl2_zone, tbl)
#else
/*
 * N.B. we are assuming that PAGE_SHIFT > (L2_BITS+3),
 * i.e. that a level 2 page table is smaller than a VM page.
 */
#define L2_GROUP_SIZE_BITS (PAGE_SHIFT-(L2_BITS+3))
#define L2_GROUP_SIZE (1<<L2_GROUP_SIZE_BITS)
#define	L2_FREE(grp)	kmem_free (user_pt_map, grp, PAGE_SIZE)
#endif	L2_ZONE


#if	PT_ZONE
#define L3_GROUP_SIZE 1
#define	PT_FREE(tbl)	zfree(lvl3_zone, tbl)
#else
/*
 * N.B. we are assuming that PAGE_SHIFT > (L3_BITS+2),
 * i.e. that a level 3 page table is smaller than a VM page.
 */
#define L3_GROUP_SIZE_BITS (PAGE_SHIFT-(L3_BITS+2))
#define L3_GROUP_SIZE (1<<L3_GROUP_SIZE_BITS)
#define	PT_FREE(grp)	kmem_free(user_pt_map, grp, PAGE_SIZE)
#endif	PT_ZONE


/*
 * VA spaces mapped by tables and table groups.
 */
#define L3_VA_SPACE       (L3_ENTRIES*PMAP_PGBYTES)
#define L3_GROUP_VA_SPACE (L3_VA_SPACE*L3_GROUP_SIZE)

#define L2_VA_SPACE       (L2_ENTRIES*L3_VA_SPACE)
#define L2_GROUP_VA_SPACE (L2_VA_SPACE*L2_GROUP_SIZE)


/*
 * Number of level 1 entries used to map user space and kernel, respectively.
 */
#define	USER_L1_ENTRIES	  L1IDX(KERNEL_BASE)
#define	KERNEL_L1_ENTRIES (L1_ENTRIES-USER_L1_ENTRIES)

#define	L1_GET_L2(lvl1_ptr)	((l2_entry_t *)(((lvl1_ptr)->table_addr)<<4))
#define	L2_GET_L3(lvl2_ptr)	((pt_entry_t *)(((lvl2_ptr)->table_addr)<<4))

/*
 * Macros to validate descriptors at each level of the page table.
 */
#define	L1_VALID(lvl1_ptr)	((lvl1_ptr)->dtype == DT_SHORT)
#define	L2_VALID(lvl2_ptr)	((lvl2_ptr)->dtype == DT_SHORT)
#define	L3_VALID(pte_ptr)	( (pte_ptr)->dtype == DT_PAGE )

/*
 * Alignment checks for page tables (must lie on 16-byte boundaries)
 * and for pages (must lie on page boundaries).
 */
#define	PT_ALIGNED(ad)   (((vm_offset_t)(ad) & 0xf)==0)
#define PAGE_ALIGNED(ad) (((vm_offset_t)(ad) & ~PMAP_PGMASK) == 0)
#define CHECK_PAGE_ALIGN(ad,who)                    \
	if (!PAGE_ALIGNED(ad))                      \
		printf("%s: addr %x not page aligned.\n", who, ad)

/*
 * Create valid PTE's for all machine pages in a VM page.
 */
#define	DO_PTES(start, template)                                        \
	{                                                               \
		int i;                                                  \
		pt_entry_t *p = start;                                  \
		for (i = ptes_per_vm_page; i>0; i--) {                  \
			*(int*)p++ = (unsigned long)(template);         \
			(unsigned long)(template) += PMAP_PGBYTES;      \
		}                                                       \
	}

/*
 *	Locking and TLB invalidation
 *
 *	Locking Protocols:
 *
 *	There are two structures in the pmap module that need locking:
 *	the pmaps themselves, and the per-page pv_lists (which are locked
 *	by locking the pv_lock_table entry that corresponds to the pv_head
 *	for the list in question.)  Most routines want to lock a pmap and
 *	then do operations in it that require pv_list locking -- however
 *	pmap_remove_all and pmap_copy_on_write operate on a physical page
 *	basis and want to do the locking in the reverse order, i.e. lock
 *	a pv_list and then go through all the pmaps referenced by that list.
 *	To protect against deadlock between these two cases, the pmap_lock
 *	is used.  There are three different locking protocols as a result:
 *
 *  1.  pmap operations only (pmap_extract, pmap_access, ...)  Lock only
 *		the pmap.
 *
 *  2.  pmap-based operations (pmap_enter, pmap_remove, ...)  Get a read
 *		lock on the pmap_lock (shared read), then lock the pmap
 *		and finally the pv_lists as needed [i.e. pmap lock before
 *		pv_list lock.]
 *
 *  3.  pv_list-based operations (pmap_remove_all, pmap_copy_on_write, ...)
 *		Get a write lock on the pmap_lock (exclusive write); this
 *		also guaranteees exclusive access to the pv_lists.  Lock the
 *		pmaps as needed.
 *
 *	At no time may any routine hold more than one pmap lock or more than
 *	one pv_list lock.  Because interrupt level routines can allocate
 *	mbufs and cause pmap_enter's, the pmap_lock and the lock on the
 *	kernel_pmap can only be held at splvm.
 */

#if	NCPUS == 1

#define	SPLVM(spl)	{ spl = splvm(); }
#define	SPLX(spl)	{ splx(spl); }

#define	PMAP_READ_LOCK(pmap, spl)	SPLVM(spl)
#define	PMAP_WRITE_LOCK(spl)		SPLVM(spl)
#define	PMAP_READ_UNLOCK(pmap, spl)	SPLX(spl)
#define	PMAP_WRITE_UNLOCK(spl)		SPLX(spl)
#define	PMAP_WRITE_TO_READ_LOCK(pmap)

#define	LOCK_PVH(index)
#define	UNLOCK_PVH(index)

extern void flush_atc_all();
#define	FLUSH_ATC_ALL			flush_atc_all
#define	FLUSH_ATC_ENTRIES(map, s, e)	flush_atc_all() /* xxx can do better than this */

#else	NCPUS == 1
/* 
 *	No multiprocessor support (yet)
 */
#endif	NCPUS == 1
                     
#endif	LOCORE

#endif	_MACHINE_MMU_
